/************************************************************************
* oagTimerUtil.h
* Added functionalities to compute cell area, cell leakage power
* and total leakage power of design
* by: Santiago Mok <smok@ee.ucla.edu>
* advised by: Prof. Puneet Gupta
*
**************************************************************************/
#if !defined(oagTimerUtil_P)
#define oagTimerUtil_P 

#include <string>
#include <vector>
#include <iostream>
#include <iomanip>
#include <map>
#include "oaDesignDB.h"
#include "oagTimerModel.h"
#include "oagTimerTimer.h"
#include "oagTimerTPointMaster.h"
#include "oagTimerLibParserInt.h"

namespace oagTimer{
/*---------------------------------------------------------*/

using namespace oa;
using namespace std;
/*---------------------------------------------------------*/
typedef enum {
    INPUT,
    OUTPUT
} io_Type;
/*---------------------------------------------------------*/
typedef vector<oaModInst*> instVector;
//typedef map<oaModInst*> Inst_Map;
/*---------------------------------------------------------*/
class Util{
  private:
      oaDesign *topDesign;
      Timer *timer;
  public:
    Util();
    ~Util();

    /* To Do: Change Util class to static*/
    void add(oaDesign *d, Timer* t){
	topDesign = d;
	timer = t;
    };
    double getCellArea(oaInst *inst);
    static double getCellLeakagePower(oaModInst *modInst);
    static double getCellLeakagePower(oaOccInst *occInst);
    static double getCellLeakagePower(oaInst *inst);
    static double getCellLeakagePower(oaModule *m);
    static double getCellLeakagePower(const char *s);
    static double getCellLeakagePower(oaString cellName);
    static double getTotalLeakPower(const oaDesign *design);

    static double getPowerSensitivity(const char *curr, const char *other);
    
    static double getCellInputCap(oaOccInstTerm *i, oaString s, Timer *timer);
    oaString getCellName(oaModInst *i);
    oaString getCellName(oaOccInst *i);

    oaOccInstTerm* findOccInTerms(oaDesign *design, oaModScalarInst *m);
    oaOccInstTerm* findOccOutTerm(oaDesign *design, oaModScalarInst *m);

    oaModInstTerm* getOutInstTerm(oaModNet *net);

    /* Query for the arrival time constraints at 
     * output of the given occInst*
     */
    TimeType getIOArr(oaModInst* i, oaDesign* design, Timer* timing, io_Type io);
    TimeType getIOArr(oaOccInst* i, oaDesign* design, Timer* timing, io_Type io);

    DelayType getIOSlack(oaModInst* i, oaDesign* design, Timer* timing, io_Type io);
    void updateFanIOTiming(oaDesign* design,Timer* t,oaModInst *m);
    void invalidateFanIOTiming(oaDesign* design,Timer* t,oaModInst *m);
    
    vector<oaModInst*> getOtherFanin(oaModInst* currFanin, oaModInst* currCell);
    /* Query for other available cell sizes in the *.lib library;
     * list must contain the current *.lib cell name when called
     * getMatchCells and the query will return the available sizes 
     * in the list vector
     * @param vector<oaString>: contains the queried sizes
     * @param sizes: 1 upsize only
     *		    2 downsize only
     *		    0 all sizes
     */
    void getMatchCells(vector<oaString> &list, int sizes);
    bool isGreaterThan(oaString current, oaString other);
    bool isLessThan(oaString current, oaString other);

    int swapCell(oaString cellName,oaModInst* currModInst);

/*! CELL DELAY RELATED FUNCTIONS */
/*---------------------------------------------------------*/
/*! This function estimate the cell delay for the given
 * new size (otherSize) with respect to the given
 * input arc (inpTerm)
 * * The estimate is based on slew and load of current
 * design configuration
 * @param oaOccInstTerm*: input arc for which delay is computed
 * @param oaString: the cell size to compute delay for
 * @param Timer: the timer object
 */
static double getCellDelayEstimate(oaOccInstTerm *inpTerm, 
			    oaString otherSize,
			    double load,
			    Timer* timer);
/*---------------------------------------------------------*/
/*! This function queries the worst (R/F) delay (from input
 * to output) with respect to the given input pin. 
 * @PARAM t: the current timer object
 * @PARAM inPin: the input from delay propagates
 * @RETURN cell delay 
 */
//static DelayType getCellDelay(Timer* t, oaOccInstTerm* inPin);
static DelayType getCellRiseDelay(oaOccInstTerm *inPin);
static DelayType getCellFallDelay(oaOccInstTerm *inPin);
static DelayType getCellDelay(oaOccInstTerm *inPin);
/*static DelayType getCellDelay(Timer *timer, 
	oaOccInstTerm *inPin, oaOccInstTerm *outPin);*/
/*---------------------------------------------------------*/
/*! This function queries the worst (R/F) delay (from input
 * to output) with respect to the given input pin and other 
 * given cell type. 
 * @PARAM t: the current timer object
 * @PARAM inPin: the input from delay propagates
 * @RETURN cell delay 
 */
//static DelayType getCellDelay(Timer* t, oaOccInstTerm* inPin);
/*static DelayType getCellDelay(Timer *timer, 
	oaOccInstTerm *inPin, oaOccInstTerm *outPin);
*/
/*---------------------------------------------------------*/
/*! This function queries the difference in arrival time 
 * between the current CellData and the same CellData 
 * with the new library size.
 * @PARAM oaDesign*: current top design 
 * @PARAM Timer: current timer object
 * @PARAM CellData*: current cell data object to query deltaAT
 * @PARAM oaString: the new lib size to query AT for on the given CellData object
 * @RETURN Arrival Time Sensitivity (newAT - currAT)
 */
static DelayType getDeltaDelay(oaDesign* design, Timer* timing, 
	    CellData *c, oaString newSize);
/*---------------------------------------------------------*/
/*! This function queries the difference in delay 
 * between the current OccObject and the same OccObject 
 * with the other TPointMaster *otherTM.
 * @PARAM OccObject: current cell block
 * @PARAM TPointMaster: timing point master for the other lib cell
 * @PARAM Timer: current timer object
 * @RETURN Arrival Time Sensitivity (newAT - currAT)
 */
static DelayType getDeltaDelay(oaOccObject *block, 
	TPointMaster *otherTM, Timer* timer);
/*---------------------------------------------------------*/
/*! This function queries the change in delay if the next
 * larger cell size was used while keeping current input slew
 * and output load cap
 * @PARAM oaDesign: the current top design hierarchy
 * @PARAM CellData: the cell to query its delta delay
 * @RETURN double: delta delay (UpSized Delay - Current Delay) 
 *		   *DBL_MAX is returned if no next size found 
 */
static double getUpSizeDeltaDelay(oaDesign* design, Timer* timing, CellData *c);
/*---------------------------------------------------------*/
/*! This function queries the worst slack at the input pins
 * of the given instance
 * @PARAM t: the current timer object
 * @PARAM inst: the current occ instance object
 * @RETURN worst slack of all inputs
 */
static DelayType getWorstInputSlack(Timer* t, oaOccInst* inst);
/*---------------------------------------------------------*/
/*! This function queries the change in delay if the next
 * smaller cell size was used while keeping current input slew
 * and output load cap
 * @PARAM oaDesign: the current top design hierarchy
 * @PARAM CellData: the cell to query its delta delay
 * @RETURN double: delta delay (DownSized Delay - Current Delay) 
 *		   *DBL_MAX is returned if no next size found 
 */
static double getDownSizeDeltaDelay(oaDesign* design, Timer* timing, CellData *c);
/*---------------------------------------------------------*/
/*! This function perform slack time check two level
 * fanout(s) downstream of the current instance and two level
 * fanin(s) upstream
 */
static bool hasValidArrTime(Timer* timer, CellData *cell);
/*! This function query for the max net delay in the fanout
 * of the given output pin
 * @param oaOccInstTerm*: output pin
 */
static double getMaxNetDelay(oaOccInstTerm* t);
/*---------------------------------------------------------*/
static oaString getInstName(oaModInst *i){
    oaNativeNS NS;
    oaString name;
    i->getName(NS,name);
    return name;
}
static oaString getInstName(oaOccInst *i){
    oaNativeNS NS;
    oaString name;
    i->getName(NS,name);
    return name;
}
/*---------------------------------------------------------*/
static void parseLibCellName(oaString name, oaString &basename, int &size){
    if(libParseData.libType.compare("nangate") == 0
	    || libParseData.libType.compare("eyechart") == 0){
	Util::parseNangateLibCellName(name,basename,size);
    }else if(libParseData.libType.compare("st") == 0){
	Util::parseSTLibCellName(name,basename,size);
    }else if(libParseData.libType.compare("st_vt") == 0){ 
	Util::parseVtSTLibCellName(name,basename,size);
    }
}
/*---------------------------------------------------------*/
static void parseVtSTLibCellName(oaString name, oaString &basename, int &size){
    size_t index1, index2;
    string curr = static_cast<string>(name);
    index1 = curr.find_first_of("_");
    index2 = curr.find_last_of("_");
    /*std::cout << "VtType:" << curr.substr(index1+1,2)
	<< " Name:" << curr.substr(index2+1) << std::endl;*/
    string currBase = curr.substr(index2+1);
    string currVt = curr.substr(index1+1,2);
    basename = currBase.c_str();
    if(currVt == "LH") size = 1;
    else if(currVt == "LS") size = 2;
    else if(currVt == "LL") size = 3;
    /*std::cout << " " << name << " -> " << basename
	<< " + " << currVt << " " << size << std::endl;*/
}
/*---------------------------------------------------------*/
static void parseSTLibCellName(oaString name, oaString &basename, int &size){
    size_t name_index, size_index;
    string curr = static_cast<string>(name);
    name_index = curr.find_last_of("X");
    //if((partition_index) == string::npos){}
    /*std::cout << "name:" << curr.substr(0,name_index)
	<< " size:" << curr.substr(name_index+2) << std::endl;*/
    string currBase = curr.substr(0,name_index);
    string currSize = curr.substr(name_index+1);
    basename = currBase.c_str();
    size = atoi(currSize.c_str());
    /*std::cout << " " << name << " -> " << basename
	<< " + " << size << std::endl;*/
}

/*---------------------------------------------------------*/
static void parseNangateLibCellName(oaString name, oaString &basename, int &size){
    size_t name_index, size_index;
    string curr = static_cast<string>(name);
    name_index = curr.find_first_of("_");
    size_index = curr.find_last_of("_");
    if((name_index && size_index) == string::npos){
	name_index = curr.find_first_of("X");
	size_index = curr.find_last_of("X");
	std::cout << "name:" << curr.substr(0,name_index)
	    << " size:" << curr.substr(size_index+2) << std::endl;
    }
    string currBase = curr.substr(0,name_index);
    string currSize = curr.substr(size_index+2);
    basename = currBase.c_str();
    size = atoi(currSize.c_str());
}
/*---------------------------------------------------------*/
/* Get combinations of size k given the cell instances.
 * @param insts: vector of ModInst
 * @param k: size of each combination
 * @return vector of all the combinations
 */
static vector<instVector> 
getCombinations(instVector insts, int k){
    vector<instVector> temp_vec;
    if(insts.size() <= k){
	temp_vec.push_back(insts);
	return temp_vec;
    }
    if(!temp_vec.empty()) temp_vec.empty();
    instVector c;
    genComb(insts,k,temp_vec,c);
    return temp_vec;
}

/* Recursively generate the combinations in the list L
 * and store them in the table T
 * @param L: list instances
 * @param k: size of combination
 * @param T: table storing each combination
 * @param C: temporary combination list during each recursion
 * @return void
 */
static void genComb(instVector L, int k, vector<instVector> &T, instVector &C){
    if(k == 1){
	for(instVector::iterator it=L.begin(); it!=L.end(); ++it){
	    C.push_back(*it);
	    T.push_back(C);
	    C.pop_back();
	}
	return;
    }
    instVector::iterator iter = L.begin();
    instVector temp_list = L;
    for(int i=0; i<=(L.size()-k); ++i){
	C.push_back(*iter);
	temp_list.erase(temp_list.begin());
	genComb(temp_list,k-1,T,C);
	C.pop_back();
	++iter;
    }
}

static void Util::downSizeAll(oaDesign *design, Timer *timer);
static void Util::changeToMinCellSize(oaDesign *design, Timer *timer);
/*---------------------------------------------------------*/
/* Report gates count statistic for the given design
 * @param oaDesign*
 * @return void
 */
static void reportSlacks(oaDesign* design,Timer *timer);
static void reportGates(oaDesign* design);
/* Debug timing information (r/f arrival, cell delay, 
 * wire delay) for each timing arc
 */
static void reportTimingInfo(oaDesign* design, Timer *t);
static void reportCellTiming(CellData *cell, Timer *t);
static void reportAllCellTiming(oaDesign *design, Timer *t);
static void reportTimingFlag(oaDesign *des);
/*! Debug statement that print all net information
 * Net load for PI and Output arc of cell
 * Net Delay for PO and Input arc of cell
 * @param oaDesign*: the top design object
 */
static void printAllNetData(oaDesign *design);
/*---------------------------------------------------------*/
/*!
  returns the name of block object
  FIXME should move out of the timer class
  \param block either an oaTerm or an oaInstTerm
*/ 
static std::string getBlockName(oaOccObject *oPtr);
/*---------------------------------------------------------*/

/*! A linear fitting routine by John Lee
 * (y=ax+b)
 * @PARAM double: a is the coefficient
 * @PARAM double: b is y-intercept
 * @PARAM double*: x is the independent variable
 * @PARAM double*: y is the dependent variable
 * @PARAM int: N is the number of data points
 */
static void linear_fit(double &a,double &b, double *x, double *y, int N);
/*---------------------------------------------------------*/
};
/*---------------------------------------------------------*/
}
#endif
