/************************************************************
* 
* File: oagTimerCellData.h
* Author: Santiago Mok (smok@ee.ucla.edu)
* Created: 05-18-2010
* Last Modified: Mon 04 Apr 2011 12:03:00 PM PDT
*
************************************************************/
#if !defined(oagTimerCellData_P)
#define oagTimerCellData_P

#include <list>

#include "oaDesignDB.h"
#include "oagTimerCellMaster.h"
#include "oagTimerUtil.h"
#include "oagTimerLibParserInt.h"
//#include "lpSA.h"
#include <map>

namespace oagTimer{
/*---------------------------------------------------------*/
typedef enum {
    CURRENT_SIZE, //Set signal to no change
    DOWN_SIZE,  //Signal to reduce cell strength
    UP_SIZE,   //Signal to increase cell strength
    ALL_SIZE,  //Signal to try all cell strengths
    NULL_SIZE
} sizingSignal;
/*---------------------------------------------------------*/
typedef list<oaString> Lib_list;
/*---------------------------------------------------------*/
using namespace std;
using namespace oa;
/*---------------------------------------------------------*/
struct netData{ 
    oaModNet *net;
    double discount;
    double forward_weight;
    double backward_weight;
    double weight;
};

class CellData{
    friend class DuetSensitivity;
    friend class DesignTool;
    friend class Opt;
    friend class PeepHole;
    friend class Timer;
    friend class lpSA;
    friend class sensInfo;
    friend class Util;
    friend class kickMove;
    friend class LR;
    friend class DP;
    friend class OptMinDelay;
    friend class SubTimer;

  public:
    CellData(){
	_signal = NULL_SIZE;
	dnCount = upCount = 0;
	index = 0;
	valid = false;
	isMark = false;
	revTopFlag = false;
	fanoutSensitivity = -1.0; //(< 0) to indicate a value has not been set yet
	unsetNode();
    }
    ~CellData(){}

    /*! The design cell occurrence instance for this cell.*/
    oaOccInst *occInst;
    /*! The design cell module instance for this cell.*/
    oaModInst *inst;
    /*! The design cell instance name for this cell.*/
    oaString instName;
    /*! The master lib cell for this cell instance.*/
    CellMaster *master;
  private:
    /*! Optimization sizing signal of this cellmaster. */
    sizingSignal _signal;
    /*! This CellMaster pointer in the cell_sizes list.*/
    Lib_list::iterator lib_this_ptr;
    /*! Current lib_list search pointer for optimizaiton.
     * Used in the peephole optimization.
     * */
    Lib_list::iterator lib_seek_ptr;
    /*! A temporary pointer pointing to the current cell size taken by
     * Opt Class
     */
    Lib_list::iterator lib_temp_ptr;
    /*! Vector of other drive strength in relation to current cell type.*/
    Lib_list cell_sizes;
    /*! This flag indicate if the current cell data is valid*/
    bool valid;

    /*! Use in Peephole OPT to indicate a root node */
    bool isRoot;
    /*! Use in Peephole OPT to indicate a fanout node */
    bool isFanout;
    /*! Use in Peephole OPT to mark the cell as sized */
    bool isMark;
    /*! Us in Peephole OPT to store highest fanout sensitivity
     * so far known
     */ 
    double fanoutSensitivity;

    /*! Indicate the number of downsizes that this cell possesses */
    int dnCount;
    /*! Indicate the number of upsizes that this cell possesses */
    int upCount;

    /*! Used in lpSA to assign gate index in the circuit */
    int index;

    /*! Used in getReverseTopological() for not double counting gates in recovergence from FF */
    bool revTopFlag;

    /*! Used in DP opt to check whether this cell has multi fanin */
    bool isMultiFanin;


    /*! Use as bookmark for topological sort in DesignTool */
    map<oaModInstTerm*,int> inputTerms;
    map<oaModInstTerm*,int> outputTerms;

    /*! Use as bottleneck net weighting in DesignTool */
    map<oaModInstTerm*,netData> inputTermsNetMap;
    //map<oaModInstTerm*,netData> outputTermsNetMap;
/*---------------------------------------------------------*/
/*! Class Function */
  public:
    /*! Initialize list of .lib cell sizes for sizing operations.*/
    void initSizing(sizingSignal s);
    /*! Clear signals for sizing operations, do not remove list of 
     * cell sizes being initialize
     */
    void clearSizingSignals(){ 
	_signal == NULL_SIZE; 
	lib_seek_ptr = cell_sizes.end();
    }
    /*! Change the current lib cell module to another lib cell module 
     * and mark current cell data as invalid
     */
    void swap(oaString cellName);
    /*! Reverse swap changes made earlier to original cell data 
     * contained within this cell
     */
    void reverseSwap();
    /*! Commit the swap made earlier and update this cell data
     * and masters
     */
    void commitSwap();
    void commitSwap(oaModInst* new_inst);
    //Unset flag to indicate that this gate is not anymore in size change
    void unsetSwapFlag();
    /*! Reset lib_seek_ptr to the list.end position*/
    void resetLibListPtr();

    /*! Return vector of all avaialable cell sizes in library*/
    vector<oaString> getAllSizes();
    /*! Return vector of other sizes*/
    vector<oaString> getOtherSizes();
    /*! Return vector of sizes greater than current size*/
    vector<oaString> getGreaterThanCurrentSize();
    /*! Return vector of sizes less than current size*/
    vector<oaString> getLessThanCurrentSize();
    /*! Return the minimum drive strength for the current cell type */
    oaString getMinSize(); 
    /*! Return the maximum drive strength for the current cell type */
    oaString getMaxSize(); 
    /*! Return the count of size smaller than current.*/
    int getDownSizeCount();
    /*! Return the count of size larger than current.*/
    int getUpSizeCount();
    /*! Return the total number of sizes available in the .lib for the current cell 
     * (excluding current cell size)
     */
    int getNumOfSizes();
    /*! Return the next sizing in the list depending the current signal.*/
    oaString getNextSize();
    /*! Check whether there are more sizes in the lib list*/
    bool hasMoreSize(); 
    /*! Check whether there are more sizes in the lib list*/
    bool hasMoreDownSize(); 
    bool hasMoreUpSize(); 
    /*! Check whether current cell size is currently maximum size*/
    bool isMaxSize();
    /*! Check whether current cell size is currently minimum size*/
    bool isMinSize();

    /*! Mark a 1 to the input instTerm  of this cell instace*/
    void mark(oaModInstTerm* i);
    /*! Check whether all input terms of this instance is all marked*/
    bool isAllInputMark();

    /*! Assign the input term forward weight of this instance */
    void setForwardWeight(oaModInstTerm* iTerm, double val,double d);
    /*! Assign the input term backward weight of this instance */
    void setBackwardWeight(double val);

    /*! Get the input term forward weight SUM of this instance */
    double getNetWeightSum();
    /*! Get the input term backward weight of this instance */
    double getBackwardWeight(oaModInstTerm* iTerm);

    /*! For Peephole OPT, check if current cell is a root node*/
    bool isRootNode();
    /*! For Peephole OPT, set current cell as a root node*/
    void setRootNode();
    /*! For Peephole OPT, check if current cell is a fanout node*/
    bool isFanoutNode();
    /*! For Peephole OPT, set current cell as a fanout node*/
    void setFanoutNode();
    
    /*! For Peephole OPT, this function verifies the current highest fanout
     * sensitivity that was rejected with this cell as root 
     * - if current sensitivity is lower than the current highest minus
     *   certain percentage delta -> reject it
     * @PARAM double: current sensitivity
     * @PARAM double: delta changes (i.e. 1.3)
     * @RETURN bool
     */
    bool validFanoutSensitivity(double sensitivity, double delta);

    /*! For Peephole OPT, this function set the current highest fanout
     * sensitivity that was rejected with this cell as root 
     * @PARAM double: current sensitivity
     * @RETURN none
     */
    void setFanoutSensitivity(double sensitivity);

    /*! DEBUG: print timing information on all the pins of this cell */
    void printTimingData(oaDesign* top, Timer* t);

  private:
    /*! Add other available drive strength lib cell for this CellMaster cell type */
    void addOtherSizes(); 

    void updateLibSeekPtr();

    /*! Comparison of drive strengths to sort in increasing order.*/
    static bool increasing_strengths(oaString a, oaString b){
	oaString a_base, b_base;
	int a_size, b_size;
	Util::parseLibCellName(a, a_base, a_size);
	Util::parseLibCellName(b, b_base, b_size);
	if(a_size < b_size) return true;
	else return false;
    }
    /*! Check whether the cell size I am trying to swap is it
     * different from what it was
     */
    bool isDiffThanCurrSize();

    /*! Set all root/fanout/mark signal for PH as false*/
    void unsetNode();
/*---------------------------------------------------------*/
/*! AppDefs Definitions */
  private:
    static oaVoidPointerAppDef<oaModInst> *_instAppDef;

    /// Initialize the appDefs associated with CellDatas.
    static void initAppDefs();
    
    /// Create a CellData for the given ModInst.
    ///
    /// It is a fatal error to try to create a CellData
    /// for a Mod which already has a CellData.
    ///
    /// \param inst The OccInst for which to create CellData.
    /// \return The new CellData.
    static CellData *create(oaOccInst *oci);

    /// Get the CellData for the given ModInst, or 0 if 
    /// the ModInst has no CellData.
    ///
    /// \param inst The ModInst to query.
    /// \return The corresponding CellData.
    static CellData *get(oaModInst *inst);
/*---------------------------------------------------------*/
};//class

typedef CellData CellType;
}//namespace
#endif
