/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Philip Chong <pchong@cadence.com>

ChangeLog:
2005-06-27: ChangeLog started
*/

#if !defined(oagUtilUnitTest_P)
#define oagUtilUnitTest_P

namespace oagUtil {

/// \brief A simple unit testing framework.
///
/// Unit test classes should derive from the UnitTest class.  The
/// derived class should be named with the name of the original class
/// followed by "UnitTest", e.g. a unit test for the oagFoo::Bar class
/// should be named oagFoo::BarUnitTest.
/// The derived class should implement the public method runAllTests().
/// This method should build the environment for testing the target class,
/// successively call logOneTest() for each test in the unit test suite,
/// and finally return the value of getReturnCode().
///
/// For example, if there are two functions f1() and f2() in class Bar
/// that need to be tested:
///
/// \verbatim
/// class BarUnitTest : public UnitTest {
///     ...
///     public:
///         int runAllTests(void) {
///             Bar myObject;
///             myObject.init();
///             int r = myObject.f1(5);
///             logOneTest(r == 3, "Checking f1(5)");
///             ...
///             r = myObject.f2(1);
///             logOneTest(r == 12, "Checking f2(1)");
///             ...
///             return getReturnCode();
///         }
///     ...
/// };
/// \endverbatim
///
/// The main() code surrounding all the unit tests for the oagFoo
/// package should look something like:
/// \verbatim
/// int main(int argc, char *argv[]) {
///     BarUnitTest myTest;
///     return myTest.runAllTests();
/// }
/// \endverbatim
///
/// If there is more than one class in oagFoo they should all be tested
/// in the same main(), and the return codes added together.
///
/// It is usually a good idea to make each unit test class a friend of
/// the class to be tested so that the unit test routines may inspect
/// the private data members of that class.

class UnitTest {
    public:
        UnitTest(void);

        ~UnitTest();

        /// Function which runs all the unit tests for the target class.
        /// The unit test writer must override this function to implement
        /// the test suite.
        /// \return return code obtained from calling getReturnCode()
        int runAllTests(void);

        /// Function to log the result of a single test.
        /// \param passed true if the test passed
        /// \param message short message describing the test
        void logOneTest(bool        passed,
                        const char  *message);

        /// Determine the return code from the count of tests which passed.
        /// \return 0 if all tests passed, otherwise 1
        int getReturnCode(void);

    private:
        unsigned int numTests;
        unsigned int numPassed;
};

}

#endif
