/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Philip Chong <pchong@cadence.com>

ChangeLog:
2005-06-27: ChangeLog started
*/

#if !defined(oagUtilOptions_P)
#define oagUtilOptions_P

#include <map>
#include <set>
#include <string>
#include <assert.h>

namespace oagUtil {

/// \brief A deprecated command line options parser. New code should use
/// OptionParser instead.
///
/// Command lines are expected to be in the form
/// \verbatim
/// command_name -opt1 [arg1] -opt2 [arg2] ...
/// \endverbatim
/// where opt{n} is the name of the {n}th option and arg{n} is the
/// corresponding argument.
///
/// Command line options can either take arguments or not;  there is
/// currently no support for options with optional arguments.  Options
/// may also be mandatory.  The parse() method flags an error if a
/// mandatory option is not present on the command line being parsed.
/// There is no support for parsing of arguments without a corresponding
/// option flag.  The order of options on the command line is ignored.
///
/// Users should first construct a Options instance, then initialize the
/// instance by adding all the options which should be accepted by the
/// parser using the add() method.  Calling the parse() method parses the
/// given command line, storing the result in the internal state of the
/// instance.  The isGiven() and getValue() methods can then be used
/// to query the instance for the results of parsing.

class Options {
    struct ParseData {
        bool        takesArgument;
        bool        isMandatory;
        bool        given;
        std::string argument;
    };

    typedef std::map<std::string, ParseData> OptionMap;

    public:
        Options(void);

        ~Options();

        /// Adds an option to the parser
        /// \param name the name of the option
        /// \param takesArgument true if the option requires an additional
        /// argument
        /// \param isMandatory true if the option is mandatory
        void add(const char *name,
                 bool       takesArgument,
                 bool       isMandatory);

        /// Parses the given command line.  The command line should
        /// include the command name itself at the front of the string.
        /// \param commandLine the command line to parse
        /// \return true if the command line was successfully parsed
        bool parse(const char *commandLine);

        /// Parses the given command line.  The command line is assumed
        /// to be already split into individual tokens in the standard
        /// argc/argv format.  Note that the command name itself should
        /// be in argv[0] and the last token is argv[argc-1].
        /// \param argc number of tokens
        /// \param argv array of tokens to parse
        /// \return true if the command line was successfully parsed
        bool parse(int          argc,
                   const char   *argv[]);

        /// Check if the specified option was encountered in the command
        /// line which was parsed.
        /// \param name the name of the option to check
        /// \return true if the option was encountered
        bool isGiven(const char *name);

        /// Gets the string argument for the specified option from the 
        /// command line which was parsed.
        /// \param name the name of the option to check
        /// \return string argument from the parsed command line, or NULL
        /// if the option was not specified on the command line
        /// or does not take an argument
        const char *getValue(const char *name);

    private:
        OptionMap registry;

        void clearParseData(void);
        bool checkMandatory(void);

        friend class OptionsUnitTest;
};

}

#endif
