/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Philip Chong <pchong@cadence.com>

ChangeLog:
2005-07-22: ChangeLog started
*/

#include "oagUtilOption.h"
#include "oagUtilOptionParser.h"

#include <cstring>
#include <cstdio>
#include <sstream>

namespace oagUtil {

///////////////////////////////////////////////////////////////////////////////

OptionParser::OptionParser(const char *helpString) :
    _helpString(helpString)
{
    assert(helpString);
}

///////////////////////////////////////////////////////////////////////////////

OptionParser::~OptionParser()
{
    for (OptionMap::iterator i = _registry.begin();
         i != _registry.end(); ++i) {
        delete *i;
    }
}

///////////////////////////////////////////////////////////////////////////////

Option *
OptionParser::add(const char   *name,
                  const char   *helpString,
                  bool         isMandatory,
                  const char   *argName)
{
    Option *t = new Option(name, helpString, isMandatory, argName);
    _registry.push_back(t);
    return t;
}

///////////////////////////////////////////////////////////////////////////////

Option *
OptionParser::getOption(const char *name)
{
    for (OptionMap::iterator i = _registry.begin();
         i != _registry.end(); ++i) {
        Option *p = *i;
        assert(p);
        if (std::strcmp(name, p->_name) == 0) {
            return p;
        }
    }
    return 0;
}

///////////////////////////////////////////////////////////////////////////////

bool
OptionParser::parse(const char *commandLine)
{
    assert(0 && "Not implemented yet");
    static_cast<void>(commandLine);
    _clearParseData();
    return _checkMandatory();
}

///////////////////////////////////////////////////////////////////////////////

bool
OptionParser::parse(int        argc,
                    const char *argv[])
{
    _clearParseData();
    for (int i = 1; i < argc; ++i) {
        assert(argv[i]);
        if (*argv[i] != '-') {
            _message << "ERROR: What is \'" << argv[i] << "\'?" << std::endl;
            _printHelp();
            return false;
        }
        const char *t = argv[i] + 1;
        if (std::strcmp(t, "h") == 0 || std::strcmp(t, "help") == 0) {
            _printHelp();
            return false;
        }
        OptionMap::iterator j;
        Option *p = 0;
        for (j = _registry.begin(); j != _registry.end(); ++j) {
            p = *j;
            assert(p);
            if (std::strcmp(p->_name, t) == 0) {
                break;
            }
        }
        if (j == _registry.end()) {
            _message << "ERROR: Unknown option \'-" << t << "\' specified." 
                << std::endl;
            _printHelp();
            return false;
        }
        if (p->_argName) {
            ++i;
            if (i >= argc) {
                _message << "ERROR: Option \'-" << t << "\' requires argument."
                    << std::endl;
                _printHelp();
                return false;
            }
            assert(argv[i]);
            p->_argValue = argv[i];
        } else {
            // set a dummy value for options which do not take args
            p->_argValue = "x";
        }
    }
    return _checkMandatory();
}

///////////////////////////////////////////////////////////////////////////////

void
OptionParser::_clearParseData()
{
    for (OptionMap::iterator i = _registry.begin();
         i != _registry.end(); ++i) {
        Option *p = *i;
        assert(p);
        p->_argValue.clear();
    }
    _message.str().clear();
}

///////////////////////////////////////////////////////////////////////////////

bool
OptionParser::_checkMandatory()
{
    for (OptionMap::iterator i = _registry.begin();
         i != _registry.end(); ++i) {
        Option *p = *i;
        assert(p);
        if (p->_isMandatory && p->_argValue.empty()) {
            _message << "ERROR: Mandatory option \'-" << p->_name
                << "\' missing" << std::endl;
            _printHelp();
            return false;
        }
    }
    return true;
}

///////////////////////////////////////////////////////////////////////////////

void
OptionParser::_printHelp() {
    _message << "Usage: " << _helpString << std::endl << std::endl;
    _message << "Syntax:" << std::endl;
    for (OptionMap::iterator i = _registry.begin();
         i != _registry.end(); ++i) {
        Option *p = *i;
        assert(p);
        char buf[1024];
        sprintf(buf, "  %s-%s%s%s%s",
                (p->_isMandatory ? "" : "["),
                p->_name,
                (p->_argName ? " " : ""),
                (p->_argName ? p->_argName : ""),
                (p->_isMandatory ? "" : "]"));
        sprintf(buf, "%-29s ", buf);
        _message << buf;
        const char *t = p->_helpString;
        while (*t) {
            const char *u;
            const char *sp = 0;
            for (u = t; (*u) && (u < t + 45); ++u) {
                if (*u == ' ') {
                    sp = u;
                }
            }
            if (!*u) {
                _message << t << std::endl;
                break;
            }
            u = sp ? sp : u;
            sprintf(buf, "%.*s", u - t, t);
            t = u;
            _message << buf << std::endl
                      << "                             ";
        }
    }
    _message << "  [-h | -help]                Print this message" 
        << std::endl;
}

///////////////////////////////////////////////////////////////////////////////

}
