/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Zhong Xiu <zxiu@andrew.cmu.edu>

ChangeLog:
2004-09-15: ChangeLog started
*/

#include <assert.h>
#include <float.h>
#include <iostream>
#include <iomanip>
#include <stdio.h>
#include <string.h>
#include <algorithm>
#include <math.h>
#include <list>
#include "oaDesignDB.h"

#include "oagTimerExtDelay.h"
#include "oagTimerCellData.h"
#include "oagTimerCellMaster.h"
#include "oagTimerTPoint.h"
#include "oagTimerTPointMaster.h"
#include "oagTimerTimer.h"
#include "oagTimerLibData.h"
#include "oagTimerLibParserInt.h"
#include "oagTimerSdcData.h"
#include "oagTimerSdcParserInt.h"
#include "oagTimerUtil.h"
#include "oagTimerDesignTool.h"

#include "oagTimerInstTermObserver.h"
#include "oagTimerTermObserver.h"

#include "oagUtilGlobMatch.h"

//#define DEBUG
//#define DEBUG_INVALIDATE
//#define DEBUG_CLOCK
//#define DEBUG_CELLREQ
//#define printDelayValues
//#define printTol

using namespace oa;
using namespace std;
using namespace oagTimer;
using namespace oagUtil;

namespace oagTimer {

  
  bool Timer::displayWarning = true;
///////////////////////////////////////////////////////////////////////////////

Timer::Timer()
{
    CellData::initAppDefs();
    CellMaster::initAppDefs();
    TPoint::initAppDefs();
    TPointMaster::initAppDefs();

    TimerExtDelay::termAppDef 
        = oaVoidPointerAppDef<oaOccTerm>::get("myExtDelay");
    TimerDrivingCell::termAppDef 
        = oaVoidPointerAppDef<oaOccTerm>::get("myDrivingCell");
    TimerFanoutLoad::termAppDef 
        = oaVoidPointerAppDef<oaOccTerm>::get("myFanoutLoad");
    TimerInputTransition::termAppDef 
        = oaVoidPointerAppDef<oaOccTerm>::get("myInputTransition");

    _wireModel = 0;
    _elmoreWireModel = 0;
    elmoreAnalysis = false;
    _termCB = new TermObserver(2, true);
    _instTermCB = new InstTermObserver(2, true);
    worstArr = 0.0;
    optIncrTimer = false;
}

///////////////////////////////////////////////////////////////////////////////

Timer::~Timer() {
    delete _termCB;
    delete _instTermCB;

    elmoreAnalysis = false;
    // remove all watched designs from watched list
    while (!_watchedDesigns.empty()) {
        oaDesign *des = *(_watchedDesigns.begin());
        assert(des);
        remove(des);
    }
}

///////////////////////////////////////////////////////////////////////////////

void
Timer::readLibrary(const char *filename,
                   const char *libertyLibname,
                   const char *libertyViewname)
{
    FILE *f = fopen(filename, "r");
    if (!f) {
        std::cerr << "ERROR: Could not open library file \'" 
            << filename << "\'." << std::endl;
        exit(0);
    }
    libParseData.libString = libertyLibname ? libertyLibname : "";
    libParseData.viewString = libertyViewname ? libertyViewname : "abstract";

    oagTimerLib_yyin = f;
    oagTimerLib_yyparse();
    stdLib.init(oaNativeNS(), libParseData.libString);
    fclose(f);
}

///////////////////////////////////////////////////////////////////////////////

/*---------------------------------------------------------*/
void
Timer::add(oaDesign *design)
{
    dsgn = design;
    assert(design);
    _watchedDesigns.insert(design);

    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);

    // create TPoints on all Terms
    /*oaIter<oaOccTerm> termIter(occ->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
        TPoint::create(i);
    }*/

    // create TPoints on all InstTerms
    oaIter<oaOccInstTerm> instTermIter(occ->getInstTerms());
    while (oaOccInstTerm *i = instTermIter.getNext()) {
        TPoint *tp = TPoint::create(i);
        if (tp && tp->type == TIMING_POINT_CLOCK) {
            clockPins.insert(i);
        }
    }
    // create TPoints on all Terms
    oaIter<oaOccTerm> termIter(occ->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
        TPoint::create(i);
        TPoint *tp = TPoint::get(i);
	if(tp->isPI()){	
	    oaOccNet *net = i->getNet();
	    oaIter<oaOccInstTerm> itIter(net->getInstTerms());
	    while(oaOccInstTerm *j = itIter.getNext()){
		TPoint *p = TPoint::get(j);
		if(p->type == TIMING_POINT_SIGNAL_IN){
		    p->isFromPI = true;
		}
	    }
	}else if(tp->isPO()){
	    oaOccNet *net = i->getNet();
	    oaIter<oaOccInstTerm> itIter(net->getInstTerms());
	    while(oaOccInstTerm *j = itIter.getNext()){
		TPoint *p = TPoint::get(j);
		if(p->type == TIMING_POINT_SIGNAL_OUT){
		    p->isToPO = true;
		}
	    }
	}
    }
    //Add flag to indicate input InstTerm is connected from output of FF
    for (std::set<oaOccObject*>::iterator i=clockPins.begin();
         i!=clockPins.end(); ++i) {
        oaOccInst *inst = (static_cast<oaOccInstTerm*>(*i))->getInst();
	oaOccNet *net = DesignTool::getOutputNet(inst);
	oaIter<oaOccInstTerm> itIter(net->getInstTerms());
	while(oaOccInstTerm *j = itIter.getNext()){
	    TPoint *p = TPoint::get(j);
	    if(p->type == TIMING_POINT_SIGNAL_IN){
		p->isFromFF = true;
	    }
	}
    }
    oaIter<oaOccInst> occIter ( occ->getInsts());
    while(oaOccInst *occInst = occIter.getNext()){
	CellData *cell = CellData::create(occInst);
	assert(cell);
	if(cell->master->isSequential){
	    oaIter<oaOccInstTerm> itIter(occInst->getInstTerms());
	    while(oaOccInstTerm *j = itIter.getNext()){
		TPoint *p = TPoint::get(j);
		if(p->type == TIMING_POINT_SIGNAL_IN){
		    oaOccNet *net = j->getNet();
		    oaIter<oaOccInstTerm> it2Iter(net->getInstTerms());
		    while(oaOccInstTerm *k = it2Iter.getNext()){
			TPoint *p = TPoint::get(k);
			if(p->type == TIMING_POINT_SIGNAL_OUT){
			    p->isToFF = true;
			}
		    }
		}
	    }
	}
    }
}

///////////////////////////////////////////////////////////////////////////////
/*
void
Timer::debug_TimingFlag(oaDesign *design)
{
#if defined DEBUG
    std::cout << "Debugging Timing Flag" << std::endl;
#endif
    assert(design);

    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);

    // create TPoints on all Terms
    oaIter<oaOccTerm> termIter(occ->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
        TPoint *tp = TPoint::get(i);
	std::cout << "TPoint(" << getBlockName(i) << ") Arrival Time: ";
	std::cout << tp->atValid << " Req Arrival Time: " << tp->ratValid << std::endl;
        if (tp && (tp->type == TIMING_POINT_PI_CLOCK || tp->type == TIMING_POINT_CLOCK)) {
	    TPoint *tp1 = TPoint::getClock(i);
	    std::cout << "TPoint(" << getBlockName(i) << ") Arrival Time: ";
	    std::cout << tp1->atValid << " Req Arrival Time: " << tp1->ratValid << std::endl;
        }
    }

    // create TPoints on all InstTerms
    oaIter<oaOccInstTerm> instTermIter(occ->getInstTerms());
    while (oaOccInstTerm *i = instTermIter.getNext()) {
        TPoint *tp = TPoint::get(i);
	TPoint *tp1 = TPoint::getClock(i);
	std::cout << "TPoint(" << getBlockName(i) << ") Arrival Time: ";
	std::cout << tp->atValid << " Req Arrival Time: " << tp->ratValid << std::endl;
	std::cout << "TPoint(" << getBlockName(i) << ") Arrival Time: ";
	std::cout << tp1->atValid << " Req Arrival Time: " << tp1->ratValid << std::endl;
        //if (tp && (tp->type == TIMING_POINT_PI_CLOCK || tp->type == TIMING_POINT_CLOCK)) {
        //if (tp && tp->type == TIMING_POINT_CLOCK) {
	//    TPoint *tp1 = TPoint::getClock(i);
	 //   std::cout << "TPoint(" << getBlockName(i) << ") Arrival Time: ";
	  //  std::cout << tp1->atValid << " Req Arrival Time: " << tp1->ratValid << std::endl;
       // }
    }
}*/
///////////////////////////////////////////////////////////////////////////////

void
Timer::remove(oaDesign *design)
{
#if defined(DEBUG)
    std::cout << "Remove (CLEANUP)" << std::endl;
#endif
    assert(design);
    int t = _watchedDesigns.erase(design);
    assert(t);
    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    oaIter<oaOccTerm> termIter(occ->getTerms());
    while (oaOccTerm *t = termIter.getNext()) {
        TPoint::destroy(t);
        // destroy external data on terms
        // FIXME should be in separate class/function?
        TimerExtDelay *p1 = static_cast<TimerExtDelay*>
                (TimerExtDelay::termAppDef->get(t));
        if (p1) {
            delete p1;
            TimerExtDelay::termAppDef->destroy(t);
        }
        TimerDrivingCell *p2 = 
            static_cast<TimerDrivingCell*>
            (TimerDrivingCell::termAppDef->get(t));
        if (p2) {
            delete p2;
            TimerDrivingCell::termAppDef->destroy(t);
        }
        TimerFanoutLoad *p3 = 
            static_cast<TimerFanoutLoad*>
            (TimerFanoutLoad::termAppDef->get(t));
        if (p3) {
            delete p3;
            TimerFanoutLoad::termAppDef->destroy(t);
        }
        TimerInputTransition *p4 = static_cast<TimerInputTransition*>
                (TimerInputTransition::termAppDef->get(t));
        if (p4) {
	    std::cout << "Remove p4" << std::endl;
            delete p4;
            TimerInputTransition::termAppDef->destroy(t);
        }
    }
    oaIter<oaOccInstTerm> instTermIter(occ->getInstTerms());
    while (oaOccInstTerm *t = instTermIter.getNext()) {
        TPoint::destroy(t);
    }
}

//-----------------------------------------------------------------------------

void
Timer::readConstraints(const char   *filename,
                       oaDesign     *design)
{
    assert(filename);
    assert(design);
    FILE *f = fopen(filename, "r");
    if (!f) {
        std::cerr << "ERROR: Could not open sdc file \'" 
            << filename << "\'." << std::endl;
        exit(0);
    }
    std::cout << "Reading sdc file \'" << filename << "\'." << std::endl;
    oagTimerSdc_yyin = f;
    oagTimerSdc_yyparse();

    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    oaIter<oaOccTerm> termIter(occ->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
        TPoint *tp = TPoint::get(i); 
        if (sdcParseData.isClock[getBlockName(i)]) {
            tp->type = TIMING_POINT_PI_CLOCK;
        }
        if (!static_cast<TimerExtDelay*>
            (TimerExtDelay::termAppDef->get(i))) {
            TimerExtDelay *p = new TimerExtDelay();
            p->delay = ModelType::ZERO_DELAY();
            oaString f;
            i->getName(oaNativeNS(), f);
            if (tp->isPI()) {
                if (sdcParseData.allInputs) {
                    p->delay = sdcParseData.allInputsDelay;
                    TimerExtDelay::termAppDef->
                        set(i,static_cast<void*>(p));
                } else {
                    delayMap::iterator it = sdcParseData.extInputDelays.find
                                            (std::string(f));
                    if (it != sdcParseData.extInputDelays.end()) {
                        p->delay = (*it).second;
                        TimerExtDelay::termAppDef->
                            set(i,static_cast<void*>(p));
                    } else {
                        for (it = sdcParseData.extInputDelays.begin();
                             it != sdcParseData.extInputDelays.end(); ++it) {
                            GlobMatch m((*it).first);
                            if (m.match(std::string(f))) {
                                p->delay = (*it).second;
                                TimerExtDelay::termAppDef->
                                    set(i, static_cast<void*>(p));
                                break;
                            }
                        }
                    }
                }
            } else if (tp->isPO()) {
                if (sdcParseData.allOutputs) {
                    p->delay = sdcParseData.allOutputsDelay;
                    TimerExtDelay::termAppDef->
                        set(i, static_cast<void*>(p));
                } else {
                    delayMap::iterator it = 
                        sdcParseData.extOutputDelays.find
                        (std::string(f));
                    if (it != sdcParseData.extOutputDelays.end()) {
                        p->delay = (*it).second;
                        TimerExtDelay::termAppDef->
                            set(i, static_cast<void*>(p));
                    } else {
                        for (it = sdcParseData.extOutputDelays.begin();
                             it != sdcParseData.extOutputDelays.end(); ++it) {
                            GlobMatch m((*it).first);
                            if (m.match(std::string(f))) {
                                p->delay = (*it).second;
                                TimerExtDelay::termAppDef->
                                    set(i, static_cast<void*>(p));
                                break;
                            }
                        }
                    }
                }
            }
            if (!static_cast<TimerExtDelay*>
                (TimerExtDelay::termAppDef->get(i))) {
                delete p;
            } 
        } // set the external delay values for PI and PO
        if (tp->isPI() && !static_cast<TimerDrivingCell*>
                           (TimerDrivingCell::termAppDef->get(i))) {
            TimerDrivingCell *p = new TimerDrivingCell();
            std::string d;
            oaNativeNS ns;
            static const oaScalarName absViewName(ns, "abstract");
            TPointMaster *driver = 0;
            oaString f;
            if (sdcParseData.allCells) {
                d = sdcParseData.allDrivingCells;
                const oaScalarName cellName(ns, d.c_str());
                oaDesign *dCell = oaDesign::open(stdLib, cellName, 
                                                 absViewName, 'r');
                assert(dCell);
                oaModule *topDMod = dCell->getTopModule();
                oaIter<oaModTerm> termIter(topDMod->getTerms());
                while (oaModTerm *i = termIter.getNext()) {
                    i->getName(ns, f);
                    TPointMaster *tm = TPointMaster::get(i);
                    if (!tm) {
                        continue;
                    }
                    oaTermType type(i->getTermType());
                    if (type == oacOutputTermType) {
                        driver = tm;
                        break;
                    }
                }
                assert(driver);
                p->driver = driver;
                TimerDrivingCell::termAppDef->
                    set(i, static_cast<void*>(p));
            } else {
                i->getName(ns, f);
                cellMap::iterator it = sdcParseData.drivingCells.find
                                       (std::string(f));
                if (it != sdcParseData.drivingCells.end()) {
                    d = (*it).second;
                    const oaScalarName cellName(ns, d.c_str());
                    oaDesign *dCell = oaDesign::open(stdLib, cellName, 
                                                     absViewName, 'r');
                    assert(dCell);
                    oaModule *topDMod = dCell->getTopModule();
                    oaIter<oaModTerm> termIter(topDMod->getTerms());
                    while (oaModTerm *i = termIter.getNext()) {
                        TPointMaster *tm = TPointMaster::get(i);
                        if (!tm) {
                            continue;
                        }
                        oaTermType type(i->getTermType());
                        if (type == oacOutputTermType) {
                            driver = tm;
                            break;
                        }
                    }
                    assert(driver);
                    p->driver = driver;
                    TimerDrivingCell::termAppDef->
                        set(i, static_cast<void*>(p));
                } else {
                    for (it = sdcParseData.drivingCells.begin();
                            it != sdcParseData.drivingCells.end(); ++it)
                    {
                        GlobMatch m((*it).first);
                        if (m.match(std::string(f))) {
                            d = (*it).second;
                            const oaScalarName cellName(ns, d.c_str());
                            oaDesign *dCell = 
                                oaDesign::open(stdLib, 
                                               cellName, absViewName, 'r');
                            assert(dCell);
                            oaModule *topDMod = dCell->getTopModule();
                            oaIter<oaModTerm> termIter(topDMod->getTerms());
                            while (oaModTerm *i = termIter.getNext()) {
                                TPointMaster *tm =
                                    TPointMaster::get(i);
                                if (!tm) {
                                    continue;
                                }
                                oaTermType type(i->getTermType());
                                if (type == oacOutputTermType) {
                                    driver = tm;
                                    break;
                                }
                            }
                            assert(driver);
                            p->driver = driver;
                            TimerDrivingCell::termAppDef->
                                set(i, static_cast<void*>(p));
                            break;
                        }
                    }
                }
            }
            if (!static_cast<TimerDrivingCell*>
                (TimerDrivingCell::termAppDef->get(i))) {
                delete p;
            }
        } // set the driving cells for PI
        if (tp->isPO() && !static_cast<TimerFanoutLoad*>
                           (TimerFanoutLoad::termAppDef->get(i))) {
            TimerFanoutLoad *p = new TimerFanoutLoad();
            p->load = 0.0;
            if (sdcParseData.allLoads) {
                p->load = sdcParseData.allFanoutLoads;
                TimerFanoutLoad::termAppDef->
                    set(i, static_cast<void*>(p));
            } else {
                oaString f;
                i->getName(oaNativeNS(), f);
                delayMap::iterator it = sdcParseData.fanoutLoads.find
                                        (std::string(f));
                if (it != sdcParseData.fanoutLoads.end()) {
                    p->load = (*it).second;
                    TimerFanoutLoad::termAppDef->
                        set(i, static_cast<void*>(p));
                } else {
                    for (it = sdcParseData.fanoutLoads.begin();
                         it != sdcParseData.fanoutLoads.end(); ++it) {
                        GlobMatch m((*it).first);
                        if (m.match(std::string(f))) {
                            p->load = (*it).second;
                            TimerFanoutLoad::termAppDef->
                                set(i, static_cast<void*>(p));
                            break;
                        }
                    }
                }
            }
            if (!static_cast<TimerFanoutLoad*>
                (TimerFanoutLoad::termAppDef->get(i))) {
                delete p;
            }
        } // set the fanout loads for PO
    }
    // Setting multi cycle constraints
    termIter = occ->getTerms();
    while (oaOccTerm *t = termIter.getNext()) {
        setMultiCycleConstraint(t);
    }
    oaIter<oaOccInstTerm> instTermIter(occ->getInstTerms());
    while (oaOccInstTerm *t = instTermIter.getNext()) {
        setMultiCycleConstraint(t); 
    }
    fclose(f);
}

/*!
  Sets a block object as a multi-cycle constraint point if 
  it is in the list of multi-cycle constraints parsed
  from the sdc file.    
  \param block pointer to the oaOccObject  
*/
void
Timer::setMultiCycleConstraint(oaOccObject *block) {
    TPoint *p = TPoint::get(block); 
    std::string name = getBlockName(block);
    termIntMap::iterator it = sdcParseData.multiCycleConstraints.find(name);
    if (it != sdcParseData.multiCycleConstraints.end()) {
        if (p->multiCycleConstraint < (*it).second) {
            p->multiCycleConstraint = (*it).second;
        }
    }
    for (it = sdcParseData.multiCycleConstraints.begin();
         it != sdcParseData.multiCycleConstraints.end(); ++it) {
        GlobMatch m((*it).first);
        if (m.match(name)) {
            if (p->multiCycleConstraint < (*it).second) {
                p->multiCycleConstraint = (*it).second;
            }
        }
    }
}

/*! 
  Finds the oaOccObject by it's name. 
  FIXME should move this out of Timer class
  \param block pointer to the oaOccObject  
  \param the name of the oaOccObject  
*/
oaOccObject *
Timer::findOccObj(oaOccurrence *occ, std::string name) {
    oaOccObject *retval;
    std::string::size_type pos = name.find("/");
    oaNativeNS ns;
    if (pos == std::string::npos) {
        oaName n(ns, name.c_str()); 
        retval = oaOccTerm::find(occ, n);
    } else {
        std::string s1;
        std::string s2;
        s1.assign(name, 0, pos);
        s2.assign(name, pos+1, name.size()-pos-1);
        oaSimpleName iName(ns, s1.c_str()); 
        oaName tName(ns, s2.c_str()); 
        oaOccInst *inst = oaOccInst::find(occ, iName);
        if (!inst) {
            return 0;
        }
        retval = oaOccInstTerm::find(inst, tName);
    }
    return retval;
}

///////////////////////////////////////////////////////////////////////////////

DelayType
Timer::getFullSlack(oaOccObject *oPtr)
{
    TPoint *p = TPoint::get(oPtr);
    updateAllArr();
    //updateArr(oPtr);
    //updateReq(oPtr);
    updateAllReq();
	
    /*std::cout << "Current:(" << getBlockName(oPtr) << ")" << std::endl;
    oaOccNet *net = getOccNet(oPtr);
    oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
    while (oaOccInstTerm *j = instTermIter.getNext()) {
	TPoint *tp = TPoint::get(j);
	if(tp->type == TIMING_POINT_CLOCK){
	    std::cout << "Clk:(" << getBlockName(j) << ")" << std::endl;

	    oaModTerm *masterTerm = j->getModInstTerm()->getTerm();
	    assert(masterTerm);
	    TPointMaster *tm = TPointMaster::get(masterTerm);
	    assert(tm);
	    oaOccInst *inst = j->getInst();
	    assert(inst);
	    TPointMaster::pathVector::iterator i;
	    for (i = tm->inPaths.begin(); i != tm->inPaths.end(); ++i) {
		assert(i->other);
		oaOccInstTerm *other = getOccFromMod(inst, i->other);
		if (!other) continue;
		p = TPoint::get(other); 
		std::cout << "Others:(" << getBlockName(other) << ")" << std::endl;
		break;
	    }
	    break;
	}
    }*/
    return p->getSlack();
}

///////////////////////////////////////////////////////////////////////////////

DelayType
Timer::getSlack(oaOccObject *oPtr)
{
    TPoint *tp = TPoint::get(oPtr);
    //return tp->getSlack();
    updateArr(oPtr);
    updateReq(oPtr);
    return tp->getSlack();
}

///////////////////////////////////////////////////////////////////////////////

TimeType
Timer::setWorstArr(oaDesign *design)
{
    assert(design);
    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    TimeType t = 0.0;
    oaIter<oaOccTerm> termIter(occ->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
	oaTermType type(i->getTermType());
	if(type == oacOutputTermType){
	    oaIter<oaOccInstTerm> instTermIter(i->getNet()->getInstTerms());
	    while (oaOccInstTerm *iTerm = instTermIter.getNext()) {
		oaTermType itType(iTerm->getTerm()->getTermType());
		if(itType == oacOutputTermType){
		    TimeType s = getArr(iTerm);
		    if(s > t) t = s;
		}
	    }
	}
    }
    worstArr = t;
    std::cout << "Worst Arr:" << worstArr << std::endl;
}
///////////////////////////////////////////////////////////////////////////////

TimeType
Timer::getArr(oaOccObject *oPtr)
{
    TPoint *tp = TPoint::get(oPtr);
    updateArr(oPtr);
    TimeType a1 = tp->getFallArr();
    TimeType a2 = tp->getRiseArr();
    TimeType s = (a1 > a2) ? a1 : a2;
    return s;
}

///////////////////////////////////////////////////////////////////////////////

TimeType
Timer::getReq(oaOccObject *oPtr)
{
    TPoint *tp = TPoint::get(oPtr);
    updateReq(oPtr);
    TimeType a1 = tp->getFallReq();
    TimeType a2 = tp->getRiseReq();
    TimeType s = (a1 < a2) ? a1 : a2;
    return s;
}

///////////////////////////////////////////////////////////////////////////////

TimeType
Timer::getFallArr(oaOccObject *oPtr)
{
     
    TPoint *tp = TPoint::get(oPtr);
    updateArr(oPtr);
    return tp->getFallArr();
}

///////////////////////////////////////////////////////////////////////////////

TimeType
Timer::getRiseArr(oaOccObject *oPtr)
{
    TPoint *tp = TPoint::get(oPtr);
    updateArr(oPtr);
    return tp->getRiseArr();
}

///////////////////////////////////////////////////////////////////////////////

TimeType
Timer::getFallReq(oaOccObject *oPtr)
{
    TPoint *tp = TPoint::get(oPtr);
    assert(tp);
    updateReq(oPtr);
    return tp->getFallReq();
}

///////////////////////////////////////////////////////////////////////////////

TimeType
Timer::getRiseReq(oaOccObject *oPtr)
{
    TPoint *tp = TPoint::get(oPtr);
    updateReq(oPtr);
    return tp->getRiseReq();
}
///////////////////////////////////////////////////////////////////////////////
DelayType
Timer::getSlew(oaOccObject *oPtr)
{
    TPoint *tp = TPoint::get(oPtr);
    updateArr(oPtr);
    DelayType a1 = tp->getFallSlew(0);
    DelayType a2 = tp->getRiseSlew(0);
    DelayType s = (a1 > a2) ? a1 : a2;
    return s;
}
///////////////////////////////////////////////////////////////////////////////

DelayType
Timer::getFallSlew(oaOccObject *oPtr)
{
     
    TPoint *tp = TPoint::get(oPtr);
    updateArr(oPtr);
    return tp->getFallSlew(0);
}

///////////////////////////////////////////////////////////////////////////////

DelayType
Timer::getRiseSlew(oaOccObject *oPtr)
{
    TPoint *tp = TPoint::get(oPtr);
    updateArr(oPtr);
    return tp->getRiseSlew(0);
}
//-----------------------------------------------------------------------------

/*! 
  This function returns the slack of a certain net. 
  This function should be used in incremental timing analysis, 
  the function first checks if the current data is valid,
  if yes, returns the value, if not, it updates the timing information
  and then returns the value.
  \param net the net we want to know its slack value
*/
DelayType
Timer::getNetSlack(const oaOccNet *net)
{
    DelayType s = ModelType::MAX_DELAY();
    DelayType t;
    oaIter<oaOccTerm> termIter(net->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
        t = getSlack(i);
        if (s > t) {
            s = t;
        }
    }
    oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
    while (oaOccInstTerm *i = instTermIter.getNext()) {
        t = getSlack(i);
        if (s > t) {
            s = t;
        }
    }
    return (s);
}

//-----------------------------------------------------------------------------

/*! 
  This function returns the worst arrival value.
  This function should be used in incremental timing analysis, 
  the function first checks if the current data is valid,
  if yes, returns the value, if not, it updates the timing information
  and then returns the value.
*/
DelayType
Timer::getWorstArr()
{
#if defined(DEBUG)
    std::cout << "Get the worst arrival" << std::endl;
#endif
    DelayType worstArr = 0.0;
    //optIncrTimer = true;
    updateAllArr();
    updateAllReq();
    for (std::set<oaOccObject*>::iterator i=clockPins.begin();
         i!=clockPins.end(); ++i) {
      //TPoint *p = static_cast<TPoint*>(myAppDef->get(*i));
        oaModTerm *masterTerm = (static_cast<oaOccInstTerm*>(*i))
            ->getModInstTerm()->getTerm();
        assert(masterTerm);
        TPointMaster *tm = TPointMaster::get(masterTerm);
        assert(tm);
        oaOccInst *inst = (static_cast<oaOccInstTerm*>(*i))->getInst();
        TPointMaster::pathVector::iterator t = tm->inPaths.begin();
        assert(t->other);
        oaOccInstTerm *other = getOccFromMod(inst, t->other);
        if (!other) {
            continue;
        }
        TPoint *otherTP = TPoint::get(other);
        assert(otherTP);
	//std::cout << getBlockName(other) << ":" << getArr(other) << std::endl;
        if (getArr(other)>worstArr) {
	    worstArr=getArr(other);
        }
        if (getArr(*i)>worstArr) {
	    worstArr=getArr(*i);
        }
	/*oaIter<oaOccInstTerm> itIter = inst->getInstTerms();
	while(oaOccInstTerm *iTerm = itIter.getNext()){
	    oaNativeNS ns; oaString f;
	    iTerm->getTermName(ns,f);
	    if(f == "SI" || f == "SE"){
		if(getArr(iTerm)>worstArr){
		    worstArr = getArr(iTerm);
		}
	    }
	}*/
    }
    for (std::set<oaDesign *>::iterator i = _watchedDesigns.begin();
         i != _watchedDesigns.end(); ++i) {
        oaDesign *des = *i;
        assert(des);
        oaOccurrence *occ = des->getTopOccurrence();
        assert(occ);
        oaIter<oaOccTerm> termIter(occ->getTerms());
        while (oaOccTerm *t=termIter.getNext()) {
            TPoint *p = TPoint::get(t);
            if (p->isPO()) {
		//std::cout << getBlockName(t) << ":" << getArr(t) << std::endl;
		double arr = getArr(t);
		if(arr == DBL_MAX) continue;
                if (arr>worstArr) {
                    worstArr=getArr(t);
                }
            }
        }
    }
    //optIncrTimer = false;
    return worstArr;
}
//-----------------------------------------------------------------------------

/*! 
  This function returns the worst slack value.
  This function should be used in incremental timing analysis, 
  the function first checks if the current data is valid,
  if yes, returns the value, if not, it updates the timing information
  and then returns the value.
*/
DelayType
Timer::getWorstSlack()
{
#if defined(DEBUG)
    std::cout << "Get the worst slack" << std::endl;
#endif
    DelayType ws = ModelType::MAX_DELAY();
#if 0
    for (std::set<oaOccObject*>::iterator i=clockPins.begin(); 
        i!=clockPins.end(); ++i) {
        getArr(*i);
    }
    for (std::set<oaDesign *>::iterator i = _watchedDesigns.begin();
         i != _watchedDesigns.end(); ++i) {
        oaDesign *des = *i;
        assert(des);
        oaOccurrence *occ = des->getTopOccurrence();
        assert(occ);
        oaIter<oaOccTerm> termIter(occ->getTerms());
        while (oaOccTerm *t=termIter.getNext()) {
            TPoint *p = TPoint::get(t);
            if (p->isPO()) {
                getArr(t);
            }
        }
    }
#endif
    //optIncrTimer = true;
    updateAllArr();
    updateAllReq();
    //std::cout << "Seeking Worst Slack" << std::endl;
    for (std::set<oaOccObject*>::iterator i=clockPins.begin();
         i!=clockPins.end(); ++i) {
      //TPoint *p = static_cast<TPoint*>(myAppDef->get(*i));
        oaModTerm *masterTerm = (static_cast<oaOccInstTerm*>(*i))
            ->getModInstTerm()->getTerm();
        assert(masterTerm);
        TPointMaster *tm = TPointMaster::get(masterTerm);
        assert(tm);
        oaOccInst *inst = (static_cast<oaOccInstTerm*>(*i))->getInst();
        TPointMaster::pathVector::iterator t = tm->inPaths.begin();
        assert(t->other);
        oaOccInstTerm *other = getOccFromMod(inst, t->other);
        if (!other) {
            continue;
        }
        TPoint *otherTP = TPoint::get(other);
        assert(otherTP);
	//std::cout << getBlockName(other) << ":" << otherTP->getSlack() << std::endl;
        if (otherTP->getSlack()<ws) {
            ws=otherTP->getSlack();
        }
	TPoint *clkPin = TPoint::get(*i);
        if (clkPin->getSlack()<ws) {
            ws=clkPin->getSlack();
        }
	/*oaIter<oaOccInstTerm> itIter = inst->getInstTerms();
	while(oaOccInstTerm *iTerm = itIter.getNext()){
	    oaNativeNS ns; oaString f;
	    iTerm->getTermName(ns,f);
	    if(f == "SI" || f == "SE"){
		TPoint *sitp = TPoint::get(iTerm);
		assert(sitp);
		if(sitp->getSlack()<ws){
		    ws = sitp->getSlack();
		}
	    }
	}*/
	
    }
    for (std::set<oaDesign *>::iterator i = _watchedDesigns.begin();
         i != _watchedDesigns.end(); ++i) {
        oaDesign *des = *i;
        assert(des);
        oaOccurrence *occ = des->getTopOccurrence();
        assert(occ);
        oaIter<oaOccTerm> termIter(occ->getTerms());
        while (oaOccTerm *t=termIter.getNext()) {
            TPoint *p = TPoint::get(t);
            if (p->isPO()) {
                if (p->getSlack()<ws) {
                    ws=p->getSlack();
		    /*cout << " " << getBlockName(t) << " ws:" << ws 
			<< " slack:" << getSlack(t) << endl;*/
                }
            }
        }
    }
    //optIncrTimer = false;
    return ws;
}

//-----------------------------------------------------------------------------

bool
Timer::isValidTiming()
{
#if defined(DEBUG)
    std::cout << "check if timing is valid" << std::endl;
#endif
    //DelayType ws = ModelType::MAX_DELAY();
    updateAllArr();
    //updateAllReq();
    for (std::set<oaDesign *>::iterator i = _watchedDesigns.begin();
         i != _watchedDesigns.end(); ++i) {
        oaDesign *des = *i;
        assert(des);
        oaOccurrence *occ = des->getTopOccurrence();
        assert(occ);
        oaIter<oaOccTerm> termIter(occ->getTerms());
        while (oaOccTerm *t=termIter.getNext()) {
            TPoint *p = TPoint::get(t);
            if (p->isPO()) {
                if (p->getSlack()<0) {
		    return false;
                }
            }
        }
    }
    return true;
}

/*!
  Insert a timing data in timing data vector.
  Insertion is done so the timing data vector is sorted in 
  increasing order of the clock index of the timing data.
  \param timingDataVector the timing data vector
  \param d the pointer to the timing data
 */
void
Timer::insertTimingData(std::vector<timingData*> &timingDataVector, 
                              timingData *d)
{
    for (std::vector<timingData*>::iterator it = timingDataVector.begin(); 
         it != timingDataVector.end(); ++it) {
        if (d->clockIndex < (*it)->clockIndex) {
            timingDataVector.insert(it, d);
            return;
        }
    }
    timingDataVector.push_back(d);
}


/*!
  Compares the slews of two timing data vectors.
  Returns false if they match.
  \param first timing data vector
  \param second timing data vector
*/
bool
Timer::timingDataCompare
(std::vector<timingData*> &v1, std::vector<timingData*> &v2) 
{
#if defined(DEBUG)
    std::cout << "timingDataCompare()" << std::endl;
#endif
    if (v1.size() != v2.size()) {
        return true;
    }
    for (unsigned int k = 0; k < v1.size(); ++k) { 
        timingData *d1 = v1[k];
        timingData *d2 = v2[k];
	/*std::cout << " riseSlew d1:" << d1->riseSlew
	    << " d2:" << d2->riseSlew << std::endl;
	std::cout << " fallSlew d1:" << d1->fallSlew
	    << " d2:" << d2->fallSlew << std::endl;*/
        if (!(d1->clockIndex == d2->clockIndex) || 
            !(d1->riseSlew == d2->riseSlew) || 
            !(d1->fallSlew == d2->fallSlew)) {
            return true;
        }
    }
    return false;
}

/*!
  Compare old and new slews for the fanout cone
  if slew differ by less than certain percentage, 
  then a flag is set at the output cone to 
  enabled to speedup incremental timing
  propagation
  \param fanout cone
*/
void
Timer::setSlewTimingValidOpt(TPoint *p)
{
#if defined DEBUG
    std::cout << "setSlewTimingValidOpt" << std::endl;
#endif
    std::vector<timingData*> v1 = p->multiClockData;
    if(v1.empty()) return;
    timingData *d1 = v1[0];
    double factor = 0.00001;
    double dRise = fabs(d1->riseSlew - p->oldRiseSlew);
    //double rTol = factor*p->oldRiseSlew;
    double rTol = factor*d1->riseSlew;
    //double rTol = -1.0;
    double dFall = fabs(d1->fallSlew - p->oldFallSlew);
    //double fTol = factor*p->oldFallSlew;
    double fTol = factor*d1->fallSlew;
    //double fTol = -1.0;
#if defined(printTol)
    std::cout << "  **SrcRise - new:" << d1->riseSlew << " old:" << p->oldRiseSlew
	<< " delta:" << dRise << " tol:" << rTol << std::endl;
    std::cout << "  **SrcFall - new:" << d1->fallSlew << " old:" << p->oldFallSlew
	<< " delta:" << dFall << " tol:" << fTol << std::endl;
#endif
    if (dRise < rTol){
#if defined(printTol)
	std::cout << " ***Setting riseSlewValid" << std::endl;
#endif
	p->riseSlewValid = true;
    }
    if (dFall < fTol){
#if defined(printTol)
	std::cout << " ***Setting fallSlewValid" << std::endl;
#endif
	p->fallSlewValid = true;
    }
}

/*!
  Clears a timing data vector.
  \param v timing data vector
*/
void
Timer::clearVector(std::vector<timingData*> &v) 
{
#if defined DEBUG
    std::cout << "clearTimingVector" << std::endl;
#endif
    for (unsigned j = 0; j < v.size(); ++j) {
        delete v[j];
    }
    v.clear();
}

/*!
  Initializes a timing data vector for update arrival time.
  \param v timing data vector
*/
void
Timer::initVector(std::vector<timingData*> &v) 
{
#if defined DEBUG
    std::cout << "initVector" << std::endl;
#endif
    for (unsigned j = 0; j < v.size(); ++j) {
        v[j]->riseArr = -ModelType::MAX_TIME();
        v[j]->fallArr = -ModelType::MAX_TIME();
        v[j]->riseSlew = ModelType::ZERO_DELAY();
        v[j]->fallSlew = ModelType::ZERO_DELAY();
    }
}

/*!
  Copies the slews of a timing data vector to another one.
  \param v1 first timing data vector
  \param v2 second timing data vector
*/
void
Timer::timingDataCopy(std::vector<timingData*> &v1, 
                            std::vector<timingData*> &v2) 
{
#if defined(DEBUG)
    std::cout << "timingDataCopy" << std::endl;
#endif
    clearVector(v2);
    for (unsigned j = 0; j < v1.size(); ++j) {
        timingData *d1 = v1[j];
        timingData *d2 = new timingData();
        d2->clockIndex = d1->clockIndex; 
        d2->riseSlew = d1->riseSlew;
        d2->fallSlew = d1->fallSlew;
        v2.push_back(d2);
	//delete d2;
    }
}

/*!
  Copies the slews of a timing data vector to another one.
  \param v1 first timing data vector
  \param v2 second timing data vector
*/
void
Timer::slewDataCopy(TPoint *p)
{
    std::vector<timingData*> v1 = p->multiClockData;
    if(v1.empty()) return;
    timingData *d1 = v1[0];
    p->oldRiseSlew = d1->riseSlew;
    p->oldFallSlew = d1->fallSlew;
}

/*!
  Return the index to the timing data (by it's clock index) in the
  timing data vector of an TPoint.
  \param p pointer to the TPoint
  \param clockIndex the index of the clock 
*/
int
Timer::findTimingData (TPoint *p, int clockIndex) 
{
    unsigned k;
    std::vector<timingData*> v = p->multiClockData;
    if (p->multiCycleConstraint == 1) {
        for (k = 0; k < v.size(); ++k) {
            if (clockIndex == v[k]->clockIndex) {
                return k;
            }
            if (clockIndex < v[k]->clockIndex) {
                return -1;
            }
        }
        return -1;
    } else {
        std::string name = sdcParseData.propagatedClocks[clockIndex]->clk->name;
        int m = sdcParseData.propagatedClocks[clockIndex]->m;
        for (k = 0; k < v.size(); ++k) {
            if (m == sdcParseData.propagatedClocks[v[k]->clockIndex]->m) {
                continue;
            }
            if (name == sdcParseData.propagatedClocks[v[k]->clockIndex]->clk->name) {
                return k;
            }
        }
        return -1;
    }
}

/*!
  This function updates the required time for an instTerm which is 
  the input of a path based on required times 
  of the other instTerm which is the output of the path. 
  \param i path iterator
  \param p pointer to the TPoint of the input of the path
  \param other pointer to the other instTerm 
*/
void
Timer::updateCellPathReq(TPointMaster::pathVector::iterator i, 
                               TPoint *p,
                               oaOccInstTerm *other)
{
#if defined(DEBUG)
    std::cout << " updateCellPathReq (" << getBlockName(other) << ")" << std::endl;
#endif
    oaModTerm *otherTerm = other->getModInstTerm()->getTerm();
    assert(otherTerm);
    TPointMaster *otherTPM = TPointMaster::get(otherTerm);
    assert(otherTPM);
    TPoint *otherTP = TPoint::get(other); 
    assert(otherTP);
    assert(otherTP->atValid);
    for (unsigned j = 0; j < p->multiClockData.size(); ++j) {
        timingData *d = p->multiClockData[j];
        int x = findTimingData (otherTP, d->clockIndex);
        assert(x != -1); 
        timingData *td = otherTP->multiClockData[x];
        DelayType delay;
        TimeType t3;
        if (i->transition & TPointMaster::TRANSITION_DST_RISE) {
            t3 = td->riseReq;
        } else if (i->transition & TPointMaster::TRANSITION_DST_FALL){
            t3 = td->fallReq;
        } else {
            assert(0);
        }
        if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
            delay = i->delay->lookup(otherTP->load, d->riseSlew, 
                                     otherTPM->loadLimit);
#if defined(DEBUG_CELLREQ)
	    std::cout << "SRC_RISE -- src: " << t3 << " delay: " << delay;
	    std::cout << " Delay breakdown -- (slew:" << d->riseSlew
		<< ", load:" << otherTP->load << ")" << std::endl;
#endif
            t3 = t3 - delay;
            if (d->riseReq > t3){
                d->riseReq = t3;
	    }
#if defined(DEBUG_CELLREQ)
	    std::cout << " ==> " << d->riseReq << "/" << t3 << std::endl;
#endif
        } else if (i->transition & TPointMaster::TRANSITION_SRC_FALL){
            delay = i->delay->lookup(otherTP->load, d->fallSlew, 
                                     otherTPM->loadLimit);
#if defined(DEBUG_CELLREQ)
	    std::cout << "SRC_FALL -- src: " << t3 << " delay: " << delay;
	    std::cout << " Delay breakdown -- (slew:" << d->fallSlew
		<< ", load:" << otherTP->load << ")" << std::endl;
#endif
            t3 = t3 - delay;
            if (d->fallReq > t3){
                d->fallReq = t3;
	    }
#if defined(DEBUG_CELLREQ)
	    std::cout << " ==> " << d->fallReq << "/" << t3 << std::endl;
#endif
        } else { 
            assert(0);
        }
    }
}

//#define Ceff
/*
  This function updates the arrival time for an instTerm which is 
  the output of a path based on arrival times 
  of the other instTerm which is the input of the path. 
  \param tm pointer to the timer point master of the input of the path
  \param p pointer to the timer point of the input of the path 
  \param i path iterator
  \param tp pointer to the timer point of the output of the path 
*/
void
Timer::updateCellPathArr(TPointMaster *tm,
            TPoint *p, 
            TPointMaster::pathVector::iterator i, 
            TPoint *tp) 
{
#if defined(DEBUG)
    std::cout << "updateCellPathArr" << std::endl;
#endif

    for (unsigned int j = 0; j < tp->multiClockData.size(); ++j) { 
        timingData *td = tp->multiClockData[j];
        TimeType t3;
        DelayType sl1, sl2, sl3, delay;
        timingData *d;
        int x = findTimingData(p, td->clockIndex);
        if (x == -1) {
            d = new timingData;
            d->clockIndex = td->clockIndex;
            d->riseArr = -ModelType::MAX_TIME();  
            d->fallArr = -ModelType::MAX_TIME();  
            d->riseSlew = ModelType::ZERO_DELAY();  
            d->fallSlew = ModelType::ZERO_DELAY();  
            insertTimingData((p->multiClockData), d);
        } else {
            d = p->multiClockData[x];
        }
        sl1 = td->riseSlew;
        sl2 = td->fallSlew;

	// Arrival Time Update
	// Incremental Timing Optimization Updated(08/26/2010) by:Santiago
        if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
	    //Original
	    delay = i->delay->lookup(p->load, sl1, tm->loadLimit);
	    t3 = delay + td->riseArr;
#if defined(printDelayValues)
		std::cout << " sl1:" << sl1 
		    <<  " SrcRiseArr:" << td->riseArr
		    << " previousDelay:" << tp->delay1 
		    << " newCalcDelay:" << delay
		    << " load:" << p->load
		    << " loadLimit" << tm->loadLimit << std::endl;
#endif
        } 
        else if (i->transition & TPointMaster::TRANSITION_SRC_FALL)
        {
	    //Original
	    delay = i->delay->lookup(p->load, sl2, tm->loadLimit);
	    t3 = delay + td->fallArr;
#if defined(printDelayValues)
		std::cout << " sl2:" << sl2   
		    <<  " SrcFallArr:" << td->fallArr
		    << " previousDelay:" << tp->delay2 
		    << " newCalcDelay:" << delay
		    << " load:" << p->load
		    << " loadLimit" << tm->loadLimit << std::endl;
#endif
        } else assert(0);
        if (i->transition & TPointMaster::TRANSITION_DST_RISE) {
            if (d->riseArr < t3)  {
                d->riseArr = t3;
            }
	    tp->riseDelay = delay;
	    if((i->transition & TPointMaster::TRANSITION_SRC_FALL)){
		tp->inverting = true;
	    }
#if defined(printDelayValues)
	    std::cout << " *** DST Rise Arr: " << d->riseArr << std::endl;
#endif
        }
        else if (i->transition & TPointMaster::TRANSITION_DST_FALL) {
            if (d->fallArr < t3) {
                d->fallArr = t3;
            }
	    tp->fallDelay = delay;
	    if((i->transition & TPointMaster::TRANSITION_SRC_RISE)){
		tp->inverting = true;
	    }
#if defined(printDelayValues)
	    std::cout << " *** DST Fall Arr: " << d->fallArr << std::endl;
#endif
        } else {
            assert(0);
        }
        /*if ( (i->transition & TPointMaster::TRANSITION_SRC_RISE) && 
		    (i->transition & TPointMaster::TRANSITION_DST_FALL)){
	    //std::cout << " SRC_RISE -> DST_FALL : INVERTING " << std::endl;
	    tp->inverting = true;
	}else if ( (i->transition & TPointMaster::TRANSITION_SRC_FALL) && 
		    (i->transition & TPointMaster::TRANSITION_DST_RISE)){
	    //std::cout << " SRC_FALL -> DST_RISE : INVERTING " << std::endl;
	    tp->inverting = true;
	}else if ( (i->transition & TPointMaster::TRANSITION_SRC_RISE) && 
		    (i->transition & TPointMaster::TRANSITION_DST_RISE)){
	    //std::cout << " SRC_RISE -> DST_RISE : NON-INVERTING " << std::endl;
	}else if ( (i->transition & TPointMaster::TRANSITION_SRC_FALL) && 
		    (i->transition & TPointMaster::TRANSITION_DST_FALL)){
	    std::cout << " SRC_FALL -> DST_FALL : NON-INVERTING " << std::endl;
	}*/
#define propagateSlew
	//d->riseSlew = 0.01; d->fallSlew = 0.01;

	// Output Slew Update
#if defined(propagateSlew)
        if (i->transition & TPointMaster::TRANSITION_DST_RISE) {
            if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
		//Original
		if (i->slew) {
		    sl3 = i->slew->lookup(p->load, sl1, tm->loadLimit);
		} else {
		    sl3 = sl1;
		}
            } 
            else if (i->transition & 
                     TPointMaster::TRANSITION_SRC_FALL) {
		// Original
		if (i->slew) {
		    sl3 = i->slew->lookup(p->load, sl2, tm->loadLimit);
		} else {
		    sl3 = sl2;
		}
            } else {
                assert(0);
            } 
            if (sl3 > d->riseSlew) {
                d->riseSlew = sl3;
            }
#if defined(printTol)
	    std::cout << " *DST Rise Slew: " << d->riseSlew << std::endl;
#endif
        } else if (i->transition & TPointMaster::TRANSITION_DST_FALL){
            if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
		//Original
		if (i->slew) {
		    sl3 = i->slew->lookup(p->load, sl1, tm->loadLimit);
		} else {
		    sl3 = sl1;
		}
            } else if (i->transition & 
                       TPointMaster::TRANSITION_SRC_FALL) {
		//Original
		if (i->slew) {
		    sl3 = i->slew->lookup(p->load, sl2, tm->loadLimit);
		} else {
		    sl3 = sl2;
		}
            } else {
                assert(0);
            } 
            if (sl3 > d->fallSlew) {
                d->fallSlew = sl3;
            }
#if defined(printTol)
	    std::cout << " *DST FALL Slew: " << d->fallSlew << std::endl;
#endif
        }
#endif
    }
}

/*
int
Timer::createNewClock(int x)
{
          propagatedClock *propClk1 = sdcParseData.propagatedClocks[x];
          clock *clk1 = propClk1->clk;
          std::string name = clk1->name + "/2";
          unsigned j, s = sdcParseData.propagatedClocks.size();
          for (j=0; j < s; j++) {   
            if (sdcParseData.propagatedClocks[j]->clk->name == name) {
              break;
            }
          } 
          if (j == s) {
            std::cout << "Creating a clock with half the period" << std::endl;
            clock *clk = new clock;
            clk->name = name;
            clk->period = clk1->period/2.0;
            propagatedClock *propClk = new propagatedClock;
            propClk->clk = clk;  
            propClk->m = propClk1->m;
            propClk->phase = false;
            sdcParseData.propagatedClocks.push_back(propClk);
          }
          return j;
}
*/


/*
  This function updates the arrival time for an instTerm which is 
  the output a path (in the fanin cone of a clock pin) based on 
  arrival times of the other instTerm which is the input of the path. 
  Returns is boolean to indicate if the path is inverting or not.
  \param tm pointer to the timer point master of the input of the path
  \param p pointer to the timer point of the input of the path 
  \param i path iterator
  \param tp pointer to the timer point of the output of the path 
*/
bool
Timer::updateClockCellPathArr(TPointMaster *tm,
            TPoint *p, 
            TPointMaster::pathVector::iterator i, 
            TPoint *tp) 
{
#if defined(DEBUG)
    std::cout << " updateClockCellPathArr Time" << std::endl;
#endif
    bool phase = false;
    if (!tp->multiClockData.size()) {
        return phase;
    }
    timingData *d, *td = tp->multiClockData[0];
    TimeType t3, t4;
    DelayType sl1, sl2, sl3, delay;
    if (!p->multiClockData.size()) {
        d = new timingData;
        d->clockIndex = td->clockIndex;
        d->riseArr = d->fallArr = -ModelType::MAX_TIME();  
        d->riseEarly = d->fallEarly = ModelType::MAX_TIME();  
        d->riseSlew = d->fallSlew = ModelType::ZERO_DELAY();  
        insertTimingData(p->multiClockData, d);
    } else {
        d = p->multiClockData[0];
        if (td->clockIndex != d->clockIndex) {
	    if (Timer::displayWarning)
	      std::cout << "Warning: both phases of clock at the clock pin." 
                      << std::endl;
            //d->clockIndex = createNewClock(td->clockIndex);
        }
    } 
    d = p->multiClockData[0];
    sl1 = td->riseSlew;
    sl2 = td->fallSlew;
    //bool CLOCK_PATH_DELAY = false;
    bool CLOCK_PATH_DELAY = true;
    
    if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
        if (CLOCK_PATH_DELAY) {
            delay = i->delay->lookup(p->load, sl1, tm->loadLimit);
        } else {
            delay = ModelType::ZERO_DELAY();
        } 
        t3 = delay + td->riseArr;
        t4 = delay + td->riseEarly;
#if defined(DEBUG_CLOCK)
	    std::cout << "**CLOCK Cell delay SRC_RISE**" << std::endl;
	    std::cout << "delay:" << delay << "(sl:" << sl1 << "," << "load:" << p->load << std::endl;
	    std::cout << "t3:" << t3 << " = " << delay <<  " + " << td->riseArr << std::endl;
	    std::cout << "t4:" << t4 << " = " << delay <<  " + " << td->riseEarly << std::endl;
#endif
    } 
    else if (i->transition & TPointMaster::TRANSITION_SRC_FALL) {
        phase = !phase;
        if (CLOCK_PATH_DELAY) {
            delay = i->delay->lookup(p->load, sl2, tm->loadLimit);
        } else {
            delay = ModelType::ZERO_DELAY();
        } 
        t3 = delay + td->fallArr;
        t4 = delay + td->fallEarly;
#if defined(DEBUG_CLOCK)
	    std::cout << "**CLOCK Cell delay SRC_RISE**" << std::endl;
	    std::cout << "delay:" << delay << "(sl:" << sl2 << "," << "load:" << p->load << std::endl;
	    std::cout << "t3:" << t3 << " = " << delay <<  " + " << td->riseArr << std::endl;
	    std::cout << "t4:" << t4 << " = " << delay <<  " + " << td->riseEarly << std::endl;
#endif

    } else {
        assert(0);
    }
    if (i->transition & TPointMaster::TRANSITION_DST_RISE) {
        if (d->riseArr < t3) {
            d->riseArr = t3;
        }
        if (d->riseEarly > t4) {
            d->riseEarly = t4;
        }
    }
    else if (i->transition & TPointMaster::TRANSITION_DST_FALL) {
        phase = !phase;
        if (d->fallArr < t3) {
            d->fallArr = t3;
        }
        if (d->fallEarly > t4) {
            d->fallEarly = t4;
        }
    } else {
        assert(0);
    }

    if (i->transition & TPointMaster::TRANSITION_DST_RISE) {
        if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
            if (i->slew) {
                sl3 = i->slew->lookup(p->load, sl1, tm->loadLimit);
            } else {
                sl3 = sl1;
            }
        } 
        else if (i->transition & TPointMaster::TRANSITION_SRC_FALL) {
            if (i->slew) {
                sl3 = i->slew->lookup(p->load, sl2, tm->loadLimit);
            } else {
                sl3 = sl2;
            }
        } else {
            assert(0);
        } 
        if (sl3 > d->riseSlew) {
            d->riseSlew = sl3;
        }
    } else if (i->transition & TPointMaster::TRANSITION_DST_FALL) {
        if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
            if (i->slew) {
                sl3 = i->slew->lookup(p->load, sl1, tm->loadLimit);
            } else {
                sl3 = sl1;
            }
        } else if (i->transition & TPointMaster::TRANSITION_SRC_FALL){
            if (i->slew) {
                sl3 = i->slew->lookup(p->load, sl2, tm->loadLimit);
            } else {
                sl3 = sl2;
            }
        } else {
            assert(0);
        } 
        if (sl3 > d->fallSlew) {
            d->fallSlew = sl3;
        }
    }
    return phase;
}




/*
  Updating the arrival time a node on a net (in the fanin cone of a clock pin).
  based on the arrival time of driver of the net.
  \param p pointer to the timer point of the node 
  \param tp pointer to the timer point of the driver of the net 
*/
void
Timer::updateClockNetPathArr(TPoint *p, TPoint *tp)
{
#if defined(DEBUG)
    std::cout << "update Clock Net Path Arrival Time" << std::endl;
#endif
    if (!tp->multiClockData.size()) {
        return;
    } 
    timingData *d, *td = tp->multiClockData[0];
    if (!p->multiClockData.size()) {
        d = new timingData;
        d->clockIndex = td->clockIndex;
        d->riseArr = td->riseArr;
        d->fallArr = td->fallArr;
        d->riseSlew = td->riseSlew;
        d->fallSlew = td->fallSlew;
        d->riseEarly = td->riseEarly;
        d->fallEarly = td->fallEarly;
        insertTimingData((p->multiClockData), d);
    } else {
        d = p->multiClockData[0];
        if (td->clockIndex != d->clockIndex) {
            if (Timer::displayWarning) 
	      std::cout << "Warning: both phases of clock at the clock pin." 
                      << std::endl;
            //d->clockIndex = createNewClock(td->clockIndex);
        }
        if (d->riseArr  < td->riseArr) {
            d->riseArr  = td->riseArr;
        }
        if (d->fallArr  < td->fallArr) {
            d->fallArr  = td->fallArr;
        }
        if (d->riseEarly  > td->riseEarly) {
            d->riseEarly  = td->riseEarly;
        }
        if (d->fallEarly  > td->fallEarly) {
            d->fallEarly  = td->fallEarly;
        }
        if (d->riseSlew < td->riseSlew) {
            d->riseSlew = td->riseSlew;
        }
        if (d->fallSlew < td->fallSlew) {
            d->fallSlew = td->fallSlew;
        }
    } 
}

/*
  Updating the arrival time a node on a net based on 
  the arrival time of driver of the net.
  \param p pointer to the timer point of the node 
  \param tp pointer to the timer point of the driver of the net 
*/
void
Timer::updateNetPathArr(TPoint *p, TPoint *tp)
{
#if defined(DEBUG)
    std::cout << "update Net Path Arrival Timer" << std::endl;
#endif
    for (unsigned k = 0; k < tp->multiClockData.size(); ++k) { 
        timingData *td = tp->multiClockData[k];
        int x = findTimingData (p, td->clockIndex);
        if (x == -1) {
            timingData *d = new timingData;
            d->clockIndex = td->clockIndex;
            d->riseArr = td->riseArr;
            d->fallArr = td->fallArr;
            d->riseSlew = td->riseSlew;
            d->fallSlew = td->fallSlew;
            insertTimingData((p->multiClockData), d);
	    //delete d;
        } else {
            timingData *d = p->multiClockData[x];
            if (d->riseArr  < td->riseArr) {
                d->riseArr  = td->riseArr;
            }
            if (d->fallArr  < td->fallArr) {
                d->fallArr  = td->fallArr;
            }
            if (d->riseSlew < td->riseSlew) {
                d->riseSlew = td->riseSlew;
            }
            if (d->fallSlew < td->fallSlew) {
                d->fallSlew = td->fallSlew;
            }

	    // Incremental Timing Optimization Updated(08/26/2010) by:Santiago
	    if(tp->riseSlewValid && !p->swapFlag){
#if defined(printTol)
		std::cout << "NetPath: Propagate riseSlewValid" << std::endl;
#endif
		p->riseSlewValid = true;
		tp->riseSlewValid = false;
	    }
	    if(tp->fallSlewValid && !p->swapFlag){
#if defined(printTol)
		std::cout << "NetPath: Propagate fallSlewValid" << std::endl;
#endif
		p->fallSlewValid = true;
		tp->fallSlewValid = false;
	    }
        }
    }
}

/*!
  Returns the net of a block object
  FIXME should move out of Timer class???
  \param block the pointer to the block object
*/
oaOccNet *
Timer::getOccNet(oaOccObject *block)
{
    TPoint *p = TPoint::get(block);
    if (p->isPI() || p->isPO() || p->type == TIMING_POINT_PI_CLOCK) {
        return (static_cast<oaOccTerm*>(block))->getNet();
    } else {
        return (static_cast<oaOccInstTerm*>(block))->getNet();
    }
}

/*!
  Returns the greatest common divider of two numbers
  FIXME should move out of Timer class
  \param d1 first number
  \param d2 second number
*/
DelayType
Timer::greatestCommonDivider(DelayType d1, DelayType d2)
{
    if (d1 == ModelType::ZERO_DELAY() ||
        d2 == ModelType::ZERO_DELAY()) {
        return d1+d2;
    }
    if (d1>d2) {
        return greatestCommonDivider(d1-d2, d2);
    } else {
        return greatestCommonDivider(d2-d1, d1);
    }
}

/*!
  Initializes the required times of a block object based on the clock periods
  and also considers the external delays.
  \param block the pointer to the block object
*/

void
Timer::initPOReq(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "initPOReq(" << getBlockName(block) << ")" << std::endl;
#endif
    TPoint *p = TPoint::get(block);
    if (p->isPO()) {
        int outClockIndex = getClockIndex(block);
        assert(outClockIndex != -1);
        DelayType outPeriod = 
            sdcParseData.propagatedClocks[outClockIndex]->clk->period;
        for (unsigned int j = 0; j < p->multiClockData.size(); ++j) {
            timingData *d = p->multiClockData[j];
            propagatedClock *inPropClk = 
                sdcParseData.propagatedClocks[d->clockIndex];
            bool inPhase = inPropClk->phase;
            int m = inPropClk->m;
            DelayType inPeriod = inPropClk->clk->period;
            DelayType d1 = 1.0;
            if (inPhase) {
                d1 = 2.0;
            }
            DelayType gcdPeriod = greatestCommonDivider(inPeriod/d1, outPeriod);
            d->riseReq = d->fallReq = ModelType::ZERO_TIME() + gcdPeriod + outPeriod * (m-1); /* OK */
            TimerExtDelay *dp = static_cast<TimerExtDelay*>
            (TimerExtDelay::termAppDef->get(static_cast<oaOccTerm*>(block)));
            if (dp) {
                d->riseReq = d->riseReq - dp->delay;
                d->fallReq = d->fallReq - dp->delay;
            }
        }
	//std::cout << "*ratValid @ initPOReq1(" << getBlockName(block) << ")" << std::endl;
        p->ratValid = true;
    } else if (p->type == TIMING_POINT_CLOCK) {
        TPoint *p1 = TPoint::getClock(block);
        assert(p1);
        if (!p1->atValid) {
            updateClockArr(block);
        }
        assert(p1->multiClockData[0]);
        for (unsigned k = 0; k < p->multiClockData.size(); ++k) {
            p->multiClockData[k]->riseReq = p1->multiClockData[0]->riseEarly;
            p->multiClockData[k]->fallReq = p1->multiClockData[0]->fallEarly;
        }
        int outClockIndex = p1->multiClockData[0]->clockIndex;
        propagatedClock *outPropClk = 
            sdcParseData.propagatedClocks[outClockIndex];
        assert(outPropClk);
        bool outPhase = outPropClk->phase;
        assert(outPropClk->clk);
        DelayType outPeriod = outPropClk->clk->period;
        for (unsigned int j = 0; j < p->multiClockData.size(); ++j) {
            timingData *d = p->multiClockData[j];
            assert(d);
            propagatedClock *inPropClk = 
                sdcParseData.propagatedClocks[d->clockIndex];
            assert(inPropClk);
            bool inPhase = inPropClk->phase;
            int m = inPropClk->m;
            assert(inPropClk->clk);
            DelayType inPeriod = inPropClk->clk->period;
            DelayType d1 = 1.0, d2 = 1.0;
	    //std::cout << "inPeriod:" << inPeriod << " outPeriod:" << outPeriod << std::endl;
            if (inPhase != outPhase) {
	    //std::cout << "inPhase:" << inPhase << std::endl;
                if (inPhase) {
                    d1 = 2.0;
                } else {
                    d2 = 2.0;
                }
            }
            DelayType gcdPeriod = greatestCommonDivider(inPeriod/d1, outPeriod/d2);
	    /*std::cout << "riseReq:" << d->riseReq << " fallReq:" << d->fallReq << std::endl;
	    std::cout << " gcdPeriod:" << gcdPeriod << " outPeriod" << outPeriod << " m:" << m << std::endl;*/
            d->riseReq = d->riseReq + gcdPeriod + outPeriod * (m-1);
            d->fallReq = d->fallReq + gcdPeriod + outPeriod * (m-1);
	    //std::cout << " riseReq:" << d->riseReq << " fallReq:" << d->fallReq << std::endl;
        }
	//std::cout << "*ratValid @ initPOReq2(" << getBlockName(block) << ")" << std::endl;
        p->ratValid = true;
    }
#if defined(DEBUG)
    std::cout << "EXIT initPOReq(" << getBlockName(block) << ")" << std::endl;
#endif
}

/*!
  Initializes the required times of a block object.
  \param block the pointer to the block object
*/
void
Timer::initReq(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "initReq(" << getBlockName(block) << ")" << std::endl;
#endif
    TPoint *p = TPoint::get(block);
    for (unsigned int j = 0; j < p->multiClockData.size(); ++j) {
        timingData *d = p->multiClockData[j];
        d->riseReq = d->fallReq = ModelType::MAX_TIME();
    }
}
/*! 
 * Get the period define in the timing constraint file (sdc)
 */
double
Timer::getPeriod()
{
    return sdcParseData.propagatedClocks[0]->clk->period;
    /*oaOccurrence *occ = dsgn->getTopOccurrence();
    assert(occ);
    oaIter<oaOccTerm> termIter(occ->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
      TPoint *p = TPoint::get(i); 
      std::cout << "getPeriod(" << getBlockName(i) << ")" << std::endl;
      if (p->isPO()) {
        int outClockIndex = getClockIndex(i);
        assert(outClockIndex != -1);
        DelayType outPeriod = 
            sdcParseData.propagatedClocks[outClockIndex]->clk->period;
	std::cout << " index:" << outClockIndex << " outPeriod:" << outPeriod << std::endl;
        for (unsigned int j = 0; j < p->multiClockData.size(); ++j) {
            timingData *d = p->multiClockData[j];
            propagatedClock *inPropClk = 
                sdcParseData.propagatedClocks[d->clockIndex];
            bool inPhase = inPropClk->phase;
            int m = inPropClk->m;
            DelayType inPeriod = inPropClk->clk->period;
	    std::cout << " inPeriod:" << inPeriod << std::endl;
            DelayType d1 = 1.0;
            if (inPhase) {
                d1 = 2.0;
            }
            DelayType gcdPeriod = greatestCommonDivider(inPeriod/d1, outPeriod);
            d->riseReq = d->fallReq = ModelType::ZERO_TIME() + gcdPeriod + outPeriod * (m-1);
            TimerExtDelay *dp = static_cast<TimerExtDelay*>
            (TimerExtDelay::termAppDef->get(static_cast<oaOccTerm*>(i)));
            if (dp) {
                d->riseReq = d->riseReq - dp->delay;
                d->fallReq = d->fallReq - dp->delay;
            }
	    std::cout << " riseReq:" << d->riseReq 
		<< " fallReq:" << d->fallReq << std::endl;
        }
      }
    }*/
}
/*!
  Updating the required time of the driver node of a net based on 
  the required time of another node on the net.
  \param p pointer to the timer point of the driver of the net 
  \param tp pointer to the timer point of the other node 
*/
void
Timer::updateNetPathReq(TPoint *p, TPoint *tp)
{
#if defined(DEBUG)
    std::cout << "updateNetPathRequire Arrival Time" << std::endl;
#endif
    for (unsigned k = 0; k < p->multiClockData.size(); ++k) {
        timingData *d = p->multiClockData[k];
        int x = findTimingData (tp, d->clockIndex);
        assert(x != -1);
        timingData *td = tp->multiClockData[x];
        if (d->riseReq > td->riseReq - tp->netDelay) {
            d->riseReq = td->riseReq - tp->netDelay;
        }
        if (d->fallReq > td->fallReq - tp->netDelay) {
            d->fallReq = td->fallReq - tp->netDelay;
        }
    }
#if defined(DEBUG)
    std::cout << "EXIT updateNetPathReq" << std::endl;
#endif
}
/*!
  returns the clock index of a block object.
  \param block the pointer to the block object
 */
int
Timer::getClockIndex(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "get Clock Index" << std::endl;
#endif
    oaOccTerm *term;
    if (block->isOccTerm()) {
        term = static_cast<oaOccTerm*>(block);
        if (term->getTermType() == oacInputTermType) {
            if (sdcParseData.allInputs) {
                return sdcParseData.allInputsClock;
            }
        } else {
            if (sdcParseData.allOutputs) {
                return sdcParseData.allOutputsClock;
            }
        } 
    } else {
        oaOccNet *net = (static_cast<oaOccInstTerm*>(block))->getNet();
        oaIter<oaOccTerm> termIter(net->getTerms());
        while ((term = termIter.getNext())) {
            if (term->getTermType() == oacInputTermType) {
                break;
            }
        }
    }
    assert(term);
    oaString f;
    term->getName(oaNativeNS(), f);
    termIntMap::iterator it = 
        sdcParseData.termClock.find(std::string(f));
    if (it != sdcParseData.termClock.end()) {
        return (*it).second;
    } 
    if (!sdcParseData.propagatedClocks.size()) {
        clock *clk = new clock;
        clk->period = 0.0;  /* PROBLEM */
        clk->name = "default";
        propagatedClock *propClk = new propagatedClock;
        propClk->clk = clk;
        propClk->m = 1;
        propClk->phase = false;
        sdcParseData.propagatedClocks.push_back(propClk);
    }
    return 0;
}

/*!
  This function updates the arrival time for an instTerm which is the output 
  or clock port of a cell (in the fanin cone of a clock pin) based on arrival 
  times of other instTerms which are inputs or clock ports of the cell. 
  \param block the pointer to the block object
*/
void
Timer::updateClockCellArr(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "updateClockCellArr(" << getBlockName(block) << ")" << std::endl;
#endif
    updateNetLoadDelay(block);
    TPoint *p = TPoint::getClock(block);
    oaModTerm *masterTerm = (static_cast<oaOccInstTerm*>(block))
            ->getModInstTerm()->getTerm();
    assert(masterTerm);
    TPointMaster *tm = TPointMaster::get(masterTerm);
    assert(tm);
    oaOccInst *inst = (static_cast<oaOccInstTerm*>(block))->getInst();
    assert(inst);
    TPointMaster::pathVector::iterator i;
    unsigned phaseCount = 0;
    for (i = tm->inPaths.begin(); i != tm->inPaths.end(); ++i) {
        assert(i->other);
        oaOccInstTerm *other = getOccFromMod(inst, i->other);
        if (!other) continue;
        TPoint *otherTP = TPoint::getClock(other); 
        assert(otherTP);
        bool phase;
#if defined(DEBUG_CLOCK)
	std::cout << "(p:" << getBlockName(block) << ", tp:" << getBlockName(other) << std::endl;
#endif
        if (otherTP->type == TIMING_POINT_CLOCK) { 
            TPoint *tp = TPoint::getClock(other);
            if (!tp->atValid) {
                initClockPIArr(other);
            }
            phase = updateClockCellPathArr(tm, p, i, tp);
        } else {
            if (!otherTP->atValid) {
                updateClockArr(other);
            }
            phase = updateClockCellPathArr(tm, p, i, otherTP);
        }
        if (phase) {
            phaseCount++;
        }
    }
    if (phaseCount > 0) {
        timingData *d = p->multiClockData[0];
        propagatedClock *propClk1 = sdcParseData.propagatedClocks[d->clockIndex];
        bool phase = !propClk1->phase;
        if (phaseCount != tm->inPaths.size()) {
            if (Timer::displayWarning)
	      std::cout << "Warning: both phases of clock at the clock pin." 
                      << std::endl;
            //createNewClock(p->multiClockData[0]);
        } else {
            unsigned j, s = sdcParseData.propagatedClocks.size();
            for (j=0; j < s; j++) {   
                if (sdcParseData.propagatedClocks[j]->clk->name == 
                    propClk1->clk->name && 
                    sdcParseData.propagatedClocks[j]->m == propClk1->m && 
                    sdcParseData.propagatedClocks[j]->phase == phase) {
                    break;
                }
            }
            if (j == s) {
                propagatedClock *propClk = new propagatedClock;
                propClk->clk = propClk1->clk;
                propClk->m = propClk1->m;
                propClk->phase = phase;
                sdcParseData.propagatedClocks.push_back(propClk);
            }
            d->clockIndex = j;
        }
    }
    p->atValid = true;
}

/*!
  This function updates the arrival time for a block object which is
  an instTerm which is the output or clock port of a cell based on 
  arrival times of other instTerms which are inputs or clock ports of the cell. 
  \param block the pointer to the block object
*/
void
Timer::updateCellArr(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "updateCellArr(" << getBlockName(block) << ")" << std::endl;
#endif
    updateNetLoadDelay(block);
    TPoint *p = TPoint::get(block);
    assert(p);
    oaModTerm *masterTerm = (static_cast<oaOccInstTerm*>(block))->getModInstTerm()->getTerm();
    TPointMaster *tm = TPointMaster::get(masterTerm);
    assert(masterTerm);
    assert(tm);
    oaOccInst *inst = (static_cast<oaOccInstTerm*>(block))->getInst();
    assert(inst);
    
    TPointMaster::pathVector::iterator i;
    for (i = tm->inPaths.begin(); i != tm->inPaths.end(); ++i) {
        assert(i->other);
        oaOccInstTerm *other = getOccFromMod(inst, i->other);
        if (!other) continue;
        TPoint *otherTP = TPoint::get(other); 
#if defined(printDelayValues)
	    std::cout << "updateCellArr(p:" << getBlockName(block) << ",tp:" << getBlockName(other) << ")" << std::endl;
#endif
        if (otherTP->type == TIMING_POINT_CLOCK) { 
            TPoint *tp = TPoint::getClock(other);
            if (!tp->atValid) {
                updateClockArr(other);
            }
	    if(optIncrTimer){
		//std::cout << " In function updateCellArr() before going into CLOCK updateCellPathArr(p:" << getBlockName(block) << ",tp:" << getBlockName(other) << ")" << std::endl;
	    }
            updateCellPathArr(tm, p, i, tp);
        } else {
            updateArr(other);
            updateCellPathArr(tm, p, i, otherTP);
        }
    }
    p->atValid = true;
    //To be remove
#if defined(DEBUG)
    std::cout << "EXIT updateCellArr(" << getBlockName(block) << ")" << std::endl;
#endif
}


/*!
  This function updates the required time for a block object which is an 
  instTerm which is the input or clock port of a cell based on required times 
  of other instTerms which are outputs or clock ports of the cell. 
  \param block the pointer to the block object
*/
void
Timer::updateCellReq(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "updateCellReq(" << getBlockName(block) << ")" << std::endl;
#endif
    TPoint *p = TPoint::get(block);
    assert(p);
    TPoint *p1 = NULL;
    if (p->type == TIMING_POINT_CLOCK) {
        p1 = TPoint::getClock(block);
        assert(p1);
        timingData *d = p1->multiClockData[0];
        d->riseReq = d->fallReq = ModelType::MAX_TIME(); 
    } else {
        initReq(block);
    }
    oaModTerm *masterTerm = (static_cast<oaOccInstTerm*>(block))
        ->getModInstTerm()->getTerm();
    assert(masterTerm);
    TPointMaster *tm = TPointMaster::get(masterTerm);
    assert(tm);
    oaOccInst *inst = (static_cast<oaOccInstTerm*>(block))->getInst();
    assert(inst);
    TPointMaster::pathVector::iterator i;
    for (i = tm->outPaths.begin(); i != tm->outPaths.end(); ++i) {
        assert(i->other);
        oaOccInstTerm *other = getOccFromMod(inst, i->other);
        if (!other) continue;
        TPoint *otherTP = TPoint::get(other); 
        if (!otherTP->atValid) {
            continue;
        }
        if (otherTP->type == TIMING_POINT_CLOCK) {
            if (!otherTP->ratValid) {
                initPOReq(other);
            }
        } else {
            updateReq(other);
        }
#if defined(DEBUG_CELLREQ)
	std::cout << "updateCellPathReq(p:" << getBlockName(block) << ",tp:" << getBlockName(other) << ")" << std::endl;
#endif
        if (p->type == TIMING_POINT_CLOCK) { 
            p1 = TPoint::getClock(block);
            updateCellPathReq(i, p1, other);
        } else {
            updateCellPathReq(i, p, other);
        }
    }
    //std::cout << "ratValid @updateCellReq(" << getBlockName(block) << ")" << std::endl;
    if (p->type == TIMING_POINT_CLOCK) {
        p1->ratValid = true;
    } else {
        p->ratValid = true;
    }
#if defined(DEBUG)
    std::cout << "EXIT updateCellReq(" << getBlockName(block) << ")" << std::endl;
#endif
}

/*!
  This function updates the required time for a block object which is 
  a primary input or an instTerm which is the output of a cell based 
  on required times of other instTerms or Terms on its net.
  \param block the pointer to the block object
*/
void
Timer::updateNetReq(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "updateNetReq(" << getBlockName(block) << ")" << std::endl;
#endif
    initReq(block);
    updateNetLoadDelay(block);
    TPoint *p = TPoint::get(block);
    oaOccNet *net = getOccNet(block);
    oaIter<oaOccTerm> termIter(net->getTerms());
    while (oaOccTerm *j = termIter.getNext()) {
        TPoint *tp = TPoint::get(j);
        if (tp->type == TIMING_POINT_PO) {
            if (tp->atValid) {
                updateReq(j);
                updateNetPathReq(p, tp);
            }
        }
    }
    oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
    while (oaOccInstTerm *j = instTermIter.getNext()) {
        oaOccTerm *masterTerm = j->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacInputTermType) {
	    oaNativeNS NS;
	    oaString termName, instName;
            TPoint *tp = TPoint::get(j);
            if (!tp->isClock()) {
                if (tp->atValid) {
                    updateReq(j);
                    updateNetPathReq(p, tp);
                }
            }
        }
    }
    //std::cout << "ratValid @updateNetReq(" << getBlockName(block) << ")" << std::endl;
    p->ratValid = true;
#if defined(DEBUG)
    std::cout << "EXIT updateNetReq(" << getBlockName(block) << ")" << std::endl;
#endif
}


/*!
  This function updates the arrival time for a primary output or
  an instTerm which is the input of a cell (in the fanin cone of a clock pin)
  based on arrival times of other instTerms or Terms on its net
  \param block the pointer to the block object
*/
/*void
Timer::updateClockNetArr(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "updateClockNetArr(" << getBlockName(block) << ")" << std::endl;
#endif
    updateNetLoadDelay(block);
    TPoint *p = TPoint::getClock(block);
    oaOccNet *net = getOccNet(block);
    oaIter<oaOccTerm> termIter(net->getTerms());
    while (oaOccTerm *j = termIter.getNext()) {
        TPoint *tp = TPoint::getClock(j); 
        assert(tp);
        if ((tp->type == TIMING_POINT_PI) ||
                (tp->type == TIMING_POINT_PI_CLOCK)) {
            if (!tp->atValid) {
                updateClockArr(j);
            }
            TPoint *tp = TPoint::get(j);
            if (!tp->isClock()) {
                if (tp->atValid) {
                    updateReq(j);
                    updateNetPathReq(p, tp);
                }
            }
        }
    }
    p->ratValid = true;
}*/


/*!
  This function updates the arrival time for a primary output or
  an instTerm which is the input of a cell (in the fanin cone of a clock pin)
  based on arrival times of other instTerms or Terms on its net
  \param block the pointer to the block object
*/
void
Timer::updateClockNetArr(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "updateClockNetArr(" << getBlockName(block) << ")" << std::endl;
#endif
    updateNetLoadDelay(block);
    TPoint *p = TPoint::getClock(block);
    oaOccNet *net = getOccNet(block);
    oaIter<oaOccTerm> termIter(net->getTerms());
    while (oaOccTerm *j = termIter.getNext()) {
        TPoint *tp = TPoint::getClock(j); 
        assert(tp);
        if ((tp->type == TIMING_POINT_PI) ||
                (tp->type == TIMING_POINT_PI_CLOCK)) {
            if (!tp->atValid) {
                updateClockArr(j);
            }
            updateClockNetPathArr(p, tp);
        }
    }
    oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
    while (oaOccInstTerm *j = instTermIter.getNext()) {
        oaOccTerm *masterTerm = j->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacOutputTermType) {
            TPoint *tp = TPoint::getClock(j); 
            assert(tp);
            if (!tp->atValid) {
                updateClockArr(j);
            }
            updateClockNetPathArr(p, tp);
        }
    }
    bool CLOCK_NET_DELAY = false;
    if (CLOCK_NET_DELAY) {
        for (unsigned int k = 0; k < p->multiClockData.size(); ++k) { 
            timingData *d = p->multiClockData[k];
            d->riseArr = d->riseArr + p->netDelay;
            d->fallArr = d->fallArr + p->netDelay;
        }
    }
    p->atValid = true;
    //To be remove
#if defined(DEBUG)
    std::cout << "EXIT updateClockNetArr(" << getBlockName(block) << ")" << std::endl;
#endif
}

/*!
  This function updates the arrival time for a block object which is 
  a primary output or an instTerm which is the input of a cell 
  based on arrival times of other instTerms or Terms on its net.
  \param block the pointer to the block object
*/
void
Timer::updateNetArr(oaOccObject *block)
{
#if defined(DEBUG)
  std::cout << "updateNetArr(" << getBlockName(block) << ")" << std::endl;   
#endif
    updateNetLoadDelay(block);
    TPoint *p = TPoint::get(block);
    oaOccNet *net = getOccNet(block);
    oaIter<oaOccTerm> termIter(net->getTerms());
    while (oaOccTerm *j = termIter.getNext()) {
        TPoint *tp = TPoint::get(j); 
        if ((tp->type == TIMING_POINT_PI) ||
            (tp->type == TIMING_POINT_PI_CLOCK)) {
            updateArr(j);
#if defined(printTol)
	    std::cout << " In function updateNetArr() before going into PI/Clock PI updateNetPathArr(p:" << getBlockName(block) << ",tp:" << getBlockName(j) << ")" << std::endl;
#endif
            updateNetPathArr(p, tp);
        }
    }
    oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
    while (oaOccInstTerm *j = instTermIter.getNext()) {
        oaOccTerm *masterTerm = j->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacOutputTermType) {
            TPoint *tp = TPoint::get(j); 
	    assert(tp);
            updateArr(j);
	    if(p->isPO()) {
#if defined(printTol)
		std::cout << " **** updateNetArr(" << getBlockName(block) << ") removing tp(" << getBlockName(j) << ") rise/fall flag" << std::endl;
#endif
		tp->riseSlewValid = false;
		tp->fallSlewValid = false;
	    }
#if defined(printTol)
	    std::cout << " In function updateNetArr() before going into SIGNAL_OUT PI updateNetPathArr(p:" << getBlockName(block) << ",tp:" << getBlockName(j) << ")" << std::endl;
#endif
            updateNetPathArr(p, tp);
        }
    }
    for (unsigned k = 0; k < p->multiClockData.size(); ++k) { 
        timingData *d = p->multiClockData[k];
	d->riseArr = d->riseArr + p->netDelay;
	d->fallArr = d->fallArr + p->netDelay;
    }
    p->atValid = true;
    //To be remove
#if defined(DEBUG)
  std::cout << "EXIT updateNetArr(" << getBlockName(block) << ")" << std::endl;   
#endif
}

/*!
  Updates the load and delay of the net and all nodes on the net of a 
  block object.
  \param block the pointer to the block object
*/
#define Constant 1.0
void
Timer::updateNetLoadDelay(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "updateNetLoadDelay(" << getBlockName(block) << ")" << std::endl;   
#endif
    oaOccNet *net = getOccNet(block);
    assert(net);
    bool vFlag = false;
    bool netDelayValid = false;
    double netLoad = 0.0;
    double netRes = 0.0;
    DelayType netDelay = ModelType::ZERO_DELAY();

    // Query Load at PO
    oaIter<oaOccTerm> termIter(net->getTerms());
    while (oaOccTerm *j = termIter.getNext()) {
        if (j->getTermType() == oacOutputTermType) {
            TPoint *op = TPoint::get(j); 
	    DelayType s = op->getRiseSlew();
            if (op->atValid) {
                vFlag = true;
                break;
            }
            TimerFanoutLoad *fp = 
                static_cast<TimerFanoutLoad*>
                (TimerFanoutLoad::termAppDef->get(j));
            if (fp) {
                netLoad += fp->load;
            }
        }
    } 

    // Query input cap load
    oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
    while (oaOccInstTerm *j = instTermIter.getNext()) {
        TPoint *op = TPoint::get(j);
        oaModTerm *masterTerm = j->getModInstTerm()->getTerm();
        assert(masterTerm);
	DelayType s = op->getRiseSlew();
        if (masterTerm->getTermType() == oacInputTermType) {
	    if(op->netValid) netDelayValid = true;
            if (op->atValid) {
                vFlag = true;
                break;
            }
            TPointMaster *tm = TPointMaster::get(masterTerm);
            assert(tm);
            netLoad += tm->cap;
        } else if (masterTerm->getTermType() == oacOutputTermType) {
	    if(op->netValid) netDelayValid = true;
            if (op->atValid) {
                vFlag = true;
                break;
            }
        }
    }
    if (!vFlag) {
	if(!elmoreAnalysis){
	    if (_wireModel) {
		oaNet *blockNet = net->getNet();
		netLoad += _wireModel->getWireCap(blockNet);
		netDelay = _wireModel->getWireDelay(blockNet);
	    }
	    termIter = net->getTerms();
	    while (oaOccTerm *j = termIter.getNext()) {
		TPoint *tp = TPoint::get(j); 
		if (tp->type == TIMING_POINT_PI)
		    tp->load = netLoad;
		else if (tp->type == TIMING_POINT_PO){
		    tp->netDelay = netDelay;
		}
	    }
	    instTermIter = net->getInstTerms();
	    while (oaOccInstTerm *j = instTermIter.getNext()) {
		oaOccTerm *masterTerm = j->getTerm();
		assert(masterTerm);
		TPoint *tp = TPoint::get(j);
		oaTermType type(masterTerm->getTermType());
		if (type == oacOutputTermType)
		    tp->load = netLoad;
		else if (type == oacInputTermType){
		    tp->netDelay = netDelay;
		}
	    }
	}else{
	    // Get Elmore Interconnect delay
	    
	    // Do not recompute wire delay if net delay is valid (due to inherent bug in OA)
	    // Just update the loading:
	    // Assign net+input cap load to PI and 
	    // Assign net+input cap load to Output Term
	    if(netDelayValid){
		//cout << "NetDelayValid" << std::endl;
		termIter = net->getTerms();
		while (oaOccTerm *j = termIter.getNext()) {
		    TPoint *tp = TPoint::get(j); 
		    if (tp->type == TIMING_POINT_PI){
			tp->load = netLoad + tp->netLoad;
			/*std::cout << getBlockName(j) << "(load:"
			    << tp->load <<"="<< netLoad << "+" << tp->netLoad << ")" << std::endl;*/
		    }
		}
		instTermIter = net->getInstTerms();
		while (oaOccInstTerm *j = instTermIter.getNext()) {
		    oaOccTerm *masterTerm = j->getTerm();
		    assert(masterTerm);
		    TPoint *tp = TPoint::get(j);
		    oaTermType type(masterTerm->getTermType());
		    if (type == oacOutputTermType){
			tp->load = netLoad + tp->netLoad;
			/*std::cout << getBlockName(j) << "(load:"
			    << tp->load <<"="<< netLoad << "+" << tp->netLoad << ")" << std::endl;*/
		    }	
		}
		return;
	    }

	    //Otherwise perform reform elmore calculation
	    //cout << "Elmore delay calc ..." << endl;
	    _outputMap outputPin; 
	    _inputMap inputPin;
	    double elmoreNetLoad = 0.0;
	    if (_elmoreWireModel) {
		oaNet *blockNet = net->getNet();
		//netRes = _elmoreWireModel->getWireRes(blockNet);
		elmoreNetLoad = _elmoreWireModel->getWireCap(blockNet);
		netDelay = _elmoreWireModel->getWireDelay(blockNet,outputPin,inputPin);
	    }
	    //Assign net+input cap load to PI and 
	    //Assign net delay to PO
	    termIter = net->getTerms();
	    while (oaOccTerm *j = termIter.getNext()) {
		oaTerm *term = j->getTerm();
		_outputMap::iterator tIter;
		TPoint *tp = TPoint::get(j); 
		if (tp->type == TIMING_POINT_PI){
		    /*std::cout << " " << getBlockName(j) << ":" 
		      << tp->netLoad << "+" << netLoad 
			<< "=" << netLoad+tp->netLoad << std::endl;*/
		    tp->netLoad = elmoreNetLoad;
		    tp->load = netLoad + tp->netLoad;
		    tp->netValid = true;
		}
		else if (tp->type == TIMING_POINT_PO){
		    tIter = outputPin.find(term);
		    if(tIter != outputPin.end()){
			tp->netDelay = Constant*tIter->second;
		    }else { tp->netDelay = Constant*netDelay; }
		}
		//tp->netValid = true;
	    }
	    
	    // Assign net+input cap load to Output Term
	    // Assign net delay to input term
	    instTermIter = net->getInstTerms();
	    while (oaOccInstTerm *j = instTermIter.getNext()) {
		oaInstTerm *instTerm = j->getInstTerm();
		_inputMap::iterator iTIter;
		oaOccTerm *masterTerm = j->getTerm();
		assert(masterTerm);
		TPoint *tp = TPoint::get(j);
		oaTermType type(masterTerm->getTermType());
		if (type == oacOutputTermType){
		    tp->netLoad = elmoreNetLoad;
		    tp->load = netLoad + tp->netLoad;
		    tp->netValid = true;
		}
		else if (type == oacInputTermType){
		    iTIter = inputPin.find(instTerm);
		    if( iTIter != inputPin.end()){
			tp->netDelay = Constant*iTIter->second;
		    }else{ tp->netDelay = Constant*netDelay; }
		    /*std::cout << Util::getBlockName(j)
			<< " netDelay:" << tp->netDelay << endl;*/
		}
		//tp->netValid = true;
	    }
	}
    }
}

/*!
  This function initializes arrival times for a primary input
  (in the fanin cone of a clock pin).
  \param block the pointer to the block object
*/
void 
Timer::initClockPIArr(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "initClockPIArr(" << getBlockName(block) << ")" << std::endl;
#endif
    TPoint *p = TPoint::getClock(block);
    if (p->type == TIMING_POINT_PI_CLOCK) {
        timingData *d = NULL;
        if (p->multiClockData.size()) {
            d = p->multiClockData[0];
        } else {
            d = new timingData;
            d->clockIndex = getClockIndex(block);
            p->multiClockData.push_back(d);
        }
        d->riseSlew = d->fallSlew = ModelType::ZERO_DELAY();
        d->riseArr = d->fallArr = ModelType::ZERO_TIME();
        d->riseEarly = d->fallEarly = ModelType::ZERO_TIME();
        p->atValid = true;
    } else if (p->type == TIMING_POINT_PI || p->type == TIMING_POINT_CLOCK) {
        p->atValid = true;
    } else {
        assert(0);
    }
}

/*!
  This function initializes arrival times for a block object which is a 
  primary inputs or a clock pin.
  It also considers possible external delay and driver.
  \param block the pointer to the block object
*/
void 
Timer::initPIArr(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "initPIArr(" << getBlockName(block) << ")" << std::endl;
#endif
    TPoint *p = TPoint::get(block);
    if ((p->type == TIMING_POINT_PI) || (p->type == TIMING_POINT_PI_CLOCK)) {
        timingData *d = NULL;
        if (p->multiClockData.size()) {
            d = p->multiClockData[0];
        } else {
            d = new timingData;
            d->clockIndex = getClockIndex(block);
            p->multiClockData.push_back(d);
        }
        d->riseArr = d->fallArr = ModelType::ZERO_TIME();
        d->riseSlew = d->fallSlew = ModelType::ZERO_DELAY();
        TimerDrivingCell *dp1 = static_cast<TimerDrivingCell*>
        (TimerDrivingCell::termAppDef->get(static_cast<oaOccTerm*>(block)));
        if (dp1) {
            TPointMaster *driver = dp1->driver;
            assert(driver);
            p->piDriverForward(driver, ModelType::ZERO_DELAY());
        }
        TimerExtDelay *dp2 = static_cast<TimerExtDelay*>
            (TimerExtDelay::termAppDef->get(static_cast<oaOccTerm*>(block)));
        if (dp2) {
            d->riseArr = d->riseArr + dp2->delay;
            d->fallArr = d->fallArr + dp2->delay;
        }
        TimerInputTransition *dp3 = static_cast<TimerInputTransition*>
            (TimerInputTransition::termAppDef->get(static_cast<oaOccTerm*>(block)));
        if (dp3) {
	    std::cout << "set PI transition: " << std::endl;
	    std::cout << dp3->transition << std::endl;
	    d->riseSlew = d->fallSlew = dp3->transition;
        }
        p->atValid = true;
    } else {
        assert(0);
    }
}

/*!
  Assigning the worst slew of all rows of the multiClockData vector
  to all rows of the vector. 
  \param block the pointer to the block object
*/
void 
Timer::takeWorstSlew(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "takeWorstSlew(" << getBlockName(block) << ")" << std::endl;   
#endif
    TPoint *p = TPoint::get(block);
    DelayType worstRiseSlew = ModelType::MAX_DELAY();
    DelayType worstFallSlew = ModelType::MAX_DELAY();
    for (unsigned int k = 0; k < p->multiClockData.size(); ++k) { 
        timingData *d = p->multiClockData[k];
        if (d->riseSlew < worstRiseSlew) worstRiseSlew = d->riseSlew;
        if (d->fallSlew < worstFallSlew) worstFallSlew = d->fallSlew;
    }
    for (unsigned int k = 0; k < p->multiClockData.size(); ++k) { 
        timingData *d = p->multiClockData[k];
        d->riseSlew = worstRiseSlew;
        d->fallSlew = worstFallSlew;
    } 
}


/*!
  updating the timing data of a block object that is a multi-cycle constraint. 
  \param block the pointer to the block object
*/
void
Timer::setMultiCyclePaths(oaOccObject *block) {
#if defined(DEBUG)
    std::cout << "setMultiCyclePaths(" << getBlockName(block) << ")" << std::endl;   
#endif
    TPoint* p = TPoint::get(block);
    if (p->type == TIMING_POINT_CLOCK) {
        TPoint* p1 = TPoint::getClock(block);
        timingData *d1 = p1->multiClockData[0];
        propagatedClock *propClk1 = sdcParseData.propagatedClocks[d1->clockIndex];
        int m1 = propClk1->m;
        int m = m1 + p->multiCycleConstraint - 1;
        unsigned j, s = sdcParseData.propagatedClocks.size();
        for (j=0; j < s; j++) {   
            if(sdcParseData.propagatedClocks[j]->clk->name == propClk1->clk->name 
               && sdcParseData.propagatedClocks[j]->m == m 
               && sdcParseData.propagatedClocks[j]->phase == propClk1->phase) {
                break;
            }
        }
        if (j == s) {
            propagatedClock *propClk = new propagatedClock;
            propClk->m = m;
            propClk->clk = propClk1->clk;
            propClk->phase = propClk1->phase;
            sdcParseData.propagatedClocks.push_back(propClk);
        }
        d1->clockIndex = j;
        p->multiCycleConstraint  = 0;
        return;
    }
    for (unsigned i = 0; i <  p->multiClockData.size(); ++i) {
        timingData *d1 = p->multiClockData[i];
        propagatedClock *propClk1 = sdcParseData.propagatedClocks[d1->clockIndex];
        int m1 = propClk1->m;
        int m = m1 + p->multiCycleConstraint - 1;
        unsigned j, s = sdcParseData.propagatedClocks.size();
        for (j=0; j < s; j++) {   
            if(sdcParseData.propagatedClocks[j]->clk->name == propClk1->clk->name 
               && sdcParseData.propagatedClocks[j]->m == m 
               && sdcParseData.propagatedClocks[j]->phase == propClk1->phase) {
                break;
            }
        }
        if (j == s) {
            propagatedClock *propClk = new propagatedClock;
            propClk->m = m;
            propClk->clk = propClk1->clk;
            propClk->phase = propClk1->phase;
            sdcParseData.propagatedClocks.push_back(propClk);
        }
        d1->clockIndex = j;
    }
    p->multiCycleConstraint  = 0;
}

/*
  Is used to consider the delay and/or phase inversion caused by net or 
  logic in the fanin cone
  of the clock pin of a register to compute correct arrival time of the
  edge of the clock as well as correct required time.
  Works on the secondary timing points of the nodes (clock timing points).
  Stores the worst arrival times at riseArr and fallArr.
  Uses fallReq and riseReq to store best arrival times (needed to compute 
  the correct reqired time at clock pin of the register.)
  At the end the best arrival times are copied over from fallReq and riseReq 
  of "clock timing points" to those of regular timing points
  \param block the pointer to the block object
*/
void 
Timer::updateClockArr(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "updateClockArr(" << getBlockName(block) << ")" << std::endl;
#endif
    TPoint *p = TPoint::getClock(block);
    if (p->atValid) {
        return;
    }
    if ((p->type == TIMING_POINT_PI) || 
            (p->type == TIMING_POINT_PI_CLOCK)) {
        initClockPIArr(block);
        return;
    } 
    initVector(p->multiClockData);
    if (p->type == TIMING_POINT_CLOCK) {
        updateClockNetArr(block);
    } else if (p->type==TIMING_POINT_SIGNAL_IN) { 
        updateClockNetArr(block);
    } else {
        updateClockCellArr(block);
    }
}

//-----------------------------------------------------------------------------

/*! 
  This function updates the arrival time for a block object recursively.
  This function should be used in incremental timing analysis, 
  the function first checks if the current data is valid,
  if not, it updates the timing information by recursively compute 
  the timing information.
  \param block the pointer to the block object
*/
void 
Timer::updateArr(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "updateArr(" << getBlockName(block) << ")" << std::endl;
#endif
    TPoint *p = TPoint::get(block);
    if (p->atValid) {
	//cout << " " << getBlockName(block) << " atValid:" << p->atValid << endl;
        return;
    }
    if ((p->type == TIMING_POINT_PI) || (p->type == TIMING_POINT_PI_CLOCK)) {
        initPIArr(block);
        if (p->multiCycleConstraint > 1) {
            setMultiCyclePaths(block); 
        }
        return;
    } 
    //cout << " *1-" << getBlockName(block) << " ratValid?: " << p->ratValid << endl;
    std::vector<timingData*> oldTimeimgDataVector; 
    timingDataCopy(p->multiClockData, oldTimeimgDataVector);
    slewDataCopy(p); 
    initVector(p->multiClockData);
    if (p->isPO() || p->type==TIMING_POINT_SIGNAL_IN) {
        updateNetArr(block);
    } else {
        updateCellArr(block);
    }
    if (p->type==TIMING_POINT_CLOCK) {
        updateClockArr(block);
    }
    if (p->multiCycleConstraint > 1) {
        setMultiCyclePaths(block); 
    }
    bool worstSlewOption = true;
    // for RTL Compiler:true,  for EinsTimer:false
    if (worstSlewOption) {
        takeWorstSlew(block); 
    }
    p->ratValid = !timingDataCompare(p->multiClockData, oldTimeimgDataVector);
    if(p->isFromPI) p->ratValid = false;   // **Added to fix "rat" propagation bug in incr timing analysis
    if (!p->ratValid) {
	p->invalidateFanin(block);
	if(p->type == TIMING_POINT_SIGNAL_OUT) 
	    invalidateAllFanin(static_cast<oaOccInstTerm*>(block)->getInst());
    }
    clearVector(oldTimeimgDataVector);
    //std::cout << "EXIT: updateArr(" << getBlockName(block) << ")" << std::endl;
}

/*!
  returns the name of block object
  FIXME should move out of the timer class
  \param block either an oaTerm or an oaInstTerm
*/ 
std::string
Timer::getBlockName(oaOccObject *oPtr) 
{
    oaString f;
    oaNativeNS ns;
    std::string s;
    if (oPtr->isOccTerm()) {
        static_cast<oaOccTerm*>(oPtr)->getName(ns, f);
        s = std::string(f);
    } else {
        oaOccInst *inst = static_cast<oaOccInstTerm*>(oPtr)->getInst();
        inst->getName(ns, f);
        s = std::string(f);
        static_cast<oaOccInstTerm*>(oPtr)->getTermName(ns, f);
        s = s + "/" + std::string(f);
    }
    return s;
}

//-----------------------------------------------------------------------------

/*! 
  This function updates the required arrival time recursively.
  This function should be used in incremental timing analysis, 
  the function first checks if the current data is valid,
  if not, it updates the timing information by recursively compute 
  the timing information.
  \param block either an oaTerm or an oaInstTerm
*/
void 
Timer::updateReq(oaOccObject *block)
{
#if defined(DEBUG)
    std::cout << "updateReq(" << getBlockName(block) << ")" << std::endl;
#endif
    TPoint *p = TPoint::get(block);
    if (p->ratValid) {
	//std::cout << "Req Valid" << std::endl;
        return;
    }
    if (p->isPO()) {
        initPOReq(block);
        return;
    } 
    if (p->type == TIMING_POINT_SIGNAL_OUT || p->type == TIMING_POINT_PI_CLOCK || p->isPI()) {
        updateNetReq(block);
    } else {
        updateCellReq(block);
    }
}

//-----------------------------------------------------------------------------

/*! 
  This function updates all arrival time in the current cell view. 
*/
void 
Timer::updateAllArr()
{
#if defined(DEBUG)
    std::cout << "update All Arrival Time" << std::endl;
#endif
    for (std::set<oaOccObject*>::iterator i = clockPins.begin();
            i != clockPins.end(); ++i) {
        updateArr(*i);
    }
    for (std::set<oaDesign *>::iterator i = _watchedDesigns.begin();
         i != _watchedDesigns.end(); ++i) {
        oaDesign *des = *i;
        assert(des);
        oaOccurrence *occ = des->getTopOccurrence();
        assert(occ);
        oaIter<oaOccTerm> termIter(occ->getTerms());
        while (oaOccTerm *t = termIter.getNext()) {
            TPoint *p = TPoint::get(t);
            if (p->isPO()) {
                updateArr(t);
            }
        }
    }
}

/*! 
  This function updates all required arrival time in the current cell view. 
*/
void 
Timer::updateAllReq()
{
#if defined(DEBUG)
    std::cout << "update All Require Arrival Timer" << std::endl;
#endif
    for (std::set<oaOccObject*>::iterator i=clockPins.begin();
            i!=clockPins.end(); ++i) {
        TPoint *p = TPoint::getClock(*i);
        if (!p->ratValid) {
            updateCellReq(*i);
        }
    }
    for (std::set<oaDesign *>::iterator i = _watchedDesigns.begin();
         i != _watchedDesigns.end(); ++i) {
        oaDesign *des = *i;
        assert(des);
        oaOccurrence *occ = des->getTopOccurrence();
        assert(occ);
        oaIter<oaOccTerm> termIter(occ->getTerms());
        while (oaOccTerm *t = termIter.getNext()) {
            TPoint *p = TPoint::get(t);
            if (p->isPI()) {
                updateReq(t);
            }
        }
    }
}
//-----------------------------------------------------------------------------

/*! 
  This function marks all the points in the fanout cone to be
  arrival time invalid. 
  \param net the net to start propagation
*/
void 
Timer::invalidateFanout(oaOccNet *net)
{
#if defined(DEBUG_INVALIDATE)
    std::cout << "*invalidate Fanout"  << std::endl;
#endif
    assert(net);
    oaIter<oaOccTerm> termIter(net->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
        TPoint *p = TPoint::get(i);
        assert(p);
        if (p->isPI()) {
            p->invalidateFanout(i);
	}else if (p->type == TIMING_POINT_PI_CLOCK) {
	    TPoint *tp = TPoint::getClock(i);
	    assert(tp);
            tp->invalidateClockFanout(i);
	}
    }

    oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
    while (oaOccInstTerm *i = instTermIter.getNext()) {
        oaOccTerm *masterTerm = i->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacOutputTermType) {
            TPoint *p = TPoint::get(i);
            assert(p);
            p->invalidateFanout(i);
        }
    }
}


//-----------------------------------------------------------------------------

/*! 
  This function marks all the points in the fanin cone to be
  required time invalid.
  \param net the net to start propagation
*/
void 
Timer::invalidateFanin(oaOccNet *net)
{
#if defined(DEBUG_INVALIDATE)
    std::cout << "*invalidate Fanin"  << std::endl;
#endif
    assert(net);
    oaIter<oaOccTerm> termIter(net->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
        TPoint *p = TPoint::get(i);
        assert(p);
        if (p->isPO()) {
            p->invalidateFanin(i);
        }
    }

    oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
    while (oaOccInstTerm *i = instTermIter.getNext()) {
        oaOccTerm *masterTerm = i->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacInputTermType) {
	    TPoint *p = TPoint::get(i);
	    assert(p);
	    if(p->isClock()){
		TPoint *tp = TPoint::getClock(i);
		assert(tp);
		tp->invalidateClockFanin(i);
		continue;
	    }
	    //p->invalidateFanin(i);
	    invalidateAllFanin(i->getInst());
        }
    }
}
/*---------------------------------------------------------*/
/*! 
  This function marks all the points in the fanin cone to be
  required time invalid for each input term.
  \param inst the instance to start propagation for each of it input
*/
void 
Timer::invalidateAllFanin(oaOccInst *inst){
#if defined(DEBUG_INVALIDATE)
    std::cout << "invalidateAllFanin"  << std::endl;
#endif
    assert(inst);
    oaIter<oaOccInstTerm> instTermIter(inst->getInstTerms());
    while (oaOccInstTerm *i = instTermIter.getNext()) {
        oaOccTerm *masterTerm = i->getTerm();
        assert(masterTerm);
        oaTermType type(masterTerm->getTermType());
        if (type == oacInputTermType) {
	    TPoint *p = TPoint::get(i);
	    assert(p);
	    p->invalidateFanin(i);
        }
    }
}
/*---------------------------------------------------------*/
//-----------------------------------------------------------------------------

/*! 
  This function finds the worst path of the current cell view. 
  \param path the worst path of the current design
*/
void
Timer::findWorstPath(nodesSlopeDir & path)
{
    //double s1, s2;
    DelayType s = ModelType::MAX_DELAY();
    oaOccObject *o;
    bool isRise;

    getWorstSlack();

    // 1. InstTerm Clocks
    for (std::set<oaOccObject*>::iterator i=clockPins.begin();
            i!=clockPins.end(); ++i) {
        //TPoint *p = static_cast<TPoint*>(myAppDef->get(*i));
        oaModTerm *masterTerm = (static_cast<oaOccInstTerm*>(*i))
                ->getModInstTerm()->getTerm();
        assert(masterTerm);
        TPointMaster *tm = TPointMaster::get(masterTerm);
        assert(tm);
        oaOccInst *inst = (static_cast<oaOccInstTerm*>(*i))->getInst();
        for (TPointMaster::pathVector::iterator t = tm->outPaths.begin();
                t != tm->outPaths.end(); ++t) {
            assert(t->other);
            oaOccInstTerm *other = getOccFromMod(inst, t->other);
            if (!other) {
                continue;
            }
            TPoint *otherTP = TPoint::get(other);
            assert(otherTP);
            compareSlack(*i, true, otherTP, isRise, s, o);
            compareSlack(*i, false, otherTP, isRise, s, o);
            isRise = true;
        }
    }

    // 2. Primary inputs
    for (std::set<oaDesign *>::iterator i = _watchedDesigns.begin();
            i != _watchedDesigns.end(); ++i) {
        oaDesign *des = *i;
        assert(des);
        oaOccurrence *occ = des->getTopOccurrence();
        assert(occ);
        oaIter<oaOccTerm> termIter(occ->getTerms());
        while (oaOccTerm *t = termIter.getNext()) {
            TPoint *p = TPoint::get(t);
            if (p->isPI() && !p->isClock()) {
                compareSlack(t, true, p, isRise, s, o);
                compareSlack(t, false, p, isRise, s, o);
            }
        }
    }

    if (s == ModelType::MAX_DELAY()) {
        return;
    } else {
        TPoint *p = TPoint::get(o);
        std::pair<oaOccObject*, bool> nPair(o, isRise);
        path.push_back(nPair); 
        if (p->isClock()) {
            fromPath(path, o, true);
        } else {
            fromPath(path, o, false);
        }
    }
}

//-----------------------------------------------------------------------------

/*! 
  This function finds the worst path from the current oaTerm or oaInstTerm,
  and stores it into the vector path.
  \param path the worst path of the current design
  \param block the current oaTerm or oaInstTerm
  \param isStartingClock whether it's the starting clock pin or not
*/
void
Timer::fromPath(nodesSlopeDir & path, oaOccObject *block,
                      bool isStartingClock)
{
  TPoint *p = TPoint::get(block);
  if (p->isPO() || (p->isClock() && !isStartingClock))
    return;

  oaOccNet *net;
  if (block->isOccTerm()) {
    net = (static_cast<oaOccTerm*> (block))->getNet();
  } else if (p->type == TIMING_POINT_SIGNAL_OUT) {
    net = (static_cast<oaOccInstTerm*> (block))->getNet();
  } else {
    net = NULL;
  }

  //double s1, s2;
  DelayType s = ModelType::MAX_DELAY();
  bool currentRise = path[path.size()-1].second;
  oaOccObject *o;
  bool isRise;

  if (net) {
    oaIter<oaOccTerm> termIter(net->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
      oaTermType type(i->getTermType());
      if (type == oacOutputTermType) {
        TPoint *tp = TPoint::get(i);
        assert(tp);
        if (currentRise)
          compareSlack(i, true, tp, isRise, s, o);
        else if (!currentRise)
          compareSlack(i, false, tp, isRise, s, o);
      }
    }

    oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
    while (oaOccInstTerm *i = instTermIter.getNext()) {
      oaOccTerm *masterTerm = i->getTerm();
      assert(masterTerm);
      oaTermType type(masterTerm->getTermType());
      if (type == oacInputTermType) {
        TPoint *tp = TPoint::get(i);
        assert(tp);
        if (currentRise)
          compareSlack(i, true, tp, isRise, s, o);
        else if (!currentRise)
          compareSlack(i, false, tp, isRise, s, o);
      }
    }
  }

  if (!block->isOccTerm()) {
    oaOccInstTerm *instTerm = static_cast<oaOccInstTerm*> (block);
    assert(instTerm);
    oaModTerm *masterTerm = instTerm->getModInstTerm()->getTerm();
    assert(masterTerm);
    TPointMaster *tm = TPointMaster::get(masterTerm);
    assert(tm);
    oaOccInst *inst = instTerm->getInst();
    assert(inst);
    for (TPointMaster::pathVector::iterator i = tm->outPaths.begin();
         i != tm->outPaths.end(); ++i) {
      assert(i->other);
      oaOccInstTerm *other = getOccFromMod(inst, i->other);
      if (!other) continue;
      TPoint *otherTP = TPoint::get(other);
      assert(otherTP);

      if ( (i->transition & TPointMaster::TRANSITION_SRC_RISE)
           && currentRise)
      {
        if (i->transition & TPointMaster::TRANSITION_DST_RISE) {
          compareSlack(other, true, otherTP, isRise, s, o);
        } else if (i->transition &
                   TPointMaster::TRANSITION_DST_FALL) {
          compareSlack(other, false, otherTP, isRise, s, o);
        } else {
          assert(0);
        }
      } else if (!currentRise && 
           (i->transition & TPointMaster::TRANSITION_SRC_FALL)) 
      {
        if (i->transition & TPointMaster::TRANSITION_DST_RISE) {
          compareSlack(other, true, otherTP, isRise, s, o);
        } else if (i->transition &
                   TPointMaster::TRANSITION_DST_FALL) {
          compareSlack(other, false, otherTP, isRise, s, o);
        } else {
          assert(0);
        }
      }

    }
  }

  if (s==ModelType::MAX_DELAY()) {
    return;
  } else { 
    //TPoint *p = static_cast<TPoint*>(myAppDef->get(o));
    std::pair<oaOccObject*, bool> nPair(o, isRise);
    path.push_back(nPair); 
    fromPath(path, o, false);
  }
}

//-----------------------------------------------------------------------------

/*! 
  This function finds the worst path from an oaTerm or an oaInstTerm.
  \param path the worst path from the oaTerm or oaInstTerm
  \param block the starting oaTerm or oaInstTerm
*/
void
Timer::fromPath(nodesSlopeDir & path, oaOccObject *block)
{
  getWorstSlack();

  TPoint *p = TPoint::get(block);
  DelayType s1 = p->getRiseSlack();
  DelayType s2 = p->getFallSlack();
  DelayType s;
  bool isRise;
  if (s1<s2) {
    isRise = true;
    s = s1;
  } else {
    isRise = false;
    s = s2;
  }
  if (p->isClock()) isRise = true;
  std::pair<oaOccObject*, bool> nPair(block, isRise);
  path.push_back(nPair);
  if (p->isClock()) {
    fromPath(path, block, true);
  } else {
    fromPath(path, block, false);
  }
}

//-----------------------------------------------------------------------------

/*! 
  This function finds the critical path to an oaTerm or an oaInstTerm. 
  \param path the worst path to the oaTerm or oaInstTerm
  \param block the ending oaTerm or oaInstTerm
*/
void
Timer::toPath(nodesSlopeDir & path, oaOccObject *block)
{
  getWorstSlack();

  TPoint *p = TPoint::get(block);
  DelayType s1 = p->getRiseSlack();
  DelayType s2 = p->getFallSlack();
  DelayType s;
  bool isRise;
  if (s1<s2) {
    isRise = true;
    s = s1;
  } else {
    isRise = false;
    s = s2;
  }
  std::pair<oaOccObject*, bool> nPair(block, isRise);
  path.insert(path.begin(), nPair);
  if (p->isClock()) {
    toPath(path, block, true);
  } else {
    toPath(path, block, false);
  }
}

//-----------------------------------------------------------------------------

/*! 
  This function finds the worst path to the current oaTerm or oaInstTerm,
  and stores it into the vector path.
  \param path the worst path to the oaTerm or oaInstTerm
  \param block the current oaTerm or oaInstTerm
  \param isEndingClock whether it's the ending clock pin or not
*/
void
Timer::toPath(nodesSlopeDir & path, oaOccObject *block,
                    bool isEndingClock)
{
  TPoint *p = TPoint::get(block);
  if (p->isPI() || (p->isClock() && !isEndingClock))
    return;

  oaOccNet *net;
  if (block->isOccTerm()) {
    net = (static_cast<oaOccTerm*> (block))->getNet();
  } else if (p->type == TIMING_POINT_SIGNAL_IN) {
    net = (static_cast<oaOccInstTerm*> (block))->getNet();
  } else {
    net = NULL;
  }

  //double s1, s2;
  DelayType s = ModelType::MAX_DELAY();
  bool currentRise = path[0].second;
  oaOccObject *o;
  bool isRise;

  if (net) {
    oaIter<oaOccTerm> termIter(net->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
      oaTermType type(i->getTermType());
      if (type == oacInputTermType) {
        TPoint *tp = TPoint::get(i);
        assert(tp);
        if (currentRise)
          compareSlack(i, true, tp, isRise, s, o);
        else if (!currentRise)
          compareSlack(i, false, tp, isRise, s, o);
      }
    }

    oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
    while (oaOccInstTerm *i = instTermIter.getNext()) {
      oaOccTerm *masterTerm = i->getTerm();
      assert(masterTerm);
      oaTermType type(masterTerm->getTermType());
      if (type == oacOutputTermType) {
        TPoint *tp = TPoint::get(i);
        assert(tp);
        if (currentRise)
          compareSlack(i, true, tp, isRise, s, o);
        else if (!currentRise)
          compareSlack(i, false, tp, isRise, s, o);
      }
    }
  }

  if (!block->isOccTerm()) {
    oaOccInstTerm *instTerm = static_cast<oaOccInstTerm*> (block);
    assert(instTerm);
    oaModTerm *masterTerm = instTerm->getModInstTerm()->getTerm();
    assert(masterTerm);
    TPointMaster *tm = TPointMaster::get(masterTerm);
    assert(tm);
    oaOccInst *inst = instTerm->getInst();
    assert(inst);
    for (TPointMaster::pathVector::iterator i = tm->inPaths.begin();
         i != tm->inPaths.end(); ++i) {
      assert(i->other);
      oaOccInstTerm *other =getOccFromMod(inst, i->other);
      if (!other) continue;
      TPoint *otherTP = TPoint::get(other);
      assert(otherTP);

      if ( (i->transition & TPointMaster::TRANSITION_DST_RISE)
           && currentRise)
      {
        if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
          compareSlack(other, true, otherTP, isRise, s, o);
        } else if (i->transition &
                   TPointMaster::TRANSITION_SRC_FALL) {
          compareSlack(other, false, otherTP, isRise, s, o);
        } else {
          assert(0);
        }
      } else if (!currentRise && 
           (i->transition & TPointMaster::TRANSITION_DST_FALL)) 
      {
        if (i->transition & TPointMaster::TRANSITION_SRC_RISE) {
          compareSlack(other, true, otherTP, isRise, s, o);
        } else if (i->transition &
                   TPointMaster::TRANSITION_SRC_FALL) {
          compareSlack(other, false, otherTP, isRise, s, o);
        } else {
          assert(0);
        }
      }

    }
  }

  if (s==ModelType::MAX_DELAY()) {
    return;
  } else { 
    std::pair<oaOccObject*, bool> nPair(o, isRise);
    path.insert(path.begin(), nPair);
    toPath(path,o,false);
  }
}

//-----------------------------------------------------------------------------

/*! 
  This function finds the critical path through an oaTerm or an oaInstTerm. 
  \param path the worst path through that oaTerm or oaInstTerm
  \param block the oaTerm or oaInstTerm
*/
void
Timer::throughPath(nodesSlopeDir & path, oaOccObject *block)
{
  toPath(path, block);
  fromPath(path, block, false);
}

//-----------------------------------------------------------------------------

/*! 
  This function compares the worst slack with the slack of the current 
  oaTerm or oaInstTerm. If the current slack is worse, update. 
  \param block the current oaTerm or oaInstTerm
  \param currentRise it's the rising or the falling slack
  \param tp the timing point of the current oaTerm or oaInstTerm
  \param isRise the worst slack is risign or falling
  \param worstSlack the worst slack
  \param blockWithWorstSlack the oaTerm or oaInstTerm with the worst slack
*/
void
Timer::compareSlack(oaOccObject *block, 
                          bool currentRise,
                          TPoint *tp,
                          bool & isRise,
                          DelayType & worstSlack, 
                          oaOccObject * & blockWithWorstSlack)
{
  if (currentRise) {
    DelayType s1 = tp->getRiseSlack();
    if (s1<worstSlack) {
      isRise = true;
      worstSlack = s1;
      blockWithWorstSlack = block;
    }
  } else if (!currentRise) {
    DelayType s2 = tp->getFallSlack();
    if (s2<worstSlack) {
      isRise = false;
      worstSlack = s2;
      blockWithWorstSlack = block;
    }
  }
}

///////////////////////////////////////////////////////////////////////////////

oaOccInstTerm *
Timer::getOccFromMod(oaOccInst *inst, oaModTerm *term) {
    assert(inst);
    assert(term);
    oaModInst *mi = inst->getModInst();
    assert(mi);
    oaModInstTerm *mit = oaModInstTerm::find(mi, term);
    if (!mit) {
        return 0;
    }
    oaIter<oaOccInstTerm> iter(inst->getInstTerms());
    while (oaOccInstTerm *t = iter.getNext()) {
        if (t->getModInstTerm() == mit) {
            return t;
        }
    }
    assert(0);
    return 0;
}

///////////////////////////////////////////////////////////////////////////////
double
Timer::getSetupTime(oaOccInstTerm *iTerm){
    TPoint *outPin = TPoint::get(iTerm);
    assert(outPin);
    if(!outPin->isToFF) return 0;
    oaOccNet *net = iTerm->getNet();

    //Search for FF instance
    oaOccInst *inst;
    oaIter<oaOccInstTerm> itIter = net->getInstTerms();
    while(oaOccInstTerm *i = itIter.getNext()){
	TPoint *tp = TPoint::get(i);
	assert(tp);
	if(tp->type == TIMING_POINT_SIGNAL_IN){
	    CellData *cell = CellData::get(i->getInst()->getModInst());
	    assert(cell);
	    if(cell->master->isSequential){
		return tp->riseDelay;
	    }
	}
    }
    return 0;    
}
///////////////////////////////////////////////////////////////////////////////
void
Timer::clearDesignTimingData(){
    oaDesign *design = *(_watchedDesigns.begin());
    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    oaIter<oaOccInstTerm> instTermIter(occ->getInstTerms());
    while (oaOccInstTerm *i = instTermIter.getNext()) {
	//std::cout << " clear " << getBlockName(i) << std::endl;
        TPoint *tp = TPoint::get(i);
	assert(tp);
	if(tp->type == TIMING_POINT_CLOCK){
	    TPoint *tp1 = TPoint::getClock(i);
	    clearVector(tp1->multiClockData);
	    tp1->atValid = false;
	    tp1->ratValid = false;
	}
	clearVector(tp->multiClockData);
	tp->atValid = false;
	tp->ratValid = false;
    }
    oaIter<oaOccTerm> termIter(occ->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
	//std::cout << " clear " << getBlockName(i) << std::endl;
        TPoint *tp = TPoint::get(i);
	assert(tp);
	clearVector(tp->multiClockData);
	tp->atValid = false;
	tp->ratValid = false;
    }
}
/*---------------------------------------------------------*/
void
Timer::printCell(oaOccInst *i){
    oaIter<oaOccInstTerm> itIter = i->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
	double slack = getSlack(iTerm);
	if(slack == DBL_MAX) slack = 10000;
	double arr = getArr(iTerm);
	if(arr == DBL_MAX) arr = 10000;
	double req = getReq(iTerm);
	if(req == DBL_MAX) req = 10000;
	cout << "  " << getBlockName(iTerm)
	    << " " << arr << " / " << req << " / " << slack << endl;
    }
}
/*---------------------------------------------------------*/
void
Timer::printDetailTiming(){
    std::cout << "All TIMING DATA" << std::endl;
    oaDesign *design = *(_watchedDesigns.begin());
    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    oaIter<oaOccTerm> tIter = occ->getTerms();
    while(oaOccTerm *tTerm = tIter.getNext()){
	cout << Util::getBlockName(tTerm) 
	    << "\n Slew(r/f):" << getRiseSlew(tTerm) << "/" << getFallSlew(tTerm)
	    << "\n Arr(r/f):" << getRiseArr(tTerm) << "/" << getFallArr(tTerm)
	    << "\n Req(r/f):" << getRiseReq(tTerm) << "/" << getFallReq(tTerm)
	    << "\n Slack:" << getSlack(tTerm) << std::endl;
    }
    oaIter<oaOccInstTerm> itIter = occ->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
	cout << Util::getBlockName(iTerm) 
	    << "\n Slew(r/f):" << getRiseSlew(iTerm) << "/" << getFallSlew(iTerm)
	    << "\n Arr(r/f):" << getRiseArr(iTerm) << "/" << getFallArr(iTerm)
	    << "\n Req(r/f):" << getRiseReq(iTerm) << "/" << getFallReq(iTerm)
	    << "\n Slack:" << getSlack(iTerm) << std::endl;
	TPoint *tp = TPoint::get(iTerm);
	cout << "netDelay:" << tp->netDelay << endl;
    }
}
/*---------------------------------------------------------*/
void
Timer::printAll(){
    std::cout << "All TIMING DATA" << std::endl;
    oaDesign *design = *(_watchedDesigns.begin());
    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    oaIter<oaOccTerm> tIter = occ->getTerms();
    while(oaOccTerm *tTerm = tIter.getNext()){
	double slack = getSlack(tTerm);
	if(slack == DBL_MAX) slack = 10000;
	double arr = getArr(tTerm);
	if(arr == DBL_MAX) arr = 10000;
	double req = getReq(tTerm);
	if(req == DBL_MAX) req = 10000;
	cout << "  " << getBlockName(tTerm)
	    << " " << arr << " / " << req << " / " << slack << endl;
    }
    oaIter<oaOccInstTerm> itIter = occ->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
	double slack = getSlack(iTerm);
	if(slack == DBL_MAX) slack = 10000;
	double arr = getArr(iTerm);
	if(arr == DBL_MAX) arr = 10000;
	double req = getReq(iTerm);
	if(req == DBL_MAX) req = 10000;
	cout << "  " << getBlockName(iTerm)
	    << " " << arr << " / " << req << " / " << slack << endl;
    }
}
/*---------------------------------------------------------*/
void
Timer::printAllPO(){
    std::cout << "DEBUG ALL PO TIMING DATA" << std::endl; 
    oaDesign *design = *(_watchedDesigns.begin());
    oaModule *mod = design->getTopModule();
    assert(mod);
    oaIter<oaModInst> iIter = mod->getInsts();
    while(oaModInst *inst = iIter.getNext()){
	CellData *cell = CellData::get(inst);
	assert(cell);
	if(cell->master->isSequential){
	    oaOccInst *occInst = DesignTool::getOccInst(design,inst);
	    oaIter<oaOccInstTerm> itIter = occInst->getInstTerms();
	    while(oaOccInstTerm *iTerm = itIter.getNext()){
		double slack = getSlack(iTerm);
		if(slack == DBL_MAX) slack = 10000;
		double arr = getArr(iTerm);
		if(arr == DBL_MAX) arr = 10000;
		double req = getReq(iTerm);
		if(req == DBL_MAX) req = 10000;
		cout << "  " << getBlockName(iTerm)
		    << " " << arr << " / " << req << " / " << slack << endl;
	    }
	}
    }
    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    oaIter<oaOccTerm> tIter = occ->getTerms();
    while(oaOccTerm *tTerm = tIter.getNext()){
	TPoint *tp = TPoint::get(tTerm);
	if(tp->type == TIMING_POINT_PO){
	    double slack = getSlack(tTerm);
	    if(slack == DBL_MAX) slack = 10000;
	    double arr = getArr(tTerm);
	    if(arr == DBL_MAX) arr = 10000;
	    double req = getReq(tTerm);
	    if(req == DBL_MAX) req = 10000;
	    cout << "  " << getBlockName(tTerm)
		<< " " << arr << " / " << req << " / " << slack << endl;
	}
    }
}
/*---------------------------------------------------------*/
void
Timer::checkRatFlags(){
    oaDesign *design = *(_watchedDesigns.begin());
    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    oaIter<oaOccTerm> tIter = occ->getTerms();
    while(oaOccTerm *tTerm = tIter.getNext()){
	TPoint *tp = TPoint::get(tTerm);
	cout << "  " << getBlockName(tTerm)
	    << " " << tp->atValid << " / " << tp->ratValid << endl;
    }
    oaIter<oaOccInstTerm> itIter = occ->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
	TPoint *tp = TPoint::get(iTerm);
	cout << "  " << getBlockName(iTerm)
	    << " " << tp->atValid << " / " << tp->ratValid << endl;
    }
}
/*---------------------------------------------------------*/
}
