/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Zhong Xiu <zxiu@andrew.cmu.edu>

ChangeLog:
2004-09-15: ChangeLog started
*/

#include <assert.h>
#include <float.h>
#include <iostream>
#include "oaDesignDB.h"

#include "oagTimerModel.h"
#include "oagTimerTPoint.h"
#include "oagTimerTPointMaster.h"
#include "oagTimerTimer.h"

//#define DEBUG
//#define DEBUG_INVALIDATE

namespace oagTimer {
/*!
  returns the name of block object
  FIXME should move out of the timer class
  \param block either an oaTerm or an oaInstTerm
*/ 
std::string
TPoint::getBlockName(oaOccObject *oPtr) 
{
    oaString f;
    oaNativeNS ns;
    std::string s;
    if (oPtr->isOccTerm()) {
        static_cast<oaOccTerm*>(oPtr)->getName(ns, f);
        s = std::string(f);
    } else {
        oaOccInst *inst = static_cast<oaOccInstTerm*>(oPtr)->getInst();
        inst->getName(ns, f);
        s = std::string(f);
        static_cast<oaOccInstTerm*>(oPtr)->getTermName(ns, f);
        s = s + "/" + std::string(f);
    }
    return s;
}

//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------

/*!
  This function clears all the data in the current timing point.
*/
void
TPoint::clear() {
#if defined(DEBUG)
    std::cout << "clear all data" << endl;
#endif
  unsigned int j;
  switch (type) {
    case TIMING_POINT_SIGNAL_IN:
    case TIMING_POINT_SIGNAL_OUT:
      for (j = 0; j < multiClockData.size(); ++j) {
        timingData *td = multiClockData[j];
        td->riseArr = td->fallArr = -ModelType::MAX_TIME();
        td->riseReq = td->fallReq = ModelType::MAX_TIME();
      }
      break;
    case TIMING_POINT_CLOCK:
      for (j = 0; j < multiClockData.size(); ++j) {
        timingData *td = multiClockData[j];
        td->riseArr = td->fallArr = 0.0;
        td->riseReq = td->fallReq = 0.0;
      }
      break;
    case TIMING_POINT_PI:
      for (j = 0; j < multiClockData.size(); ++j) {
        timingData *td = multiClockData[j];
        td->riseArr = td->fallArr = 0.0;
        td->riseReq = td->fallReq = ModelType::MAX_TIME();
      }
      break;
    case TIMING_POINT_PO:
      for (j = 0; j < multiClockData.size(); ++j) {
        timingData *td = multiClockData[j];
        td->riseArr = td->fallArr = -ModelType::MAX_TIME();
        td->riseReq = td->fallReq = 0.0;
      }
      break;
    case TIMING_POINT_PI_CLOCK:
    default:
      break;
  }
  for (j = 0; j < multiClockData.size(); ++j) {
    timingData *td = multiClockData[j];
    td->riseSlew = td->fallSlew = 0.0;
  }
}

void
TPoint::clearSlew() 
{
#if defined DEBUG
    std::cout << "clearTimingVector" << std::endl;
#endif
    for (unsigned int j = 0; j < multiClockData.size(); ++j) {
        timingData *d =  multiClockData[j];
	if(d->riseSlew == 0.0) d->riseSlew = -1.0;
	if(d->fallSlew == 0.0) d->fallSlew = -1.0;
    }
    //multiClockData.clear();
}
//-----------------------------------------------------------------------------

/*!
  This function propagates the slew rate and delay at the PI, 
  here the slew rate and the delay come from the driving cell.
  \param tm this is the TPointMaster of the driving cell's output
  \param defaultSlew the default slew of the input, use 0 now.
*/
void
TPoint::piDriverForward(TPointMaster *tm, 
                                  DelayType defaultSlew) 
{
#if defined(DEBUG)
    std::cout << "PI Driver Forward" << std::endl;
#endif
  assert(tm);

  for(TPointMaster::pathVector::iterator i = tm->inPaths.begin();
      i != tm->inPaths.end(); ++i) {
    assert(i->delay);
    DelayType t = i->delay->lookup(load, defaultSlew, tm->loadLimit);
    DelayType t0 = i->delay->lookup(0.0, defaultSlew, tm->loadLimit);
    if (i->transition & TPointMaster::TRANSITION_DST_RISE) {
      for (unsigned int j = 0; j < multiClockData.size(); ++j) {
        timingData *d = multiClockData[j];
        if (t-t0 > d->riseArr - ModelType::ZERO_TIME()) {
          d->riseArr = ModelType::ZERO_TIME() + t-t0;
        }
      }
    }
    else if (i->transition & TPointMaster::TRANSITION_DST_FALL) {
      for (unsigned int j = 0; j < multiClockData.size(); ++j) {
        timingData *d = multiClockData[j];
        if (t-t0 > d->fallArr - ModelType::ZERO_TIME()) {
          d->fallArr = ModelType::ZERO_TIME() + t-t0;
        }
      }
    }
    else {
      assert(0);
    }
    if (i->slew) {
      t = i->slew->lookup(load, defaultSlew, tm->loadLimit);
      if (i->transition & TPointMaster::TRANSITION_DST_RISE) {
        for (unsigned int j = 0; j < multiClockData.size(); ++j) {
          timingData *d = multiClockData[j];
          if (t > d->riseSlew) {
            d->riseSlew = t;
          }
        }
      }
      else if (i->transition & TPointMaster::TRANSITION_DST_FALL) {
        for (unsigned int j = 0; j < multiClockData.size(); ++j) {
          timingData *d = multiClockData[j];
          if (t > d->fallSlew) {
            d->fallSlew = t;
          }
        }
      }
    }
  }
}

///////////////////////////////////////////////////////////////////////////////

void
TPoint::invalidateFanout(oaOccObject *oPtr)
{
#if defined(DEBUG_INVALIDATE)
    std::cout << "Invalidate Fanout (" << getBlockName( oPtr ) << ")" << endl;
#endif
    assert(oPtr);
    if (!atValid) {
	return;
    }
    atValid = false;
    if (type == TIMING_POINT_PO || isClock()) {
	    //cout << "PO or isClock --> returned" << endl;
	    return;
    }
    if (type == TIMING_POINT_SIGNAL_IN || type == TIMING_POINT_SIGNAL_OUT) {
	//cout << "SIGNAL_IN or SIGNAL_OUT" << endl;
        oaOccInstTerm *instTerm = static_cast<oaOccInstTerm*>(oPtr);
        assert(instTerm);
        oaModTerm *masterTerm = instTerm->getModInstTerm()->getTerm();
        assert(masterTerm);
        TPointMaster *tm = TPointMaster::get(masterTerm);
        assert(tm);
        oaOccInst *inst = instTerm->getInst();
        assert(inst);
        for (TPointMaster::pathVector::iterator i = tm->outPaths.begin();
             i != tm->outPaths.end(); ++i) {
	    //std::cout << ++counter << std::endl;
            assert(i->other);
            oaOccInstTerm *other = Timer::getOccFromMod(inst, i->other);
            if (!other) {
		//cout << "continued" << endl;
                continue;
            }
            TPoint *otherTP = TPoint::get(other);
            assert(otherTP);
	    //cout << "other" << endl;
            otherTP->invalidateFanout(other);
        }
    }

    if (type == TIMING_POINT_PI || type == TIMING_POINT_SIGNAL_OUT) {
	//cout << "PI or SIGNAL_OUT" << endl;
        oaOccNet* net;
        if (oPtr->isOccTerm()) {
            net = (static_cast<oaOccTerm*>(oPtr))->getNet();
        } else {
            net = (static_cast<oaOccInstTerm*>(oPtr))->getNet();
        }
        if (net) {
            oaIter<oaOccTerm> termIter(net->getTerms());
            while (oaOccTerm *i = termIter.getNext()) {
                oaTermType type(i->getTermType());
                if (type == oacOutputTermType) {
                    TPoint *tp = TPoint::get(i);
                    assert(tp);
                    tp->invalidateFanout(i);
                }
            }
            oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
            while (oaOccInstTerm *i = instTermIter.getNext()) {
                oaOccTerm *masterTerm = i->getTerm();
                assert(masterTerm);
                oaTermType type(masterTerm->getTermType());
                if (type == oacInputTermType) {
                    TPoint *tp = TPoint::get(i);
                    assert(tp);
                    tp->invalidateFanout(i);
                }
            }
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
void 
TPoint::invalidateFanin(oaOccObject *oPtr)
{
#if defined(DEBUG_INVALIDATE)
    std::cout << "Invalidate Fanin(" << getBlockName( oPtr ) << ")" << endl;
#endif
    assert(oPtr);
    if (!ratValid) {
        return;
    }
    ratValid = false;
    //clearSlew();
    if (type == TIMING_POINT_PI || isClock()) {
    //std::cout << "PI or CLOCK"  << endl;
        return;
    }

    if (type == TIMING_POINT_SIGNAL_OUT) {
	//cout << "SIGNAL_OUT" << endl;
        oaOccInstTerm *instTerm = static_cast<oaOccInstTerm*>(oPtr);
        assert(instTerm);
        oaModTerm *masterTerm = instTerm->getModInstTerm()->getTerm();
        assert(masterTerm);
        TPointMaster *tm = TPointMaster::get(masterTerm);
        assert(tm);
        oaOccInst *inst = instTerm->getInst();
        assert(inst);
        for (TPointMaster::pathVector::iterator i = tm->inPaths.begin();
             i != tm->inPaths.end(); ++i) {
            assert(i->other);
            oaOccInstTerm *other = Timer::getOccFromMod(inst, i->other);
            if (!other) {
                continue;
            }
            TPoint *otherTP = TPoint::get(other);
            assert(otherTP);
            if (otherTP->isClock()) {
                otherTP = getClock(other);
            }
            assert(otherTP);
            // otherTP is the point which we need to set to invalid
            otherTP->invalidateFanin(other);
        }
    }

    if (type == TIMING_POINT_SIGNAL_IN || type == TIMING_POINT_PO) {
	//cout << "PO or SIGNAL_IN" << endl;
        oaOccNet* net;
        if (oPtr->isOccTerm()) {
            net = (static_cast<oaOccTerm*>(oPtr))->getNet();
        } else {
            net = (static_cast<oaOccInstTerm*>(oPtr))->getNet();
        }
        if (net) {
            oaIter<oaOccTerm> termIter(net->getTerms());
            while (oaOccTerm *i = termIter.getNext()) {
                oaTermType type(i->getTermType());
                if (type == oacInputTermType) {
                    TPoint *tp = TPoint::get(i);
                    assert(tp);
                    tp->invalidateFanin(i);
                }
            }
            oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
            while (oaOccInstTerm *i = instTermIter.getNext()) {
                oaOccTerm *masterTerm = i->getTerm();
                assert(masterTerm);
                oaTermType type(masterTerm->getTermType());
                if (type == oacOutputTermType) {
                    TPoint *tp = TPoint::get(i);
                    assert(tp);
                    tp->invalidateFanin(i);
                }
            }
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
void
TPoint::invalidateClockFanout(oaOccObject *oPtr)
{
#if defined(DEBUG_INVALIDATE)
    std::cout << "InvalidateClockFanout (" << getBlockName( oPtr ) << ")" << endl;
#endif
    assert(oPtr);
    if (!atValid) {
	//std::cout << "CLOCK !atValid" << std::endl;
	return;
    }
    atValid = false;
    if (type == TIMING_POINT_PO) {
	//cout << "PO --> returned" << endl;
	return;
    }else if(type == TIMING_POINT_CLOCK){
	//cout << "CLOCKFANOUT TIMING_POINT_CLOCK" << endl;
        oaOccInstTerm *instTerm = static_cast<oaOccInstTerm*>(oPtr);
        assert(instTerm);
        oaModTerm *masterTerm = instTerm->getModInstTerm()->getTerm();
        assert(masterTerm);
        TPointMaster *tm = TPointMaster::get(masterTerm);
        assert(tm);
        oaOccInst *inst = instTerm->getInst();
        assert(inst);
        for (TPointMaster::pathVector::iterator i = tm->outPaths.begin();
             i != tm->outPaths.end(); ++i) {
            assert(i->other);
            oaOccInstTerm *other = Timer::getOccFromMod(inst, i->other);
            if (!other) {
		//cout << "continued" << endl;
                continue;
            }
            TPoint *otherTP = TPoint::get(other);
            assert(otherTP);
	    //cout << "other" << endl;
            otherTP->invalidateFanout(other);
        }
    }

    if (type == TIMING_POINT_SIGNAL_IN || type == TIMING_POINT_SIGNAL_OUT) {
	//cout << "CLOCKFANOUT SIGNAL_IN or SIGNAL_OUT" << endl;
        oaOccInstTerm *instTerm = static_cast<oaOccInstTerm*>(oPtr);
        assert(instTerm);
        oaModTerm *masterTerm = instTerm->getModInstTerm()->getTerm();
        assert(masterTerm);
        TPointMaster *tm = TPointMaster::get(masterTerm);
        assert(tm);
        oaOccInst *inst = instTerm->getInst();
        assert(inst);
        for (TPointMaster::pathVector::iterator i = tm->outPaths.begin();
             i != tm->outPaths.end(); ++i) {
            assert(i->other);
            oaOccInstTerm *other = Timer::getOccFromMod(inst, i->other);
            if (!other) {
		//cout << "continued" << endl;
                continue;
            }
            TPoint *otherTP = TPoint::getClock(other);
            assert(otherTP);
	    //cout << "other" << endl;
            otherTP->invalidateClockFanout(other);
        }
    }

    if (type == TIMING_POINT_PI_CLOCK || type == TIMING_POINT_SIGNAL_OUT) {
	//cout << "PI_CLOCK or SIGNAL_OUT" << endl;
        oaOccNet* net;
        if (oPtr->isOccTerm()) {
            net = (static_cast<oaOccTerm*>(oPtr))->getNet();
        } else {
            net = (static_cast<oaOccInstTerm*>(oPtr))->getNet();
        }
        if (net) {
            oaIter<oaOccTerm> termIter(net->getTerms());
            while (oaOccTerm *i = termIter.getNext()) {
                oaTermType type(i->getTermType());
                if (type == oacOutputTermType) {
                    TPoint *tp = TPoint::getClock(i);
                    assert(tp);
                    tp->invalidateClockFanout(i);
                }
            }
            oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
            while (oaOccInstTerm *i = instTermIter.getNext()) {
                oaOccTerm *masterTerm = i->getTerm();
                assert(masterTerm);
                oaTermType type(masterTerm->getTermType());
                if (type == oacInputTermType) {
                    TPoint *tp = TPoint::getClock(i);
                    assert(tp);
                    tp->invalidateClockFanout(i);
                }
            }
        }
    }
}
///////////////////////////////////////////////////////////////////////////////
void 
TPoint::invalidateClockFanin(oaOccObject *oPtr)
{
#if defined(DEBUG_INVALIDATE)
    std::cout << "InvalidateCLOCKFanin(" << getBlockName( oPtr ) << ")" << endl;
#endif
    assert(oPtr);
    if (!ratValid) {
    //std::cout << "!ratValid"  << endl;
        return;
    }
    ratValid = false;
    if (type == TIMING_POINT_PI) {
	//std::cout << "PI"  << endl;
        return;
    }

    if (type == TIMING_POINT_SIGNAL_OUT || type == TIMING_POINT_CLOCK) {
	//cout << "SIGNAL_OUT or TIMING_POINT_CLOCKC" << endl;
        oaOccInstTerm *instTerm = static_cast<oaOccInstTerm*>(oPtr);
        assert(instTerm);
        oaModTerm *masterTerm = instTerm->getModInstTerm()->getTerm();
        assert(masterTerm);
        TPointMaster *tm = TPointMaster::get(masterTerm);
        assert(tm);
        oaOccInst *inst = instTerm->getInst();
        assert(inst);
        for (TPointMaster::pathVector::iterator i = tm->inPaths.begin();
             i != tm->inPaths.end(); ++i) {
            assert(i->other);
            oaOccInstTerm *other = Timer::getOccFromMod(inst, i->other);
            if (!other) {
                continue;
            }
            TPoint *otherTP = TPoint::get(other);
            assert(otherTP);
            if (otherTP->isClock()) {
                otherTP = getClock(other);
            }
            assert(otherTP);
            // otherTP is the point which we need to set to invalid
            otherTP->invalidateFanin(other);
        }
    }

    if (type == TIMING_POINT_SIGNAL_IN || type == TIMING_POINT_PO || type == TIMING_POINT_CLOCK) {
	//cout << "PO or SIGNAL_IN or TIMING_POINT_CLOCK" << endl;
        oaOccNet* net;
        if (oPtr->isOccTerm()) {
            net = (static_cast<oaOccTerm*>(oPtr))->getNet();
        } else {
            net = (static_cast<oaOccInstTerm*>(oPtr))->getNet();
        }
        if (net) {
            oaIter<oaOccTerm> termIter(net->getTerms());
            while (oaOccTerm *i = termIter.getNext()) {
                oaTermType type(i->getTermType());
                if (type == oacInputTermType) {
                    TPoint *tp = TPoint::getClock(i);
                    assert(tp);
                    tp->invalidateClockFanin(i);
                }
            }
            oaIter<oaOccInstTerm> instTermIter(net->getInstTerms());
            while (oaOccInstTerm *i = instTermIter.getNext()) {
                oaOccTerm *masterTerm = i->getTerm();
                assert(masterTerm);
                oaTermType type(masterTerm->getTermType());
                if (type == oacOutputTermType) {
                    TPoint *tp = TPoint::getClock(i);
                    assert(tp);
                    tp->invalidateClockFanin(i);
                }
            }
        }
    }
    //std::cout << "EXIT InvalidateCLOCKFanin(" << getBlockName( oPtr ) << ")" << endl;
}
///////////////////////////////////////////////////////////////////////////////
oaVoidPointerAppDef<oaOccInstTerm> *TPoint::_instTermAppDef = 0;
oaVoidPointerAppDef<oaOccTerm> *TPoint::_termAppDef = 0;
oaVoidPointerAppDef<oaOccInstTerm> *TPoint::_clockInstTermAppDef = 0;
oaVoidPointerAppDef<oaOccTerm> *TPoint::_clockTermAppDef = 0;

///////////////////////////////////////////////////////////////////////////////

void
TPoint::initAppDefs()
{
#if defined(DEBUG)
    std::cout << "initialize App Def" << std::endl;
#endif
    assert(!_instTermAppDef);
    _instTermAppDef =
        oaVoidPointerAppDef<oaOccInstTerm>::get("oagTimerInstTerm");
    assert(_instTermAppDef);

    assert(!_termAppDef);
    _termAppDef =
        oaVoidPointerAppDef<oaOccTerm>::get("oagTimerTerm");
    assert(_termAppDef);

    assert(!_clockInstTermAppDef);
    _clockInstTermAppDef =
        oaVoidPointerAppDef<oaOccInstTerm>::get("oagTimerClockInstTerm");
    assert(_clockInstTermAppDef);

    assert(!_clockTermAppDef);
    _clockTermAppDef =
        oaVoidPointerAppDef<oaOccTerm>::get("oagTimerClockTerm");
    assert(_clockTermAppDef);
}

///////////////////////////////////////////////////////////////////////////////

TPoint *
TPoint::create(oaOccObject *oPtr)
{
#if defined(DEBUG)
    std::cout << "Creating TPoint OccObject ()" << std::endl;
#endif
    assert(oPtr);
    if (oPtr->isOccTerm()) {
        return create(static_cast<oaOccTerm *>(oPtr));
    } else {
        return create(static_cast<oaOccInstTerm *>(oPtr));
    }
}

///////////////////////////////////////////////////////////////////////////////

TPoint *
TPoint::create(oaOccInstTerm *oPtr)
{
#if defined(DEBUG)
    std::cout << "Creating Timing Point of OccInstTerm" << std::endl;
#endif
    assert(oPtr);
    oaModInstTerm *iTerm = oPtr->getModInstTerm();
    assert(iTerm);
    oaModTerm *masterTerm = iTerm->getTerm();
    assert(masterTerm);
    oaModNet *masterNet = masterTerm->getNet();
    assert(masterNet);
    oaSigType s(masterNet->getSigType());

    if (s == oacPowerSigType || s == oacGroundSigType) {
        return 0;
    }

    TPoint *tp = new TPoint;

    assert(!_instTermAppDef->get(oPtr));
    _instTermAppDef->set(oPtr, static_cast<void *>(tp));

    oaTermType t(masterTerm->getTermType());
    if (t == oacInputTermType) {
        if (s == oacClockSigType) {
            tp->type = TIMING_POINT_CLOCK;
        } else if (s == oacSignalSigType) {
            tp->type = TIMING_POINT_SIGNAL_IN;
        } else {
            assert(0);
        }
    } else if (t == oacOutputTermType) {
        assert(s == oacSignalSigType);
        tp->type = TIMING_POINT_SIGNAL_OUT;
    } else {
        assert(0);
    }
    return tp;
}

///////////////////////////////////////////////////////////////////////////////

TPoint *
TPoint::create(oaOccTerm *oPtr)
{
#if defined(DEBUG)
    std::cout << "Create TPoint of OccTerm" << endl;
#endif
    assert(oPtr);
    oaOccNet *net = oPtr->getNet();
    assert(net);
    oaSigType s(net->getSigType());
    if (s == oacPowerSigType || s == oacGroundSigType) {
        return 0;
    }

    TPoint *tp = new TPoint;
    assert(!_termAppDef->get(oPtr));
    _termAppDef->set(oPtr, static_cast<void *>(tp));

    oaTermType t(oPtr->getTermType());
    if (t == oacInputTermType) {
        tp->type = TIMING_POINT_PI;
    } else if (t == oacOutputTermType) {
        tp->type = TIMING_POINT_PO;
    } else {
        assert(0);
    }
    return tp;
}

//////////////////////////////////////////////////////////////////////////////

TPoint *
TPoint::get(oaOccObject *oPtr)
{
#if defined(DEBUG)
    //std::cout << "Get ("  << ")" << std::endl;
#endif
    assert(oPtr);
    TPoint *tp;
    if (oPtr->isOccTerm()) {
        oaOccTerm *term = static_cast<oaOccTerm *>(oPtr);
        tp = static_cast<TPoint *>(_termAppDef->get(term));
    } else {
        oaOccInstTerm *instTerm = static_cast<oaOccInstTerm *>(oPtr);
        tp = static_cast<TPoint *>(_instTermAppDef->get(instTerm));
    }
    assert(tp);
    return tp;
}

//////////////////////////////////////////////////////////////////////////////

TPoint *
TPoint::getClock(oaOccObject *oPtr)
{
#if defined(DEBUG)
    //std::cout << "Get Clock TPoint ()" << std::endl;
#endif
    assert(oPtr);
    TPoint *tp;
    if (oPtr->isOccTerm()) {
        oaOccTerm *term = static_cast<oaOccTerm *>(oPtr);
        tp = static_cast<TPoint *>(_clockTermAppDef->get(term));
        if (!tp) {
	    //std::cout << "isOccTerm() and !tp" << std::endl;
            tp = new TPoint;
            TPoint *p1 = get(oPtr);
            tp->type = p1->type;
            _clockTermAppDef->set(term, static_cast<void*>(tp));
        }
    } else {
        oaOccInstTerm *instTerm = static_cast<oaOccInstTerm *>(oPtr);
        tp = static_cast<TPoint *>(_clockInstTermAppDef->get(instTerm));
        if (!tp) {
	    //std::cout << "isOccInstTerm() and !tp" << std::endl;
            tp = new TPoint;
            TPoint *p1 = get(oPtr);
            tp->type = p1->type;
            _clockInstTermAppDef->set(instTerm, static_cast<void*>(tp));
        }
    }
    assert(tp);
    return tp;
}

///////////////////////////////////////////////////////////////////////////////

void
TPoint::destroy(oaOccObject *oPtr)
{
#if defined(DEBUG)
    //std::cout << "Destroy TPoint" << std::endl;
#endif
    assert(oPtr);
    TPoint *tp;
    if (oPtr->isOccTerm()) {
        oaOccTerm *term = static_cast<oaOccTerm *>(oPtr);
        tp = static_cast<TPoint *>(_termAppDef->get(term));
        if (tp) {
            delete tp;
            _termAppDef->destroy(term);
        }
        tp = static_cast<TPoint *>(_clockTermAppDef->get(term));
        if (tp) {
            delete tp;
            _clockTermAppDef->destroy(term);
        }
    } else {
        oaOccInstTerm *instTerm = static_cast<oaOccInstTerm *>(oPtr);
        tp = static_cast<TPoint *>(_instTermAppDef->get(instTerm));
        if (tp) {
            delete tp;
            _instTermAppDef->destroy(instTerm);
        }
        tp = static_cast<TPoint *>(_clockInstTermAppDef->get(instTerm));
        if (tp) {
            delete tp;
            _clockInstTermAppDef->destroy(instTerm);
        }
    }
}

///////////////////////////////////////////////////////////////////////////////

}
