/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Zhong Xiu <zxiu@andrew.cmu.edu>

ChangeLog:
2004-09-15: ChangeLog started
*/

#if !defined(oagTimerModel_P)
#define oagTimerModel_P

#include <float.h>
#include <vector>

namespace oagTimer {

/*! This structure collects the necessary information of a pin. */
struct pinData {
  char *name; //!< the pin's name
  double cap; //!< the pin's capacitance
  double loadLimit; //!< the pin's load limit
  bool isClock; //!< whether the pin is a clock pin or not
};


/* TEMPORARY time and delay type definitions */
typedef double TimeType;
typedef double DelayType;

/*! This is the vector of the pinData */
typedef std::vector<pinData> pinDataVector;

/*!
  The TimerModel class contains the lookup table information, which 
  can be used to compute the slew rate and the delay value of an internal arc. 
*/
class TimerModel {
  public:
    TimerModel();
    TimerModel(const TimerModel &c);
    /*! The desctructor, which is empty now. */
    ~TimerModel() {}

    int pushLoadAxis(double d);
    int pushClockSlewAxis(double d);
    int pushInputSlewAxis(double d);
    int pushSlewAxis(double d);

    void reserveData();

    int pushData(double d);

    void clear();

    DelayType lookup(double load, DelayType slew, double loadLimit);

    /*! maximum time value TWEAK */
    static TimeType MAX_TIME() { return DBL_MAX; }
    /*! zero time value TWEAK */
    static TimeType ZERO_TIME() { return 0.0; }
    /*! maximum delay value TWEAK */
    static DelayType MAX_DELAY() { return DBL_MAX; }
    /*! zero delay value TWEAK */
   static DelayType ZERO_DELAY() { return 0.0; }

  private:
    unsigned int loadSize, slewSize, dataSize;
    std::vector<double> loadAxis, slewAxis, tableData;
    /*! this flag shows whether slew appears first or not */
    bool originalSlewMajor;
    /*! a pointer which points to the next empty data position */
    unsigned int ptr;

};

/*! 
  This structure contains the necessary information of an internal arc.
*/
struct pathData {
  char *src; //!< the name of the source node
  char *dst; //!< the name of the destination node
  const TimerModel *delay; //!< the timing model for the delay value
  const TimerModel *slew; //!< the slew model for the slew rate
  int srcTrans; //!< the transition type at the source node
  int dstTrans; //!< the transition type at the destination node
};

/*! This is the vector of the paths */
typedef std::vector<pathData> pathDataVector;

/* TEMPORARY */
typedef TimerModel ModelType;
}

#endif
