/************************************************************
* 
* File: oagTimerDesignTool.cpp
* Author: Santiago Mok (smok@ee.ucla.edu)
* Created: 07-08-2010
* Last Modified: Mon 04 Apr 2011 05:24:28 PM PDT
*
************************************************************/
#include "oagTimerDesignTool.h"
#include "oagTimerUtil.h"

#include <iostream>
#include <vector>

//#define DEBUG
#define MODE_READ 'r'
namespace oagTimer{
/*---------------------------------------------------------*/
using namespace oa;
using namespace std;
/*---------------------------------------------------------*/

int
DesignTool::getMaxDepth(oaDesign *design){
    assert(design);
    vector<oaModInst*> PI_insts;
    vector<oaModInst*>::iterator it;
    vector<oaModTerm*> PI = getAllModPI(design);
    for(vector<oaModTerm*>::iterator tIter=PI.begin();
	    tIter!=PI.end(); ++tIter){
	oaModNet* net = (*tIter)->getNet();
	oaIter<oaModInstTerm> instTermIter(net->getInstTerms());
	while(oaModInstTerm* i = instTermIter.getNext()){
	    oaTermType type(i->getTerm()->getTermType());
	    if(type == oacInputTermType){
		oaModInst* inst = i->getInst();
		it = find(PI_insts.begin(), PI_insts.end(), inst);
		if(it==PI_insts.end()) PI_insts.push_back(inst);
	    }
	}
    }
    int max_depth = 0;
    for(it=PI_insts.begin(); it!=PI_insts.end(); ++it){
	int depth = 0;
	getDepth(*it,0,depth);
	if(depth > max_depth) max_depth = depth;
    }
    return max_depth;
}
/*---------------------------------------------------------*/
void
DesignTool::getDepth(oaModInst* inst,int curr, int &max){
    oaModNet* net = getOutputNet(inst);
    if(isPO(net)){ 
	if(curr > max) max = curr;
	return;
    }
    oaIter<oaModInstTerm> instTermIter(net->getInstTerms());
    while(oaModInstTerm *i = instTermIter.getNext()){
	oaTermType type(i->getTerm()->getTermType());
	if(type == oacInputTermType){
	    oaModInst* inst = i->getInst();
	    getDepth(inst,curr+1,max);
	}
    }
}
/*---------------------------------------------------------*/
bool
DesignTool::isPO(oaModNet* net){
    oaIter<oaModTerm> termIter(net->getTerms());
    while(oaModTerm* t = termIter.getNext()){
	oaTermType type(t->getTermType());
	if(type == oacOutputTermType) return true;
    }
    return false;
}
/*---------------------------------------------------------*/
vector<oaModInst*>
DesignTool::getReverseTopological(oaDesign *design, Timer *timer){
#if defined(DEBUG)
    std::cout << "getReverseTopological()" << std::endl;
#endif
    assert(design);
    vector<oaModNet*> nets;
    //For all input nets (D-input) of FF
    // **NEED TO CHANGE THIS TAKEN FROM "getTopological()"
    for (std::set<oaOccObject*>::iterator i=timer->clockPins.begin();
         i!=timer->clockPins.end(); ++i) {
        oaOccInst *inst = (static_cast<oaOccInstTerm*>(*i))->getInst();
	oaIter<oaOccInstTerm> itIter = inst->getInstTerms();
	while(oaOccInstTerm *iTerm = itIter.getNext()){
	    TPoint *pin = TPoint::get(iTerm);
	    if(pin->type != TIMING_POINT_CLOCK || pin->type != TIMING_POINT_SIGNAL_OUT){
		oaModNet *net = iTerm->getNet()->getModNet();	
		nets.push_back(net);
	    }
	}
    }
    //For all PO
    vector<oaModTerm*> PO = getAllModPO(design);
    for(vector<oaModTerm*>::iterator tIter=PO.begin();
	    tIter!=PO.end(); ++tIter){
	nets.push_back((*tIter)->getNet());
    }
    vector<oaModInst*> design_insts;
    while(!nets.empty()){
	oaModNet *net = nets.back();
	nets.pop_back();

	oaIter<oaModInstTerm> itIter = net->getInstTerms();
	while(oaModInstTerm *iTerm = itIter.getNext()){ 
	    oaOccInstTerm *occInstTerm = getOccInstTerm(design,iTerm);
	    TPoint *pin = TPoint::get(occInstTerm);
	    assert(pin);
	    if(pin->type == TIMING_POINT_SIGNAL_OUT){
		oaIter<oaOccNet> occNetIter = net->getOccNets(design->getTopOccurrence());
		oaOccNet *n = occNetIter.getNext();
		if(allInPinValid(n)){ 
		    oaModInst *inst = iTerm->getInst();
		    CellData *cell = CellData::get(inst);
		    assert(cell);
		    //std::cout << " cell " << cell->instName << " is valid" << std::endl;
		    if(cell->master->isSequential || cell->revTopFlag) continue;
		    
		    //Mark input of all current cell instance as valid
		    oaIter<oaModInstTerm> itIter2 = inst->getInstTerms();
		    while(oaModInstTerm *inITerm = itIter2.getNext()){ 
			oaOccInstTerm *occInstTerm2 = getOccInstTerm(design,inITerm);
			TPoint *inPin = TPoint::get(occInstTerm2);
			assert(inPin);
			if(inPin->type == TIMING_POINT_SIGNAL_IN){
			    inPin->revTopValid = true;
			}
		    }
		    
		    design_insts.push_back(inst);
		    cell->revTopFlag = true;
		    vector<oaModNet*> inPinNets = getInputNets(inst);
		    for(int i=0; i<inPinNets.size(); i++){	
			nets.push_back(inPinNets[i]);
		    }
		}
	    }
	}
    }
    /*! DEBUG*/
    /*std::cout << "Topologically Sorted:" << std::endl;
    for(vector<oaModInst*>::iterator it=design_insts.begin();
	    it!=design_insts.end(); ++it){
	std::cout << Util::getInstName(*it) << std::endl;
    }*/
    //Clean flags created in TPoint in case getReverseTopological get call again
    oaOccurrence *top = design->getTopOccurrence();
    oaIter<oaOccInstTerm> desIter = top->getInstTerms();
    while(oaOccInstTerm *i = desIter.getNext()){
	TPoint *tp = TPoint::get(i);
	tp->revTopValid = false;
    }
    oaModule *topMod = design->getTopModule();
    oaIter<oaModInst> modIter = top->getInsts();
    while(oaModInst *j = modIter.getNext()){
	CellData *cell = CellData::get(j);
	assert(cell);
	cell->revTopFlag = false;
    }
    return design_insts;
}
/*---------------------------------------------------------*/
vector<oaModInst*>
DesignTool::getTopological(oaDesign *design, Timer *timer){
#if defined(DEBUG)
    std::cout << "getTopological()" << std::endl;
#endif
    assert(design);
    vector<oaModNet*> pi_nets;
    //For all output nets of FF
    for (std::set<oaOccObject*>::iterator i=timer->clockPins.begin();
         i!=timer->clockPins.end(); ++i) {
        oaOccInst *inst = (static_cast<oaOccInstTerm*>(*i))->getInst();
	oaModNet *net = getOutputNet(inst)->getModNet();
	pi_nets.push_back(net);
    }
    //For all PI
    vector<oaModTerm*> PI = getAllModPI(design);
    for(vector<oaModTerm*>::iterator tIter=PI.begin();
	    tIter!=PI.end(); ++tIter){
	pi_nets.push_back((*tIter)->getNet());
    }
    vector<oaModInst*> design_insts;
    buildTopologicalDesign(design_insts,pi_nets);
    /*! DEBUG*/
    /*std::cout << "Topologically Sorted:" << std::endl;
    for(vector<oaModInst*>::iterator it=design_insts.begin();
	    it!=design_insts.end(); ++it){
	std::cout << Util::getInstName(*it) << std::endl;
    }*/
    return design_insts;
}
/*---------------------------------------------------------*/
void
DesignTool::buildTopologicalDesign(vector<oaModInst*> &insts,vector<oaModNet*> nets){
#if defined(DEBUG)
    std::cout << "buildTopologicalDesign()" << std::endl;
#endif
    vector<oaModNet*> output_nets; 
    for(vector<oaModNet*>::iterator netIter=nets.begin(); netIter!=nets.end(); ++netIter){
	oaIter<oaModInstTerm> instTermIter((*netIter)->getInstTerms()); 
	while(oaModInstTerm *i = instTermIter.getNext()){
	    oaTermType type(i->getTerm()->getTermType());
	    if(type == oacInputTermType){
		oaModInst* inst = i->getInst();
		CellData *cell = CellData::get(inst);
		assert(cell);
		cell->mark(i);
		if(cell->master->isSequential) continue;
		if(cell->isAllInputMark()){ 
		    vector<oaModInst*>::iterator it;
		    it = find(insts.begin(), insts.end(), inst);
		    if(it == insts.end()) insts.push_back(inst);
		    output_nets.push_back(getOutputNet(inst));
		}
	    } 
	}
    }
    if(!output_nets.empty()){
	buildTopologicalDesign(insts, output_nets);
    }
}
/*---------------------------------------------------------*/
vector<instTimingData>
DesignTool::getIncreasingSlack(oaDesign *design, Timer *timer){

    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    vector<instTimingData> design_insts;
    oaIter<oaOccInst> instIter ( occ->getInsts());
    while(oaOccInst *occInst = instIter.getNext()){
	instTimingData data;
	DelayType min_slack = ModelType::MAX_DELAY();
	oaIter<oaOccInstTerm> instTermIter (occInst->getInstTerms());
	while(oaOccInstTerm *iTerm = instTermIter.getNext()){
	    DelayType slack = timer->getSlack(iTerm);
	    if(slack < min_slack) min_slack = slack;
	}
	data.inst = occInst->getModInst();
	data.out_Slack = min_slack;
	design_insts.push_back(data);
    }
    sort(design_insts.begin(), design_insts.end(), incr_slack_sort);
    return design_insts;
}
/*---------------------------------------------------------*/
vector<instTimingData>
DesignTool::getDecreasingSlack(oaDesign *design, Timer *timer){

    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    vector<instTimingData> design_insts;
    oaIter<oaOccInst> instIter ( occ->getInsts());
    while(oaOccInst *occInst = instIter.getNext()){
	instTimingData data;
	DelayType min_slack = ModelType::MAX_DELAY();
	oaIter<oaOccInstTerm> instTermIter (occInst->getInstTerms());
	while(oaOccInstTerm *iTerm = instTermIter.getNext()){
	    DelayType slack = timer->getSlack(iTerm);
	    if(slack < min_slack) min_slack = slack;
	}
	data.inst = occInst->getModInst();
	data.out_Slack = min_slack;
	design_insts.push_back(data);
    }
    sort(design_insts.begin(), design_insts.end(), decr_slack_sort);
    return design_insts;
}
/*---------------------------------------------------------*/
vector<instTimingData>
DesignTool::getDecreasingSlew(oaDesign *design, Timer *timer){

    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);
    vector<instTimingData> design_insts;
    oaIter<oaOccInst> instIter ( occ->getInsts());
    while(oaOccInst *occInst = instIter.getNext()){
	instTimingData data;
	DelayType max_slew = ModelType::ZERO_DELAY();
	oaIter<oaOccInstTerm> instTermIter (occInst->getInstTerms());
	while(oaOccInstTerm *iTerm = instTermIter.getNext()){
	    DelayType slew = timer->getSlew(iTerm);
	    if(slew > max_slew) max_slew = slew;
	}
	data.inst = occInst->getModInst();
	data.out_Slew = max_slew;
	design_insts.push_back(data);
    }
    sort(design_insts.begin(), design_insts.end(), decr_slew_sort);
    return design_insts;
}
/*---------------------------------------------------------*/
vector<instTimingData>
DesignTool::getDecreasingFanoutWeight(oaDesign *design, Timer *timer, int level){
    //printSensitivityList(sensitivity_Insts);
    oaModule *mod = design->getTopModule();
    assert(mod);
    vector<instTimingData> design_insts;
    oaIter<oaModInst> instIter ( mod->getInsts());
    while(oaModInst *modInst = instIter.getNext()){
	CellData *cell = CellData::get(modInst);
	assert(cell);
	if(cell->master->isSequential) continue;
	instTimingData data;
	data.inst = modInst;
	//cell->unsetNode();
	if(cell->getUpSizeCount()){
	    double fanoutWeight = getFanoutWeight(data,modInst,level);
	    //if(data.fanoutWeight != 0) 
	    //design_insts.push_back(data);

	    //Get input slack and input slew
	    oaOccInst* occInst = getOccInst(design,modInst);
	    DelayType min_slack = ModelType::MAX_DELAY();
	    DelayType max_slew = ModelType::ZERO_DELAY();

	    oaIter<oaOccInstTerm> instTermIter (occInst->getInstTerms());
	    while(oaOccInstTerm *iTerm = instTermIter.getNext()){
		oaTermType type(iTerm->getTerm()->getTermType());
		if(type == oacOutputTermType){
		    DelayType slack = timer->getSlack(iTerm);
		    if(slack < min_slack) min_slack = slack;

		    DelayType slew = timer->getSlew(iTerm);
		    if(slew > max_slew) max_slew = slew;
		}
	    }
	    data.out_Slack = min_slack;
	    data.out_Slew = max_slew;
	    //data.fanoutWeight = fabs(fanoutWeight/min_slack);
	    data.fanoutWeight = fanoutWeight*max_slew;
	    //data.fanoutWeight = fanoutWeight*max_slew/min_slack;
	    
	    design_insts.push_back(data);
	}
    }
    sort(design_insts.begin(), design_insts.end(), decr_fanoutWeight_sort);
    return design_insts;
}
/*---------------------------------------------------------*/
vector<instTimingData>
DesignTool::getDecreasingFanoutSlack(oaDesign *design, Timer *timer, int level){
    //printSensitivityList(sensitivity_Insts);
    oaModule *mod = design->getTopModule();
    assert(mod);
    vector<instTimingData> design_insts;
    oaIter<oaModInst> instIter ( mod->getInsts());
    while(oaModInst *modInst = instIter.getNext()){
	CellData *cell = CellData::get(modInst);
	assert(cell);
	if(cell->master->isSequential) continue;
	instTimingData data;
	data.inst = modInst;
	double fanoutWeight = getFanoutSlack(design,timer,modInst,level);
	if(fanoutWeight == DBL_MAX) continue;

	//Get input slack and input slew
	oaOccInst* occInst = getOccInst(design,modInst);
	DelayType min_slack = ModelType::MAX_DELAY();
	DelayType max_slew = ModelType::ZERO_DELAY();

	oaIter<oaOccInstTerm> instTermIter (occInst->getInstTerms());
	while(oaOccInstTerm *iTerm = instTermIter.getNext()){
	    oaTermType type(iTerm->getTerm()->getTermType());
	    if(type == oacOutputTermType){
		DelayType slack = timer->getSlack(iTerm);
		if(slack < min_slack) min_slack = slack;

		DelayType slew = timer->getSlew(iTerm);
		if(slew > max_slew) max_slew = slew;
	    }
	}
	data.out_Slack = min_slack;
	data.out_Slew = max_slew;
	data.fanoutWeight = fanoutWeight*max_slew;
	design_insts.push_back(data);
    }
    sort(design_insts.begin(), design_insts.end(), decr_fanoutWeight_sort);
    return design_insts;
}
/*---------------------------------------------------------*/
vector<instTimingData>
DesignTool::getDecreasingFanoutGain(oaDesign *design, Timer *timer, int level){
    oaModule *mod = design->getTopModule();
    assert(mod);
    vector<instTimingData> design_insts;
    oaIter<oaModInst> instIter ( mod->getInsts());
    //oaString myCell = "starCell_2527";
    while(oaModInst *modInst = instIter.getNext()){
	CellData *cell = CellData::get(modInst);
	assert(cell);
	if(cell->master->isSequential) continue;
	oaOccInst* occInst = getOccInst(design,modInst);
	DelayType max_slew;
	oaIter<oaOccInstTerm> instTermIter (occInst->getInstTerms());
	while(oaOccInstTerm *iTerm = instTermIter.getNext()){
	    oaTermType type(iTerm->getTerm()->getTermType());
	    if(type == oacOutputTermType){
		max_slew = timer->getSlew(iTerm);
	    }
	}
	instTimingData data;
	data.inst = modInst;
	/*if(cell->instName == myCell) cout << cell->instName << " found" << endl;
	else continue;*/
	if(cell->getUpSizeCount()){
	    double upDelta = Util::getUpSizeDeltaDelay(design,timer,cell);
	    double gain = getFanoutGain(design,timer,modInst,upDelta,level);
	    if(gain == DBL_MAX) continue;
	    //cout << " updDelay:" << upDelta << " Weight:" << gain << std::endl;
	    data.fanoutWeight = gain*max_slew;
	    design_insts.push_back(data);
	}
    }
    sort(design_insts.begin(), design_insts.end(), decr_fanoutWeight_sort);
    return design_insts;
}
/*---------------------------------------------------------*/
void
DesignTool::getNetWeight(oaDesign *design,Timer *timer){
    vector<oaOccTerm*> PI = getAllOccPI(design);
    vector<oaOccNet*> pi_nets;
    for(vector<oaOccTerm*>::iterator tIter=PI.begin();
	    tIter!=PI.end(); ++tIter){
	pi_nets.push_back((*tIter)->getNet());
    }
    vector<oaOccInst*> design_insts;
    traverseForwardTopologically(design_insts,pi_nets,timer);
    traverseBackwardTopologically(design_insts,timer);
    
    /*! DEBUG*/
    vector<netData> nd;
    std::cout << "Net Weight: Topologically Sorted:" << std::endl;
    for(vector<oaOccInst*>::iterator it=design_insts.begin();
	    it!=design_insts.end(); ++it){
	std::cout << Util::getInstName((*it)->getModInst()) << std::endl;
	CellData *cd = CellData::get((*it)->getModInst());
	assert(cd);
	/*
	oaModInst *inst = (*it)->getModInst();
	oaIter<oaModInstTerm> instTermIter = inst->getInstTerms();
	while(oaModInstTerm* i = instTermIter.getNext()){
	    oaTermType type(i->getTerm()->getTermType());
	    if(type == oacInputTermType){
		std::cout << " " << timer->getBlockName(getOccInstTerm(design,i)) << "-";
		std::cout << " Discout " << cd->inputTermsNetMap[i].discount;
		std::cout << " FWD Weight: " << cd->inputTermsNetMap[i].forward_weight;
		std::cout << " BWD Weight: " << cd->inputTermsNetMap[i].backward_weight;
		std::cout << " NET Weight: " << cd->inputTermsNetMap[i].weight << std::endl;
	    }
	}*/
    }
    //return design_insts;
}
/*---------------------------------------------------------*/
void
DesignTool::traverseForwardTopologically(vector<oaOccInst*> &insts,vector<oaOccNet*> nets,
	Timer* timer){
#if defined(DEBUG)
    std::cout << "traverseForwardTopologically()" << std::endl;
#endif
    vector<oaOccNet*> output_nets; 
    for(vector<oaOccNet*>::iterator netIter=nets.begin(); netIter!=nets.end(); ++netIter){
	oaIter<oaOccInstTerm> instTermIter((*netIter)->getInstTerms()); 
	while(oaOccInstTerm *i = instTermIter.getNext()){
	    oaTermType type(i->getTerm()->getTermType());
	    if(type == oacInputTermType){
		oaModInst* inst = i->getInst()->getModInst();
		CellData *cell = CellData::get(inst);
		assert(cell);
		cell->mark(i->getModInstTerm());
		if(cell->isAllInputMark()){ 
		    vector<oaOccInst*>::iterator it;
		    it = find(insts.begin(), insts.end(), i->getInst());
		    if(it == insts.end()) insts.push_back(i->getInst());
		    output_nets.push_back(getOutputNet(i->getInst()));
		}
		/*Compute Net Weight*/
		DelayType slack = timer->getSlack(i);
		TimeType worst_delay = timer->worstArr;
		double val = -1 * (slack/worst_delay);
		double discount = exp(val);
		double fwd_weight = discount*getPrevTermWeightSum(i);
		cell->setForwardWeight(i->getModInstTerm(),fwd_weight,discount);
	    } 
	}
    }
    if(!output_nets.empty()){
	traverseForwardTopologically(insts, output_nets,timer);
    }
}
/*---------------------------------------------------------*/
void
DesignTool::traverseBackwardTopologically(vector<oaOccInst*> &insts, Timer* timer){
#if defined(DEBUG)
    std::cout << "traverseBackwardTopologically()" << std::endl;
#endif
    //vector<oaOccTerm*> PO = getAllOccPO();
    for(vector<oaOccInst*>::reverse_iterator rev_it=insts.rbegin();
	    rev_it!=insts.rend(); ++rev_it){
	oaIter<oaOccInstTerm> instTermIter = (*rev_it)->getInstTerms();	
	while(oaOccInstTerm* instTerm = instTermIter.getNext()){
	    oaTermType type(instTerm->getTerm()->getTermType());
	    if(type == oacOutputTermType){
		oaModInst* inst = instTerm->getInst()->getModInst();
		CellData *cell = CellData::get(inst);
		assert(cell);
		/*DelayType slack = timer->getSlack(instTerm);
		TimeType worst_delay = timer->worstArr;
		double val = -1 * (slack/worst_delay);
		double discount = exp(val);
		*/
		double back_weight = getPrevTermWeightSum(instTerm);
		cell->setBackwardWeight(back_weight);
	    }
	}
    }
}
/*---------------------------------------------------------*/
/*
double
DesignTool::computeWeight(oaOccInstTerm* iTerm, Timer* t){
}*/
/*---------------------------------------------------------*/
double
DesignTool::getPrevTermWeightSum(oaOccInstTerm* iTerm){
    oaTermType currTermType(iTerm->getTerm()->getTermType());
    if(currTermType == oacInputTermType){
	oaIter<oaOccInstTerm> instTermIter = iTerm->getNet()->getInstTerms();
	while(oaOccInstTerm* otherTerm = instTermIter.getNext()){
	    oaTermType type(otherTerm->getTerm()->getTermType());
	    if(type == oacOutputTermType){
		oaModInst *inst = otherTerm->getInst()->getModInst();
		CellData *cell = CellData::get(inst);
		return cell->getNetWeightSum();
	    }
	}
	double Ff = 0.0;
	oaIter<oaOccTerm> termIter = iTerm->getNet()->getTerms();
	while(oaOccTerm* otherTerm = termIter.getNext()){
	    oaTermType type(otherTerm->getTermType());
	    if(type == oacInputTermType){
		Ff += 1.0;
	    }
	}
	return Ff;
    }else if(currTermType == oacOutputTermType){
	/*Compute Backward Net Weight at PO*/
	oaIter<oaOccTerm> termIter = iTerm->getNet()->getTerms();
	while(oaOccTerm* otherTerm = termIter.getNext()){
	    oaTermType type(otherTerm->getTermType());
	    if(type == oacOutputTermType){
		return 1.0;
	    }
	}
	double Bb = 0.0;	
	oaIter<oaOccInstTerm> instTermIter = iTerm->getNet()->getInstTerms();
	while(oaOccInstTerm* otherTerm = instTermIter.getNext()){
	    oaTermType type(otherTerm->getTerm()->getTermType());
	    if(type == oacInputTermType){
		oaModInst *inst = otherTerm->getInst()->getModInst();
		CellData *cell = CellData::get(inst);
		Bb += cell->getBackwardWeight(otherTerm->getModInstTerm());
	    }
	}
	return Bb;
    }
}
/*---------------------------------------------------------*/
void
DesignTool::getFanins(oaModInst* head, vector<oaModInst*> &insts, int lvl){
    vector<oaModNet*> nets = getInputNets(head);
    for(vector<oaModNet*>::iterator it=nets.begin();
	    it!=nets.end(); ++it){
	oaIter<oaModInstTerm> instTermIter((*it)->getInstTerms());
	while(oaModInstTerm *i = instTermIter.getNext()){
	    oaTermType type(i->getTerm()->getTermType());
	    if(type == oacOutputTermType){
		oaModInst* inst = i->getInst();
		if(lvl == 1){
		    vector<oaModInst*>::iterator it;
		    it = find(insts.begin(), insts.end(),inst); 
		    if(it==insts.end()) insts.push_back(inst);
		}else getFanins(inst,insts,lvl-1);
	    }
	}
    }
}
/*---------------------------------------------------------*/
void
DesignTool::getFanouts(oaModInst* head, vector<oaModInst*> &insts, int lvl){
    oaModNet* net = getOutputNet(head);
    oaIter<oaModInstTerm> instTermIter(net->getInstTerms());
    while(oaModInstTerm *i = instTermIter.getNext()){
	oaTermType type(i->getTerm()->getTermType());
	if(type == oacInputTermType){
	    oaModInst* inst = i->getInst();
	    if(lvl == 1){
		vector<oaModInst*>::iterator it;
		it = find(insts.begin(), insts.end(),inst); 
		if(it==insts.end()) insts.push_back(inst);
	    }else getFanouts(inst,insts,lvl-1);
	}
    }
}
/*---------------------------------------------------------*/
double
DesignTool::getFanoutSlack(oaDesign *top, Timer *timer, oaModInst* inst, int level){
    vector<oaModInst*> fanouts;
    getFanouts(inst,fanouts,level);
    if(fanouts.empty()) return DBL_MAX;
    double weight = 0.0;
    for(vector<oaModInst*>::iterator it=fanouts.begin();
	    it!=fanouts.end(); ++it){
	CellData *cell = CellData::get(*it);
	assert(cell);
	if(cell->getDownSizeCount()){
	    oaOccInst *occInst = DesignTool::getOccInst(top,*it);
	    oaOccInstTerm *outTerm = DesignTool::getOccOutInstTerm(occInst);
	    DelayType slack = timer->getSlack(outTerm);
	    weight += slack;
	}
    }
    return weight;
}
/*---------------------------------------------------------*/
double
DesignTool::getFanoutWeight(instTimingData &data, oaModInst* inst, int level){
    //printSensitivityList(sense_Vector);
    //return 0.0;
    int dnSzCnt = 0;
    int dnSzable = 0;
    data.downSizeable = dnSzable;
    vector<oaModInst*> fanouts;
    if(!fanouts.empty()) fanouts.clear();

    getFanouts(inst,fanouts,level);
    if(fanouts.empty()) return 0;
    double weight = 0.0;
    for(vector<oaModInst*>::iterator it=fanouts.begin();
	    it!=fanouts.end(); ++it){
	CellData *cell = CellData::get(*it);
	assert(cell);
	dnSzCnt = cell->getDownSizeCount();
	weight += (double)dnSzCnt;
	if(dnSzCnt) ++dnSzable;
    }
    data.downSizeable = dnSzable;
    return weight;
}
/*---------------------------------------------------------*/
double
DesignTool::getFanoutGain(oaDesign *top, Timer *timer, oaModInst* inst, 
	double upDelta, int level){
    vector<oaModInst*> fanouts;
    getFanouts(inst,fanouts,level);
    if(fanouts.empty()) return DBL_MAX;
    bool validFanout = false;
    double weight = 0.0;
    for(vector<oaModInst*>::iterator it=fanouts.begin();
	    it!=fanouts.end(); ++it){
	CellData *cell = CellData::get(*it);
	assert(cell);
	if(cell->master->isSequential) return DBL_MAX;
	if(cell->getDownSizeCount()){
	    validFanout = true;
	    oaOccInst *occInst = DesignTool::getOccInst(top,*it);
	    oaOccInstTerm *outTerm = DesignTool::getOccOutInstTerm(occInst);
	    DelayType slack = timer->getSlack(outTerm);
	    double downDelta = Util::getDownSizeDeltaDelay(top,timer,cell);
	    if(downDelta < ((-1.0*upDelta)+slack)){ 
		//++weight;
		vector<oaString> size_vec = cell->getLessThanCurrentSize();
		vector<oaString>::iterator it=size_vec.begin();
		weight += Util::getCellLeakagePower(*it); 
	    }
	    /*std::cout << " Slack:" << slack << "Check" << ((-1.0*upDelta)+slack) << std::endl;
	    std::cout << " Weight:" << weight << std::endl;*/
	}
    }
    if(!validFanout) return DBL_MAX;
    return weight;
}
/*---------------------------------------------------------*/
vector<oaModNet*>
DesignTool::getInputNets(oaModInst* inst){
#if defined(DEBUG)
    std::cout << "getInputNets" << std::endl;
#endif
    vector<oaModNet*> nets;
    oaIter<oaModInstTerm> instTermIter(inst->getInstTerms());
    while(oaModInstTerm* i = instTermIter.getNext()){
	oaTermType type(i->getTerm()->getTermType());
	if(type == oacInputTermType){
	    nets.push_back(i->getNet());
	}
    }
    return nets;
}
/*---------------------------------------------------------*/
vector<oaOccNet*>
DesignTool::getInputNets(oaOccInst* inst){
#if defined(DEBUG)
    std::cout << "getInputNets" << std::endl;
#endif
    vector<oaOccNet*> nets;
    oaIter<oaOccInstTerm> instTermIter(inst->getInstTerms());
    while(oaOccInstTerm* i = instTermIter.getNext()){
	oaTermType type(i->getTerm()->getTermType());
	if(type == oacInputTermType){
	    nets.push_back(i->getNet());
	}
    }
    return nets;
}
/*---------------------------------------------------------*/
oaModNet*
DesignTool::getOutputNet(oaModInst* inst){
#if defined(DEBUG)
    std::cout << "getOutputNet" << std::endl;
#endif
    oaIter<oaModInstTerm> instTermIter(inst->getInstTerms());
    while(oaModInstTerm* i = instTermIter.getNext()){
	oaTermType type(i->getTerm()->getTermType());
	if(type == oacOutputTermType){
	    return i->getNet();
	}
    }
}
/*---------------------------------------------------------*/
oaOccNet*
DesignTool::getOutputNet(oaOccInst* inst){
#if defined(DEBUG)
    std::cout << "getOutputNet" << std::endl;
#endif
    oaIter<oaOccInstTerm> instTermIter(inst->getInstTerms());
    while(oaOccInstTerm* i = instTermIter.getNext()){
	oaTermType type(i->getTerm()->getTermType());
	if(type == oacOutputTermType){
	    return i->getNet();
	}
    }
}
/*---------------------------------------------------------*/
vector<oaModTerm*>
DesignTool::getAllModPI(oaDesign* design){
#if defined(DEBUG)
    std::cout << "getAllModPI" << std::endl;
#endif
    assert(design);
    oaModule *mod = design->getTopModule();
    assert(mod);

    vector<oaModTerm*> PI;
    oaIter<oaModTerm> termIter(mod->getTerms());
    while (oaModTerm *i = termIter.getNext()) {
	oaTermType type(i->getTermType());
	if(type == oacInputTermType){
	    PI.push_back(i);
	}
    }
    return PI;
}
/*---------------------------------------------------------*/
vector<oaOccTerm*>
DesignTool::getAllOccPI(oaDesign* design){
#if defined(DEBUG)
    std::cout << "getAllOccPI" << std::endl;
#endif
    assert(design);
    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);

    vector<oaOccTerm*> PI;
    oaIter<oaOccTerm> termIter(occ->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
	oaTermType type(i->getTermType());
	if(type == oacInputTermType){
	    PI.push_back(i);
	}
    }
    return PI;
}
/*---------------------------------------------------------*/
vector<oaModTerm*>
DesignTool::getAllModPO(oaDesign* design){
#if defined(DEBUG)
    std::cout << "getAllModPO" << std::endl;
#endif
    assert(design);
    oaModule *mod = design->getTopModule();
    assert(mod);

    vector<oaModTerm*> PO;
    oaIter<oaModTerm> termIter(mod->getTerms());
    while (oaModTerm *i = termIter.getNext()) {
	oaTermType type(i->getTermType());
	if(type == oacOutputTermType){
	    PO.push_back(i);
	}
    }
    return PO;
}
/*---------------------------------------------------------*/
vector<oaOccTerm*>
DesignTool::getAllOccPO(oaDesign* design){
#if defined(DEBUG)
    std::cout << "getAllOccPO" << std::endl;
#endif
    assert(design);
    oaOccurrence *occ = design->getTopOccurrence();
    assert(occ);

    vector<oaOccTerm*> PO;
    oaIter<oaOccTerm> termIter(occ->getTerms());
    while (oaOccTerm *i = termIter.getNext()) {
	oaTermType type(i->getTermType());
	if(type == oacOutputTermType){
	    PO.push_back(i);
	}
    }
    return PO;
}
/*---------------------------------------------------------*/
DelayType
DesignTool::getMinSlack(oaDesign* design, Timer* timer, oaModInst* inst){
    DelayType min_slack = ModelType::MAX_DELAY();
    oaOccInst* occInst = getOccInst(design, inst);
    oaIter<oaOccInstTerm> instTermIter ( occInst->getInstTerms());
    while(oaOccInstTerm *instTerm = instTermIter.getNext()){
	oaTermType type(instTerm->getTerm()->getTermType());
	if(type == oacInputTermType){
	    DelayType t = timer->getSlack(instTerm);
	    if(t < min_slack) min_slack = t;
	}
    }
    //return min_slack;
}
/*---------------------------------------------------------*/
DelayType
DesignTool::getMaxSlew(oaDesign* design, Timer* timer, oaModInst* inst){
    DelayType max_slew = ModelType::ZERO_DELAY();
    oaOccInst* occInst = getOccInst(design, inst);
    oaIter<oaOccInstTerm> instTermIter ( occInst->getInstTerms());
    while(oaOccInstTerm *instTerm = instTermIter.getNext()){
	oaTermType type(instTerm->getTerm()->getTermType());
	if(type == oacInputTermType){
	    DelayType t = timer->getSlew(instTerm);
	    if(t > max_slew) max_slew = t;
	}
    }
    return max_slew;
}
/*---------------------------------------------------------*/
oaOccInst*
DesignTool::getOccInst(oaDesign *design, oaModInst *inst){
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInst> occInstIter (inst->getOccInsts(occ));
    while(oaOccInst *occInst = occInstIter.getNext() ){
	return occInst;
    }
}
/*---------------------------------------------------------*/
oaOccInstTerm*
DesignTool::getOccInstTerm(oaDesign *design, oaModInstTerm *instT){
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> occInstIter (instT->getOccInstTerms(occ));
    while(oaOccInstTerm *instT = occInstIter.getNext() ){
	return instT;
    }
}
/*---------------------------------------------------------*/
vector<oaOccInstTerm*>
DesignTool::getOccInInstTerm(oaOccInst *inst){
    assert(inst);
    vector<oaOccInstTerm*> inTerms;
    oaIter<oaOccInstTerm> iTermIter = inst->getInstTerms();
    while(oaOccInstTerm* i = iTermIter.getNext()){
	oaTermType type(i->getTerm()->getTermType());
	if(type == oacInputTermType) inTerms.push_back(i);
    }
    return inTerms;
}
/*---------------------------------------------------------*/
oaOccInstTerm* 
DesignTool::getOccOutInstTerm(oaOccInst *inst){
    assert(inst);
    oaIter<oaOccInstTerm> iTermIter = inst->getInstTerms();
    while(oaOccInstTerm* i = iTermIter.getNext()){
	oaTermType type(i->getTerm()->getTermType());
	if(type == oacOutputTermType) return i;
    }
}
/*---------------------------------------------------------*/
oaModule*
DesignTool::getCellTopMod(oaString name){
    oaNativeNS NS;
    const oaScalarName libertyName(NS, libParseData.libString);
    const oaScalarName libertyView(NS, libParseData.viewString);
    const oaScalarName libertyCell(NS, name);

    oaDesign *lib_cell = oaDesign::open(
	    libertyName, libertyCell, libertyView, MODE_READ);
    assert(lib_cell);
    oaModule* top = lib_cell->getTopModule();
    lib_cell->close();
    return top;
}
/*---------------------------------------------------------*/
int
DesignTool::getGateCount(oaDesign *des){
    oaOccurrence *occ = des->getTopOccurrence();
    assert(occ);
    int cnt = 0;
    oaIter<oaOccInst> instIter(occ->getInsts());
    while(oaOccInst *i = instIter.getNext()) ++cnt;
    return cnt;
}
/*---------------------------------------------------------*/
void
DesignTool::printSensitivityList(vector<sensitivityData> vec){
    std::cout << "DownSize Sorted Sensitivity" << std::endl;
    vector<sensitivityData>::iterator it;
    Util ut;
    for(it=vec.begin(); it!=vec.end(); ++it){
	std::cout << Util::getInstName(it->inst) << " " << ut.getCellName(it->inst) << " "
	    << it->otherLibCellSize << " " << it->sensitivity << std::endl;
    }
    /*
    std::cout << "Upsized Sorted Sensitivity" << std::endl;
    for(it=upsized_sensitivity_Insts.begin(); it!=upsized_sensitivity_Insts.end(); ++it){
	std::cout << Util::getInstName(it->inst) << " " << ut.getCellName(it->inst) << " "
	    << it->otherLibCellSize << " " << it->sensitivity << std::endl;
    }*/
}
/*---------------------------------------------------------*/
void
DesignTool::checkFLAGS(oaDesign *d){
    oaOccurrence *occ = d->getTopOccurrence();
    oaIter<oaOccInstTerm> instTermIter(occ->getInstTerms());
    while (oaOccInstTerm *i = instTermIter.getNext()) {
        TPoint *tp = TPoint::get(i);
	if(tp->riseSlewValid){
	    std::cout << "Error instTerm:" << DesignTool::getBlockName(i)
		<< " still have RISE flag on" << std::endl;
	}
	if(tp->fallSlewValid){
	    std::cout << "Error instTerm:" << DesignTool::getBlockName(i)
		<< " still have FALL flag on" << std::endl;
	}
    }
}
/*---------------------------------------------------------*/
bool 
DesignTool::allInPinValid(oaOccNet *net){
    oaIter<oaOccInstTerm> itIter = net->getInstTerms();
    while(oaOccInstTerm *iTerm = itIter.getNext()){
	TPoint *pin = TPoint::get(iTerm);
	assert(pin);
	if(pin->type == TIMING_POINT_SIGNAL_IN){
	    CellData *cell = CellData::get(iTerm->getInst()->getModInst());
	    assert(cell);
	    if(cell->master->isSequential) continue;
	    if(!pin->revTopValid) return false;

	    //std::cout << Util::getBlockName(iTerm) << " pin revTopValid" << std::endl;
	}
    }
    return true;
}
/*---------------------------------------------------------*/
/*!
  returns the name of block object
  FIXME should move out of the timer class
  \param block either an oaTerm or an oaInstTerm
*/ 
std::string
DesignTool::getBlockName(oaOccObject *oPtr) 
{
    oaString f;
    oaNativeNS ns;
    std::string s;
    if (oPtr->isOccTerm()) {
        static_cast<oaOccTerm*>(oPtr)->getName(ns, f);
        s = std::string(f);
    } else {
        oaOccInst *inst = static_cast<oaOccInstTerm*>(oPtr)->getInst();
        inst->getName(ns, f);
        s = std::string(f);
        static_cast<oaOccInstTerm*>(oPtr)->getTermName(ns, f);
        s = s + "/" + std::string(f);
    }
    return s;
}
/*---------------------------------------------------------*/
}
