/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Zhong Xiu <zxiu@andrew.cmu.edu>

ChangeLog:
2004-09-15: ChangeLog started
*/
#include <map>

#include <assert.h>
#include <iostream>
#include <vector>
#include <cstdlib>
#include "oaDesignDB.h"

#include "oagTimerTimer.h"
#include "oagTimerWireModel.h"
#include "oagTimerElmoreWireModel.h"
#include "oagTimerReport.h"
#include "oagUtilOption.h"
#include "oagUtilOptionParser.h"
#include "oagTimerSensitivity.h"
#include "oagTimerDuetSensitivity.h"
#include "oagTimerOpt.h"
#include "oagTimerUtil.h"
#include "oagTimerTimerUtil.h"
#include "oagTimerKickMove.h"
#include "lpSA.h"
#include "oagTimerLR.h"
#include "oagTimerDP.h"
#include "oagTimerOptMinDelay.h"

#define MODE_READ 'r'
using namespace oa;
using namespace std;
//-----------------------------------------------------------------------------
int
main(int argc, const char **argv) {

    oagUtil::OptionParser options("Analyze the timing of an OpenAccess "
            "design");
    oagUtil::Option *libOpt = options.add("lib", "Input library name", true, 
            "library");
    oagUtil::Option *cellOpt = options.add("cell", "Input cell name", true, 
            "cell");
    oagUtil::Option *viewOpt = options.add("view", "Input view name", true, 
            "view");
    oagUtil::Option *libertyOpt = options.add("liberty", ".lib library file", 
            true, "file");
    oagUtil::Option *libertyLibOpt = options.add("liberty_lib",
            "Lib name for timing data (default: from .lib file)",
            false, "library");
    oagUtil::Option *libertyCellOpt = options.add("liberty_cell",
            "Cell name for swapping instance",
            false, "cell");
    oagUtil::Option *libertyViewOpt = options.add("liberty_view",
            "View name for timing data (default: abstract)",
            false, "view");
    oagUtil::Option *libertyTypeOpt = options.add("liberty_type",
            "Cell name convention for timing library",
            false, "nangate | footprint | st | st_vt");
    oagUtil::Option *instOpt = options.add("instance",
            "Instance to be swap",false, "inst");
    oagUtil::Option *sdcOpt = options.add("sdc", "SDC constraint file", false, 
            "file");
    oagUtil::Option *reportFromOpt = options.add("report_from", "Report worst "
            "timing path from this pin or port", false, "pin/port");
    oagUtil::Option *reportToOpt = options.add("report_to", "Report worst "
            "timing path to this pin or port", false, "pin/port");
    oagUtil::Option *reportThroughOpt = options.add("report_through", "Report "
            "worst timing path through this pin or port", false, "pin/port");
    oagUtil::Option *reportOpt = options.add("report", "Report the timing of "
            "all points, of all endpoints path with the worst slack", false, 
            "timing | endpoint_slack | critical");
    oagUtil::Option *wireOpt = options.add("wire", "elmore: Include elmore wire model in timing analysis;Default: 0 wire delay", false,"model");
    oagUtil::Option *slewOpt = options.add("slew", "Include slew degradation effect in timing analysis calculations", false,"degr");
    oagUtil::Option *powerOpt = options.add("power", "Display cell or design leakage power", false,"design");
    oagUtil::Option *duetOpt = options.add("duet", "Gate sizing based on (power/delay ratio)", false,"power_delay");
    oagUtil::Option *debugDuetOpt = options.add("debugDuet", "Debug Duet power_slack sensitivity", 
            false,"true | false");
    oagUtil::Option *peepholeOpt = options.add("peephole", "Multi-Gate sizing through PeepHole optimization", 
            false,"(1 or 2) fanout level depth");
    oagUtil::Option *optLeakageOpt = options.add("optLeakage", "Perform gate sizing or vt assignment to optimize for Leakage power", 
            false,"lp | lr | power_delay | power_slack | peephole | kickMove | vt");
    oagUtil::Option *kickMovesOpt = options.add("KickMoves", "Specify the number of iteration to let kickmove run", 
            false,"i.e. 10 | 250 | 500 | 1000");
    oagUtil::Option *debugOpt = options.add("debug", "Display design information (i.e. gates count, slack histogram, etc.) for debugging purposes", false,"gatesCount | slackHist | timingInfo");
    oagUtil::Option *statsOpt = options.add("stats", "Display design information (i.e. gates count, slack histogram, etc.", false,"gatesCount | slackHist | downsizeAllCell");
    oagUtil::Option *rootSelectOpt = options.add("rootselect", "root node selection method", false,"0 / 1 / 2/ 3");
    oagUtil::Option *saveOpt = options.add("save", "Write to specified oa design file", 
            false,"design");
    oagUtil::Option *warningMsg = options.add("warning", "Display warning flags", false, 
            "true | false");
    if (!options.parse(argc, argv)) {
        std::cerr << options.getMessage();
        exit(0);
    }

    //try {
        oa::oaDesignInit();
        oa::oaLibDefList::openLibs();

        const oa::oaNativeNS nativeNS;
        const oa::oaScalarName cellName(nativeNS, cellOpt->getValue());
        const oa::oaScalarName libName(nativeNS, libOpt->getValue());
        const oa::oaScalarName viewName(nativeNS, viewOpt->getValue());

        // take top level cellView
        oa::oaDesign *cv = oa::oaDesign::open(libName, cellName, viewName, 'a');
        assert(cv);
        oa::oaOccurrence *occ = cv->getTopOccurrence();
        assert(occ);
        
        // initiate the timing project
        oagTimer::Timer *myTiming = new oagTimer::Timer();
        oagTimer::ElmoreWireModel *EWM = new oagTimer::ElmoreWireModel(); 
        
	if(warningMsg->isGiven()) {
	  if (strcmp(warningMsg->getValue(), "false") == 0) {
	    oagTimer::Timer::displayWarning = false;
	  }
	}
	
	if(libertyTypeOpt->isGiven()){
            libParseData.libType = libertyTypeOpt->getValue();
        }
        myTiming->readLibrary(libertyOpt->getValue(),
                              libertyLibOpt->getValue(),
                              libertyViewOpt->getValue());
        myTiming->add(cv);
        if(wireOpt->isGiven()){
            if(strcmp(wireOpt->getValue(),"elmore")==0)
                myTiming->setWireModel(EWM);
        }
        if(slewOpt->isGiven()){
            if(strcmp(slewOpt->getValue(),"degr")==0)
                myTiming->setSlewDegr();
        }
        if (sdcOpt->isGiven()) {
            myTiming->readConstraints(sdcOpt->getValue(), cv);
        }
        
        oagTimer::Report myReport(cv, myTiming);
        oa::oaOccObject *t;
        oa::oaOccObject *in;

        if (reportFromOpt->isGiven()) {
            t = oagTimer::Timer::findOccObj(occ, reportFromOpt->getValue());
            if (!t) {
                std::cout << " Not found: " << reportFromOpt->getValue() 
                    << std::endl;
            } else {
                oagTimer::nodesSlopeDir path;
                myTiming->fromPath(path, t);
                myReport.pathReport(path);
            }
        }

        if (reportToOpt->isGiven()) {
            t = oagTimer::Timer::findOccObj(occ, reportToOpt->getValue());
            if (!t) {
                std::cout << " Not found: " << reportToOpt->getValue() 
                    << std::endl;
            } else {
                oagTimer::nodesSlopeDir path;
                myTiming->toPath(path, t);
                myReport.pathReport(path);
            }
        }

        if (reportThroughOpt->isGiven()) {
            t = oagTimer::Timer::findOccObj(occ, reportThroughOpt->getValue());
            if (!t) {
                std::cout << " Not found: " << reportThroughOpt->getValue() 
                    << std::endl;
            } else {
                oagTimer::nodesSlopeDir path;
                myTiming->throughPath(path, t);
                myReport.pathReport(path);
            }
        }

        if (reportOpt->isGiven()) {
            if (strcmp(reportOpt->getValue(), "timing")==0) {
                std::cout << "Report Timing:" << std::endl;
                myReport.reportAll();
            } else if (strcmp(reportOpt->getValue(), "endpoint_slack") == 0) {
                myReport.reportEndpointsSlacks();
            } else if (strcmp(reportOpt->getValue(), "critical") == 0) {
                oagTimer::nodesSlopeDir path;
                myTiming->findWorstPath(path);
                myReport.pathReport(path);
            } else {
                std::cerr << "Check the parameter after -report"
                    << " [timing | endpoint_slack | critical]: " 
                    << reportOpt->getValue() << std::endl;
            }
        }
        
        // Report Design Statistics Options
        //Debug timing data on i/o arc
        if (debugOpt->isGiven()) {
            if (strcmp(debugOpt->getValue(), "timingInfo")==0) {
                //Util::reportTimingInfo(cv,myTiming);
                Util::reportAllCellTiming(cv,myTiming);
            }
        }

        if (statsOpt->isGiven()) {
            if(!libertyTypeOpt->isGiven()){
                std::cout << "Error: missing library type!" << std::endl;
                return 0;
            }
            if (strcmp(statsOpt->getValue(), "downsizeAllCell") == 0){
                Util::reportGates(cv);
                Util::changeToMinCellSize(cv,myTiming);
                cout << "Done Downsizing all Cell Instances " << endl;
            }
            if (strcmp(statsOpt->getValue(), "gatesCount") == 0){
              Util::reportGates(cv);
            }
            if (strcmp(statsOpt->getValue(), "slackHist") == 0){
              Util::reportSlacks(cv,myTiming);
            }
        }
        

        // Leakage Optimization Options
/* Swapping and Power Utilities */
        const char* libertyViewName = libertyViewOpt->getValue();
        oaString viewString = libertyViewName ? libertyViewName : "abstract"; 
        const oaScalarName libertyView(nativeNS,viewString);
        if(libertyLibOpt->isGiven())
        const oaScalarName libertyName(nativeNS, libertyLibOpt->getValue()); 

        if(powerOpt->isGiven()){
            Util ut;
            if(strcmp(powerOpt->getValue(), "design")==0){
                std::cout << "Total Design Leakage Power: " << ut.getTotalLeakPower(cv) << "(in library units)" << std::endl;
            }
        }

        if(duetOpt->isGiven()){
            if(!libertyTypeOpt->isGiven()){
                std::cout << "Error: missing library type!" << std::endl;
                return 0;
            }
            oagTimer::DuetSensitivity d(cv,myTiming);
            if(debugDuetOpt->isGiven()){
                d.enableDebug();
            }
            if(strcmp(duetOpt->getValue(),"power_delay")==0){
                d.run();
                if(saveOpt->isGiven()){
                    const char* saveName = saveOpt->getValue();
                    cv->saveAs(libName,oaScalarName(nativeNS,saveName),viewName);
                    std::cout << "Modified Netlist saved to " << saveName << std::endl;
                }
            }
        }

        if(optLeakageOpt->isGiven()){
            if(!libertyTypeOpt->isGiven()){
                std::cout << "Error: missing library type!" << std::endl;
                return 0;
            }
            if(strcmp(optLeakageOpt->getValue(),"lp")==0){
                std::cout << "Performing LP optimization ..." << std::endl;
                lpSA myLpOpt(cv,myTiming);
                myLpOpt.run();
            }
            else if(strcmp(optLeakageOpt->getValue(),"lrdebug")==0){
                std::cout << "Performing Debug ..." << std::endl;
                LR myLROpt(cv,myTiming);
                myLROpt.runDebug();
            }else if(strcmp(optLeakageOpt->getValue(),"lr0")==0){
                std::cout << "Performing Lagrangian optimization ..." << std::endl;
                LR myLROpt(cv,myTiming);
                myLROpt.runDP(0);
            }else if(strcmp(optLeakageOpt->getValue(),"lr1")==0){
                std::cout << "Performing Lagrangian optimization ..." << std::endl;
                LR myLROpt(cv,myTiming);
                myLROpt.runDP(1);
            }else if(strcmp(optLeakageOpt->getValue(),"DP")==0){
                std::cout << "Performing DP - maximizing slack ..." << std::endl;
                DP myDPOpt(cv,myTiming);
                myDPOpt.run();
            }else if(strcmp(optLeakageOpt->getValue(),"kickMove")==0){
                if(kickMovesOpt->isGiven()){
                    std::cout << "Performing kickMove optimization ..." << std::endl;
                    kickMove myOpt(cv,myTiming);
                    int iterations = atoi(kickMovesOpt->getValue());
                    myOpt.run(iterations);
                }
            }else if(strcmp(optLeakageOpt->getValue(),"delay")==0){
                std::cout << "Performing Delay optimization ..." << std::endl;
                OptMinDelay myDelay(cv,myTiming);
                myDelay.run();
            }
            if(saveOpt->isGiven()){
                const char* saveName = saveOpt->getValue();
                cv->saveAs(libName,oaScalarName(nativeNS,saveName),viewName);
                std::cout << "Modified Netlist saved to " << saveName << std::endl;
            }
        }

        if(peepholeOpt->isGiven()){
            if(!libertyTypeOpt->isGiven()){
                std::cout << "Error: missing library type!" << std::endl;
                return 0;
            }
            /*assert(peepholeOpt->getValue());
            int lvl = atoi(peepholeOpt->getValue());
            int rs = -1;
            if(!rootSelectOpt->isGiven()){ 
                std::cerr << " No Root Node Selected" << std::endl;
                exit(1);
            }else{
                rs = atoi(rootSelectOpt->getValue());
            }*/
            oagTimer::Opt ph(cv,myTiming);
            ph.run();
            //ph.peepHoleOpt(lvl,rs);

            if(saveOpt->isGiven()){
                const char* saveName = saveOpt->getValue();
                cv->saveAs(libName,oaScalarName(nativeNS,saveName),viewName);
                std::cout << "Modified Netlist saved to " << saveName << std::endl;
            }
        }
        
        if (reportOpt->isGiven()) {
            if (strcmp(reportOpt->getValue(), "timing")==0) {
                std::cout << "Report Timing:" << std::endl;
                myReport.reportAll();
            } else if (strcmp(reportOpt->getValue(), "endpoint_slack") == 0) {
                myReport.reportEndpointsSlacks();
            } else if (strcmp(reportOpt->getValue(), "critical") == 0) {
                oagTimer::nodesSlopeDir path;
                myTiming->findWorstPath(path);
                myReport.pathReport(path);
            } else {
                std::cerr << "Check the parameter after -report"
                    << " [timing | endpoint_slack | critical]: " 
                    << reportOpt->getValue() << std::endl;
            }
        }
        
        if(powerOpt->isGiven()){
            Util ut;
            if(strcmp(powerOpt->getValue(), "design")==0){
                std::cout << "Total Design Leakage Power: " << ut.getTotalLeakPower(cv) << "(in library units)" << std::endl;
            }
        }
        
        delete myTiming;
        delete EWM;
        
        cv->close();
    //}
    //catch (oa::oaException &e) {
    //    std::cerr << "ERROR: " << e.getMsg() << std::endl;
    //    assert(false);
    //    return 1;
    //}
    return 0;
}

//-----------------------------------------------------------------------------
