#include <cstdlib>
#include <iostream>
#include <ctime>
#include "oaDesignDB.h"
#include "oagFuncOccGraph.h"
#include "oagFuncPrint.h"
#include "oagFuncQueryOcc.h"
#include "oagSswAiGraphUtil.h"
#include "oagSswAiReasoningEngine.h"
#include "oagSswObsAiGraph.h"
#include "oagSswOdcSatSweepingEngine.h"
#include "oagSswSatSweepingEngine.h"
#include "oagUtilOptionParser.h"

using namespace std;
using namespace oa;

namespace {
oaNativeNS ns;
oaString   str;

template <class T>
const char *
getName(T* inst) {
    inst->getName(ns, str);
    return str;
}
}

// *****************************************************************************
void
runSatSweeping(oaDesign              *design,
               oagUtil::OptionParser &options)
{
    using namespace oagFunc;
    using namespace oagSsw;

    oaOccurrence * occ = design->getTopOccurrence();
    list<OccRef> inputs, outputs, states;
    OccGraph::getInputs(occ, inputs);
    OccGraph::getOutputs(occ, outputs);
    OccGraph::getStates(occ, states);

    // Build the graph.
    ObsAiGraph graph;
    graph.enableStructuralHashing();
    AiReasoningEngine aiEngine(graph);
    QueryOcc query(design, &aiEngine);
    for (list<OccRef>::iterator it = inputs.begin(); it != inputs.end(); ++it) {
        query.set(*it, aiEngine.genericNewTerminal());
        oagAi::Ref ref = aiEngine.getRef(query.get(*it));
	printf("input %d\n", ref);
    }
    for (list<OccRef>::iterator it = states.begin(); it != states.end(); ++it) {
        query.set(*it, aiEngine.genericNewTerminal());
        oagAi::Ref ref = aiEngine.getRef(query.get(*it));
	printf("input %d\n", ref);
    }

    for (list<OccRef>::iterator it = outputs.begin(); it != outputs.end(); ++it) {
        OccRef output = *it;
        oagAi::Ref ref = aiEngine.getRef(query.get(output));
	printf("output %d\n", ref);
        graph.setObservable(ref, true);
    }
    for (list<OccRef>::iterator it = states.begin(); it != states.end(); ++it) {
        OccRef state = *it;
        OccRef nextState = OccGraph::getNextState(state);
        oagAi::Ref ref = aiEngine.getRef(query.get(nextState));
	printf("state %d\n", ref);
        graph.setObservable(ref, true);
    }
    graph.print();

    // Run ODC SAT sweeping.
    oagUtil::Option *seedOpt = options.getOption("randomSeed");
    if (seedOpt->isGiven()) {
        oaUInt4 seed = atoi(seedOpt->getValue());
        srand(seed);
    } else {
        srand(time(0));
    }

    list<oagAi::Ref> observable;
    graph.getAllObservable(observable);

    list<oagAi::Ref> tfi;
    AiGraphUtil::getTransitiveFanin(graph, observable, tfi);
    tfi.insert(tfi.end(), observable.begin(), observable.end());

    if (options.getOption("obsLevels")->isGiven()) {
        /*
        SatSweepingEngine plainSswEngine(graph, tfi);
        plainSswEngine.getLogger()->enable();
        plainSswEngine.run();
        */
        
        oaTimer totalTimer;
        OdcSatSweepingEngine sswEngine(graph, tfi);
        sswEngine.getLogger()->enable();

        oaUInt4 numObsLevels = atoi(options.getOption("obsLevels")->getValue());
        sswEngine.run(numObsLevels);

        cout << sswEngine.stats.numMerges << " nodes were merged." << endl;
        cout << "Total elapsed time: " << totalTimer.getElapsed() << "s" << endl;
    } else {
        SatSweepingEngine sswEngine(graph, tfi);
        sswEngine.getLogger()->enable();

        sswEngine.run();

        cout << sswEngine.stats.numMerges << " nodes were merged." << endl;
    }
}

// *****************************************************************************
// main
// *****************************************************************************
int main(int argc, const char *argv[]) {
    using namespace oagUtil;

    OptionParser options("Run SAT sweeping on a design.");
    Option *libOpt          = options.add("lib", "Input library name", true, "LIB");
    Option *cellOpt         = options.add("cell", "Input cell name", true, "CELL");
    Option *viewOpt         = options.add("view", "Input view name (default: netlist)", false, "VIEW");
    Option *libDefFileOpt   = options.add("libDefFile", "Use a specific library definitions file", false, "FILE");
    Option *obsLevelsOpt    = options.add("obsLevels", 
                                          "Use ODC SAT sweeping with NUM levels of observability", false, "NUM");
    Option *seedOpt         = options.add("randomSeed", "Seed for random number generator", false, "NUM");

    // Some options are not used in this function.
    (void) obsLevelsOpt;

    if (!options.parse(argc, argv)) {
        cerr << options.getMessage();
        exit(EXIT_FAILURE);
    }

    try {
        oaDesignInit();
        if (libDefFileOpt->isGiven()) {
            oaLibDefList::openLibs(libDefFileOpt->getValue());
        } else {
            oaLibDefList::openLibs();
        }

        oagFunc::initialize();

        if (seedOpt->isGiven()) {
            oaUInt4 seed = atoi(seedOpt->getValue());
            cout << "Random seed: " << seed << endl;
            srand(seed);
        } else {
            srand(time(0));
        }

        const oaVerilogNS  ns;
        const oaScalarName libName(ns, libOpt->getValue());
        const oaScalarName cellName(ns, cellOpt->getValue());
        const oaScalarName viewName(ns, viewOpt->isGiven() ? viewOpt->getValue()
                                                           : "netlist");

        oaLib *lib = oaLib::find(libName);
        if (!lib) {
            cerr << "Error: Library not found: " << libOpt->getValue() << endl;
            exit(EXIT_FAILURE);
        }

        oaDesign *design = oaDesign::open(libName, cellName, viewName, 'a');

        printFunctionality(design, false, true);
        runSatSweeping(design, options);
    } catch(oaException &e) {
        cerr << "OA Exception: " << e.getMsg() << endl;
        exit(EXIT_FAILURE);
    }   

    return 0;
}
