#if !defined(oagSswSatSweepingEngine_P)
#define oagSswSatSweepingEngine_P

#include <vector>
#include "oaBase.h"
#include "oagSswAiLevelizer.h"
#include "oagSswAiSimEngine.h"
#include "oagSswAiSatSolver.h"
#include "oagSswLogger.h"
#include "oagSswMap.h"
#include "oagSswSet.h"

namespace oagSsw
{

/// \brief Performs SAT sweeping on an AI graph.
class SatSweepingEngine {
  public:
    /// \brief Constructs an engine that will sweep all nodes in the graph.
                            SatSweepingEngine(oagAi::Graph           &graph);
    /// \brief Constructs an engine that will sweep a set of nodes in the graph.
                            SatSweepingEngine(oagAi::Graph           &graph,
                                              const list<oagAi::Ref> &nodes);

    /// \brief Runs SAT sweeping.
    ///
    /// Searches for functionally equivalent nodes.  When found, they are merged:
    /// The fanout of one is moved to the other.
    /// Only combinational equivalences are detected.
    ///
    /// Structural hashing is assumed to be enabled when propagating merges
    /// through the graph.  However, after merging it is still possible for
    /// two nodes in the graph to have the same inputs.
    void                    run();
    /// \brief Returns true if the node has been replaced by an equivalent.
    bool                    hasRep(oagAi::Ref x);
    /// \brief Returns the replacement of a node, or a null ref if it has not been
    /// replaced.
    oagAi::Ref              getRep(oagAi::Ref x);

    /// Returns the output Logger used for SAT sweeping with LODCs.
    Logger                  *getLogger()    { return logger; }

    /// \name Tuning Parameters
    //@{
    class Params {
      public:
                            Params() {
                                initNumSimBits      = 32;
                                maxSatBacktracks    = INT_MAX;
                                maxCumSatBacktracks = INT_MAX;
                            }

        oa::oaUInt4         initNumSimBits;
        /// \brief Backtracks per SAT query.
        oa::oaUInt4         maxSatBacktracks;
        /// \brief Total (cumulative) backtracks in all SAT queries.
        oa::oaUInt4         maxCumSatBacktracks;
    };
    Params                  params;
    //@}
    
    /// \brief Statistics.
    struct Stats {
        /// \brief The number of merges during the last run.
        oa::oaUInt4         numMerges;

        struct {
            /// \brief The number of SAT checks with each result during the most
            /// recent run.
            //@{
            oa::oaUInt4     numSatisfiable;
            oa::oaUInt4     numUnsatisfiable;
            oa::oaUInt4     numUndecidable;
            //@}

            /// \brief The total number of backtracks in all SAT checks during the
            /// most recent run.
            oa::oaUInt4     numBacktracks;
        } sat;
        struct {
            oa::oaFloat     simulation;
            oa::oaFloat     satChecks;
            oa::oaFloat     computeClusters;
            oa::oaFloat     sortClusters;
        } runtime;
    };

    Stats                   stats;

    /// \brief Resources allocated to each run.
    class Resources {
      public:
                            Resources() {
                                numSatBacktracks = 1024;
                            }

        oa::oaUInt4         numSatBacktracks;
    };
    Resources               resources;
   
  private:
    /// \brief Function object that indicates whether a node has been merged.
    class HasRep {
      public:
                            HasRep(SatSweepingEngine &engine) : engine(engine) {}
        bool                operator() (oagAi::Ref x) const { return engine.hasRep(x); }

      private:
        mutable SatSweepingEngine   &engine;
    };

    /// \brief Function object for sorting AI nodes into clusters.
    ///
    /// The order of nodes is determined by the following criteria, in order of
    /// priority:
    /// \li Simulation vector (modulo complementation)
    /// \li Graph level
    /// \li Reference value
    class LessNode {
      public:
                            LessNode(SatSweepingEngine &engine) : engine(engine) {}
        bool                operator() (oagAi::Ref x,
                                        oagAi::Ref y) const;

      private:
        mutable SatSweepingEngine &engine;
    };

    // Probably should be named IndexRange instead.
    class Cluster {
      public:
                            Cluster() {}
                            Cluster(oa::oaUInt4 begin,
                                    oa::oaUInt4 end)
                                : begin(begin),
                                  end(end) {}

        oa::oaUInt4         size() const    { return end - begin; }
                                
        oa::oaUInt4         begin;
        /// exclusive
        oa::oaUInt4         end;
    };

    /// \brief Function object that returns true if a cluster contains only
    /// one element.
    class IsSingleton {
      public:
        bool                operator() (const Cluster &cluster) const;
    };

    /// \brief Function object for sorting clusters.
    /// 
    /// Clusters are ordered by the graph level of their second elements.
    class LessCluster {
      public:
                            LessCluster(SatSweepingEngine &engine) : engine(engine) {}
        bool                operator() (const Cluster &c1,
                                        const Cluster &c2) const;

      private:
        mutable SatSweepingEngine &engine;
    };


                            
    void                    clearStats();

    /// \brief Collects combinational inputs from among a list of nodes.
    ///
    /// Inserts into the inputs field each node in the list that is:
    /// \li A terminal node whose driver is null
    /// \li Sequential
    /// The list given may contain nodes that are not inputs; they will be
    /// ignored.
    ///
    /// \param pool a list of nodes
    void                    addInputsFrom(const list<oagAi::Ref> &pool);
    
    /// Groups the search and replacement nodes by level.
    void                    levelize();

    /// \brief Initializes the simulation vectors with random inputs.
    void                    initializeSimVectors();

    /// \brief Removes merged nodes from \a active.
    void                    updateActive();

    /// \brief Computes clusters of active nodes with the simulation vectors.
    void                    computeClusters();
    
    /// \brief Returns true if all the resources allocated to SAT sweeping have
    /// been used.
    bool                    reachedResourceLimit();
    /// \brief Increase the number of resources allocated to SAT sweeping.
    void                    increaseResources();

    /// \brief Merges equivalent nodes and finds input assignments to
    /// distinguish inequivalent nodes;
    void                    refineClusters();

    /// \brief Adds an assignment of input values to the simulation vectors.
    ///
    /// Inputs not mentioned in the assignment are assigned randomly.
    /// The simulation vectors of non-inputs are not changed.
    void                    extendSimVectors(const Map<oagAi::Ref, bool>::Type &assignment);

    /// \brief Add transitive fanin of a node to solver.
    void                    addTransitiveFaninToSolver(AiSatSolver           &satSolver,
                                                       oagAi::Ref            node,
                                                       Set<oagAi::Ref>::Type &alreadyAddedTransitiveFanin);
    
    /// \brief Verifies equivalence of two refs.
    /// 
    /// Returns true if two refs are functionally equivalent, or false if they
    /// are inequivalent or equivalence could not be determined under the 
    /// current resource limits.  If they are verified to be inequivalent,
    /// an input assignment that distinguishes them is stored in \a assignment.
    /// 
    /// \param x 
    /// \param y
    /// \retval assignment contains a distinguishing assignment when the refs
    /// are not equivalent
    /// \return true if \a x can be safely merged onto \a y
    bool                    verifyEquivalence(oagAi::Ref        x,
                                              oagAi::Ref        y,
                                              Map<oagAi::Ref,
                                                  bool>::Type   &assignment);

    /// \brief Merges one node to another.
    ///
    /// The fanout of the first node is moved to the replacement ref.
    void                    merge(oagAi::Ref x,
                                  oagAi::Ref replacement);
    /// Merge recursively.
    void                    mergeRec(oagAi::Ref x,
                                     oagAi::Ref replacement);


    oagAi::Graph            &graph;
    Set<oagAi::Ref>::Type   nodes;
    Set<oagAi::Ref>::Type   inputs;

    vector<oagAi::Ref>      activeNodes;
    vector<Cluster>         clusters;
    
    Map<oagAi::Ref, oagAi::Ref>::Type reps;
    
    AiLevelizer                     levelizer;
    vector<Set<oagAi::Ref>::Type >  levels;

    AiSimEngine                 simEngine;

    Logger                      *logger;

    static const oa::oaDouble   BACKTRACKS_INCR_FACTOR;

    friend class LessNode;
    friend class LessCluster;
};

}

#endif

// vim: set tw=80:
