#include <iomanip>
#include <iostream>
#include <stdexcept>
#include <sstream>
#include <string>
#include "oagSswAiSimEngine.h"

using namespace std;
using namespace oa;
using namespace oagAi;

namespace oagSsw
{

const BitVector::Word AiSimEngine::FULL_WORD  = ~0;

oaUInt4
AiSimEngine::getIndex(Ref e)
{
    return (oaUInt4)e >> 1;
}

oaUInt4
AiSimEngine::wordLength(oaUInt4 bitLen)
{ 
    return bitLen == 0 ? 0 : 1 + BitVector::getWordIndex(bitLen - 1); 
}

oaUInt4
AiSimEngine::shiftl(BitVector::Word w, oaUInt4 n)
{ 
    return n >= BitVector::BITS_PER_WORD ? 0 : (w << n); 
}

BitVector::Word
AiSimEngine::wordMask(oaUInt4 from, oaUInt4 to)
{ 
    return shiftl(FULL_WORD, BitVector::BITS_PER_WORD - to) 
        & ~shiftl(FULL_WORD, BitVector::BITS_PER_WORD - from);
}

AiSimEngine::AiSimEngine(Graph & graph)
: graph(graph)
{
    numBits = 0;
    //simVectors.resize(10000);
}

void
AiSimEngine::setSimVectorListLength(oaUInt4 length)
{
    simVectors.resize(length);
}

oaUInt4
AiSimEngine::getLength() 
{
    return numBits;
}

void
AiSimEngine::setLength(oaUInt4 length)       
{
    oaUInt4 oldLength;

    if (length == numBits) {
        return;
    }

    oldLength = numBits;
    numBits = length;
    list<Ref> allRefs;
    graph.getAll(allRefs); 
    for (list<Ref>::const_iterator it = allRefs.begin(); it != allRefs.end(); ++it)
    {
        simVectors[getIndex(*it)].setSize(numBits);
    }
}

BitVector::Word
AiSimEngine::getWordInternal(Ref e, oaUInt4 wordIdx)
{
    BitVector::Word word, mask;

    word = simVectors[getIndex(e)].words[wordIdx];
    if (graph.isInverted(e) && (wordIdx == BitVector::getWordIndex(numBits))){
        mask = wordMask(0, numBits % BitVector::BITS_PER_WORD);
        return ~word & mask;
    } else if (graph.isInverted(e)) {
        return ~word; 
    } else {
        return word;
    }
}

oaUInt4
AiSimEngine::getWord(Ref e, oaUInt4 wordIdx)
{
    return (oaUInt4)getWordInternal(e, wordIdx);
}

bool
AiSimEngine::getBit(Ref e, oaUInt4 bitIdx) 
{
    bool bit;

    bit = simVectors[getIndex(e)].getBit(bitIdx);
    return graph.isInverted(e) ? !bit : bit;
}

void
AiSimEngine::setBit(Ref e, oaUInt4 bitIdx, bool val)
{
    assert(bitIdx < numBits);
    if (graph.isInverted(e)) {
        val = !val;
    }
    simVectors[getIndex(e)].setBit(bitIdx, val);
}

void  
AiSimEngine::simulateAll(bool onlyLastWord)
{
    list<Ref> allRefs;
    graph.getAll(allRefs);
    vector<bool> updated(simVectors.size());

    for (list<Ref>::const_iterator it = allRefs.begin(); it != allRefs.end(); ++it)
    {
        simulateTransitiveFanInRec(*it, onlyLastWord, updated);
    }
}

void  
AiSimEngine::simulateTransitiveFanIn(Ref e, bool onlyLastWord)
{
    vector<bool> updated;
    simulateTransitiveFanInRec(e, onlyLastWord, updated);
}

void  
AiSimEngine::simulateTransitiveFanInRec(Ref e, bool onlyLastWord, vector<bool> &updated)
{
    if (updated[getIndex(e)]) {
        return;
    }

    switch (graph.getNodeType(e)) {
      case Node::AND :
        simulateTransitiveFanInRec(graph.getAndLeft(e), onlyLastWord, updated);
        simulateTransitiveFanInRec(graph.getAndRight(e), onlyLastWord, updated);
        simulateAnd(e, onlyLastWord);
        break;

      case Node::TERMINAL :
        {
            Ref driver = graph.getTerminalDriver(e);

            if (!graph.isNull(driver)) {
                simulateTransitiveFanInRec(driver, onlyLastWord, updated);

                BitVector &eVec      = simVectors[e];
                BitVector &driverVec = simVectors[driver];

                if (graph.isInverted(e) == graph.isInverted(driver)) {
                    if (onlyLastWord) {
                        eVec.words.back() = driverVec.words.back();
                    } else {
                        eVec.words = driverVec.words;
                    }
                } else {
                    assert(eVec.words.size() == driverVec.words.size());

                    Ref negDriver = graph.notOf(graph.getNonInverted(driver));

                    if (onlyLastWord) {
                        eVec.words.back() = getWordInternal(negDriver, eVec.words.size() - 1);
                    } else {
                        for (oaUInt4 i = 0; i < eVec.words.size(); ++i) {
                            BitVector::Word word = getWordInternal(negDriver, i);
                            eVec.words[i] = word;
                        }
                    }
                }
            }
        }
        break;

      default :
        break;
        // do not need simulate SEQUENTIAL nodes, CONSTANT0
        // how about merged node? -- little redundant work, but will not affect much...
    } 
    updated[getIndex(e)] = true; 
}

void
AiSimEngine::simulateAnd(Ref e, bool onlyLastWord)
{
    oaUInt4 i;
    Ref left, right;
    BitVector::Word wordLeft, wordRight, wordAnd;

    oaUInt4 numWords = wordLength(numBits);
    assert(graph.getNodeType(e) == Node::AND);
    left = graph.getAndLeft(e);
    right = graph.getAndRight(e);
    if (!onlyLastWord) {
        for (i = 0; i < numWords; i++) {
            wordLeft = getWordInternal(left, i);
            wordRight = getWordInternal(right, i);
            wordAnd = wordLeft & wordRight;
            simVectors[getIndex(e)].words[i] = wordAnd;
        }
    } else {
        wordLeft = getWordInternal(left, numWords - 1);
        wordRight = getWordInternal(right, numWords - 1);
        wordAnd = wordLeft & wordRight;
        simVectors[getIndex(e)].words[numWords - 1] = wordAnd;
    }
}

void         
AiSimEngine::randomizeVars(oaUInt4 from, oaUInt4 to) 
{
    list<Ref> allRefs;
    graph.getAll(allRefs);
    BitVector::Word mask;
    oaUInt4 i, maskFrom, maskTo;

    assert(to < numBits);
    assert(from <= to);

    oaUInt4 numWords = wordLength(numBits);
    for (list<Ref>::const_iterator it = allRefs.begin(); it != allRefs.end(); ++it)
    {
        if (graph.isTerminal(*it) && graph.isNull(graph.getTerminalDriver(*it))
                || graph.isSequential(*it)) {
            Ref e = *it;
            for (i = BitVector::getWordIndex(from); i < numWords && i <= BitVector::getWordIndex(to); i++) {
                if (i == BitVector::getWordIndex(from)) {
                    maskFrom = from % BitVector::BITS_PER_WORD;
                } else {
                    maskFrom = 0;
                }
                if (i == BitVector::getWordIndex(to)) {
                    maskTo = to % BitVector::BITS_PER_WORD;
                } else {
                    maskTo = BitVector::BITS_PER_WORD;
                }
                mask = wordMask(maskFrom, maskTo);
                simVectors[getIndex(e)].words[i] = 
                    (simVectors[getIndex(e)].words[i] & ~mask) | (randomWord() & mask);
            }
        }
    }
}

BitVector::Word
AiSimEngine::randomWord()
{
    BitVector::Word word;
    oaUInt4 j;

    word = 0;
    for (j = 0; j < BitVector::BITS_PER_WORD; j += 16) {
        word |= ((rand() & 0xffff00) >> 8) << j;
    }
    return word;
}

int
AiSimEngine::compareRef(Ref e1,
                        Ref e2,
                        bool modCompl) 
{
    if (modCompl) {
        if (getBit(e1, 0)) {
            e1 = graph.notOf(e1);
        }
        if (getBit(e2, 0)) {
            e2 = graph.notOf(e2);
        }
    }

    oaUInt4 i;
    BitVector::Word word1, word2;

    for (i = 0; i < wordLength(numBits); i++) {
        word1 = getWordInternal(e1, i);
        word2 = getWordInternal(e2, i);
        if (word1 < word2) {
            return -1;
        }
        if (word1 > word2) {
            return 1;
        }
    }
    return 0;
}

void
AiSimEngine::clear()
{
    simVectors.clear();
}  

void
AiSimEngine::dumpAll()
{
    oaUInt4 i;
    list<Ref> allRefs;
    graph.getAll(allRefs);

    ios::fmtflags flags = cout.flags();

    oaUInt4 numWords = wordLength(numBits);
    for (list<Ref>::const_iterator it = allRefs.begin(); it != allRefs.end(); ++it)
    {
        graph.print(*it);
        for (i = 0; i < numWords; i++) {
            cout << setfill('0') << setw(8) << setbase(16)
                 << simVectors[getIndex(*it)].words[i];
        }
        cout << endl;
    }

    cout.flags(flags);
}

oaUInt4  
AiSimEngine::getNumWords()
{
    return wordLength(numBits);
}

bool
AiSimEngine::maskedEquivalent(Ref &e1,
                              Ref &e2,
                              const BitVector &mask)
{
    if (numBits != mask.size()) {
        ostringstream oss;

        oss << "BitVector lengths do not match: "
            << numBits << " and " << mask.size();

        throw invalid_argument(oss.str());
    }

    for (oaUInt4 i = 0; i < wordLength(numBits); i++) {
        // Compare words from last to first instead of vice versa because the
        // differences are most likely to be at the end, in the most recently
        // added bits.
        oaUInt4 j       = wordLength(numBits) - 1 - i;

        if (((getWordInternal(e1, j) ^ getWordInternal(e2, j)) 
            & mask.words[j]) != (BitVector::Word)0) {
            return false;
        }
    }
 
    return true;
}

}
