
/*
Author: Kai-hui Chang <changkh@eecs.umich.edu>
*/

#include "oagResynUtil.h"
#include "oagUtilOaUtil.h"
#include <iostream>
#include <fstream>
#include <sys/times.h>
#include <unistd.h>


using namespace std;
using namespace oa;

namespace oagResyn {


// *****************************************************************************
// getCogNoDriver()
//
/// \brief Get the Center-of-gravity of sinks.
///
/// Get location of center-of-gravity of a net without driver, in other words, the COG of sinks.
//
// *****************************************************************************
void util::getCogNoDriver(oaNet* iNet, oaCoord& x, oaCoord& y)
{
    int count;
    oaCoord nx, ny, bx= 0, by= 0;

    x= 0; y= 0;
    count= 0;
    
    oaIter<oaTerm> termIter(iNet->getTerms());
    while (oaTerm *i = termIter.getNext()) 
    {
        oaTermType termType = i->getTermType();
        getTermCoord(i, nx, ny);
        if (termType == oacInputTermType)
        {
            x+= nx;
            y+= ny;
            count++;
        }
        else
        {
            bx= nx;
            by= ny;
        }
    }
    oaIter<oaInstTerm>   oaITIter(iNet->getInstTerms());
    while (oaInstTerm*  instTerm = oaITIter.getNext())
    {
        oaTerm* term = instTerm->getTerm();
        oaTermType termType = term->getTermType();
        getInstTermCoord(instTerm, nx, ny);
        if (termType == oacInputTermType)
        {
            x+= nx;
            y+= ny;
            count++;
        }
        else
        {
            bx= nx;
            by= ny;
        }
    }
    if (count > 0)
    {
        x /= count;
        y /= count;
    }
    else
    {
        x= bx;
        y= by;
    }
}

// *****************************************************************************
// getCog()
//
/// \brief Get location of center-of-gravity of a net.
//
// *****************************************************************************
void util::getCog(oaNet* iNet, oaCoord& x, oaCoord& y)
{
    int count;
    oaCoord nx, ny;

    x= 0; y= 0;
    count= 0;
    
    oaIter<oaTerm> termIter(iNet->getTerms());
    while (oaTerm *i = termIter.getNext()) 
    {
        oaTermType termType = i->getTermType();
        {
            getTermCoord(i, nx, ny);
            x+= nx;
            y+= ny;
            count++;
        }
    }
    oaIter<oaInstTerm>   oaITIter(iNet->getInstTerms());
    while (oaInstTerm*  instTerm = oaITIter.getNext())
    {
        oaTerm* term = instTerm->getTerm();
        oaTermType termType = term->getTermType();
        {
            getInstTermCoord(instTerm, nx, ny);
            x+= nx;
            y+= ny;
            count++;
        }
    }
    x /= count;
    y /= count;
    
}

// *****************************************************************************
// getInstSize()
//
/// \brief Get size of an oaInst.
//
// *****************************************************************************
void util::getInstSize(oaInst* iInst, oaUInt4& x, oaUInt4& y)
{
    oaBox box;
   
    iInst->getBBox(box);
    x= box.getWidth();
    y= box.getHeight();
}

// *****************************************************************************
// getInstCoord()
//
/// \brief Get location of an oaInst. Use bottom left.
//
// *****************************************************************************
void util::getInstCoord(oaInst* iInst, oaCoord& x, oaCoord& y)
{
    oaBox box;
   
    iInst->getBBox(box);
    x= box.bottom();
    y= box.left();
}

// *****************************************************************************
// getDriverInst()
//
/// \brief Get a net's driver instance.
//
// *****************************************************************************
oaInst* util::getDriverInst(oaNet* iNet)
{
    
    oaIter<oaInstTerm>   oaITIter(iNet->getInstTerms());
    while (oaInstTerm*  instTerm = oaITIter.getNext())
    {
        oaTerm* term = instTerm->getTerm();
        oaTermType termType = term->getTermType();
        if (termType == oacOutputTermType)
        {
            return (instTerm->getInst());
        }
    }
    return NULL;
}


// *****************************************************************************
// getHPWL()
//
/// \brief Return the HPWL of a net.
//
// *****************************************************************************
oaUInt4 util::getHPWL(oaNet* iNet)
{
    return oagUtil::OaUtil::getHPWL(iNet);
}

// *****************************************************************************
// moveInst()
//
/// \brief Move an instance to the specified location.
//
// *****************************************************************************
void util::moveInst(oaInst* iInst, oaCoord x, oaCoord y)
{
    oaPoint myPoint;

    myPoint.set(x, y);
    iInst->setOrigin(myPoint);
}

// *****************************************************************************
// getTermCoord()
//
/// \brief Get coordinates of Term.
//
// *****************************************************************************
void util::getTermCoord(oaTerm* iTerm, oaCoord& x, oaCoord& y)
{
    oaBox box;
    oaPoint point;
    
    box.init();
    oaIter<oaPin> pinIter(iTerm->getPins());
    while (oaPin *j = pinIter.getNext()) 
    {
        oaIter<oaPinFig> figIter(j->getFigs());
        oaPinFig *pinFig = figIter.getNext();
        assert(pinFig);
        oaBox pinBox;
        pinFig->getBBox(pinBox);
        box.merge(pinBox);
    }
    box.getCenter(point);
    x= point.x();
    y= point.y();
    
}

// *****************************************************************************
// getInstTermCoord()
//
/// \brief Get coordinates of instTerm.
//
// *****************************************************************************
void util::getInstTermCoord(oaInstTerm* iTerm, oaCoord& x, oaCoord& y)
{
    oaBox box;
    oaTerm *masterTerm = iTerm->getTerm();
    oaPoint point;

    box.init();
    assert(masterTerm);
    oaInst *inst= iTerm->getInst();
    assert(inst);
    oaTransform instTrans;
    inst->getTransform(instTrans);
    oaIter<oaPin> pinIter(masterTerm->getPins());
    while (oaPin *j = pinIter.getNext()) 
    {
        oaIter<oaPinFig> figIter(j->getFigs());
        oaPinFig *pinFig = figIter.getNext();
        assert(pinFig);
        oaBox pinBox;
        pinFig->getBBox(pinBox);
        pinBox.transform(instTrans);
        box.merge(pinBox);
    }
    box.getCenter(point);
    x= point.x();
    y= point.y();
}


// *****************************************************************************
// getDist()
//
/// \brief Returns the distance of two points. 
//
// *****************************************************************************
double util::getDist(oaCoord x1, oaCoord y1, oaCoord x2, oaCoord y2)
{
    return (fabs(static_cast<double>(x1 - x2)) + fabs (static_cast<double>(y1 - y2)));
}

// *****************************************************************************
// intVecAddVec()
//
/// \brief Add an Simulation vector to a vector<vector<SimulationVector>>. 
///
/// Note: newSize must <= SimVectorSize
//
// *****************************************************************************
void util::intVecAddVec(vector<vector<SimulationVector> >& origVec, vector<SimulationVector>& newVec, int origSize, int newSize)
{
    int begin, end;
    int pat1;
    SimulationVector mask; 
    unsigned int i;
    int pattern;
    vector<SimulationVector> addVec;
    bool newInt;

    if (newSize == 0)
        return;
    begin= origSize / SimVectorSize;
    end= (origSize + newSize - 1) / SimVectorSize;
    newInt= origSize % SimVectorSize == 0 ? 1 : 0;
    if (newInt && newSize > 0)
    {
        for (i= 0; i < newVec.size(); i++)
        {
            addVec.push_back(0);
        }
        origVec.push_back(addVec);
        addVec.clear();
    }
    if (begin != end)
    {
        pat1= SimVectorSize - (origSize % SimVectorSize);
        for (i= 0; i < newVec.size(); i++)
        {
            pattern= newVec[i];
            pattern= pattern << (origSize % SimVectorSize);
            origVec[begin][i] |= pattern;
            pattern= newVec[i] >> pat1;
            mask= (1 << (newSize - pat1)) - 1;
            pattern= pattern & mask;
            addVec.push_back(pattern);
        }
        origVec.push_back(addVec);
    }
    else
    {
        for (i= 0; i < newVec.size(); i++)
        {
            pattern= newVec[i];
            mask= (1 << newSize) - 1;
            pattern= pattern & mask;
            pattern= pattern << (origSize % SimVectorSize);
            origVec[begin][i] |= pattern;
        }
    }
}

// *****************************************************************************
// vecAddInt()
//
/// \brief Add a SimulationVector to a vector<SimulationVector>
//
// *****************************************************************************
void util::vecAddInt(vector<SimulationVector>& origVec, SimulationVector newInt, int origSize, int newSize)
{
    int begin, end;
    int pat1;
    SimulationVector mask; 
    SimulationVector pattern;
   
    if (newSize == 0)
        return;
    begin= origSize / SimVectorSize;
    end= (origSize + newSize - 1) / SimVectorSize;
    if (begin != end)
    {
        pat1= SimVectorSize - (origSize % SimVectorSize);
        pattern= newInt;
        pattern= pattern << (origSize % SimVectorSize);
        mask= (1 << pat1) - 1;
        mask= mask << (origSize % SimVectorSize);
        mask= ~mask;
        origVec[begin] &= mask;
        origVec[begin] |= pattern;
        pattern= newInt >> pat1;
        mask= (1 << (newSize - pat1)) - 1;
        pattern= pattern & mask;
        origVec.push_back(pattern);
    }
    else
    {   
        pattern= newInt;
        pattern= pattern << (origSize % SimVectorSize);
        mask= (1 << newSize) - 1;
        mask= mask << (origSize % SimVectorSize);
        mask= ~mask;
        if (origSize % SimVectorSize == 0)
            origVec.push_back(pattern);
        else
        {
            origVec[begin] &= mask;
            origVec[begin] |= pattern;
        }
    }
}

// *****************************************************************************
// vecAddVec()
//
/// \brief Add an vector<SimulationVector> to a vector<SimulationVector>
//
// *****************************************************************************
void util::vecAddVec(vector<SimulationVector>& origVec, vector<SimulationVector>& newVec, int origSize, int newSize)
{
    int i;
    int intSize;
    int begin, end;
    int oSize= origSize;

    if (newSize == 0)
        return;
    begin= 0;
    end= (newSize - 1) / SimVectorSize;
    intSize= end - begin + 1;
    for (i= 0; i < intSize; i++)
    {
        if (newSize - begin > static_cast<int>(SimVectorSize))
            vecAddInt(origVec, newVec[i], oSize, SimVectorSize);
        else
            vecAddInt(origVec, newVec[i], oSize, newSize - begin);
        begin+= SimVectorSize;
        oSize+= SimVectorSize;
    }
}

// *****************************************************************************
// util::bitVecToIntVec()
//
/// \brief Convert a vector<bool> to a vector<SimulationVector>
//
// *****************************************************************************
void util::bitVecToIntVec(vector<SimulationVector>& intVec, vector<bool>& bitVec)
{
    int i, j;
    int intSize;
    int last;
    int begin= 0;
    int size;
    int newSize;
    SimulationVector pat;
    int count= 0;

    newSize= bitVec.size();
    intSize= newSize / SimVectorSize;
    last= newSize % SimVectorSize;
    if (last > 0)
        intSize++;
    intVec.clear();
    for (i= 0; i < intSize; i++)
    {
        if (newSize - begin > static_cast<int>(SimVectorSize))
            size= SimVectorSize;
        else
            size= newSize - begin;
        pat= 0;
        for (j= 0; j < size; j++)
        {
            if (bitVec[count++])
                pat|= (1 << j);
        }
        intVec.push_back(pat);
        begin+= SimVectorSize;
    }
}

// *****************************************************************************
// util::printIntVec()
//
/// \brief Print vector<SimulationVector>
//
// *****************************************************************************
void util::printIntVec(vector<SimulationVector>& inVec)
{
    vector<SimulationVector>::iterator intIter;

    for (intIter= inVec.begin(); intIter != inVec.end(); intIter++)
        cout<<*intIter<<" ";
    cout<<endl;
}


// *****************************************************************************
// util::intVecGetBit()
//
/// \brief Get a specific bit in the SimulationVector vector.
//
// *****************************************************************************
bool util::intVecGetBit(vector<SimulationVector>& inVec, int bitNO)
{
    int ints;
    int last;
    SimulationVector bitmask;

    ints= bitNO / SimVectorSize;
    last= bitNO % SimVectorSize;
    bitmask= 1 << last;
    if ((inVec[ints] & bitmask) != 0)
        return true;
    else
        return false;
}

// *****************************************************************************
// util::getToken()
//
/// \brief Get a token from a file. 
/// 
/// \param ifile2 The file to read from.
/// \param token Returns the token.
/// \return 0: normal token, 1: end-of-line, 2: end-of-file
//
// *****************************************************************************
int util::getToken(ifstream& ifile2, string& token)
{
    int chr= -1;
    bool inToken= false;

    token.clear();
    // skip heading white space
    while (ifile2.eof() == false)
    {
        chr= ifile2.get();
        if (chr != ' ' && chr != '\t')
            break;
    }
    if (chr != -1)
        ifile2.unget();
    while (!ifile2.eof())
    {
        chr= ifile2.get();
        if (chr == '\r')
            chr= ifile2.get();
        if (chr == '\n')
        {
            if (inToken == true)
            {
                ifile2.unget();
                return 0;
            }
            else
                return 1;
        }
        if (inToken == true && (chr == ' ' || chr == '\t'))
            return 0;
        if (inToken == false)
            inToken= true;
        token= token + static_cast<char>(chr);
    }
    return 2;
}

// *****************************************************************************
// util::getClockTime()
//
/// \brief Get clock time (machine time)
//
// *****************************************************************************
int util::getClockTime()
{
    return (time(NULL));
}

// *****************************************************************************
// util::util::netGetInstDriver()
//
/// \brief Given a net, return its driver (instTerm). 
///
/// If it is driven by term, it will not be returned. Assume that the first output of instance it encountered is the driver.
//
// *****************************************************************************
oaInstTerm* util::netGetInstDriver(oaNet* inNet)
{
    oaIter<oaInstTerm>   oaITIter(inNet->getInstTerms((oacInstTermIterNotImplicit | oacInstTermIterEquivNets)));
    while (oaInstTerm*  instTerm = oaITIter.getNext())
    {
        oaTerm* term = instTerm->getTerm();
        oaTermType termType = term->getTermType();

        if (termType == oacOutputTermType)
            return instTerm;
    }
    return NULL;
}

} // End of namespace
// vim: ci et sw=4 sts=4
