
/* 
Author: Kai-hui Chang <changkh@eecs.umich.edu>
*/

///
/// \file oagResynUserIntf.h
/// Classes for user interface.

#if !defined(oagResynUserIntf_P)
#define oagResynUserIntf_P

#include "oaDesignDB.h"
#include "oagResynType.h"
#include "oagResynUserPattern.h"
#include <vector>
#include <exception>
#include <map>

namespace oagResyn {

class oagResynErr: public std::exception
{
public:
    oagResynErr(const std::string& message):_message(message) {}
    virtual const char* what() const throw ()
    {
       return std::string("Error in oagResyn package: " + _message).c_str();
    }
    ~oagResynErr() throw () {}
private:
   std::string _message;
};

/// Interface for simulator.
class simulatorInterface
{
public:
    typedef oagResyn::SimulationVector SimulationVector;
    /// Perform random simulation
    virtual void doRandomSimulation() = 0;
    /// Perform incremental simulation
    virtual void incrementalSim(oa::oaNet* net, SimulationVector newVec) = 0;
    /// Run full simulation
    virtual void runFull()= 0;
    /// Run next cycle, for sequential simulation
    virtual void nextCycle()= 0;
    /// Set a vector value to a state instance
    virtual void setStateVector(oa::oaInst* inst, SimulationVector vec)= 0;
    /// Set a vector value to a state net
    virtual void setVector(oa::oaNet* net, SimulationVector vec, int isw)= 0;
    /// Get vector value of a net
    virtual void getVector(oa::oaNet* net, SimulationVector& vec)= 0;
    /// Get state instances
    virtual void getStates(std::vector<oa::oaInst*>& result)= 0;
    /// Get state nets
    virtual void getStateNets(std::vector<oa::oaNet*>& result)= 0;
    /// Get primary inputs
    virtual void getInputs(std::vector<oa::oaNet*>& result)= 0;
    /// Get primary outputs
    virtual void getOutputs(std::vector<oa::oaNet*>& result)= 0;
    /// Get nets identified as clocks
    virtual void getClockNets(std::vector<oa::oaNet*>& result)= 0;
    /// Get all the bitNets in this design
    virtual void getAllNets(std::vector<oa::oaNet*>& result)= 0;
    /// Generate a SimulationVector
    virtual SimulationVector randomVector()= 0;
    /// Set user specified pattern file. 
    virtual bool setUserPatternFile(const char* fileName)=0;
    /// Set initial state file
    virtual bool useStateFile(const char* fileName)= 0;
    /// Advance inputs using either random or user specified patterns.
    virtual void advanceInputs()=0;
    /// Perform advanceInputs and nextCycle
    virtual void step()=0;
    /// If true, user specified patterns are exhausted, and random inputs will be used.
    virtual bool isFinished()=0;
};

/// Interface for equivalence checker outputs
class equiCheckOutputInterface
{
public:
    typedef oagResyn::SimulationVector SimulationVector;
    /// True if the equivalence checking is for two nets
    virtual bool isTwoNet()=0;
    /// True if the equivalence checking is for two designs
    virtual bool isTwoDesign()=0;                 
    /// Get counterexamples
    virtual const std::vector<SimulationVector>& getCounterExample()=0;
    /// Get the outputs of design 1 for the counterexamples
    virtual const std::vector<SimulationVector>& getCounterExampleOutput1()=0;
    /// Get the outputs of design 2 for the counterexamples
    virtual const std::vector<SimulationVector>& getCounterExampleOutput2()=0;
    /// Get primary inputs for design 1
    virtual const std::vector<oa::oaNet*>& getPrimaryInputs1()=0;
    /// Get primary inputs for design 2 
    virtual const std::vector<oa::oaNet*>& getPrimaryInputs2()=0; 
    /// Get primary outputs for design 2 
    virtual const std::vector<oa::oaNet*>& getPrimaryOutputs1()=0;
    /// Get primary outputs for design 2 
    virtual const std::vector<oa::oaNet*>& getPrimaryOutputs2()=0; 
    /// Get similarity factor (for checking 2 designs only)
    virtual double getSimilarityFactor()=0;
    /// If true, two designs/nets are equivalent
    virtual bool isEquivalent()= 0;
    /// Get library name of design 1
    virtual std::string getLib1()= 0;
    /// Get library name of design 2 
    virtual std::string getLib2()= 0;
    /// Get cell name of design 1
    virtual std::string getCell1()= 0;
    /// Get cell name of design 1
    virtual std::string getCell2()= 0;
    /// Get net name of design 1
    virtual std::string getNet1()= 0;
    /// Get net name of design 2
    virtual std::string getNet2()= 0;
    /// Get the wires in design 1 with signatures not exist in design 2
    virtual const std::vector<oa::oaNet*>& getDiffNets1()= 0;
    /// Get the wires in design 2 with signatures not exist in design 1
    virtual const std::vector<oa::oaNet*>& getDiffNets2()= 0;
    /// Get the level of logic of instances in design 1
    virtual const std::map<oa::oaInst*, int>& getInstLevel1()= 0;
    /// Get the level of logic of instances in design 2
    virtual const std::map<oa::oaInst*, int>& getInstLevel2()= 0;

};

}
#endif

// vim: ci et sw=4 sts=4
