/*
Author: Kai-hui Chang <changkh@eecs.umich.edu>
*/

#include <iostream>
#include <fstream>
#include <string>
#include <set>
#include "oagResynOptSim.h"
#include "oagResynCktMan.h"
#include "oagResynUtil.h"

using namespace std;
using namespace oa;

// #define DEBUG

namespace oagResyn {

// *****************************************************************************
// Sim()
//
/// \brief Constructor.
//
// *****************************************************************************
optSim::optSim(oa::oaDesign *design) {
    myUserPattern= NULL;
    setDesign(design);
}

// *****************************************************************************
// Sim()
//
/// \brief Constructor.
//
// *****************************************************************************
optSim::optSim() {
    myUserPattern= NULL;
}

// *****************************************************************************
// setDesign()
//
/// \brief Set the design for simulation
//
// *****************************************************************************
void optSim::setDesign(oa::oaDesign *design) {
    assert(design);
    block= design->getTopBlock();
    assert(block);
    this->design = design;
    myCkt= cktManager.designGetOptInternalCkt(design);
}

// *****************************************************************************
// getStates()
//
/// \brief Gets all state instances inside this design. 
// *****************************************************************************
void optSim::getStates(vector<oaInst*>& result) {
    result= myCkt->stateInsts;
}

// *****************************************************************************
// getAllNets()
//
/// \brief Gets all nets in the circuit (oaBitNet).
// *****************************************************************************
void optSim::getAllNets(vector<oaNet*>& result) {
    set<oaNet*>::iterator setIter;
    
    result.clear();
    for (setIter= myCkt->allBitNets.begin(); setIter != myCkt->allBitNets.end(); setIter++)
        result.push_back(*setIter);
}

// *****************************************************************************
// getClockNets()
//
/// \brief Gets all nets marked as clock by DFF's clock.
// *****************************************************************************
void optSim::getClockNets(vector<oaNet*>& result) {
    set<oaNet*>::iterator setIter;
    
    result.clear();
    for (setIter= myCkt->clockNets.begin(); setIter != myCkt->clockNets.end(); setIter++)
        result.push_back(*setIter);
}

// *****************************************************************************
// getInputs()
//
/// \brief Gets all PI nets inside this design.
// *****************************************************************************
void optSim::getInputs(vector<oaNet*>& result) {
    result= myCkt->inputNets;
}

// *****************************************************************************
// getOutputs()
//
/// \brief Gets all PO nets inside this design.
// *****************************************************************************
void optSim::getOutputs(vector<oaNet*>& result) {
    result= myCkt->outputNets;
}

// *****************************************************************************
// simulateInst()
//
/// \brief Simulate an internal instance. 
///
/// Inputs are from its input wires and outputs are sent to output wires. If isw is 1, copy the value to nextSimVec instead of simVec.
// *****************************************************************************
void optSim::simulateInst(optSimInst* inInst, int isw = 0)
{
    libCell* cell;
    unsigned long type= 0;
    unsigned int i;
    bool processed= false;
    SimulationVector result= 0, result2;

    cell= inInst->cell;
    #ifdef DEBUG
    myCkt->printInternalInst(inInst);
    cout<<"Inputs: ";
    for (i= 0; i < inInst->inputs.size(); i++)
        cout<<inInst->inputs[i]->wire1->simVec<<" ";
    cout<<endl;
    cout<<"Outputs: ";
    #endif
    for (i= 0; i < inInst->outputs.size(); i++)
    {
        if (inInst->outputs[i]->wire1 == NULL)
            continue;
        if (inInst->inputs.size() <= 4)
            type= cell->tTables[i].to_ulong();
        if (inInst->inputs.size() == 1)
        {
            switch (type)
            {
            case 1: // INV
                result= ~inInst->inputs[0]->wire1->simVec;
                processed= true;
                break;
            case 2: // BUF
                result= inInst->inputs[0]->wire1->simVec;
                processed= true;
                break;
            }
        }
        else if (inInst->inputs.size() == 2)
        {
            switch (type)
            {
            case 8: // AND
                result= inInst->inputs[0]->wire1->simVec & inInst->inputs[1]->wire1->simVec;
                processed= true;
                break;
            case 7: //NAND
                result= inInst->inputs[0]->wire1->simVec & inInst->inputs[1]->wire1->simVec;
                result= ~result;
                processed= true;
                break;
            case 14: // OR
                result= inInst->inputs[0]->wire1->simVec | inInst->inputs[1]->wire1->simVec;
                processed= true;
                break;
            case 1: // NOR
                result= inInst->inputs[0]->wire1->simVec | inInst->inputs[1]->wire1->simVec;
                result= ~result;
                processed= true;
                break;
            case 6: //XOR
                result= inInst->inputs[0]->wire1->simVec ^ inInst->inputs[1]->wire1->simVec;
                processed= true;
                break;
            case 9: //XNOR
                result= ~(inInst->inputs[0]->wire1->simVec ^ inInst->inputs[1]->wire1->simVec);
                processed= true;
                break;
            }
        }
        else if (inInst->inputs.size() == 3)
        {
            switch (type)
            {
            case 7: // AOI21
                result= inInst->inputs[0]->wire1->simVec & inInst->inputs[1]->wire1->simVec;
                result |= inInst->inputs[2]->wire1->simVec;
                result= ~result;
                processed= true;
                break;
            case 202: // MX2X1
                result= inInst->inputs[1]->wire1->simVec & inInst->inputs[2]->wire1->simVec;
                result2= inInst->inputs[0]->wire1->simVec & ~inInst->inputs[2]->wire1->simVec;
                result |= result2;
                processed= true;
                break;
            case 1: // NOR
                result= inInst->inputs[0]->wire1->simVec |
                        inInst->inputs[1]->wire1->simVec | 
                        inInst->inputs[2]->wire1->simVec;
                result= ~result;
                processed= true;
                break;
            case 127: // NAND
                result= inInst->inputs[0]->wire1->simVec & 
                        inInst->inputs[1]->wire1->simVec & 
                        inInst->inputs[2]->wire1->simVec;
                result= ~result;
                processed= true;
                break;
            case 31: // OAI21
                result= ~((inInst->inputs[0]->wire1->simVec | 
                        inInst->inputs[1]->wire1->simVec) & 
                        inInst->inputs[2]->wire1->simVec);
                processed= true;
                break;
            }
        }
        else if (inInst->inputs.size() == 4)
        {
            switch (type)
            {
            case 4383: // OAI22
                result= inInst->inputs[0]->wire1->simVec | inInst->inputs[1]->wire1->simVec;
                result2= inInst->inputs[2]->wire1->simVec | inInst->inputs[3]->wire1->simVec;
                result= result & result2;
                result= ~result;
                processed= true;
                break;
            case 1911: // AOI22
                result= inInst->inputs[0]->wire1->simVec & inInst->inputs[1]->wire1->simVec;
                result2= inInst->inputs[2]->wire1->simVec & inInst->inputs[3]->wire1->simVec;
                result= result | result2;
                result= ~result;
                processed= true;
                break;
            case 32767: // NAND4
                result= inInst->inputs[0]->wire1->simVec & inInst->inputs[1]->wire1->simVec;
                result2= inInst->inputs[2]->wire1->simVec & inInst->inputs[3]->wire1->simVec;
                result= result & result2;
                result= ~result;
                processed= true;
                break;
            case 65534: // OR4
                result= inInst->inputs[0]->wire1->simVec | inInst->inputs[1]->wire1->simVec;
                result2= inInst->inputs[2]->wire1->simVec | inInst->inputs[3]->wire1->simVec;
                result= result | result2;
                processed= true;
                break;
            case 1: // NOR4
                result= inInst->inputs[0]->wire1->simVec | inInst->inputs[1]->wire1->simVec;
                result2= inInst->inputs[2]->wire1->simVec | inInst->inputs[3]->wire1->simVec;
                result= result | result2;
                result= ~result;
                processed= true;
                break;
            }
        }
        if (processed == false)
        {
            // Use truth table, calculate bit-by-bit
            unsigned int minterm, bitCount;
            vector<optSimPort*>::reverse_iterator portIter;
            optSimWire* myWire;

            result2= 0;
            for (bitCount= 0; bitCount < sizeof(SimulationVector) * 8; bitCount++)
            {
                minterm= 0;
                for (portIter= inInst->inputs.rbegin(); 
                     portIter != inInst->inputs.rend(); 
                     portIter++)
                {
                    minterm= minterm << 1;
                    myWire= (*portIter)->wire1;
                    if ((myWire->simVec & (1 << bitCount)) != 0)
                    {
                        minterm |= 1;
                    }
                }
                if (cell->tTables[i][minterm] == true)
                    result2 |= (1 << bitCount);
            }
            result= result2;
        }
        #ifdef DEBUG
        cout<<result<<" ";
        #endif
        if (inInst->outputs[i]->wire1)
        {
            if (isw == 0)
                inInst->outputs[i]->wire1->simVec= result;
            else
                inInst->outputs[i]->wire1->nextSimVec= result;
        }
    }
    #ifdef DEBUG
    cout<<endl;
    #endif
}

// *****************************************************************************
// setVector()
//
/// \brief Set a vector to a net. 
//
// *****************************************************************************
void optSim::setVector(oaNet* net, SimulationVector vec, int isw)
{
    optSimWire* myWire;

    if (net->getNumBits() > 1)
    {
        cout<<"Simulation vector must be set on single-bit nets."<<endl;
        cout<<"The problematic net name is: "<<getNameFromOA(net)<<endl;
        return;
    }
    myWire= myCkt->wireMap[net];
    if (myWire != myCkt->tie0Wire && myWire != myCkt->tie1Wire)
    {
        if (isw == 0)
            myWire->simVec= vec;
        else
            myWire->nextSimVec= vec;
    }
}

// *****************************************************************************
// getStateVectorD()
//
/// \brief Get vector of a state register. 
///
/// Returns its current D value.
//
// *****************************************************************************
void optSim::getStateVectorD(oaInst* inst, SimulationVector& vec)
{
    optSimInst* myInst;
    optSimWire* myWire;
    vector<optSimPort*>::iterator portIter;

    myInst= myCkt->instMap[inst];
    for (portIter= myInst->inputs.begin(); portIter != myInst->inputs.end(); portIter++)
    {
        myWire= (*portIter)->wire1;
        if (myWire->isRegD)
            vec= myWire->simVec;
    }
}

// *****************************************************************************
// getStateVector()
//
/// \brief Get vector of a state register. 
///
/// Returns its current Q value.
//
// *****************************************************************************
void optSim::getStateVector(oaInst* inst, SimulationVector& vec)
{
    optSimInst* myInst;
    optSimWire* myWire;
    vector<optSimPort*>::iterator portIter;

    myInst= myCkt->instMap[inst];
    for (portIter= myInst->outputs.begin(); portIter != myInst->outputs.end(); portIter++)
    {
        myWire= (*portIter)->wire1;
        if (myWire->isRegOutQN)
            vec= ~myWire->simVec;
        else
            vec= myWire->simVec;
    }
}
// *****************************************************************************
// setStateVector()
//
/// \brief Set vector to a state register. 
///
/// Will search Q and QN.
//
// *****************************************************************************
void optSim::setStateVector(oaInst* inst, SimulationVector vec)
{
    optSimInst* myInst;
    optSimWire* myWire;
    vector<optSimPort*>::iterator portIter;

    myInst= myCkt->instMap[inst];
    for (portIter= myInst->outputs.begin(); portIter != myInst->outputs.end(); portIter++)
    {
        myWire= (*portIter)->wire1;
        if (myWire == NULL)
            continue;
        if (myWire->isRegOutQN)
            myWire->simVec= ~vec;
        else
            myWire->simVec= vec;
    }
}

// *****************************************************************************
// getVector()
//
/// \brief Get vector from a net. 
//
// *****************************************************************************
void optSim::getVector(oaNet* net, SimulationVector& vec)
{
    optSimWire* myWire;

    if (net->getNumBits() > 1)
    {
        cout<<"Simulation vector only available on single-bit nets."<<endl;
        cout<<"The problematic net name is: "<<getNameFromOA(net)<<endl;
        return;
    }
    myWire= myCkt->wireMap[net];
    vec= myWire->simVec;
}

// *****************************************************************************
// generateRandomInputVectors()
//
/// \brief Generate random input vectors.
//
// *****************************************************************************
void optSim::generateRandomInputVectors()
{
    vector<oaNet*>::iterator netIter;

    for (netIter= myCkt->inputNets.begin(); netIter != myCkt->inputNets.end(); netIter++)
    {
        setVector(*netIter, randomVector());
    }
}

// *****************************************************************************
// generateRandomStateVectors()
//
/// \brief Generate random state vectors.
//
// *****************************************************************************
void optSim::generateRandomStateVectors()
{
    vector<oaInst*>::iterator instIter;

    for (instIter= myCkt->stateInsts.begin(); instIter != myCkt->stateInsts.end(); instIter++)
    {
        setStateVector(*instIter, randomVector());
    }
}

// *****************************************************************************
// runFull()
//
/// \brief Do one-cycle full simulation.
//
// *****************************************************************************
void optSim::runFull()
{
    vector<optSimInst*>::iterator instIter;

    for (instIter= myCkt->levelizedInst.begin(); instIter != myCkt->levelizedInst.end();
         instIter++)
    {
        simulateInst(*instIter);
    }
}

// *****************************************************************************
// propagateState()
//
/// \brief Propogate state values across state variables.
///
/// Values will pass through state registers.
//
// *****************************************************************************
void optSim::propagateState()
{
    // Propagate state register's inputs to outputs
    vector<optSimWire*>::iterator wireIter;
    optSimWire* myWire, *myWire2;
    optSimInst* myInst;
    vector<optSimPort*>::iterator portIter, portIter2;
    SimulationVector vec;
    vector<SimulationVector> origVecs;
    int i;

    #ifdef DEBUG
    cout<<"Propagate states"<<endl;
    #endif
    for (wireIter= myCkt->stateInWires.begin(); wireIter != myCkt->stateInWires.end(); wireIter++)
    {
        myWire= *wireIter;
        vec= myWire->simVec;
        origVecs.push_back(vec);
    }
    i= 0;
    for (wireIter= myCkt->stateInWires.begin(); wireIter != myCkt->stateInWires.end(); wireIter++)
    {
        myWire= *wireIter;
        vec= origVecs[i];
        i++;
        for (portIter= myWire->outputs.begin(); portIter != myWire->outputs.end(); portIter++)
        {
            myInst= (*portIter)->inst1;
            if (myInst->isStateReg)
            {        
                for (portIter2= myInst->outputs.begin(); 
                     portIter2 != myInst->outputs.end(); 
                     portIter2++)
                {
                    myWire2= (*portIter2)->wire1;
                    if (myWire2 == NULL)
                        continue;
                    if (myWire2->isRegOutQN)
                        myWire2->simVec= ~vec;
                    else if (myWire2->isRegOut)
                        myWire2->simVec= vec;
                    #ifdef DEBUG
                    if (myWire2->isRegOut || myWire2->isRegOutQN)
                    {
                        cout<<"Instance is "<<getNameFromOA(myInst->origInst)<<endl;
                        cout<<getNameFromOA(myWire->origWire)<<": "<<vec<<endl;
                        cout<<getNameFromOA(myWire2->origWire)<<": "<<myWire2->simVec<<endl;
                    }
                    #endif
                }
            }
        }
    }
}

// *****************************************************************************
// nextCycle()
//
/// \brief Do another cycle of full simulation. 
///
/// Values will pass through state registers.
//
// *****************************************************************************
void optSim::nextCycle()
{
    propagateState();
    runFull();
}

// *****************************************************************************
// randomVector()
//
/// \brief Generate a random vector.
//
// *****************************************************************************
SimulationVector optSim::randomVector()
{
    unsigned int i;
    const int N_RANDS = UINT_MAX / RAND_MAX;

    SimulationVector result = 0;
    for (i= 0; i < sizeof(SimulationVector) / sizeof(int); i++)
    {
        for(int i = 0; i<N_RANDS; ++i)
            result += static_cast<SimulationVector>(rand()) << (i * sizeof(int) * 8);
    }
    return result;
}

// *****************************************************************************
// getStateNets()
//
/// \brief Get nets connected to the outputs of state registers.
//
// *****************************************************************************
void optSim::getStateNets(vector<oaNet*>& result)
{
    result= myCkt->stateNets;
}

// *****************************************************************************
// cleanUp()
//
/// \brief Destroy datastructures built and free the memory.
//
// *****************************************************************************
void optSim::cleanUp()
{
    if (myUserPattern)
        delete(myUserPattern);
}

// *****************************************************************************
// ~sim()
//
/// \brief Destructor.
//
// *****************************************************************************
optSim::~optSim()
{
    cleanUp();
}

// *****************************************************************************
// doRandomSimulation()
//
/// \brief Do random simulation for one cycle.
//
// *****************************************************************************
void optSim::doRandomSimulation()
{
    vector<oaNet*> nets;
    
    generateRandomInputVectors();
    generateRandomStateVectors();
    runFull();
}

// *****************************************************************************
// incrementalSim
//
/// \brief Do incremental simulation for the net.
///
///  It is assumed that its value is changed to newVec. Simulation values of all other nets must have been available before calling incrementalSim. For example, runFull() has been called before.
//
// *****************************************************************************
void optSim::incrementalSim(oaNet* net, SimulationVector newVec)
{
    optSimWire* wire2;

    wire2= myCkt->wireMap[net];
    incrementalSim(wire2, newVec);
}

// *****************************************************************************
// incrementalSim
//
/// \brief Do incremental simulation for the wire.
//
// *****************************************************************************
void optSim::incrementalSim(optSimWire* wire, SimulationVector newVec)
{
    // See my master's thesis for the simulation algorithm
    list<optSimWire*> activeNodes;
    list<optSimInst*> activeGates;
    optSimWire* wire2;
    vector<optSimPort*>::iterator portIter;
    optSimInst* inst2;

    wire->nextSimVec= newVec;
    activeNodes.push_back(wire);

    do
    {
        // Update active Nodes
        while (activeNodes.empty() == false)
        {
            wire2= activeNodes.front();
            wire2->marked= false;
            activeNodes.pop_front();
            wire2->simVec= wire2->nextSimVec;
            for (portIter= wire2->outputs.begin(); portIter != wire2->outputs.end(); portIter++)
            {
                inst2= (*portIter)->inst1;
                if (inst2 && inst2->marked == false && inst2->isStateReg == false)
                {
                    inst2->marked= true;
                    activeGates.push_back(inst2);
                }
            }
        }
        while (activeGates.empty() == false)
        {
            
            inst2= activeGates.front();
            activeGates.pop_front();
            inst2->marked= false;
            simulateInst(inst2, 1);
            for (portIter= inst2->outputs.begin(); portIter != inst2->outputs.end(); portIter++)
            {
                wire2= (*portIter)->wire1;
                if (wire2->nextSimVec != wire2->simVec && wire2->marked == false)
                {
                    wire2->marked= true;
                    activeNodes.push_back(wire2);
                }
            }
        }
    } while (activeNodes.empty() == false);
}

/// \brief Advance input to next cycle.
/// Get new input vectors and apply them to primary inputs. If the user
/// has provided data file, inputs will be from the file. Otherwise random
/// inputs will be returned. If user data has been exhausted, random
/// inputs will be used.
///
void optSim::advanceInputs()
{
    vector<SimulationVector> vecs;
    unsigned int i;
    
    if (myUserPattern && myUserPattern->isFinished() == false)
    {
        if (myUserPattern->getVector(vecs) == false)
        {
            generateRandomInputVectors();
            return;
        }   
        for (i= 0; i < inputVecNets.size(); i++)
        {
            setVector(inputVecNets[i], vecs[i]);
        }
    }
    else
    {
        generateRandomInputVectors();
    }
}

/// Set user pattern file.
/// \param fileName file name 
/// \return true if success.
bool optSim::setUserPatternFile(const char* fileName)
{
    vector<oaInst*>::iterator instIter;

    myUserPattern= new (userPattern);
    // If input file not available, set states to random
    if (myUserPattern->setFile(design, fileName) == false)
    {
        delete(myUserPattern);
        myUserPattern= NULL;
        generateRandomStateVectors();
        return false;
    }
    // if input file available, set states to 0
    for (instIter= myCkt->stateInsts.begin(); instIter != myCkt->stateInsts.end(); instIter++)
    {
        setStateVector(*instIter, 0);
    }
    myUserPattern->getNets(inputVecNets);
    return true;
}

/// \brief Step to next cycle.
/// Will update inputs and do next cycle simulation
void optSim::step()
{
    advanceInputs();
    nextCycle();
}

/// \brief If true, user patterns have been exhausted
bool optSim::isFinished()
{
    if (myUserPattern)
    {
        if (myUserPattern->isFinished())
        {
            return true;   
        }
    }
    return false;
}

/// \brief use state file to initialize states.
bool optSim::useStateFile(const char* fileName)
{
    ifstream iFile;
    vector<oaInst*> insts;
    oaInst* inst;
    vector<oaInst*>::iterator instIter;
    SimulationVector simVal;
    int result;
    
    string token;
    oaBlock* block= design->getTopBlock();
    iFile.open(fileName, ios::in);
    if (iFile.is_open())
    {
        while (util::getToken(iFile, token) == 0)
        {
            const oa::oaNativeNS nativeNS;
            oaSimpleName name1(nativeNS, static_cast<const oaChar*>(token.c_str()));
            inst= oaInst::find(block, name1);
            if  (inst == NULL)
            {
                cout<<"Instance "<<token<<" not found, ignored."<<endl;
            }
            insts.push_back(inst);
        }
        // Read values
        for (instIter= insts.begin(); instIter != insts.end(); instIter++)
        {
            result= util::getToken(iFile, token);
            if (result == 1)
                continue;
            else if (result == 2)
            {
                iFile.close();
                break;
            }
            else if (result == 0)
            {
                if (SimVectorSize == 32)
                    simVal= atol(token.c_str());
                else
                    simVal= atoll(token.c_str());
                if (*instIter)
                    setStateVector(*instIter, simVal);
            }
        }
        return true;
    }
    else
        return false;
}

} // End of namespace
// vim: ci et sw=4 sts=4
