
/* 
Author: Kai-hui Chang <changkh@eecs.umich.edu>
*/

///
/// \file oagResynOptInternalCkt.h
/// Internal data structure for optimized simulation.

#if !defined(oagResynoptInternalCkt_P)
#define oagResynoptInternalCkt_P

#include "oaDesignDB.h"
#include "oagResyn.h"
#include <vector>
#include <map>

namespace oagResyn {

/*
template <> std::string getNameFromOA<oa::oaDesign*>(oa::oaDesign* i);

template <> std::string getNameFromOA<oa::oaDesign*>(oa::oaDesign* i)
{
    oa::oaNativeNS ns;
    oa::oaString name;
    i->getCellName(ns,name);
    return std::string(static_cast<const char*>(name) );
}
*/


/// Internal port data structure 
class optSimPort 
{
public:
    /// Points to the wire it connects to
    class optSimWire* wire1;
    /// Points to the instance it connects to
    class optSimInst* inst1;
    /// Points to oaTerm
    oa::oaTerm* origPort;
    oa::oaInstTerm* origInstTerm;
    void connectWire(optSimWire* newWire);
    void removeFromWire(); 
    void destroy(); 
};

/// Internal wire data structure
class optSimWire
{
public:
    optSimWire() {isPI= false; isPO= false; isRegOut= false; regQWire= NULL; marked= false; isRegOutQN= false; isRegD= false; ext= NULL;};
    void removePort(optSimPort* inPort);
    void removeDriver();
    /// The driver of the wire
    optSimPort* driver;
    /// The fanout ports of the wire
    std::vector<optSimPort*> outputs;
    /// Simulation values
    SimulationVector simVec, nextSimVec; // nextSimVec is for event-driven simulation
    /// Signature of the wire
    std::vector<SimulationVector> signature;
    /// If true, the wire is a primary input
    bool isPI;
    /// If true, the wire is a primary output
    bool isPO;
    bool isRegOut;  ///< Is register's out (QN or Q)
    bool isRegOutQN; ///< Is register's out and is QN
    bool isRegD; ///< Is register's D
    bool marked; ///< Misc. usage, please clear after use.
    optSimWire* regQWire; ///< If this wire is QN, it points to the Q wire
    oa::oaNet* origWire; ///< Points to original oaNet
    void* ext; ///< Pointer to extension data structure
    int tmpflag1, tmpflag2; /// Temp flags
};

/// Internal data structure for instance
class optSimInst 
{
public:
    optSimInst() {visited= false; isStateReg= false; marked= false; level= 0;};
    bool visited; 
    std::vector<optSimPort*> inputs;  ///< Inputs of the instance
    std::vector<optSimPort*> outputs; ///< Outputs of the instance
    libCell* cell; ///< Points to library cell
    bool isStateReg; ///< If true, it is a state register
    oa::oaInst* origInst; ///< Points to original oaInst
    int level; ///< Its level of logic (determined during levelization)
    bool marked; ///< Temp flag, please clear after use
};


/// Internal circuit for optimized simulation
class optInternalCkt {
friend class optSim;
friend class equiCheck;
friend class optCktOper;

public:
    optInternalCkt(oa::oaDesign *design);
    ~optInternalCkt();
    void setup();
    void cleanUp();

    void analyzeCircuit();
    void buildInternalCircuit();
    void printInOutState();
    void printInternalCircuit();
    void levelizeInternalCircuit();
    void getStates(std::vector<oa::oaInst*>& result);
    void getStateNets(std::vector<oa::oaNet*>& result);
    void getStateInNets(std::vector<oa::oaNet*>& result);
    void getInputs(std::vector<oa::oaNet*>& result);
    void getOutputs(std::vector<oa::oaNet*>& result);
    void getClockNets(std::vector<oa::oaNet*>& result);
    void getAllNets(std::vector<oa::oaNet*>& result);
    void getAllInsts(std::vector<oa::oaInst*>& result);
    void getAllCombiInsts(std::vector<oa::oaInst*>& result);
    void getInputConeSimInsts(std::vector<oa::oaNet*>& nets, 
                              std::vector<optSimInst*>& insts);
    void optSimInstsGetSimWireIO(std::vector<optSimInst*>& insts, 
                                 std::vector<optSimWire*>& inputs, 
                                 std::vector<optSimWire*>& outputs, 
                                 std::vector<optSimWire*>& allNet);
    /// Get optSimWire from oaNet
    optSimWire* netGetSimWire(oa::oaNet* net) { return wireMap[net];}
    /// Get optSimInst from oaInst
    optSimInst* instGetSimInst(oa::oaInst* inst) { return instMap[inst];}
    void findWireClose(optSimWire* origWire, int maxNumber, 
                       std::vector<optSimWire*>& wires);
    void getOutputConeWires(optSimWire* origWire, 
                       std::set<optSimWire*>& outputConeWires);
    void getOutputConeInstWires(optSimWire* origWire, 
                       std::set<optSimInst*>& insts, 
                       std::set<optSimWire*>& outputConeWires);
    void setVector(oa::oaNet* net, SimulationVector vec, int isw);
    void getInputConeInsts(std::vector<oa::oaNet*>& nets,
         std::vector<oa::oaInst*>& insts, std::vector<oa::oaNet*>& inputNets);
    void getInputSimInsts(std::vector<optSimWire*>& wires,
         std::vector<optSimInst*>& insts, int instNO);
    optSimWire* addWire(std::string& name);
    void destroyWire(optSimWire* iWire);
    optSimInst* addInst(std::string& name, std::string& cellType);
    void destroyInst(optSimInst* iInst);
    void generateBlif(std::string fileName, std::vector<optSimInst*>& insts,
                      std::vector<optSimWire*>& inputs,
                      std::vector<optSimWire*>& outputs);
    void optSimInstsGetSubcktIO(std::vector<optSimInst*>& insts,
                                std::vector<optSimWire*>& inputs,
                                std::vector<optSimWire*>& outputs,
                                std::vector<optSimWire*>& others);
    void unconnectNetInputCone(oa::oaNet* net);
    int getMaxLevel();
    optSimWire *tie0Wire, *tie1Wire;
protected:  
    oa::oaDesign *design;
    oa::oaBlock *block;
    void instGetModName(oa::oaInst* inInst, std::string& modName);
    void printInternalInst(optSimInst* inst);
    std::vector<oa::oaInst*> stateInsts;
    std::vector<oa::oaNet*> inputNets, outputNets, stateNets, stateInNets;
    std::map<oa::oaNet*, optSimWire*> wireMap;
    std::map<oa::oaInst*, optSimInst*> instMap;
    std::vector<optSimInst*> levelizedInst;
    std::vector<optSimWire*> stateInWires;
    std::set<oa::oaNet*> clockNets; 
    std::set<oa::oaNet*> allBitNets;
};

/// Set vector value for a net.
inline void optInternalCkt::setVector(oa::oaNet* net,
                                   SimulationVector vec,
                                   int isw= 0)
{
    optSimWire* myWire;

    myWire= wireMap[net];
    if (myWire == tie1Wire || myWire == tie0Wire)
        return;
    if (isw == 0)
        myWire->simVec= vec;
    else
        myWire->nextSimVec= vec;
}


}

#endif

// vim: ci et sw=4 sts=4
