#include "oagRedunRTG.h"

#define foreach(i,x) for(typeof((x).begin()) i=(x).begin(); i!=(x).end(); ++i)

using namespace std;

namespace oagRedun {

class RTGoag : public RTG {
    typedef oagAi::Graph     Graph;
    typedef oagAi::Ref       Ref;
    typedef oagAi::Node      Node;
    static unsigned index(Ref r) { return Graph::getNodeID(r); }

    /// AI graph
    Graph                  &_graph;
    const std::vector<Ref> &_cos;
    unsigned                _dirtyMarker;
    std::vector<bool>       _isCO;

    /// DFS ordering of graph nodes
    std::vector<Ref>        _dfsNum2ref;
    std::vector<unsigned>   _idx2dfsNum; 
    void dfs(Ref);

    /// simulation data
    class SimData {
        unsigned _data;
    public:
        void randomize() { _data = random(); }
        void reset()     { _data =  0; }
        void set()       { _data = ~0; }

        void print(FILE *fp = stdout) const {
            unsigned d = _data;
            for (unsigned i=0; i<sizeof(unsigned)*CHAR_BIT; i++, d>>=1) {
                fprintf(fp, "%d", d&1);
            }
        }

        SimData operator~() const { 
            SimData s; s._data = ~_data; return s; 
        }
        SimData operator^(bool b) const { 
            SimData s; s._data = b ? ~_data: _data; return s; 
        }
        SimData operator&=(const SimData &b) { 
            _data &= b._data; return *this; 
        }
        bool operator!=(const SimData &b) const {
            return _data != b._data;
        }
    };
    std::vector<SimData>    _goodData;
    std::vector<SimData>    _badData;

    /// event queue
    // 80% of the RTGoag runtime is probably spent here
    std::set<unsigned> _q;

    bool propagate(Ref ref);
    void init();
public:
    RTGoag(Graph &_g, const std::vector<Ref> &o, unsigned seed);
    void randomSim();
    bool isTested(Ref ref, bool SA1);
    void removeTestSite(Ref ref);
};

void
RTGoag::dfs(Ref ref)
{
    if (_graph.isVisited(ref)) return;
    _graph.markVisited(ref);

    switch (_graph.getNodeType(ref)) {
        case Node::CONSTANT0:
            break;
        case Node::AND: 
            dfs(_graph.getAndLeft (ref));
            dfs(_graph.getAndRight(ref));
            break;;
        case Node::TERMINAL: 
            dfs(_graph.getTerminalDriver(ref));
            break;
        case Node::NONE:
        case Node::SEQUENTIAL:
            assert(0);
    }

    if (_idx2dfsNum.size() <= index(ref)) {
        _idx2dfsNum.resize(index(ref)+1, UINT_MAX);
    }
    _idx2dfsNum[index(ref)] = _dfsNum2ref.size();
    _dfsNum2ref.push_back(Graph::getNonInverted(ref));
}

void
RTGoag::init()
{
    _dirtyMarker = _graph.getDirtyMarker();
    _isCO       .clear();
    _dfsNum2ref .clear();
    _idx2dfsNum.clear();
    _goodData   .clear();
    _badData    .clear();

    _graph.newTraversalID();
    _graph.markVisited(_graph.getNull());
    foreach (it, _cos) {
        dfs(*it);
    }

    const unsigned n = _idx2dfsNum.size();
    _goodData.resize(n);
    _badData .resize(n);
    
    _isCO.resize(n, false);
    foreach (it, _cos) {
        _isCO[index(*it)] = true;
    }
}

RTGoag::RTGoag(Graph &g, const vector<Ref> &o, unsigned seed)
    : _graph(g), _cos(o)
{
    srandom(seed);
    init();
}

void
RTGoag::randomSim()
{
    assert(_graph.getDirtyMarker() == _dirtyMarker);
    foreach (it, _dfsNum2ref) {
        Ref      ref = *it;
        unsigned i   = index(ref);
        switch (_graph.getNodeType(ref)) {
            case Node::CONSTANT0:
                _goodData[i].reset();
                break;
            case Node::AND: {
                Ref l = _graph.getAndLeft (ref);
                Ref r = _graph.getAndRight(ref);
                _goodData[i]  = _goodData[index(l)] ^ Graph::isInverted(l);
                _goodData[i] &= _goodData[index(r)] ^ Graph::isInverted(r);
                break;
            }
            case Node::TERMINAL: {
                Ref l = _graph.getTerminalDriver(ref);
                if (l) {
                    _goodData[i] = _goodData[index(l)] ^ Graph::isInverted(l);
                } else {
                    _goodData[i].randomize();
                }
                break;
            }
            default:
                assert(0);
        }
        _badData[i] = _goodData[i];
        //printf("%d\t: ", i);
        //_goodData[i].print();
        //printf("\n");
    }

#if 0
    // observability
    vector<SimData> obs(_goodData.size());
    // all outputs observable
    foreach (it, _cos) 
        obs[index(*it)].set();

    for (int i=_dfsNum2ref.size()-1; i>=0; i--) {
        Ref ref = _dfsNum2ref[i];
    }
#endif
}

void
RTGoag::removeTestSite(Ref ref)
{
    assert(_graph.getNodeType(ref) == Node::TERMINAL);
    _graph.removeEquivalent(ref);
    _graph.resubstitute(ref, _graph.getTerminalDriver(ref));
    _graph.setTerminalDriver(ref, _graph.getNull());
    _dirtyMarker = _graph.getDirtyMarker();
}

bool
RTGoag::propagate(Ref ref)
{
    const unsigned i = index(ref);

    if (_isCO[i] and _badData[i]!=_goodData[i]) 
        return true;

    assert(_q.empty());
    foreach (it, _graph.getFanout(ref)) {
        // floating logic?
        if (_idx2dfsNum[index(*it)] == UINT_MAX) continue;
        _q.insert(_idx2dfsNum[index(*it)]);
    }

    // try to propagate fault
    foreach (it, _q) {

        Ref      ref = _dfsNum2ref[ *it];
        unsigned i   = index(ref);

        switch (_graph.getNodeType(ref)) {
            case Node::AND: {
                Ref l = _graph.getAndLeft (ref);
                Ref r = _graph.getAndRight(ref);
                _badData[i]  = _badData[index(l)] ^ Graph::isInverted(l);
                _badData[i] &= _badData[index(r)] ^ Graph::isInverted(r);
                break;
            }
            case Node::TERMINAL: {
                Ref l = _graph.getTerminalDriver(ref);
                assert(l != _graph.getNull());
                _badData[i] = _badData[index(l)] ^ Graph::isInverted(l);
                break;
            }
            default:
                assert(0);
        }

        if (_badData[i] != _goodData[i]) {
            if (_isCO[i]) return true;
            foreach (it, _graph.getFanout(ref)) {
                // floating logic?
                if (_idx2dfsNum[index(*it)] == UINT_MAX) continue;
                _q.insert(_idx2dfsNum[index(*it)]);
            }
        }
    }
    return false;
}

bool
RTGoag::isTested(Ref ref, bool SA1)
{
    const unsigned i = index(ref);
    assert(i < _badData.size());
    assert(_graph.getDirtyMarker() == _dirtyMarker);

    if (SA1) _badData[i].set();
    else     _badData[i].reset();

    bool res = propagate(ref);

    // restore data
    _badData[i] = _goodData[i];
    foreach (it, _q) {
        unsigned i = index(_dfsNum2ref[*it]);
        _badData[i] = _goodData[i];
    }
    _q.clear();

    return res;
}

RTG*
RTG::createOAG(oagAi::Graph &_g, const std::vector<oagAi::Ref> &o, unsigned seed)
{
    return new RTGoag(_g, o, seed);
}

}
// vim:et:
