/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-05-31: ChangeLog started
*/

#if !defined(oagFuncVerilogSynthesis_P)
#define oagFuncVerilogSynthesis_P

#include "oaDesignDB.h"
#include "oagAiGraph.h"
#include "oagFunc.h"
#include "oagFuncMultiRef.h"
#include "oagFuncVerilogDesign.h"
#include "oagFuncManager.h"
#include "oagFuncSynthesis.h"
#include <list>
#include <set>
#include <string>

using namespace std;

namespace oagFunc {

// *****************************************************************************
// VerilogSynthesis
//
/// \brief Constructs OpenAccess objects and a functional description from a given VerilogDesign object.  
///
/// All calls are static functions.
//
// *****************************************************************************

class VerilogSynthesis : public Synthesis {

    static const bool IMPLICIT_NET_WARNINGS = false;

  protected:

// *****************************************************************************
// Type definitions for various Verilog objects.
// *****************************************************************************

    typedef map<string, ConstantValue> ParameterValues;

    class FunctionVariableAssignment : public map<int, MultiRef> {
      public:
  
        int     start, stop;
        
        FunctionVariableAssignment() { start = stop = 0; }
        FunctionVariableAssignment(int st, int sp) { start = st; stop = sp; }
    };

// *****************************************************************************
// ProceduralState
//
/// This data structure holds information about the state of variables within a
/// procedural section of code.
//
// *****************************************************************************

class ProceduralState {
  public:

    set<oa::oaModBitNet*>                   dependencies;
    map<oa::oaModBitNet*, MultiRef>         blockingAssignments, 
                                            nonblockingAssignments;
    bool                                    isRegister;
    bool                                    isFunction;
    map<string, FunctionVariableAssignment> functionAssignments;
    
                                    ProceduralState() { isRegister = false; isFunction = false; }
};


// *****************************************************************************
// ConditionalLvalRef
//
/// This represents the destination of a left-hand assignment and the condition
/// under which it should be assigned.
//
// *****************************************************************************

class ConditionalLvalRef {
  public:
  
    oa::oaModBitNet*    lval;
    MultiRef            condition;
                        ConditionalLvalRef(oa::oaModBitNet *n, MultiRef c) { lval = n; condition = c; }
};

typedef enum{LVAL_UNKNOWN, LVAL_UNCONDITIONAL, LVAL_CONDITIONAL, LVAL_FUNCTION} LvalType;

// *****************************************************************************
// LvalRef
//
/// This represents the destination of a left-hand assignment.
//
// *****************************************************************************

class LvalRef {
  public:
    LvalType                 type;
    
    oa::oaModBitNet*         unconditionalLval;
    list<ConditionalLvalRef> conditionalLval;
    string                   functionLvalName;
    int                      functionLvalBit;
    
                             LvalRef(string c, int b);
                             LvalRef(oa::oaModBitNet* n);
                             LvalRef();
    void                     addConditional(oa::oaModBitNet *n, MultiRef c);
};

typedef list<LvalRef*> LvalRefBus;

// *****************************************************************************

  public:
  
    static void                    synthesize(VerilogDesign *design);

  protected:
  
    // synthesize particular subcomponents
    static void                    synthesizeModule(VerilogDesign::Module *module,
                                                    list<ConstantValue> &parameters);
    static void                    synthesizeModuleNets();
    static void                    synthesizeModuleTerms();
    static void                    synthesizeModuleAssigns();
    static void                    synthesizeModuleInsts();
    static void                    synthesizeModuleFunc();

    static void                    synthesizeBehavioral(VerilogDesign::Statement *statement, 
                                                        ProceduralState &state, 
                                                        ParameterValues *parameters);
    static string                  getParameterizedModuleName(VerilogDesign::Module *vModule, 
                                                              list<ConstantValue> &parameters);
    
    // parse expression trees
    static void                    evaluateConstantExpression(ConstantValue &result, 
                                                              VerilogDesign::Expression *expression,
                                                              ParameterValues *parameters);
    static bool                    isConstantExpression(VerilogDesign::Expression *expression, 
                                                        ParameterValues *parameters);
    static void                    evaluateExpression(MultiRefBus &result, VerilogDesign::Expression *expression, 
                                                      ProceduralState *state, ParameterValues *parameters);
    static void                    evaluateLval(LvalRefBus &result, VerilogDesign::Expression *expression, 
                                                ProceduralState *state, ParameterValues *parameters);
    static MultiRef                getContextualValue(oa::oaModBitNet *net, ProceduralState *state);
    
    // utility
    static void                    appendSuffix(string &root, const string suffix);

    /// \name Intermediate state
    /// @{

    // The names of the finished modules
    static set<string>             finishedModules;

    /// The Verilog module currently being synthesized
    static VerilogDesign::Module  *currentVmodule;
    /// The parameters of the module currently being synthesized
    static ParameterValues        *currentParams;

    static map<string, Bounds>     twoDimRegisters;

    /// @}
};

}

#endif
