/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-05-31: ChangeLog started
*/

#include "oaDesignDB.h"
#include "oagFuncManager.h"
#include "oagFuncSynthesis.h"
#include <list>
#include <set>

//#define DEBUG

#include "oagFuncDebug.h"

#define sign(a)  ( (a)==0?0: ((a)<0?-1:1) )
#define abs(a)   ( (a)>=0?(a):-(a) )
#define max(a,b) ( (a)>(b)?(a):(b) )
#define min(a,b) ( (a)>(b)?(b):(a) )

namespace oagFunc {

// *****************************************************************************
// Class static variable definitions.
// *****************************************************************************

oa::oaNameSpace                     *Synthesis::nameSpace = NULL;
oa::oaLib                           *Synthesis::currentLibrary = NULL;
oa::oaView                          *Synthesis::currentView = NULL;
Manager                             *Synthesis::currentManager = NULL;
oa::oaModule                        *Synthesis::currentModule = NULL;
bool                                 Synthesis::overwriteStructure = true;

list<oa::oaLib*>                     Synthesis::leafLibs;
list<oa::oaScalarName>               Synthesis::leafViews;


// ***************************************************************************** 
// setOverwriteStructure() 
// 
/// \brief Set whether synthesis should clear and overwrite or reuse existing OA objects.
////// Upon creating a new module, the functional description being created
/// will always overwrite any existing functional description.
///
/// The synthesis process will also create structure (new nets, instances, 
/// terminals, etc.) as necessary.  This function sets the behavior towards the 
/// OpenAccess objects that are present inside a pre-existing module.  If
/// the overwriteStructure flag is set (the default), upon synthesizing a
/// module, the existing module will be cleared of all pre-existing OpenAccess
/// objects.
///
/// The alternative is to append functional information to pre-existing
/// structures.  This may be useful when information and structure need to be
/// read from multiple sources (such as first loading the physical description
/// of a cell library and then later appending the functional information for the
/// cells).
///
// ***************************************************************************** 
void
Synthesis::setOverwriteStructure(bool overwrite) {
    overwriteStructure = overwrite;
}


// *****************************************************************************
// zeroExpand()
//
/// \brief Expands the smaller of two bit vectors to the length of the larger.
///
/// The additional bits wired to constant zero.
//
// *****************************************************************************
void 
Synthesis::zeroExpand(MultiRefBus &l1, MultiRefBus &l2) {
    if(l1.size() < l2.size()) {
        for(int i=l2.size()-l1.size(); i>0; i--) {
            l1.push_front( constantZero() );
        }
    } else if(l1.size() > l2.size()) {
        for(int i=l1.size()-l2.size(); i>0; i--) {
            l2.push_front( constantZero() );
        }
    }
}


///////////////////////////////////////////////////////////////////////////////
/// Checks if a Design exist by opening and closing the design.
/// This function is only a temporary wrapper for the function
/// oaDesign::exists, because oaDesign::exists is broken.
///
/// \param libName The name of the lib
/// \param cellName The name of the cell
/// \param viewName The name of the view

bool
Synthesis::checkForExistenceOfDesign(oa::oaScalarName libName,
                                     oa::oaScalarName cellName,
                                     oa::oaScalarName viewName)
{
    try {
        oa::oaDesign *design;
        design = oa::oaDesign::open(libName, cellName, viewName, 'a');
        design->close();
        return true;
    } catch (oa::oaException &e) {
        return false;
    }
}


// *****************************************************************************
// findModule()
//
/// \brief Find an existing module.
///
/// Looks for the given name in:
///   1. a module in the current design (if it exists)
///   2. a design in the current library / view
///   3. a design in the leaf libraries / leaf views (if any have been provided)
///
/// If the module does not exist, NULL is returned.
///
/// \param identifier the module name
/// \return the module, or NULL if it does not exist
//
// *****************************************************************************
oa::oaModule*           
Synthesis::findModule(const string identifier) {
    assert(nameSpace);

    DEBUG_PRINTLN("find module " << identifier);

    if (!currentLibrary || !currentView) {
      cerr << "ERROR: The design library and view both must first be set" << endl;
      QUIT_ON_ERROR;
    }

    oa::oaScalarName    cellName(*nameSpace, identifier.c_str());

    // 1. a module in the current design (if it exists)
    if (currentModule) {
        oa::oaModule *module = oa::oaModule::find(currentModule->getDesign(), cellName);
        if (module) {
            return module;
        }
    }

    // 2. search current design library
    oa::oaNativeNS      nativeNS;
    oa::oaScalarName    libName;
    currentLibrary->getName(libName);
    oa::oaScalarName    viewName;
    currentView->getName(viewName);

    oa::oaDesign *design;
    
    // find or open design
    if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
        return design->getTopModule();
    } else {
        try {
            design = oa::oaDesign::open(libName, cellName, viewName, 'a');
            assert(design);
            oa::oaModule *module = design->getTopModule();
            return module;
        } catch (oa::oaException &e) {
        }
    }

    // 3. search leaf libraries
    for(list<oa::oaLib*>::iterator libIter=leafLibs.begin();
        libIter!=leafLibs.end(); libIter++) {
      oa::oaLib *leafLib = *libIter;
      oa::oaScalarName leafLibName;
      leafLib->getName(leafLibName);

      // search each view
      for(list<oa::oaScalarName>::iterator viewIter=leafViews.begin();
          viewIter!=leafViews.end(); viewIter++) {

        if ((design = oa::oaDesign::find(leafLibName, cellName, *viewIter))) {
            return design->getTopModule();
        }

        try {
            design = oa::oaDesign::open(leafLibName, cellName, *viewIter, 'r');
            assert(design);
            oa::oaModule *module = design->getTopModule();
            return module;
        } catch (oa::oaException &e) {
            // design didn't exist
        }
      }

      cout << endl;
    }
    
    return NULL;
}


// *****************************************************************************
// createModule()
//
/// \brief Create a new module.
///
/// A design with the given identifier is either opened or created in the 
/// current library with the current view.  A module with the same name
/// is either opened or created and set as the top module.
/// 
/// If the overwriteStructure flag is set and the design already exists,
/// the previous version will be closed and destroyed, and a new one will
/// be created.
///
/// If the overwriteStructure flag is not set and the design already exists, 
/// any associated manager is deleted, and
/// all references from nets to the now deleted nodes are cleared.  Nothing
/// else is touched.
///
/// \param identifier the module name
/// \return the created/overwritten module
//
// *****************************************************************************
oa::oaModule*           
Synthesis::createModule(const string identifier) {
    assert(nameSpace);

    DEBUG_PRINTLN("create module " << identifier);

    if (!currentLibrary || !currentView) {
      cerr << "ERROR: The design library and view both must first be set" << endl;
      QUIT_ON_ERROR;
    }

    oa::oaScalarName    cellName(*nameSpace, identifier.c_str());

    // open design
    oa::oaNativeNS      nativeNS;
    oa::oaScalarName    libName;
    currentLibrary->getName(libName);
    oa::oaScalarName    viewName;
    currentView->getName(viewName);
    oa::oaViewType     *viewType = oa::oaViewType::get(oa::oacNetlist);
    oa::oaDesign       *design;
    
    // find or create design
    if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
        return design->getTopModule();
    } else {
        try {
            design = oa::oaDesign::open(libName, cellName, viewName, 'a');
        } catch (oa::oaException &e) {
            // create
            design = oa::oaDesign::open(libName, cellName, viewName, viewType, 'w');
        }
    }
    if (!design) {
      cerr << "ERROR: Could not find/open design: " << identifier << endl;
      QUIT_ON_ERROR;
    }
        
    // get/create top module
    oa::oaModule *module = design->getTopModule();
    if (!module) {
        module = oa::oaModule::create(design, cellName);
        design->setTopModule(module);
    } else {

        // any existing structure needs to be discarded (if overwriteStructure)
        if (overwriteStructure) {
            // destroy the existing design and create a new one
            design->purge();
            if (checkForExistenceOfDesign(libName, cellName, viewName)) {
                oa::oaDesign::destroy(libName, cellName, viewName);
            }
            design = oa::oaDesign::open(libName, cellName, viewName, viewType, 'w');
            module = oa::oaModule::create(design, cellName); 
            design->setTopModule(module); 
        }

        // any existing functional information needs to be discarded
        
        // clear existing manager
        if (oagFunc::Manager::hasManager(design)) {
            oagFunc::Manager *manager = oagFunc::Manager::get(design);
            assert(manager);
            delete manager;
            managerAppDef->destroy(design);
        }

        // clear connections from nets to nodes
        oa::oaModNet *net;
        oa::oaIter<oa::oaModNet> netIter(module->getNets(oacNetIterSingleBit));
        while ((net = netIter.getNext())) {
            AiRefAppDef->destroy(net);
        }
    }
    assert(module);

    // create global "tie0" and "tie1" nets if they don't exist
    oa::oaScalarName tie0Name(nativeNS, "tie0");
    if (oa::oaModScalarNet::find(module, tie0Name) == NULL) {
        oa::oaModScalarNet::create(module, tie0Name, oa::oacTieLoSigType, false);
    }
    oa::oaScalarName tie1Name(nativeNS, "tie1");
    if (oa::oaModScalarNet::find(module, tie1Name) == NULL) {
        oa::oaModScalarNet::create(module, tie1Name, oa::oacTieHiSigType, false);
    }

    return module;
}


// *****************************************************************************
// instantiateModule()
//
/// \brief Instantiates a module.
///
/// \param master the design to be instantiated in this module
/// \param name the name of the instance
///
/// \return the new ModInst
// *****************************************************************************
oa::oaModInst*          
Synthesis::instantiateModule(oa::oaDesign *master, const string name) {
    assert(nameSpace);
    assert(currentModule);
    assert(master);

    // find or create design
    oa::oaModInst *inst;
    if (name.empty()) {
        inst = oa::oaModScalarInst::create(currentModule, master);
    } else {
        inst = oa::oaModScalarInst::create(currentModule, master, 
                                           oa::oaScalarName(*nameSpace,name.c_str()));
    }

    return inst;
}
    

// *****************************************************************************
// createScalarNet()
//
/// \brief Creates a new scalar net or finds it if it already exists.
//
// *****************************************************************************
oa::oaModScalarNet*
Synthesis::createScalarNet(const string identifier) {
    assert(nameSpace);
    assert(currentModule);

    // does this net exist?
    oa::oaScalarName netName(*nameSpace, oa::oaString(identifier.c_str()));
    oa::oaModScalarNet *net = oa::oaModScalarNet::find(currentModule, netName);
    if (!net) {
        // create the net
        net = oa::oaModScalarNet::create(currentModule, netName);
    }

    // create a terminal node if module has functional description
    if (currentManager != NULL) {
        currentManager->setNetToAiConnection(net, 
            currentManager->ai.newTerminal(currentManager->ai.getNull()) );
    }       

    return net;
}


// *****************************************************************************
// createBusNet()
//
/// \brief Creates a new bus net or finds it if it already exists.
//
// *****************************************************************************
oa::oaModBusNet*
Synthesis::createBusNet(const string identifier, int start, int stop) {
    assert(nameSpace);
    assert(currentModule);
    
    oa::oaScalarName netName(*nameSpace, oa::oaString(identifier.c_str()));
    
    // does a bus net of this name already exist?
    oa::oaModBusNet *bus = oa::oaModBusNet::find(currentModule, netName, start, stop, 1);
    if (bus) {
        cerr << "ERROR: net " << identifier << " defined twice" << endl;
        QUIT_ON_ERROR;
    }
    
    // create the net
    bus = oa::oaModBusNet::create(currentModule, netName, start, stop, 1);

    oa::oaString modName;
    currentModule->getName(oa::oaVerilogNS(), modName);

    // create terminal AI nodes if functional manager exists
    if (currentManager != NULL) {
        for(int b = 0; b<=abs(stop-start); b++) {
            currentManager->setNetToAiConnection(bus->getBit(b), 
                currentManager->ai.newTerminal(currentManager->ai.getNull()) );
        }
    }

    return bus;
}


// *****************************************************************************
// createTerm()
//
/// \brief Create a new terminal or finds it if it already exists.
//
// *****************************************************************************
oa::oaModTerm*
Synthesis::createTerm(oa::oaModNet *net, const string identifier, oa::oaTermType type, 
                      unsigned int portPosition) {
    assert(nameSpace);
    assert(net);
    
    if (type == oa::oacInputOutputTermType) {
       cerr << "WARNING: Functionality not supported for inout ports" << endl;
    }
    
    oa::oaModTerm *term;

    oa::oaModule *module = net->getModule();
    oa::oaUInt4 numBits = net->getNumBits();
    oa::oaModTerm *newTerm = 0;
    
    // create term
    if (numBits == 1) {
        oa::oaScalarName sName(*nameSpace, identifier.c_str());
        term = oa::oaModTerm::find(module, sName);
        if (!term) {
            newTerm = oa::oaModTerm::create(net, sName, type);
        }
    } else {
        oa::oaVectorName vName(*nameSpace, identifier.c_str(), 0, numBits - 1);
        term = oa::oaModTerm::find(module, vName);
        if (!term) {
            newTerm = oa::oaModTerm::create(net, vName, type);
        }
    }

    if (term) {
        // FIXME perhaps we should do more checking here to make sure the
        // term has the right direction, bus width, position, etc.
        return term;
    }

    // set port position if given
    if (portPosition != oacNullIndex) {
        newTerm->setPosition(portPosition);
    }
    return newTerm;
}   


// *****************************************************************************
// findBusNet()
//
/// \brief Find a bus net.  
///
//
// *****************************************************************************
oa::oaModBusNet*
Synthesis::findBusNet(const string identifier) {
  assert(currentModule);
  assert(nameSpace);

  oa::oaScalarName baseName(*nameSpace,identifier.c_str());

  oa::oaModBusNetDef *busNetDef = oa::oaModBusNetDef::find(currentModule, baseName);
  if (busNetDef) {
    oa::oaIter<oa::oaModBusNet> busIter(busNetDef->getBusNets());
    oa::oaModBusNet *busNet = busIter.getNext();
    if (busNet) {
      return busNet;
    }
  }
   
  return NULL;
}


// *****************************************************************************
// findBusNetBit()
//
/// \brief Find an existing bus net bit.
//
// *****************************************************************************
oa::oaModBusNetBit*
Synthesis::findBusNetBit(const string identifier, const int bit) {
    assert(nameSpace);
    assert(currentModule);

    // create the net
    oa::oaScalarName netName(*nameSpace, identifier.c_str());
    return oa::oaModBusNetBit::find(currentModule, netName, bit);
}


// *****************************************************************************
// findScalarNet()
//
/// \brief Find an existing scalar net.
//
// *****************************************************************************
oa::oaModScalarNet*
Synthesis::findScalarNet(const string identifier) {
    assert(nameSpace);
    assert(currentModule);

    // create the net
    oa::oaScalarName netName(*nameSpace, identifier.c_str());
    return oa::oaModScalarNet::find(currentModule, netName);
}


// *****************************************************************************
// findNet()
//
/// \brief Find a net, either a scalar or a bus
//
// *****************************************************************************
oa::oaModNet*
Synthesis::findNet(const string identifier) {
  oa::oaModNet *net;
  net = findScalarNet(identifier);
  if (!net)
    net = findBusNet(identifier);  
  return net;
}


// *****************************************************************************
// assignMultiRef()
//
/// \brief Assigns the functionality of a net.
///
/// Attaches a MultiRef to a particular net.  If the MultiRef refers to an 
/// oagAi::Ref, store it in the AiRefAppDef application definition.  
/// If it refers to another net, mark the two nets as equivalent.
//
// *****************************************************************************
void
Synthesis::assignMultiRef(oa::oaModBitNet *net, const MultiRef e) {
    assert(nameSpace);
    assert(net);
    
    oa::oaString str;
    net->getName(*nameSpace, str);
    
    // has this net already been assigned?
    if (currentManager && net->getAppDefs().includes(AiRefAppDef)) {
        oagAi::Ref node = AiRefAppDef->get(net);
        if (!currentManager->ai.isNull(currentManager->ai.getTerminalDriver(node))) {
            cerr << "ERROR: net " << str << " is being assigned more than once" << endl;
            QUIT_ON_ERROR;
        }
    }
    
    if (e.type == AIREF) {
        assert(currentManager);
        // if we're being asked to assign an AIREF, we know this module must already be functional
        assert( net->getAppDefs().includes(AiRefAppDef) );
    
        oagAi::Ref node = AiRefAppDef->get(net);
        
        // repoint the node to the MultiRef
        currentManager->ai.setTerminalDriver(node, e.ai);
        DEBUG_PRINTLN("\t\t . " << str << " (functional) = " << e.ai)
    } else {
        net->makeEquivalent(e.net);
        DEBUG_PRINTLN("\t\t . " << str << " (structural)")
    }
}


// *****************************************************************************
// notOf()
//
/// \brief Return the not of a MultiRef.
//
// *****************************************************************************
MultiRef
Synthesis::notOf(MultiRef e) {
    assert(currentModule);

    // have we created a functional manager for this design yet?
    if (currentManager == NULL) {
        currentManager = Manager::get(currentModule->getDesign());
    }

    // if this MultiRef is a net, convert it to an oagAi::Ref
    if (e.type == NET) {
        // there must already be an existing oagAi::Ref
        oa::oaModBitNet *net = e.net;
        assert(net);
        assert(net->getAppDefs().includes(AiRefAppDef));
        e.ai = AiRefAppDef->get(net);
        e.type = AIREF;
    }

    assert(e.type == AIREF);
    return MultiRef(currentManager->ai.notOf(e.ai));
}

// *****************************************************************************
// andOf()
//
/// \brief Return the logical AND of two MultiRefs.
///
/// The necessary oagAI structures are created to represent the function.
//
// *****************************************************************************
MultiRef
Synthesis::andOf(MultiRef e1, MultiRef e2) {
    assert(currentModule);

    // have we created a functional manager for this design yet?
    if (currentManager == NULL) {
        currentManager = Manager::get(currentModule->getDesign());
    }

    // if this either MultiRef is a net, convert it to an oagAi::Ref
    if (e1.type == NET) {
        // there must already be an existing oagAi::Ref
        oa::oaModBitNet *net = e1.net;
        assert(net);
        assert(net->getAppDefs().includes(AiRefAppDef));
        e1.ai = AiRefAppDef->get(net);
        e1.type = AIREF;
    }
    if (e2.type == NET) {
        // there must already be an existing oagAi::Ref
        oa::oaModBitNet *net = e2.net;
        assert(net);
        assert(net->getAppDefs().includes(AiRefAppDef));
        e2.ai = AiRefAppDef->get(net);
        e2.type = AIREF;
    }
    
    assert(e1.type == AIREF);
    assert(e2.type == AIREF);
    return MultiRef(currentManager->ai.andOf(e1.ai, e2.ai));
}


// *****************************************************************************
// orOf()
//
/// \brief Return the logical OR of two MultiRefs.
///
/// The necessary oagAI structures are created to represent the function.
//
// *****************************************************************************
MultiRef
Synthesis::orOf(MultiRef e1, MultiRef e2) {
    assert(currentModule);

    // have we created a functional manager for this design yet?
    if (currentManager == NULL) {
        currentManager = Manager::get(currentModule->getDesign());
    }

    // if this either MultiRef is a net, convert it to an oagAi::Ref
    if (e1.type == NET) {
        // there must already be an existing oagAi::Ref
        oa::oaModBitNet *net = e1.net;
        assert(net);
        assert(net->getAppDefs().includes(AiRefAppDef));
        e1.ai = AiRefAppDef->get(net);
        e1.type = AIREF;
    }
    if (e2.type == NET) {
        // there must already be an existing oagAi::Ref
        oa::oaModBitNet *net = e2.net;
        assert(net);
        assert(net->getAppDefs().includes(AiRefAppDef));
        e2.ai = AiRefAppDef->get(net);
        e2.type = AIREF;
    }

    assert(e1.type == AIREF);
    assert(e2.type == AIREF);
    return MultiRef(currentManager->ai.notOf(currentManager->ai.andOf(
                       currentManager->ai.notOf(e1.ai), currentManager->ai.notOf(e2.ai))));
}


// *****************************************************************************
// xorOf()
//
/// \brief Return the logical XOR of two MultiRefs.
///
/// The necessary oagAI structures are created to represent the function.
//
// *****************************************************************************
MultiRef
Synthesis::xorOf(MultiRef e1, MultiRef e2) {
    assert(currentModule);

    // have we created a functional manager for this design yet?
    if (currentManager == NULL) {
        currentManager = Manager::get(currentModule->getDesign());
    }

    // if this either MultiRef is a net, convert it to an oagAi::Ref
    if (e1.type == NET) {
        // there must already be an existing oagAi::Ref
        oa::oaModBitNet *net = e1.net;
        assert(net);
        assert(net->getAppDefs().includes(AiRefAppDef));
        e1.ai = AiRefAppDef->get(net);
        e1.type = AIREF;
    }
    if (e2.type == NET) {
        // there must already be an existing oagAi::Ref
        oa::oaModBitNet *net = e2.net;
        assert(net);
        assert(net->getAppDefs().includes(AiRefAppDef));
        e2.ai = AiRefAppDef->get(net);
        e2.type = AIREF;
    }

    assert(e1.type == AIREF);
    assert(e2.type == AIREF);   
    oagAi::Ref t1 = currentManager->ai.andOf(currentManager->ai.notOf(e1.ai), e2.ai);
    oagAi::Ref t2 = currentManager->ai.andOf(currentManager->ai.notOf(e2.ai), e1.ai);
    return MultiRef(currentManager->ai.notOf(currentManager->ai.andOf(
                       currentManager->ai.notOf(t1), currentManager->ai.notOf(t2))));
}


// *****************************************************************************
// connectPort()
//
/// \brief Connects an oaModNet to an InstTerm by port position.  
///
///  The InstTerm may not preexist if the
///  master module of the Inst has not been defined.
//
// *****************************************************************************
void
Synthesis::connectPort(oa::oaModInst *inst, oa::oaModNet *net, 
                       unsigned int portPosition) {
    oa::oaModInstTerm::create(net, inst, portPosition);

    // check that port widths match
    oa::oaModule *module = inst->getMasterModule();
    assert(module);
    oa::oaModTerm *term = oa::oaModTerm::find(module, portPosition);
    assert(term);
    if(term->getNumBits() != net->getNumBits()) {
        cout << "ERROR: Port connection width mismatch" << endl;
        QUIT_ON_ERROR;
    }
}


// *****************************************************************************
// connectPort()
//
/// \brief Connects an oaModNet to an InstTerm by name.
///
///  The InstTerm may not preexist if the
///  master module of the Inst has not been defined.
//
// *****************************************************************************
void 
Synthesis::connectPort(oa::oaModInst *inst, oa::oaModNet *net, 
                       const string portString) {
    assert(nameSpace);
    
    oa::oaName netName, termName;
    net->getName(netName);
    if (net->getNumBits() == 1) {
        termName = oa::oaScalarName(*nameSpace, portString.c_str());
        oa::oaModInstTerm::create(net, inst, termName);
  
    } else if (netName.getType() == oa::oacVectorNameType) {
        termName = oa::oaVectorName(*nameSpace, portString.c_str(), 0,net->getNumBits()-1);
        oa::oaModInstTerm::create(net, inst, termName);
    } else {
        QUIT_ON_INTERNAL_ERROR;
    }
    
    // check that port widths match
    oa::oaModule *module = inst->getMasterModule();
    assert(module);
    oa::oaModTerm *term = oa::oaModTerm::find(module, termName);
    assert(term);
    if(term->getNumBits() != net->getNumBits()) {
        cout << "ERROR: Port connection width mismatch" << endl;
        QUIT_ON_ERROR;
    }
}


// *****************************************************************************
// reductionOr()
//
/// \brief Returns the logical OR of a list of MultiRefs.
///
/// OR's together a list of MultiRef.  This is especially useful in determining
/// logical truth.  Build graph as a tree to minimize unnecessary depth.
/// The function will destroy the contents of list l.
///
// *****************************************************************************
MultiRef
Synthesis::reductionOr(MultiRefBus &l) {
    assert(currentModule);
    assert(l.size() > 0);
    
    // have we created a functional manager for this design yet?
    if (currentManager == NULL) {
        currentManager = Manager::get(currentModule->getDesign());
    }
    
    MultiRefBus::iterator it, completed;
    while(l.size() > 1) {
        it = l.begin();
        while(it != l.end()) {
            MultiRef e1(*it);
            completed = it++;
            l.erase(completed);
            if (it == l.end()) {
                l.push_front(e1);
                break;
            }
            l.push_front(orOf(e1, *it));
            completed = it++;
            l.erase(completed);
        }
    }
    
    return l.front();
}


// *****************************************************************************
// reductionXor()
//
/// \brief Returns the logical OR of a list of MultiRefs.
///
/// OR's together a list of MultiRef.  This is especially useful in determining
/// logical truth.  Build graph as a tree to minimize unnecessary depth.
/// The function will destroy the contents of list l.
//
// *****************************************************************************
MultiRef
Synthesis::reductionXor(MultiRefBus &l) {
    assert(currentModule);
    assert(l.size() > 0);
    
    // have we created a functional manager for this design yet?
    if (currentManager == NULL) {
        currentManager = Manager::get(currentModule->getDesign());
    }
    
    MultiRefBus::iterator it, completed;
    while(l.size() > 1) {
        it = l.begin();
        while(it != l.end()) {
            MultiRef e1 = *it;
            completed = it++;
            l.erase(completed);
            if (it == l.end()) {
                l.push_front(e1);
                break;
            }
            l.push_front(xorOf(e1, *it));
            completed = it++;
            l.erase(completed);
        }
    }
    
    return l.front();
}


// *****************************************************************************
// reductionAnd()
//
/// \brief Returns the logical OR of a list of MultiRefs.
///
/// OR's together a list of MultiRef.  This is especially useful in determining
/// logical truth.  Build graph as a tree to minimize unnecessary depth.
/// The function will destroy the contents of list l.
//
// *****************************************************************************
MultiRef
Synthesis::reductionAnd(MultiRefBus &l) {
    assert(currentModule);
    assert(l.size() > 0);
    
    // have we created a functional manager for this design yet?
    if (currentManager == NULL) {
        currentManager = Manager::get(currentModule->getDesign());
    }
    
    MultiRefBus::iterator it, completed;
    while(l.size() > 1) {
        it = l.begin();
        while(it != l.end()) {
            MultiRef e1 = *it;
            completed = it++;
            l.erase(completed);
            if (it == l.end()) {
                l.push_front(e1);
                break;
            }
            l.push_front(andOf(e1, *it));
            completed = it++;
            l.erase(completed);
        }
    }
    
    return l.front();
}


// *****************************************************************************
// constantZero()
//
/// \brief Returns an MultiRef that represents a constant 0.  
///
/// The MultiRef refers to a net named "tie0".
///
/// This behavior is modeled after the OA to Verilog translator.
//
// *****************************************************************************
MultiRef
Synthesis::constantZero() {
    oa::oaNativeNS      nativeNS;
    oa::oaScalarName    name(nativeNS, "tie0");
    MultiRef e(oa::oaModScalarNet::find(currentModule, name));
    return e;
}


// *****************************************************************************
// constantOne()
//
//
/// \brief Returns an MultiRef that represents a constant 1.  
///
/// The MultiRef refers to a net named "tie1".
///
/// This behavior is modeled after the OA to Verilog translator.
//
// *****************************************************************************
MultiRef
Synthesis::constantOne() {
    oa::oaNativeNS      nativeNS;
    oa::oaScalarName    name(nativeNS, "tie1");
    MultiRef e(oa::oaModScalarNet::find(currentModule, name));
    return e;
}


// *****************************************************************************
// lessThan()
//
/// \brief Compares the values of two lists of MultiRefs.
///
/// Returns an MultiRef that is true if l1 < l2.  The lists must be of the
/// same size.  Destroys their contents but does not deallocate.
//
/// This generates an unbalanced architecture.  It should be improved.
//
// *****************************************************************************
MultiRef
Synthesis::lessThan(MultiRefBus &l1, MultiRefBus &l2) {
    assert(l1.size() == l2.size() && l1.size() > 0);
    
    MultiRef last(constantZero());
    while(!l1.empty()) {
        // a'b + last(ab + a'b')
        last = orOf(andOf(notOf(l1.back()), l2.back()),
                    andOf(last, notOf(xorOf(l1.back(), l2.back()))));
        l1.pop_back();
        l2.pop_back();
    }
    
    return last;
}


// *****************************************************************************
// equalTo()
//
/// \brief Compares the values of two lists of MultiRefs.
///
/// Returns an MultiRef that is true if l1 == l2.  The lists must be of the
/// same size.  Destroys their contents but does not deallocate.
//
/// This generates an unbalanced architecture.  It should be improved.
//
// *****************************************************************************
MultiRef
Synthesis::equalTo(MultiRefBus &l1, MultiRefBus &l2) {
    assert(l1.size() == l2.size() && l1.size() > 0);
    
    MultiRefBus l;
    MultiRefBus::iterator l1Iter = l1.begin(), l2Iter = l2.begin();
    while(l1Iter != l1.end() && l2Iter != l2.end()) {
        assert(l1.front().type != NEITHER);
        assert(l2.front().type != NEITHER);
        l.push_back( notOf(xorOf(*l1Iter, *l2Iter)) );
        ++l1Iter; ++l2Iter;
    }
    
    return reductionAnd(l);
}


// *****************************************************************************
// arithmeticAdd()
//
/// \brief Returns the sum of two MultiRefBuses.
///
/// A ripply carry adder is built.  The operands l1 and l2 must be of the same
/// width |l1|=|l2|.  The result will be one bit wider, |l1|+1.
///
// *****************************************************************************
void
Synthesis::arithmeticAdd(MultiRefBus &result, MultiRefBus &l1, MultiRefBus &l2) {
    assert(l1.size() == l2.size());
    result.clear();
    
    // this constructs a ripple-carry adder
    MultiRef sum, carryOut, carryIn = constantZero();
    MultiRefBus::reverse_iterator   it1 = l1.rbegin();
    MultiRefBus::reverse_iterator   it2 = l2.rbegin();
    while(it1 != l1.rend() && it2 != l2.rend()) {
        assert(it1->type == NET || it1->type == AIREF);
        assert(it2->type == NET || it2->type == AIREF);
        fullAdder(sum, carryOut, *it1, *it2, carryIn);
        result.push_front(sum);
        carryIn = carryOut;
        it1++; it2++;
    }
    result.push_front(carryOut);

}


// *****************************************************************************
// arithmeticSubtract()
//
/// \brief Returns the difference of two MultiRefBuses.
///
/// Builds a ripply carry subtractor.  The operands l1 and l2 must be of the same
/// width.  The result, l1-l2, will be of the same width as the operands.
//
// *****************************************************************************
void
Synthesis::arithmeticSubtract(MultiRefBus &result, MultiRefBus &l1, MultiRefBus &l2) {
    MultiRef discard;
    arithmeticSubtract(result, discard, l1, l2);
}


// *****************************************************************************
// arithmeticSubtract()
//
/// \brief Returns the difference of two MultiRefBuses and a sign flag.
///
/// Builds a ripply carry subtractor.  The operands l1 and l2 must be of the same
/// width.  The result, l1-l2, will be of the same width as the operands.
//
// *****************************************************************************
void
Synthesis::arithmeticSubtract(MultiRefBus &result, MultiRef &negFlag, 
                              MultiRefBus &l1, MultiRefBus &l2) {
    assert(l1.size() == l2.size());
    result.clear();
    
    // this constructs a ripple-carry subtractor
    MultiRef sum, carryOut, carryIn = constantOne();
    MultiRefBus::reverse_iterator   it1 = l1.rbegin();
    MultiRefBus::reverse_iterator   it2 = l2.rbegin();
    while(it1 != l1.rend() && it2 != l2.rend()) {
        assert(it1->type == NET || it1->type == AIREF);
        assert(it2->type == NET || it2->type == AIREF);
        fullAdder(sum, carryOut, *it1, notOf(*it2), carryIn);
        result.push_front(sum);
        carryIn = carryOut;
        it1++; it2++;
    }
    
    negFlag = notOf(carryIn);
}


// *****************************************************************************
// arithmeticMultiply()
//
/// \brief Returns the product of two MultiRefBuses.
///
/// Builds an array multiplier.  There is no requirement on the widths of the
/// operands, |l1| and |l2|.  The result l1*l2 will be of width |l1|+|l2|;
//
// *****************************************************************************
void
Synthesis::arithmeticMultiply(MultiRefBus &result, MultiRefBus &l1, MultiRefBus &l2) {
    assert(l1.size() > 1 && l1.size() > 1);

    result.clear();

    MultiRefBus arrayRow;
    MultiRefBus::reverse_iterator colIter, rowIter = l2.rbegin();
    for(colIter = l1.rbegin(); colIter != l1.rend(); ++colIter) {
        arrayRow.push_front(andOf(*colIter, *rowIter));
    }
    arrayRow.push_front(constantZero());
    ++rowIter;

    while(rowIter != l2.rend()) {
        result.push_front(arrayRow.back());
        arrayRow.pop_back();
        MultiRefBus lastArrayRow(arrayRow);
        MultiRefBus andRow;
        for(colIter = l1.rbegin(); colIter != l1.rend(); ++colIter) {
            andRow.push_front(andOf(*colIter, *rowIter));
        }
        assert(andRow.size() == lastArrayRow.size());
        arrayRow.clear();
        arithmeticAdd(arrayRow, andRow, lastArrayRow);
        ++rowIter;
    }

    while(!arrayRow.empty()) {
        result.push_front(arrayRow.back());
        arrayRow.pop_back();
    }
}


// *****************************************************************************
// arithmeticDivide()
//
/// \brief Returns the dividend of two MultiRefBuses.
///
/// Builds an array divider.  There is no requirement on the widths of the
/// operands, |l1| and |l2|.  The result l1/l2 will be of width |l1|;
//
// *****************************************************************************
void        
Synthesis::arithmeticDivide(MultiRefBus &result, MultiRefBus &l1, MultiRefBus &l2) {
    result.clear();
    // l1 is the quotient
    // l2 is the divisor

    MultiRefBus           arrayRow;
    zeroExpand(arrayRow, l2);
    MultiRefBus::iterator quotientIter = l1.begin();

    while(quotientIter != l1.end()) {
        arrayRow.pop_front();
        arrayRow.push_back(*quotientIter);
        MultiRefBus lastArrayRow(arrayRow);
        MultiRefBus afterSubtract;
        MultiRef    negFlag;
        arithmeticSubtract(afterSubtract, negFlag, lastArrayRow, l2);
        mux(arrayRow, negFlag, afterSubtract, lastArrayRow);
        result.push_back(notOf(negFlag));
        ++quotientIter;
    }
}


// *****************************************************************************
// arithmeticDivide()
//
/// \brief Returns the remainder after division of two MultiRefBuses.
///
/// Builds an array divider.  There is no requirement on the widths of the
/// operands, |l1| and |l2|.  The result l1%l2 will be of width |l2|;
//
// *****************************************************************************
void
Synthesis::arithmeticModulo(MultiRefBus &result, MultiRefBus &l1, MultiRefBus &l2) {
    result.clear();
    // l1 is the quotient
    // l2 is the divisor

    MultiRefBus           arrayRow;
    zeroExpand(arrayRow, l2);
    MultiRefBus::iterator quotientIter = l1.begin();

    while(quotientIter != l1.end()) {
        arrayRow.pop_front();
        arrayRow.push_back(*quotientIter);
        MultiRefBus lastArrayRow(arrayRow);
        MultiRefBus afterSubtract;
        MultiRef    negFlag;
        arithmeticSubtract(afterSubtract, negFlag, lastArrayRow, l2);
        mux(arrayRow, negFlag, afterSubtract, lastArrayRow);
        ++quotientIter;
    }
    
    // the remainder is set of remaining values
    while(!arrayRow.empty()) {
        result.push_front(arrayRow.back());
        arrayRow.pop_back();
    }
}


// *****************************************************************************
// latch()
//
/// \brief Creates a simple latch.  The latch is transparent when enable is true.
///
/// The latch enable is appended as an asynchronous signal with the label "enable".
///
/// \param enable
/// \param in
/// \param name an optional name for the element
/// \return reference to new sequential node
//
// *****************************************************************************
MultiRef
Synthesis::latch(MultiRef enable, MultiRef in, const string name) {
    MultiRef seqRef = seq(in, name);
    assert(seqRef.type == AIREF);
    annotateAsynchronousSignal(seqRef.ai, string("enable"), enable);  
    return seqRef;
}


// *****************************************************************************
// seq()
//
/// \brief Creates a generic sequential element.
///
/// \param in
/// \param name an optional name for the element
/// \return reference to new sequential node
//
// *****************************************************************************
MultiRef
Synthesis::seq(MultiRef in, const string name) {
    assert(nameSpace);
    assert(currentModule);
    static_cast<void>(name);

    // have we created a functional manager for this design yet?
    if (currentManager == NULL) {
        currentManager = Manager::get(currentModule->getDesign());
    }

    // if this MultiRef is a net, convert it to an oagAi::Ref
    if (in.type == NET) {
        // there must already be an existing oagAi::Ref
        oa::oaModBitNet *net = in.net;
        assert(net);
        assert(net->getAppDefs().includes(AiRefAppDef));
        in.ai = AiRefAppDef->get(net);
        in.type = AIREF;
    }

    // create a sequential node
    oagAi::Ref seqRef(currentManager->ai.newSequential(in.ai));

    DEBUG_PRINTLN("\t\t\t created state bit")

    return MultiRef(seqRef);
}


// *****************************************************************************
// annotateAsynchronousSignal()
//
/// \brief Annotates an asychronous signal to a sequential AI node.
///
/// \param sequential a reference to a sequential node
/// \param label the label for the asychronous input
/// \param signal the signal that drives this input
//
// *****************************************************************************
void
Synthesis::annotateAsynchronousSignal(oagAi::Ref sequential, const string label, MultiRef signal) {
    assert(currentManager);

    // if the MultiRef is a net, convert it to an oagAi::Ref
    if (signal.type == NET) {
        // there must already be an existing oagAi::Ref
        oa::oaModBitNet *net = signal.net;
        assert(net);
        assert(net->getAppDefs().includes(AiRefAppDef));
        signal.ai = AiRefAppDef->get(net);
        signal.type = AIREF;
    }
    
    // annotate the asynchronous input
    DEBUG_PRINTLN("\tannotating node " << sequential 
                  << " with asychronous signal " << label 
                  << " = " << signal.ai);
    oagAi::Node::SequentialData *seqData = currentManager->ai.getSequentialData(sequential);
    assert(seqData);
    assert(seqData->abstractionLevel == oagAi::Node::SequentialData::BLACK_BOX);
    seqData->signals[label] = signal.ai;
}

    
// *****************************************************************************
// mux()
//
/// \brief Multiplexes two bits.
//
// *****************************************************************************
MultiRef
Synthesis::mux(MultiRef select, MultiRef in0, MultiRef in1) {
    if (in0 == in1) {
        return in0;
    }
    return orOf(andOf(in1, select), andOf(in0, notOf(select)));
}


// *****************************************************************************
// mux()
//
/// \brief Multiplexes two buses.
//
// *****************************************************************************
void
Synthesis::mux(MultiRefBus &result, MultiRef select, MultiRefBus &in0, MultiRefBus &in1) {
    assert(in0.size() == in1.size());
    result.clear();
    
    MultiRefBus::iterator in0Iter = in0.begin(), in1Iter = in1.begin();
    while(in0Iter != in0.end() && in1Iter != in1.end()) {
        result.push_back(mux(select, *in0Iter, *in1Iter));
        ++in0Iter; ++in1Iter;
    }
}


// *****************************************************************************
// fullAdder()
//
/// \brief Creates a full adder.  
///
/// The signal carryIn is given the fastest path.
//
// *****************************************************************************
void
Synthesis::fullAdder(MultiRef &sum, MultiRef &carryOut, 
                     MultiRef e1, MultiRef e2, MultiRef carryIn) {
    MultiRef intermediate = xorOf(e1, e2);
    sum = xorOf(carryIn, intermediate);
    carryOut = orOf(andOf(e1,e2), andOf(intermediate, carryIn));
}


// *****************************************************************************
// multiBitConstant()
//
/// \brief Converts an integer value into a constant bits of the specified width.
///
/// If the number of bits is zero, the function uses the smallest number of
/// bits required.
//
// *****************************************************************************
void
Synthesis::multiBitConstant(MultiRefBus &result, int value, int bits) {
    result.clear();
    for(int i=0; (bits == 0 ? value != 0 : i<bits); i++, value = value >> 1) {
        if (value % 2) {
            result.push_front(constantOne());
        } else {
            result.push_front(constantZero());
        }
    }
    if (result.empty()) {
        result.push_back(constantZero());
    }
}

}
