/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-11-08: ChangeLog started
*/

#include "oagFuncSimMod.h"
#include <iostream>

#include "oagFuncDebug.h"

using namespace std;

namespace oagFunc {

// *****************************************************************************
// SimMod()
//
/// \brief Constructor.
///
/// Simulating modules requires that the index SIM_USER_DATA_INDEX is allocated
/// and available to be modified on oagAi::Nodes.
///
/// \param design
//
// *****************************************************************************
SimMod::SimMod(oa::oaDesign *design) { 
    assert(design);
    assert(design->getTopModule());
    this->design = design;
    clear();
    // determine BITS_PER_RAND
    BITS_PER_RAND = 0;
    for(unsigned int i=1; i<sizeof(int)*8; i++) {
      if (RAND_MAX >= (1<<i)) {
        BITS_PER_RAND++;
      }
    }
}


// *****************************************************************************
// clear()
//
/// \brief Clears all previously set and computed vectors.
//
// *****************************************************************************   
void
SimMod::clear() {
    toBeSimulated.clear();
    /*
    // refresh constant node
    ModRef constant;
    constant = ModGraph::constantZero(design->getTopModule());
    ModGraph::setUserData(constant, SIM_USER_DATA_INDEX, 0);
    toBeSimulated.push_back(constant);
    */
}


// *****************************************************************************
// generateRandomStateVectors()
//
/// \brief Generates a set of random vectors at all state bits.
//
// *****************************************************************************
void
SimMod::generateRandomStateVectors() {
    list<ModRef> states;
    ModGraph::getLocalStates(design->getTopModule(), states);
    for(list<ModRef>::iterator it=states.begin(); it!=states.end(); ++it) {
        setVector(*it, randomVector());
    }
}


// *****************************************************************************
// generateRandomInputVectors()
//
/// \brief Generates a set of random vectors at all inputs.
//
// *****************************************************************************
void
SimMod::generateRandomInputVectors() {
    list<ModRef> inputs;
    ModGraph::getInputs(design->getTopModule(), inputs);
    for(list<ModRef>::iterator it=inputs.begin(); it!=inputs.end(); ++it) {
        setVector(*it, randomVector());
    }
}


// *****************************************************************************
// randomVector()
//
/// \brief Returns a random simulation vector.
//
/// \return a random vector
//
// *****************************************************************************
SimMod::SimVec
SimMod::randomVector() {
    const int RAND_MASK = (1 << BITS_PER_RAND) - 1;

    SimVec result = 0;
    int bits = sizeof(int)*8;
    while(bits>0) {
      result = (result << BITS_PER_RAND) | (rand() & RAND_MASK);
      bits -= BITS_PER_RAND;
    }
    return result;
}

    
// *****************************************************************************
// runOne()
//
/// \brief Compute the simulation vector at one node and store the result.
///
/// The reference should be non-inverted.
///
/// \param ref
//
// *****************************************************************************
void
SimMod::runOne(const ModRef & ref) {
    assert(!ModGraph::isInverted(ref));
    assert(ref.module);
    
#ifdef DEBUG
    oa::oaString nameString;
    ref.module->getName(nameString);
    DEBUG_PRINT("- ai " << nameString << "." << ref.ref << "\t");
#endif

    switch(ModGraph::getNodeType(ref)) {
    case oagAi::Node::AND: {
      DEBUG_PRINTMORE("and");
      
      ModRef left = ModGraph::getAndLeft(ref);
      ModRef nonInvLeft = ModGraph::getNonInverted(left);
      ModRef right = ModGraph::getAndRight(ref);
      ModRef nonInvRight = ModGraph::getNonInverted(right);
      
      SimVec leftVec = 0;
      if(ModGraph::getNodeType(nonInvLeft) == oagAi::Node::CONSTANT0) {
        leftVec = 0;
      } else {
        leftVec = ModGraph::getUserData(nonInvLeft, SIM_USER_DATA_INDEX);
      }
      leftVec = ModGraph::isInverted(left) ? ~leftVec : leftVec;
            
      SimVec rightVec = 0;
      if(ModGraph::getNodeType(nonInvRight) == oagAi::Node::CONSTANT0) {
        rightVec = 0;
      } else {
        rightVec = ModGraph::getUserData(nonInvRight, SIM_USER_DATA_INDEX);
      }
      rightVec = ModGraph::isInverted(right) ? ~rightVec : rightVec;
            
      ModGraph::setUserData(ref, SIM_USER_DATA_INDEX, leftVec & rightVec);
            
      break;
    }

    case oagAi::Node::SEQUENTIAL: {
      DEBUG_PRINTMORE("seq")
      break;
    }
            
    case oagAi::Node::TERMINAL: {
      DEBUG_PRINTMORE("terminal")

      ModRef driver = ModGraph::getTerminalDriver(ref);
      ModRef nonInvDriver = ModGraph::getNonInverted(driver);
      SimVec driverVec = 0;
      if(ModGraph::getNodeType(nonInvDriver) == oagAi::Node::CONSTANT0) {
        driverVec = 0;
      } else {
        driverVec = ModGraph::getUserData(nonInvDriver, SIM_USER_DATA_INDEX);
      }
      driverVec = ModGraph::isInverted(driver) ? ~driverVec : driverVec;

      ModGraph::setUserData(ref, SIM_USER_DATA_INDEX, driverVec);

      break;
    }
            
    default:
      QUIT_ON_INTERNAL_ERROR;
      break;
    }
    
    DEBUG_PRINTMORE(" vector=" << ModGraph::getUserData(ref, SIM_USER_DATA_INDEX));
}


// *****************************************************************************
// runFull()
//
/// \brief Propagates simulation vectors through all combinational logic.
//
// *****************************************************************************
void
SimMod::runFull() {

    toBeSimulated.clear();

    // seed all the start points
    ModGraph::getInputs(design->getTopModule(), toBeSimulated);
    ModGraph::getLocalStates(design->getTopModule(), toBeSimulated);
    
    runIncremental();
}


// *****************************************************************************
// nextCycle()
//
/// \brief Advances all of the state bits to the next state value.
//
// *****************************************************************************
void    
SimMod::nextCycle() {
    list<ModRef> stateList;
    ModGraph::getLocalStates(design->getTopModule(), stateList);
    for(list<ModRef>::iterator it = stateList.begin();
        it != stateList.end(); ++it) {
        assert(!ModGraph::isInverted(*it));
        ModRef nextState = ModGraph::getNextState(*it);
        ModRef nonInvNextState = ModGraph::getNonInverted(nextState);
        if (ModGraph::isNull(nextState)) {
            // no next state input.  do nothing
        } else if (ModGraph::isInverted(nextState)) {
          ModGraph::setUserData(*it, SIM_USER_DATA_INDEX,
                                ~ModGraph::getUserData(nonInvNextState, 
                                                       SIM_USER_DATA_INDEX));
        } else {
          ModGraph::setUserData(*it, SIM_USER_DATA_INDEX,
                                ModGraph::getUserData(nonInvNextState, 
                                                      SIM_USER_DATA_INDEX));
        }
        toBeSimulated.push_back(*it);
    }    
}


// *****************************************************************************
// runIncremental()
//
/// \brief Propagates the simulation vectors from any node that has been updated.
///
/// Nodes are marked updated after a call to setVector() or after nextCycle().
//
// *****************************************************************************
void
SimMod::runIncremental() {
  if (toBeSimulated.empty()) return;

  vector<ModRef> topoOrder;
  ModGraph::getTransitiveFanout(toBeSimulated, topoOrder, true);

  // simulate in forward topological order
  for(vector<ModRef>::iterator it = topoOrder.begin();
      it != topoOrder.end(); it++) {
    // compute vector at node
    runOne(*it);
  }

  toBeSimulated.clear();
}

}
