/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-05-31: ChangeLog started
*/

#include <iostream>
#include "oagFuncQueryMod.h"
#include "oagFuncModGraph.h"
#include <set>

#include "oagFuncDebug.h"

namespace oagFunc {

// *****************************************************************************
// QueryMod()
//
/// \brief Constructor.
///
/// \param design the OpenAccess design of interest
/// \param engine a ReasoningEngine to manipulate functional information
//
// *****************************************************************************
QueryMod::QueryMod(oa::oaDesign *design, GenericReasoningEngine *engine) { 
    assert(design);
    assert(engine);
    this->design = design; this->engine = engine;
}

 
// *****************************************************************************
// getNextState()
//
/// \brief Returns the next state function for a state bit
///
/// The net provided must be a state bit net, such as returned by getStateBits,
/// otherwise an error will be generated.
///
/// \param net a state bit
/// \return the next state function
//
// *****************************************************************************
GenericFunction
QueryMod::getNextState(oa::oaModBitNet *net) {
    assert(net);
    
    // construct the ModRef to the next state function
    ModRef ref = ModGraph::getNetToAiConnection(net);
    if (ModGraph::isNull(ref)) {
        cerr << "ERROR: Net is not connected to graph" << endl;
        QUIT_ON_ERROR;
    }
    assert(ModGraph::isTerminal(ref));
    ref = ModGraph::getTerminalDriver(ref);
    if (!ModGraph::isSequential(ref)) {
        cerr << "ERROR: Net is not connected to a sequential node" << endl;
        QUIT_ON_ERROR;
    }
    ref = ModGraph::getNextState(ref);
    return get(ref);
}


// *****************************************************************************
// get()
//
/// \brief Returns the function at a bit net
///
/// The function at the desired net is derived by recursing backward through 
/// the circuit structure until points are reached that have had their
/// function explicitly defined by the user.  If a terminal point is reached
/// (such as a primary input, state bit, or floating net) that does not have 
/// a function explicitly assigned by the user, an error will be generated.
/// If the backward propogation is successful, these known functions are
/// then propogated forward to derive the function at the desired net.
///
/// \param net
/// \return the function of net
//
// *****************************************************************************
GenericFunction
QueryMod::get(oa::oaModBitNet *net) {
    assert(net);
    
    // find the ModRef that this net is associated with
    ModRef ref = ModGraph::getNetToAiConnection(net);
    if (ModGraph::isNull(ref)) {
        cerr << "ERROR: Net is not connected to graph" << endl;
        QUIT_ON_ERROR;
    }
    assert(ModGraph::isTerminal(ref));
    
    return get(ref);
}


// *****************************************************************************
// set()
//
/// \brief Explcitly sets the logic function associated with a net.
///
/// \param net the oaModBitNet to set
/// \param func its function
//
// *****************************************************************************
void
QueryMod::set(oa::oaModBitNet *net, GenericFunction func) {
    assert(net);
    assert(func);

    // find the ModRef that this net is associated with
    ModRef ref = ModGraph::getNetToAiConnection(net);
    if (ModGraph::isNull(ref)) {
        cerr << "ERROR: Net is not connected to graph" << endl;
        QUIT_ON_ERROR;
    }   
    
    setFunctionsOnRefs[ref] = func;
    precomputedCache.clear();
}


// *****************************************************************************
// unset()
//
/// \brief Removes any logic function explicitly associated with a net.
///
/// \param net
//
// *****************************************************************************
void
QueryMod::unset(oa::oaModBitNet *net) {
    assert(net);

    // find the ModRef that this net is associated with
    ModRef ref = ModGraph::getNetToAiConnection(net);
    if (ModGraph::isNull(ref)) {
        cerr << "ERROR: Net is not connected to graph" << endl;
        QUIT_ON_ERROR;
    }   
    
    setFunctionsOnRefs.erase(ref);    
    precomputedCache.clear();
}


// *****************************************************************************
// getNextState()
//
/// \brief Returns the next state function for a state bit
///
/// The ModRef provided must be a state bit, otherwise an error will be 
/// generated.
///
/// \param ref a state bit
/// \return the next state function
//
// *****************************************************************************
GenericFunction
QueryMod::getNextState(ModRef &ref) {

    if (!ModGraph::isSequential(ref)) {
        cerr << "ERROR: Reference is not to a sequential node" << endl;
        QUIT_ON_ERROR;
    }

    // initialize loop detection
    visitedSet.clear();

    ModRef nextState = ModGraph::getNextState(ref);
    return getRecursively(nextState);
}


// *****************************************************************************
// get()
//
/// \brief Returns the function at an ModRef
///
/// The function at the desired ModRef is derived by recursing backward through 
/// the circuit structure until points are reached that have had their
/// function explicitly defined by the user.  If a terminal point is reached
/// (such as a primary input, state bit, or floating net) that does not have 
/// a function explicitly assigned by the user, an error will be generated.
/// If the backward propogation is successful, these known functions are
/// then propogated forward to derive the function at the desired ModRef.
///
/// \param ref
/// \return the function
//
// *****************************************************************************
GenericFunction
QueryMod::get(ModRef &ref) {
    
    // initialize loop detection
    visitedSet.clear(); 
    
    return getRecursively(ref);
}


// *****************************************************************************
// set()
//
/// \brief Explcitly sets the logic function associated with an ModRef
///
/// \param ref the ModRef to set
/// \param func its function
//
// *****************************************************************************
void
QueryMod::set(ModRef &ref, GenericFunction func) {
    assert(func);

    // also set all the connected refs as well to create a proper cut
    std::set<oagFunc::ModRef> connectedRefs;
    std::set<oa::oaModBitNet*> connectedNets;
    ModGraph::getAllConnections(ref, connectedNets, connectedRefs,
                                false, true, true, false);
    for(std::set<ModRef>::iterator connectedIter = connectedRefs.begin();
        connectedIter != connectedRefs.end(); ++connectedIter) {
        setFunctionsOnRefs[*connectedIter] = func;
    }

    precomputedCache.clear();    
}


// *****************************************************************************
// unset()
//
/// \brief Removes any logic function explicitly associated with an ModRef
///
/// \param ref
//
// *****************************************************************************
void
QueryMod::unset(ModRef &ref) {
    setFunctionsOnRefs.erase(ref);
    precomputedCache.clear();
}


// *****************************************************************************
// getRecursively()
//
/// \brief Returns the function at ModRef, a reference to an and/inverter graph in the module domain
///
/// The function at the desired oagAi::Graph point is derived by recursing backward through 
/// the oagAi::Graph and circuit structure until points are reached that have had their
/// function explicitly defined by the user.  If a terminal point is reached 
/// (such as a primary input, state bit, or floating net) that does not have 
/// a function explicitly assigned by the user, an error will be generated.
/// If the backward propogation is successful, these known functions are
/// then propogated forward to derive the function of the supplied ModRef.
///
/// \param ref an ModRef
/// \return the function
//
// *****************************************************************************
GenericFunction
QueryMod::getRecursively(ModRef &ref) {
    assert(ref.module);

    // is this function set by the user?
    if (setFunctionsOnRefs.find(ref) != setFunctionsOnRefs.end()) {
        DEBUG_PRINTLN("- known func");
        assert(setFunctionsOnRefs[ref]);
        return setFunctionsOnRefs[ref];
    }
    
    // is this result currently cached?
    if (precomputedCache.find(ref) != precomputedCache.end()) {
        DEBUG_PRINTLN("- cached func");
        assert(precomputedCache[ref]);
        return precomputedCache[ref];
    }

    oa::oaString str;
    ref.module->getName(oa::oaVerilogNS(), str);
    DEBUG_PRINT("- ai " << str << "." << ref.ref << "\t")

    GenericFunction result = NULL;

    if (ModGraph::isInverted(ref)) {
        DEBUG_PRINTMORE("not " << endl)
        ModRef not_ref = ModGraph::notOf(ref);
        result = engine->genericNotOf(getRecursively(not_ref));
        precomputedCache[ref] = result;
        return result;
    }

    switch(ModGraph::getNodeType(ref)) {
      case oagAi::Node::AND: {
        DEBUG_PRINTMORE("and" << endl)
        
        // check for cycle
        if (visitedSet.find(ref) != visitedSet.end()) {
            cerr << "ERROR: Combinational cycle detected" << endl;
            QUIT_ON_ERROR;
        } else {
            visitedSet.insert(ref);
        }
        
        ModRef left(ModGraph::getAndLeft(ref)), 
               right(ModGraph::getAndRight(ref));
        result = engine->genericAndOf(getRecursively(left), getRecursively(right));
        break;
        }
      case oagAi::Node::CONSTANT0:
        DEBUG_PRINTMORE("zero" << endl)
        result = engine->genericGetZero();
        break;  
      case oagAi::Node::SEQUENTIAL:
        DEBUG_PRINTMORE("seq" << endl)
        cerr << "ERROR: Unassigned state bit" << endl;
        QUIT_ON_ERROR;
        break;
      case oagAi::Node::TERMINAL: {
        DEBUG_PRINTMORE("terminal")
      
        std::set<oagFunc::ModRef> connectedRefs;
        std::set<oa::oaModBitNet*> connectedNets;
        ModGraph::getAllConnections(ref, connectedNets, connectedRefs,
            false, true, true, false);
        
        DEBUG_PRINTMORE(" (nets " << connectedNets.size() << " nodes " << connectedRefs.size() << ") ")
        
        // for every connected net...
        const oa::oaVerilogNS verilogNS;
        for(std::set<oa::oaModBitNet*>::iterator connectedIter = connectedNets.begin();
                connectedIter != connectedNets.end(); ++connectedIter) {
            oa::oaString connectedNetString;
            (*connectedIter)->getName(verilogNS, connectedNetString);
            
            // is it a constant 0 or 1?
            if (connectedNetString == oa::oaString("tie0")) {
                DEBUG_PRINTMORE("tied to 0" << endl)
                return engine->genericGetZero();
            } else if (connectedNetString == oa::oaString("tie1")) {
                DEBUG_PRINTMORE("tied to 1" << endl)            
                return engine->genericNotOf(engine->genericGetZero());
            }          
        }

        // for every connected ref...        
        for(std::set<ModRef>::iterator connectedIter = connectedRefs.begin();
                connectedIter != connectedRefs.end(); ++connectedIter) {
            ModRef connectedRef = *connectedIter;
            
            // is this function set by the user?
            if (setFunctionsOnRefs.find(connectedRef) != setFunctionsOnRefs.end()) {
                DEBUG_PRINTMORE(" connected to known func" << endl);
                assert(setFunctionsOnRefs[connectedRef]);
                return setFunctionsOnRefs[connectedRef];
            }
    
            // is this result currently cached?
            if (precomputedCache.find(connectedRef) != precomputedCache.end()) {
                DEBUG_PRINTMORE(" connected to cached func" << endl);
                assert(precomputedCache[connectedRef]);
                precomputedCache[ref] = precomputedCache[connectedRef];
                return precomputedCache[connectedRef];
            }
            
            // continue evaluation...
            if (ModGraph::getNodeType(connectedRef) != oagAi::Node::TERMINAL ||
                 ModGraph::isInverted(connectedRef)) {
                DEBUG_PRINTMORE(" got driver" << endl)
                result = getRecursively(connectedRef);
                break;
            }
        }

        // if we haven't discovered a driver or set function, this node is dangling
        if (!result) {
            cerr << "ERROR: Floating graph node" << endl;
            QUIT_ON_ERROR;
        }

        break;
      }
      default:
        cerr << "ERROR: Broken AI node: " << ref.ref << endl;
        QUIT_ON_INTERNAL_ERROR;
    }
    
    precomputedCache[ref] = result;
    
    assert(result);
    return result;
}

}
