/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-08-24: ChangeLog started
*/

#if !defined(oagFuncOccGraph_P)
#define oagFuncOccGraph_P

#include <stdlib.h>

#include "oaDesignDB.h"
#include "oagFuncOccRef.h"
#include "oagFuncManager.h"
#include "oagAiGraph.h"
#include <list>
#include <set>

using namespace std;

namespace oagFunc {

// *****************************************************************************
// OccGraph
//
/// \brief Object for manipulating graphs in the occurrence domain view.
///
/// The OccGraph class is not a data structure on its own, but instead provides
/// an interface for manipulating functionality graphs in the unfolded occurrence 
/// domain view.  All methods in the OccGraph class are static.
/// The OccRef structure provides the mechanism for referring to graph nodes
/// within this view, and most OccGraph methods operate on OccRefs.
///
/// The occurrence view is appropriate for working with the global
/// functionality of hierarchical designs.  Two different
/// instantiations (occurrences) may implement the same local behavior, but
/// the global functions being implemented on each of their pins depends on their
/// connectivity and context in the larger design.  To manipulate global
/// functionality, it does not suffice to traverse the folded (module) design
/// view; the unfolded (occurrence) hierarchy is better suited.
///
/// The occurrence view is not appropriate for making modifications to the local 
/// functional descriptions of design objects.  There aren't any methods in the 
/// OccGraph class to modify graphs from within the occurrence view.  Any changes
/// must be made to the module (using the ModGraph interface) and thereby all
/// occurrences of the module.  If a change is to be made to only this particular 
/// instance (e.g. an observability-don't-care simplification),
/// it should be uniquified (via oagFunc::uniqify) into its own module, and the
/// changes made on that unique module.
///
// *****************************************************************************

class OccGraph {

  private:
        
                                   OccGraph();   // this is a purely static
                                                 // class and can not be
                                                 // instantiated
  public:

    typedef set<OccRef> Cut;
    typedef list<Cut>   CutSet;

    /// \name Functional operations
    /// @{

    static OccRef           getNull(oa::oaOccurrence *occurrence) {
        /// \brief Returns a null reference inside the given occurrence.
        ///
        /// There is a no global OccGraph null reference; one unique null
        /// reference exists for each occurrence.  This function returns the
        /// unique null reference for the given occurrence.
        /// \param occurrence
        /// \return a null reference inside the given module
        return OccRef(oagAi::Graph::getNull(), occurrence);
    }
    static inline bool      isNull(OccRef x) {
        /// \brief Tests a reference to see if it is null.
        /// \param x an Ref
        /// \return true, if the reference is to a null function
        return x.occurrence == NULL || oagAi::Graph::isNull(x.ref);
    }
    static inline OccRef    constantOne(oa::oaOccurrence *occurrence) {
        /// \brief Returns a reference in the given occurrence to the function that is always true.
        /// \param occurrence
        /// \return a reference to a function that is always true
        return OccRef(getGraph(occurrence)->constantOne(), occurrence);
    }
    static inline OccRef    constantZero(oa::oaOccurrence *occurrence) {
        /// \brief Returns a reference in the given occurrence to the function that is always false.
        /// \param occurrence
        /// \return a reference to a function that is always false        
        return OccRef(getGraph(occurrence)->constantZero(), occurrence);
    }
    static inline OccRef notOf(OccRef x) {
        /// \brief Returns the complement of a function.  
        ///
        /// The structure of the graph will not
        /// be modified and no new nodes will be created.
        /// \param x the function to complement
        /// \return an Ref for the resulting function
        return OccRef(oagAi::Graph::notOf(x.ref), x.occurrence);
    }
    static inline bool       isInverted(OccRef x) {
        /// \brief Tests whether a reference that points to a node is inverted.
        /// \param x
        /// \return true if the reference is inverted
        return oagAi::Graph::isInverted(x.ref);
    }
    static inline OccRef     getNonInverted(OccRef x) {
        /// \brief Returns a reference that is not inverted.
        /// \param x
        /// \return non-inverted reference
        return OccRef(oagAi::Graph::getNonInverted(x.ref), x.occurrence);
    }
    
    /// @}
    /// \name Intra-module incremental graph traversal
    /// @{

    static inline oagAi::Node::Type     getNodeType(OccRef x) {
        /// \brief Returns the type of the referenced graph node.
        /// \param x an Ref
        /// \return the type of the referenced node
        return getGraph(x)->getNodeType(x.ref);
    }

    //    on terminal nodes...
    static OccRef     getTerminalDriver(OccRef terminal, 
                                        bool searchThroughHierarchy = false);
    static inline bool       isTerminal(OccRef terminal) {
        /// \brief Tests if a reference is to a TERMINAL node.
        /// \param terminal
        /// \return true, if the reference is to a TERMINAL node    
        return getGraph(terminal)->isTerminal(terminal.ref);
    }
    //    on sequential nodes...
    static inline OccRef     getNextState(OccRef sequential) {
        /// \brief Returns the next state input of a SEQUENTIAL node.
        ///
        /// If the reference is not to a SEQUENTIAL node, an
        /// error will be generated.
        /// \param sequential
        /// \return the reference to the next state, or null if none exists    
        assert(isSequential(sequential));
        return OccRef(getGraph(sequential)->getNextState(sequential.ref), sequential.occurrence);
    }
    static inline bool       isSequential(OccRef sequential) {
        /// \brief Tests if a reference is to a SEQUENTIAL node.
        /// \param sequential
        /// \return true, if the reference is to a SEQUENTIAL node    
        return getGraph(sequential)->isSequential(sequential.ref);
    }
    static oagAi::Node::SequentialData *getSequentialData(OccRef sequential) {
        /// \brief Returns the sequential description of a SEQUENTIAL node.
        ///
        /// If the reference is not to a SEQUENTIAL node, an
        /// error will be generated.
        /// \param sequential
        /// \return sequential description
        return getGraph(sequential)->getSequentialData(sequential.ref);    
    }
    //    on and nodes...
    static inline OccRef     getAndLeft(OccRef x) {
        /// \brief Returns the left operand of an AND node.
        /// \param x
        /// \return reference to left operand
        return OccRef(getGraph(x)->getAndLeft(x.ref), x.occurrence);
    }
    static inline OccRef     getAndRight(OccRef x) {
        /// \brief Returns the right operand of an AND node.
        /// \param x
        /// \return reference to right operand
        return OccRef(getGraph(x)->getAndRight(x.ref), x.occurrence);
    }
    
    static void              getFanout(OccRef x, list<OccRef> &result);
    static inline bool       hasFanout(OccRef x) {
        /// \brief Returns the fanout of a node.
        ///
        /// An error is generated if fanout is not being maintained.
        /// \param x
        /// \return true, if the node has any fanout, false otherwise
        return getGraph(x)->hasFanout(x.ref);
    }

    /// @}
    /// \name Explicitly managing graph-to-net connections
    /// @{

    // Net-to-ai connections
    static oa::oaOccBitNet *   getNetToAiConnection(OccRef ref);
    static OccRef              getNetToAiConnection(oa::oaOccBitNet *net);
    
    static void                getNetToAiAllEquivConnections(OccRef ref, list<oa::oaOccBitNet *> &result);
    
    /// @}
    /// \name Global graph queries
    /// @{

    static void         getOutputs(oa::oaOccurrence *top, list<OccRef> &result);
    static void         getOutputs(oa::oaOccurrence *top, list<oa::oaOccBitNet*> &result);
    static void         getInputs(oa::oaOccurrence *top, list<OccRef> &result);
    static void         getInputs(oa::oaOccurrence *top, list<oa::oaOccBitNet*> &result);
    static void         getStates(oa::oaOccurrence *top, list<OccRef> &result);
    static void         getLocalStates(oa::oaOccurrence *top, list<OccRef> &result);
    static void         getConstants(oa::oaOccurrence *top, list<OccRef> &zeros, list<OccRef> &ones);

    static void                getAllConnections(oa::oaOccBitNet *net,
                                    set<oa::oaOccBitNet*> &connectedNets,
                                    set<OccRef> &connectedRefs,
                                    bool searchForwardThroughGraph = true,
                                    bool searchBackwardThroughGraph = true,
                                    bool searchThroughHierarchy = true,
                                    bool searchThroughEquivNets = true,
                                    bool searchThroughEquivRefs = false,
                                    bool includeSelf = true);
    static void                getAllConnections(OccRef x, 
                                    set<oa::oaOccBitNet*> &connectedNets,
                                    set<OccRef> &connectedRefs,
                                    bool searchForwardThroughGraph = true,
                                    bool searchBackwardThroughGraph = true,
                                    bool searchThroughHierarchy = true,                                    
                                    bool searchThroughEquivNets = true,
                                    bool searchThroughEquivRefs = false,
                                    bool includeSelf = true);
                                    
    /// @}
    /// \name Advanced graph traversal
    /// @{

    static void         enumerateKfeasibleCuts(OccRef x, unsigned int maxCutSize, CutSet &result);
    static void         clearKfeasibleCuts(oa::oaOccurrence *occurrence);

    static void         getTransitiveFanin(OccRef x, list<OccRef> &transitiveFanin);
    static void         getTransitiveFanout(OccRef x, list<OccRef> &transitiveFanout);
    static void         getFaninRoots(OccRef x, list<OccRef> &faninRoots);
    static void         getFanoutRoots(OccRef x, list<OccRef> &fanoutRoots);

    /// @}
    /// \name Equivalence marking
    /// @{

    static bool         testEquivalence(OccRef x, OccRef y);
    static void         getEquivalents(OccRef x, list<OccRef> &result);

    static void         getFanoutOfEquivalentNodes(OccRef x, list<OccRef> &result);

    /// @}

  private:
                       
    static inline oagAi::Graph *   getGraph(oa::oaOccurrence *occurrence) {
        /// \brief Returns the graph of a occurrence's module's design.
        ///
        /// This module's functional description will be contained
        /// within this graph, along with the functional descriptions
        /// of other modules in the same design.
        /// \param module
        /// \return the graph of a module's design    
        return &(Manager::get(occurrence->getModule()->getDesign())->ai);
    }
    static inline oagAi::Graph *   getGraph(const OccRef &x) {
        /// \brief Returns the graph that a reference is in.
        /// \param x    
        return &(Manager::get(x.occurrence->getModule()->getDesign())->ai);
    }
    
    static void             convertRefListToOccRefList(const list<oagAi::Ref> &source,
                                                       oa::oaOccurrence *occurrence,
                                                       list<OccRef> &result);
};

}

#endif
