/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-08-25: ChangeLog started
*/

#include "oagFunc.h"
#include "oagFuncModGraph.h"
#include "oagFuncOccGraph.h"
#include <memory>

#include "oagFuncDebug.h"

namespace oagFunc {

// Used in OccGraph::getNetToAiConnection
typedef oa::oaIter<oa::oaOccNet> OccNetIter;


// *****************************************************************************
// convertRefListToOccRefList()
//
/// \brief Converts a list of Refs in a given occurrence into a list of OccRefs.
///
/// \param source the list of original Refs
/// \param occurrence the occurrence
/// \param result the list of resulting OccRefs
//
// *****************************************************************************
void
OccGraph::convertRefListToOccRefList(const list<oagAi::Ref> &source,
                                     oa::oaOccurrence *occurrence,
                                     list<OccRef> &result) {
    for(list<oagAi::Ref>::const_iterator it = source.begin(); it != source.end(); ++it) {
        result.push_back(OccRef(*it, occurrence));
    }
}

/*
// *****************************************************************************
// enumerateKfeasibleCuts()
//
/// \brief UNIMPLEMENTED.  Returns the list of cuts in the transitive fan-in of a given node up to a given size.
///
/// Information about cut sets is cached between subsequent calls.  This cache
/// can be reset by clearKfeasibleCuts().
///
/// \param x
/// \param maxCutSize
/// \param a set of cuts
//
// *****************************************************************************
void
OccGraph::enumerateKfeasibleCuts(OccRef x, unsigned int maxCutSize, CutSet &result) {
    // UNIMPLEMENTED
    QUIT_ON_INTERNAL_ERROR;
}


// *****************************************************************************
// clearKfeasibleCuts()
//
/// \brief UNIMPLEMENTED.  Clears the cached sets of cuts in a given occurrence.
///
/// \param occurrence
//
// *****************************************************************************
void
OccGraph::clearKfeasibleCuts(oa::oaOccurrence *occurrence) {
    // UNIMPLEMENTED
    QUIT_ON_INTERNAL_ERROR;
}
*/

// *****************************************************************************
// getFanout()
//
/// \brief Returns the fan-out of a node that lies within the same occurrence.
///
/// An error is generated if fanout is not being maintained.
///
/// \param x
/// \param result the fanout list
//
// *****************************************************************************
void
OccGraph::getFanout(OccRef x, list<OccRef> &result) {
    convertRefListToOccRefList(getGraph(x)->getFanout(x.ref), x.occurrence, result);
}
    
    
// *****************************************************************************
// getTransitiveFanin()
//
/// \brief Returns the transitive fan-in of a node that lies within the same occurrence.
///
/// The transitive traversal is confined to the local occurrence.  It does not
/// follow OpenAccess structures outside of the graph.
///
/// See oagAi::Graph::getTransitiveFanin for further details.
///
/// \param x
/// \param transitiveFanin the resulting list of nodes in the transitive fan-in
//
// *****************************************************************************
void
OccGraph::getTransitiveFanin(OccRef x, list<OccRef> &transitiveFanin) {
    list<oagAi::Ref> flatResult;
    getGraph(x)->getTransitiveFanin(x.ref, flatResult);
    
    convertRefListToOccRefList(flatResult, x.occurrence, transitiveFanin);
}


// *****************************************************************************
// getTransitiveFanout()
//
/// \brief Returns the transitive fan-out of a node that lies within the same occurrence.
///
/// The transitive traversal is confined to the local occurrence.  It does not
/// follow OpenAccess structures outside of the graph.
///
/// See oagAi::Graph::getTransitiveFanout for further details.
///
/// \param x
/// \param transitiveFanout the resulting list of nodes in the transitive fan-out
//
// *****************************************************************************
void
OccGraph::getTransitiveFanout(OccRef x, list<OccRef> &transitiveFanout) {
    list<oagAi::Ref> flatResult;
    getGraph(x)->getTransitiveFanout(x.ref, flatResult);
    
    convertRefListToOccRefList(flatResult, x.occurrence, transitiveFanout);
}


// *****************************************************************************
// getFaninRoots()
//
/// \brief Returns the roots of the transitive fan-in of a node within the same occurrence.
///
/// The transitive traversal is confined to the local occurrence.  It does not
/// follow OpenAccess structures outside of the graph.
///
/// See oagAi::Graph::getFaninRoots for further details.
///
/// \param x
/// \param faninRoots the resulting list of roots of the transitive fan-in
//
// *****************************************************************************
void
OccGraph::getFaninRoots(OccRef x, list<OccRef> &faninRoots) {
    list<oagAi::Ref> flatResult;
    getGraph(x)->getFaninRoots(x.ref, flatResult);
    
    convertRefListToOccRefList(flatResult, x.occurrence, faninRoots);
}


// *****************************************************************************
// getFanoutRoots()
//
/// \brief Returns the roots of the transitive fan-out of a node within the same occurrence.
///
/// The transitive traversal is confined to the local occurrence.  It does not
/// follow OpenAccess structures outside of the graph.
///
/// See oagAi::Graph::getFanoutRoots for further details.
///
/// \param x
/// \param fanoutRoots the resulting list of roots of the transitive fan-out
//
// *****************************************************************************
void
OccGraph::getFanoutRoots(OccRef x, list<OccRef> &fanoutRoots) {
    list<oagAi::Ref> flatResult;
    getGraph(x)->getFanoutRoots(x.ref, flatResult);
    
    convertRefListToOccRefList(flatResult, x.occurrence, fanoutRoots);
}


// *****************************************************************************
// testEquivalence()
//
/// \brief Tests if two references are explicitly marked as equivalent.
///
/// \param x
/// \param y
//
// *****************************************************************************
bool
OccGraph::testEquivalence(OccRef x, OccRef y) {

    return x.occurrence == y.occurrence && getGraph(x)->testEquivalence(x.ref, y.ref);
}


// *****************************************************************************
// getEquivalents()
//
/// \brief Returns all references that have been marked equivalent.
///
/// The set of equivalent references will be appended to the provided list.
/// The list will include the reference provided.
///
/// \param x
/// \param result
//
// *****************************************************************************
void
OccGraph::getEquivalents(OccRef x, list<OccRef> &result) {
    list<oagAi::Ref> flatResult;
    getGraph(x)->getEquivalents(x.ref, flatResult);
    
    convertRefListToOccRefList(flatResult, x.occurrence, result);
}


// *****************************************************************************
// getFanoutOfEquivalentNodes()
//
/// \brief Returns the fan-out of a node and all of its equivalents.
///
/// \param x
/// \param result the fan-out of a node and all of its equivalents
//
// *****************************************************************************
void
OccGraph::getFanoutOfEquivalentNodes(OccRef x, list<OccRef> &result) {
    list<oagAi::Ref> flatResult;
    getGraph(x)->getFanoutOfEquivalentNodes(x.ref, flatResult);
    
    convertRefListToOccRefList(flatResult, x.occurrence, result);
}


// *****************************************************************************
// getNetToAiConnection()
//
/// \brief Returns the OpenAccess net connected to a given graph TERMINAL.
///
/// \param x
/// \return the occurrence net, or NULL if there is none
//
// *****************************************************************************
oa::oaOccBitNet *   
OccGraph::getNetToAiConnection(OccRef x) {
    Manager *manager = Manager::get(x.occurrence->getModule()->getDesign());
    oa::oaModBitNet *modNet = manager->getNetToAiConnection(x.ref);
    if (!modNet) {
        return NULL;
    }

    oa::oaOccurrence *topOcc = x.occurrence->getTopOccurrence();
    auto_ptr<OccNetIter> occNetIter;
    if (x.occurrence == topOcc) {
        // let the database find the net in the occurrence that corresponds to
        // the modNet
        occNetIter.reset(new OccNetIter(modNet->getOccNets(topOcc)));
    } else {
        // search all nets in the occurrence for the one that corresponds to
        // the modNet
        occNetIter.reset(
                new OccNetIter(x.occurrence->getNets(oacNetIterSingleBit)));
    }
    oa::oaOccNet *occNet;
    while((occNet = occNetIter->getNext())) {
        oa::oaOccBitNet *occBitNet = toBitNet(occNet);
        if (occBitNet->getModNet() == modNet) {
            return occBitNet;
        }
    }

    // one of the occNets must correspond to the modNet!
    QUIT_ON_INTERNAL_ERROR;
    return NULL;
}


// *****************************************************************************
// getNetToAiConnection()
//
/// \brief Returns the graph node connected to a given OpenAccess net.
///
/// \param net
/// \return a reference to the connected graph TERMINAL node, or a null OccRef if none exists
//
// *****************************************************************************
OccRef
OccGraph::getNetToAiConnection(oa::oaOccBitNet *net) {
    Manager *manager = Manager::get(net->getModNet()->getDesign());
    oa::oaModBitNet *modNet = static_cast<oa::oaModBitNet*>(net->getModNet());
    return OccRef(manager->getNetToAiConnection(modNet), net->getOccurrence());
}


// *****************************************************************************
// getOutputs()
//
/// \brief Gets all output terminals inside an occurrence.
///
/// An output terminal is a TERMINAL node that is connected to an output oaOccTerm.
///
/// If the bit net attached to the oaOccTerm does not have a connection to the
/// graph, one is created in the corresponding module.
///
/// \param top the occurrence
/// \param result the list of all output terminals inside the occurrence
//
// *****************************************************************************
void         
OccGraph::getOutputs(oa::oaOccurrence *top, list<OccRef> &result) {
    assert(top);
    oa::oaOccTerm *term;
    oa::oaIter<oa::oaOccTerm> termIter(top->getTerms(oacTermIterSingleBit));
    while((term = termIter.getNext())) {
        if(term->getTermType() == oa::oacOutputTermType) {
            ModRef modRef = oagFunc::ModGraph::prepareNetToAiConnection(toBitNet(term->getNet()->getModNet()));
            result.push_back(OccRef(modRef.ref, top));
        }
    }
}


// *****************************************************************************
// getOutputs()
//
/// \brief Gets all output nets inside an occurrence.
///
/// An output net is a net that is connected to an output oaTerm.
///
/// If the bit net attached to the oaOccTerm does not have a connection to the
/// graph, one is created in the corresponding module.
///
/// \param top the occurrence
/// \param result the list of all output nets inside the occurrence
//
// *****************************************************************************
void         
OccGraph::getOutputs(oa::oaOccurrence *top, list<oa::oaOccBitNet *> &result) {
    assert(top);
    oa::oaOccTerm *term;
    oa::oaIter<oa::oaOccTerm> termIter(top->getTerms(oacTermIterSingleBit));
    while((term = termIter.getNext())) {
        if(term->getTermType() == oa::oacOutputTermType) {
            result.push_back(toBitNet(term->getNet()));
        }
    }
}


// *****************************************************************************
// getInputs()
//
/// \brief Gets all input terminals inside an occurrence.
///
/// An input terminal is a TERMINAL node that is connected to an input oaTerm.
///
/// \param top the occurrence
/// \param result the list of all input terminals inside the occurrence
//
// *****************************************************************************
void         
OccGraph::getInputs(oa::oaOccurrence *top, list<OccRef> &result) {
    assert(top);
    oa::oaOccTerm *term;
    oa::oaIter<oa::oaOccTerm> termIter(top->getTerms(oacTermIterSingleBit));
    while((term = termIter.getNext())) {
        if(term->getTermType() == oa::oacInputTermType) {
            ModRef modRef = oagFunc::ModGraph::prepareNetToAiConnection(toBitNet(term->getNet()->getModNet()));
            result.push_back(OccRef(modRef.ref, top));
        }
    }
}


// *****************************************************************************
// getInputs()
//
/// \brief Gets all input nets inside an occurrence.
///
/// An input net is a net that is connected to an input oaTerm.
///
/// \param top the occurrence
/// \param result the list of all input nets inside the occurrence
//
// *****************************************************************************
void         
OccGraph::getInputs(oa::oaOccurrence *top, list<oa::oaOccBitNet *> &result) {
    assert(top);
    oa::oaOccTerm *term;
    oa::oaIter<oa::oaOccTerm> termIter(top->getTerms(oacTermIterSingleBit));
    while((term = termIter.getNext())) {
        if(term->getTermType() == oa::oacInputTermType) {
            result.push_back(toBitNet(term->getNet()));
        }
    }
}


// *****************************************************************************
// getConstants()
//
/// \brief Gets all constant 0 refs in the hierarchy.
///
/// \param occurrence
/// \param zeros constant zeros
/// \param ones constant ones
///
// *****************************************************************************
void         
OccGraph::getConstants(oa::oaOccurrence *occurrence, list<OccRef> &zeros, list<OccRef> &ones) {
    assert(occurrence);
    
    const oa::oaNativeNS nativeNS;
    const oa::oaScalarName tie0Name(nativeNS, "tie0");
    const oa::oaScalarName tie1Name(nativeNS, "tie1");

    list<oa::oaOccurrence*> toBeVisited;
    toBeVisited.push_back(occurrence);

    while(!toBeVisited.empty()) {
        oa::oaOccurrence *currentOcc = toBeVisited.front();
        toBeVisited.pop_front();
        
        // add the constant zero inside this object
        zeros.push_back(OccGraph::constantZero(currentOcc));
        oa::oaOccScalarNet *net;
        if ((net = oa::oaOccScalarNet::find(currentOcc, tie0Name))) {
            OccRef ref = getNetToAiConnection(net);
            assert(!isNull(ref));
            zeros.push_back(ref);
        }
        if ((net = oa::oaOccScalarNet::find(currentOcc, tie1Name))) {
            OccRef ref = getNetToAiConnection(net);
            assert(!isNull(ref));
            ones.push_back(ref);
        }
               
        // recursively descend into the instances inside this object
        oa::oaOccInst *inst;
        oa::oaIter<oa::oaOccInst> instIter(currentOcc->getInsts());
        while((inst = instIter.getNext())) {
            oa::oaOccurrence *subOcc = inst->getMasterOccurrence();
            if(!subOcc) {
                oa::oaString str;
                inst->getInst()->getCellName(oa::oaVerilogNS(), str);
                cerr << "ERROR: Could not find instantiated design \"" << str << "\" in OA database" << endl;
                QUIT_ON_ERROR;
            }
            toBeVisited.push_back(subOcc);
        }
    }
}


// *****************************************************************************
// getStates()
//
/// \brief Gets all state bits inside this occurrence and its subinstances.
///
/// \param occurrence
/// \param result the list of all state bits inside the occurrence
///
// *****************************************************************************
void         
OccGraph::getStates(oa::oaOccurrence *occurrence, list<OccRef> &result) {
    assert(occurrence);
    
    list<oa::oaOccurrence*> toBeVisited;
    toBeVisited.push_back(occurrence);

    while(!toBeVisited.empty()) {
        oa::oaOccurrence *currentOcc = toBeVisited.front();
        toBeVisited.pop_front();
        
        // add the state bits inside this object
        list<oagAi::Ref> currentResult;
        getGraph(currentOcc)->getAllSequential(currentResult);
        convertRefListToOccRefList(currentResult, currentOcc, result);
        
        // recursively descend into the instances inside this object
        oa::oaOccInst *inst;
        oa::oaIter<oa::oaOccInst> instIter(currentOcc->getInsts());
        while((inst = instIter.getNext())) {
            oa::oaOccurrence *subOcc = inst->getMasterOccurrence();
            if(!subOcc) {
                oa::oaString str;
                inst->getInst()->getCellName(oa::oaVerilogNS(), str);
                cerr << "ERROR: Could not find instantiated design \"" << str << "\" in OA database" << endl;
                QUIT_ON_ERROR;
            }
            toBeVisited.push_back(subOcc);
        }
    }
}    


// *****************************************************************************
// getLocalStates()
//
/// \brief Gets all state bits inside this occurrence.
///
/// \param occurrence
/// \param result the list of all state bits inside the occurrence
//
// *****************************************************************************
void         
OccGraph::getLocalStates(oa::oaOccurrence *occurrence, list<OccRef> &result) {
    assert(occurrence);

    // add the state bits inside this object
    list<oagAi::Ref> flatResult;
    getGraph(occurrence)->getAllSequential(flatResult);
    convertRefListToOccRefList(flatResult, occurrence, result);
}    


// *****************************************************************************
// getAllConnections()
//
/// \brief Returns the set of nets and refs that are logically connected.
///
/// \param net
/// \param connectedNets the resulting list of connected nets
/// \param connectedRefs the resulting list of connected refs
/// \param searchForwardThroughGraph follow TERMINAL fan-out to functionally identical refs
/// \param searchBackwardThroughGraph follow TERMINAL drivers to functionally identical refs
/// \param searchThroughHierarchy traverse OpenAccess object hierarchy
/// \param searchThroughEquivNets search all nets that are marked equivalent in the database
/// \param searchThroughEquivRefs follow all Refs that are marked equivalent in the graph
/// \param includeSelf include the starting node
//
// *****************************************************************************
void
OccGraph::getAllConnections(oa::oaOccBitNet *net, 
                            set<oa::oaOccBitNet*> &connectedNets,
                            set<OccRef> &connectedRefs,
                            bool searchForwardThroughGraph,
                            bool searchBackwardThroughGraph,
                            bool searchThroughHierarchy,
                            bool searchThroughEquivNets,
                            bool searchThroughEquivRefs,
                            bool includeSelf) {
    assert(net);
    
    // has this net been visited?
    if (connectedNets.find(net) == connectedNets.end()) {
        connectedNets.insert(net);
    } else {
        return;
    }

    // --- directly connected refs
    getAllConnections(getNetToAiConnection(net), connectedNets, connectedRefs,
            searchForwardThroughGraph, searchBackwardThroughGraph, searchThroughHierarchy, 
            searchThroughEquivNets, searchThroughEquivRefs, true);
    
    // --- equivalent nets (if searchThroughEquivNets)
    if (searchThroughEquivNets) {
        oa::oaOccBitNet *equivNet;
        oa::oaIter<oa::oaOccBitNet> equivIter(net->getEquivalentNets());
        while((equivNet = equivIter.getNext())) {
            getAllConnections(equivNet, connectedNets, connectedRefs,
                searchForwardThroughGraph, searchBackwardThroughGraph, searchThroughHierarchy, 
                searchThroughEquivNets, searchThroughEquivRefs, true);
        }
    }
    
    // --- across hierarchy (if searchThroughHierarchy)
    if (searchThroughHierarchy) {

        // down hierarchy
        oa::oaOccInstTerm   *nextInstTerm;
        oa::oaIter<oa::oaOccInstTerm> instTermIter(net->getInstTerms(oacInstTermIterAll));                                                                                 
        while((nextInstTerm = instTermIter.getNext())) {
            oa::oaOccTerm *nextTerm = nextInstTerm->getTerm();
            assert(nextTerm);
            assert(nextTerm->getNumBits() == 1);
            oa::oaOccBitNet *nextNet = toBitNet(nextTerm->getNet());
            assert(nextNet);
            getAllConnections(nextNet, connectedNets, connectedRefs,
                searchForwardThroughGraph, searchBackwardThroughGraph, searchThroughHierarchy, 
                searchThroughEquivNets, searchThroughEquivRefs, true);
        }      
        
        // up hierarchy
        oa::oaOccTerm   *nextTerm;
        oa::oaIter<oa::oaOccTerm> termIter(net->getTerms(oacTermIterAll));
        while((nextTerm = termIter.getNext())) {
            oa::oaOccInst *parent = net->getOccurrence()->getOccInst();
            if (parent) {
             
                // find the corresponding InstTerm in the parent
                oa::oaOccInstTerm *nextInstTerm;
                unsigned int bitIndex = oacNullIndex;
                if (nextTerm->isImplicit()) {
                    // this is an implicit bit in a bus
                          
                    // find the corresponding bus term
                    oa::oaOccTerm *busTerm = findBusTerm(static_cast<oa::oaOccBusTermBit*>(nextTerm));
                    assert(busTerm);

                    // which bit within the bus term?
                    oa::oaName generalName;
                    nextTerm->getName(generalName);
                    assert(generalName.getVectorBit());
                    bitIndex = generalName.getVectorBit()->getIndex();
                    
                    if (parent->usesTermPositions()) {   
                        // connected by position
                        unsigned int portPosition = busTerm->getPosition();
                        assert(portPosition != oacNullIndex);
                        nextInstTerm = oa::oaOccInstTerm::find(parent, portPosition);
                    } else {
                        // connected by name
                        oa::oaName termName;
                        busTerm->getName(termName);
                        nextInstTerm = oa::oaOccInstTerm::find(parent, termName);
                    }

                    // select appropriate bit out of instTerm
                    if (nextInstTerm) {
                      nextInstTerm = nextInstTerm->getBit(bitIndex);
                    }

                } else {
                    // this is a scalar term

                    if (parent->usesTermPositions()) {
                        // connected by position
                        unsigned int portPosition = nextTerm->getPosition();
                        assert(portPosition != oacNullIndex);
                        nextInstTerm = oa::oaOccInstTerm::find(parent, portPosition);
                    } else {
                        // connected by name
                        oa::oaName termName;
                        nextTerm->getName(termName);
                        nextInstTerm = oa::oaOccInstTerm::find(parent, termName);
                    }
                }
                   
                if (!nextInstTerm) {
                    // this Term wasn't connected to an InstTerm
                    DEBUG_PRINTLN("\tterminal isn't bound to any instTerm!")
                    continue;
                }

                assert(nextInstTerm->getNumBits() == 1);
                oa::oaOccBitNet *nextNet = toBitNet(nextInstTerm->getNet());
                assert(nextNet);

                getAllConnections(nextNet, connectedNets, connectedRefs,
                    searchForwardThroughGraph, searchBackwardThroughGraph, searchThroughHierarchy, 
                    searchThroughEquivNets, searchThroughEquivRefs, true);
                 
            } else {
                DEBUG_PRINTLN("\t\talready at top of hierarchy")
            }

        }
    }
 
    // -- exclude self
    if (!includeSelf) {
        connectedNets.erase(net);
    } 
}


// *****************************************************************************
// getAllConnections()
//
/// \brief Returns the set of nets and refs that are logically connected.
///
/// \param x
/// \param connectedNets the resulting list of connected nets
/// \param connectedRefs the resulting list of connected refs
/// \param searchForwardThroughGraph follow TERMINAL fan-out to functionally identical refs
/// \param searchBackwardThroughGraph follow TERMINAL drivers to functionally identical refs
/// \param searchThroughHierarchy traverse OpenAccess object hierarchy
/// \param searchThroughEquivNets search all nets that are marked equivalent in the database
/// \param searchThroughEquivRefs follow all Refs that are marked equivalent in the graph
/// \param includeSelf include the starting node
//
// *****************************************************************************    
void 
OccGraph::getAllConnections(OccRef x, 
                            set<oa::oaOccBitNet*> &connectedNets,
                            set<OccRef> &connectedRefs,
                            bool searchForwardThroughGraph,
                            bool searchBackwardThroughGraph,
                            bool searchThroughHierarchy,
                            bool searchThroughEquivNets,
                            bool searchThroughEquivRefs,
                            bool includeSelf) {
    if (isNull(x)) {
        return;
    }

    // has this ref been visited?
    if (connectedRefs.find(x) == connectedRefs.end()) {
        connectedRefs.insert(x);
    } else {
        return;
    }
    
    // --- directly connected nets
    if (isTerminal(x) && !isInverted(x)) {
        oa::oaOccBitNet *net;
        if ((net = getNetToAiConnection(x))) {
            getAllConnections(net, connectedNets, connectedRefs,
                searchForwardThroughGraph, searchBackwardThroughGraph, searchThroughHierarchy, 
                searchThroughEquivNets, searchThroughEquivRefs, true);
        }
    }

    // --- connected backward through graph (if searchBackwardThroughGraph)
    if (searchBackwardThroughGraph) {
        // terminal drivers
        if (isTerminal(x)) {
            OccRef driver = getTerminalDriver(x);
            driver = isInverted(x) ? notOf(driver) : driver;
            getAllConnections(driver, connectedNets, connectedRefs,
                searchForwardThroughGraph, searchBackwardThroughGraph, searchThroughHierarchy, 
                searchThroughEquivNets, searchThroughEquivRefs, true);
        }
    }

    // --- connected forward through graph (if searchForwardThroughGraph)
    if (searchForwardThroughGraph) {
        // terminals in fanout
        list<OccRef> fanoutList;
        getFanout(x, fanoutList);
        while(!fanoutList.empty()) {
            OccRef fanout(fanoutList.front());
            fanoutList.pop_front();
            if(isTerminal(fanout)) {
                fanout = isInverted(x) ? notOf(fanout) : fanout;
                fanout = isInverted(getTerminalDriver(fanout)) ? notOf(fanout) : fanout;
                getAllConnections(fanout, connectedNets, connectedRefs,
                    searchForwardThroughGraph, searchBackwardThroughGraph, searchThroughHierarchy, 
                    searchThroughEquivNets, searchThroughEquivRefs, true);
            }
        }
    }

    // --- equivalent nodes (if searchThroughEquivRefs)
    if (searchThroughEquivRefs) {
        list<OccRef> equivList;
        getEquivalents(x, equivList);
        while(!equivList.empty()) {
            OccRef equiv(equivList.front());
            equivList.pop_front();
            getAllConnections(equiv, connectedNets, connectedRefs,
                searchForwardThroughGraph, searchBackwardThroughGraph, searchThroughHierarchy, 
                searchThroughEquivNets, searchThroughEquivRefs, true);
        }
    }
    
    // -- exclude self
    if (!includeSelf) {
        connectedRefs.erase(x);
    }
}


// *****************************************************************************
// getTerminalDriver()
//
/// \brief Returns the driver of a TERMINAL node.
///
/// If searchHierarchy is false, this function only returns the graph reference
/// that is driving this terminal.  If the node is driven by a connected 
/// OpenAccess net, a null reference is returned.
///
/// If the reference is not to a TERMINAL node, an
/// error will be generated.
///
/// \param terminal
/// \param searchHierarchy
/// \return the reference to the driver, or null if none exists   
//
// *****************************************************************************    
OccRef
OccGraph::getTerminalDriver(OccRef terminal,
                            bool searchHierarchy) {
    assert(isTerminal(terminal));

    oagAi::Ref result = getGraph(terminal)->getTerminalDriver(terminal.ref);
    if (!oagAi::Graph::isNull(result)) {
        return OccRef(result, terminal.occurrence);    
    }
    
    if (searchHierarchy) {
        QUIT_ON_ERROR
    }
    
    return OccRef();
}  


// *****************************************************************************
// print()
//
/// \brief Prints full information about a OccRef.
///
/// \param lineFeed
//
// *****************************************************************************
void                
OccRef::print(bool lineFeed) {

    oa::oaString occPathString;
    if (!occurrence) {
        occPathString = "(null)";
    } else if (occurrence->getOccInst()) {
        occurrence->getOccInst()->getPathName(oa::oaVerilogNS(), occPathString);
    } else {
        occPathString = "(top)";
    }

    if (OccGraph::getNodeType(*this) == oagAi::Node::AND) {
        cout << "AND " << occPathString << "." << ref << " :\t"
                << OccGraph::getAndLeft(*this).ref << ", " << OccGraph::getAndLeft(*this).ref;
    } else if (OccGraph::getNodeType(*this) == oagAi::Node::SEQUENTIAL) {
        cout    << "SEQ " << occPathString << "." << ref << " :\t"
                << OccGraph::getNextState(*this).ref;
    } else if (OccGraph::getNodeType(*this) == oagAi::Node::CONSTANT0) {
        cout    << "ZRO " << occPathString << "." << ref << " :\t";
    } else if (OccGraph::getNodeType(*this) == oagAi::Node::TERMINAL) {
        cout    << "TRM " << occPathString << "." << ref << " :\t"
                << OccGraph::getTerminalDriver(*this).ref;
    } else if (OccGraph::getNodeType(*this) == oagAi::Node::NONE) {
        cout    << "NUL " << occPathString << "." << ref << " :\t";
    }

    if (OccGraph::isInverted(*this)) {
        cout << " (inverted)";
    }
    
    if (lineFeed) {
        cout << endl;
    }
}


// *****************************************************************************
// printName()
//
/// \brief Prints concise information about a OccRef.
///
/// \param lineFeed
//
// *****************************************************************************
void                
OccRef::printName(bool lineFeed) {

    oa::oaString occPathString;
    if (!occurrence) {
        occPathString = "(null)";
    } else if (occurrence->getOccInst()) {
        occurrence->getOccInst()->getPathName(oa::oaVerilogNS(), occPathString);
    } else {
        occPathString = "(top)";
    }

    cout << occPathString << "." << ref;
    
    if (lineFeed) {
        cout << endl;
    }
}


}
