/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-08-24: ChangeLog started
*/

#if !defined(oagFuncModGraph_P)
#define oagFuncModGraph_P

#include <stdlib.h>

#include "oaDesignDB.h"
#include "oagFuncModRef.h"
#include "oagFuncManager.h"
#include "oagAiGraph.h"
#include <list>
#include <set>

using namespace std;

namespace oagFunc {

// *****************************************************************************
// ModGraph
//
/// \brief Object for manipulating graphs in the module domain view.
///
/// The ModGraph class is not a data structure on its own, but instead provides
/// an interface for manipulating functionality graphs in the folded module 
/// domain view.  All methods in the ModGraph class are static.
/// The ModRef structure provides the mechanism for referring to graph nodes
/// within this view, and most ModGraph methods operate on ModRefs.
///
/// The module view is appropriate for making modifications to the local functional
/// descriptions of design objects.  The functional graphs describe the local
/// behavior of each logical object (module), with every instantiation (occurrence)
/// of a given object type (module) behaving identically.  Every occurrence
/// inherits its local functional behavior from the module type that it
/// implements.  This is analagous to the role that physical descriptions play
/// in the block domain.
///
/// The module view is not appropriate for working with the global
/// functionality of hierarchical designs.  Two different
/// instantiations (occurrences) may implement the same local behavior, but
/// the global functions being implemented on each of their pins depends on their
/// connectivity and context in the larger design.  To manipulate global
/// functionality, it does not suffice to traverse the folded (module) design
/// view; the unfolded (occurrence) hierarchy is better suited.  The OccGraph
/// class provided the interface for manipulating graphs in this view.
//
// *****************************************************************************

class ModGraph {
  friend class SimMod;

  private:
        
                                   ModGraph();   // this is a purely static
                                                 // class and can not be
                                                 // instantiated
  public:

    typedef set<ModRef> Cut;
    typedef list<Cut>   CutSet;
    
    /// \name Functional operations
    /// @{

    static ModRef           getNull(oa::oaModule *module) {
        /// \brief Returns a null reference inside the given module.
        ///
        /// There is a no global ModGraph null reference; one unique null
        /// reference exists for each module.  This function returns the
        /// unique null reference for the given module.  Some ModGraph
        /// operations are dependent on the module context of their
        /// parameters, even if they are null.
        /// \param module
        /// \return a null reference inside the given module
        return ModRef(oagAi::Graph::getNull(), module);
    }
    static inline bool      isNull(ModRef x) {
        /// \brief Tests a reference to see if it is null.
        ///
        /// The module that the reference is in is irrelevant.
        /// \param x
        /// \return true, if the reference is null
        return oagAi::Graph::isNull(x.ref);
    }
    static inline ModRef    constantOne(oa::oaModule *module) {
        /// \brief Returns a reference in the given module to the function that is always true.
        /// \param module
        /// \return a reference to a function that is always true
        return ModRef(getGraph(module)->constantOne(), module);
    }
    static inline ModRef    constantZero(oa::oaModule *module) {
        /// \brief Returns a reference in the given module to the function that is always false.
        /// \param module
        /// \return a reference to a function that is always false        
        return ModRef(getGraph(module)->constantZero(), module);
    }
    static inline ModRef notOf(ModRef x) {
        /// \brief Returns the complement of a reference. 
        ///
        /// The structure of the graph will not
        /// be modified and no new nodes will be created.
        /// \param x
        /// \return a reference to the resulting complement
        return ModRef(oagAi::Graph::notOf(x.ref), x.module);
    }
    static inline bool       isInverted(ModRef x) {
        /// \brief Tests whether a reference that points to a node is inverted.
        /// \param x
        /// \return true, if the reference is inverted
        return oagAi::Graph::isInverted(x.ref);
    }
    static inline ModRef     getNonInverted(ModRef x) {
        /// \brief Returns a reference that is not inverted.
        /// \param x
        /// \return a reference that is not inverted
        return ModRef(oagAi::Graph::getNonInverted(x.ref), x.module);
    }
    
    /// @}
    /// \name User-data
    /// @{
    
    static inline unsigned int getUserData(ModRef x, unsigned int index) {
        /// \brief Returns the user data field on a node.
        ///
        /// \param x
        /// \param index the data word index
        /// \return the user data
        return getGraph(x)->getUserData(x.ref, index);
    }
    static inline void         setUserData(ModRef x, unsigned int index, unsigned int data) {
        /// \brief Sets the user data field on a node.
        ///
        /// \param x
        /// \param index the data word index
        /// \param data
        getGraph(x)->setUserData(x.ref, index, data);
    }
    
    /// @}
    /// \name Graph modification
    /// @{
    
    static ModRef           newSequential(ModRef nextState) {
        /// \brief Creates a new SEQUENTIAL node.
        ///
        /// The node will be created in the same module as the
        /// nextState parameter.  Its next state input will
        /// be set to nextState.
        /// \param nextState
        /// \return a reference to a new sequential node
        return ModRef(getGraph(nextState)->newSequential(nextState.ref), nextState.module);
    }
    static ModRef           newTerminal(ModRef terminal) {
        /// \brief Creates a new TERMINAL node.
        /// \param terminal
        /// \return a reference to a new terminal node
        return ModRef(getGraph(terminal)->newTerminal(terminal.ref), terminal.module);    
    }
    static ModRef           newAnd(ModRef x, ModRef y);
    static ModRef           andOf(ModRef x, ModRef y);
    static void             setTerminalDriver(ModRef terminal, ModRef driver);    
    static void             setNextState(ModRef sequential, ModRef nextState);
    static void             setAndLeft(ModRef x, ModRef left);
    static void             setAndRight(ModRef x, ModRef left);
    
    static void             resubstitute(ModRef original, ModRef replacement);
    static void             resubstitute(ModRef original, ModRef replacement, ModRef target);
    static void             detach(ModRef x);
    
    /// @}
    /// \name Intra-module incremental graph traversal
    /// @{
    
    static inline oagAi::Node::Type     getNodeType(ModRef x) {
        /// \brief Returns the type of the referenced graph node.
        /// \param x
        /// \return the type of the referenced node
        return getGraph(x)->getNodeType(x.ref);
    }    
    
    //    on terminal nodes...
    static ModRef         getTerminalDriver(ModRef terminal) {
        /// \brief Returns the driver of a TERMINAL node.
        ///
        /// This function only returns the graph reference
        /// that is driving this terminal.  If the node is
        /// driven by a connected OpenAccess net, 
        /// a null reference is returned.
        ///
        /// If the reference is not to a TERMINAL node, an
        /// error will be generated.
        /// \param terminal
        /// \return the reference to the driver, or null if none exists
        return ModRef(getGraph(terminal)->getTerminalDriver(terminal.ref), terminal.module);    
    }
    static bool           isTerminal(ModRef x) {
        /// \brief Tests if a reference is to a TERMINAL node.
        /// \param x
        /// \return true, if the reference is to a TERMINAL node
        return getGraph(x)->isTerminal(x.ref);    
    }
    //    on sequential nodes...
    static ModRef         getNextState(ModRef sequential) {
        /// \brief Returns the next state input of a SEQUENTIAL node.
        ///
        /// If the reference is not to a SEQUENTIAL node, an
        /// error will be generated.
        /// \param sequential
        /// \return the reference to the next state, or null if none exists
        return ModRef(getGraph(sequential)->getNextState(sequential.ref), sequential.module);    
    }
    static bool           isSequential(ModRef x) {
        /// \brief Tests if a reference is to a SEQUENTIAL node.
        /// \param x
        /// \return true, if the reference is to a SEQUENTIAL node
        return getGraph(x)->isSequential(x.ref);    
    }
    static oagAi::Node::SequentialData *getSequentialData(ModRef sequential) {
        /// \brief Returns the sequential description of a SEQUENTIAL node.
        ///
        /// If the reference is not to a SEQUENTIAL node, an
        /// error will be generated.
        /// \param sequential
        /// \return sequential description
        return getGraph(sequential)->getSequentialData(sequential.ref);    
    }
    //    on and nodes...
    static bool           isAnd(ModRef x) {
        /// \brief Tests if a reference is to an AND node.
        /// \param x
        /// \return true, if the reference is to an AND node
        return getGraph(x)->isAnd(x.ref);    
    }
    static inline ModRef         getAndLeft(ModRef x) {
        /// \brief Returns the left operand of an AND node.
        ///
        /// If the reference is not to an AND node, an
        /// error will be generated.        
        /// \param x
        /// \return reference to left operand
        return ModRef(getGraph(x)->getAndLeft(x.ref), x.module);
    }
    static inline ModRef         getAndRight(ModRef x) {
        /// \brief Returns the right operand of an AND node.
        ///
        /// If the reference is not to an AND node, an
        /// error will be generated.        
        /// \param x
        /// \return reference to right operand
        return ModRef(getGraph(x)->getAndRight(x.ref), x.module);
    }
    // on all nodes...
    static list<ModRef>      getFanout(ModRef x);
    static void              getFanout(ModRef x, list<ModRef> &result);
    static inline bool       hasFanout(ModRef x) {
        /// \brief Returns the fanout of a node.
        ///
        /// An error is generated if fanout is not being maintained.
        /// \param x
        /// \return true, if the node has any fanout
        return getGraph(x)->hasFanout(x.ref);
    }

    /// @}
    /// \name Explicitly managing graph-to-net connections
    /// @{
    
    static ModRef              prepareNetToAiConnection(oa::oaModBitNet *net);
    static oa::oaModBitNet *   getNetToAiConnection(ModRef ref);
    static ModRef              getNetToAiConnection(oa::oaModBitNet *net);
    static void                setNetToAiConnection(oa::oaModBitNet *net, ModRef ref);
    static void                removeNetToAiConnection(oa::oaModBitNet *net);

    static oa::oaModBitNet *   findDriverOfEquivalentNets(oa::oaModBitNet *net);
    static void                connectEquivalentNetsInGraph(oa::oaModule *module);
                                    
    /// @}
    /// \name Advanced graph traversal
    /// @{
    
    static bool         hasCombinationalCycle(oa::oaModule *module);

    static CutSet       enumerateKfeasibleCuts(ModRef x, unsigned int maxCutSize,
                                               int maxCutCout = -1, int maxCutDepth = -1,
                                               bool includeConstantNode = true);
    static void         clearKfeasibleCuts(oa::oaModule *module);

    static void         getTransitiveFanin(ModRef x, list<ModRef> &transitiveFanin, 
                                           bool includeRoots = true, bool crossSequential = false);
    static void         getTransitiveFanin(ModRef x, vector<ModRef> &transitiveFanin, 
                                           bool includeRoots = true, bool crossSequential = false);
    static void         getTransitiveFanin(list<ModRef> x, list<ModRef> &transitiveFanin, 
                                           bool includeRoots = true, bool crossSequential = false);
    static void         getTransitiveFanin(list<ModRef> x, vector<ModRef> &transitiveFanin, 
                                           bool includeRoots = true, bool crossSequential = false);

    static void         getTransitiveFanout(ModRef x, list<ModRef> &transitiveFanout, 
                                            bool includeRoots = true, bool crossSequential = false);
    static void         getTransitiveFanout(ModRef x, vector<ModRef> &transitiveFanout, 
                                            bool includeRoots = true, bool crossSequential = false);
    static void         getTransitiveFanout(list<ModRef> x, list<ModRef> &transitiveFanout, 
                                            bool includeRoots = true, bool crossSequential = false);
    static void         getTransitiveFanout(list<ModRef> x, vector<ModRef> &transitiveFanout, 
                                            bool includeRoots = true, bool crossSequential = false);

    static void         getFaninCone(ModRef x, const list<ModRef> &coneRoots, 
                                     list<ModRef> &transitiveFanin, bool includeRoots = true);
    static void         getFanoutCone(ModRef x, const list<ModRef> &coneRoots, 
                                      list<ModRef> &transitiveFanout, bool includeRoots = true);
    
    static void         getFaninRoots(ModRef x, list<ModRef> &faninRoots);
    static void         getFanoutRoots(ModRef x, list<ModRef> &fanoutRoots);

    /// @}
    /// \name Equivalence marking
    /// @{
    
    static bool         testEquivalence(ModRef x, ModRef y);
    static void         setEquivalent(ModRef x, ModRef y);
    static void         getEquivalents(ModRef x, list<ModRef> & result);
    static void         removeEquivalences(ModRef x) {
        /// \brief Removes any equivalences between this and other nodes.
        ///
        /// This function destroys equivalences in both directions.  Not only
        /// will this node no longer point to any equivalents, but any of
        /// its previous equivalents will no longer point to this node.
        /// \param x
        getGraph(x)->removeEquivalences(x.ref);
    }
    static void         chooseEquivalent(ModRef x) {
        /// \brief Moves the fan-out of all equivalent nodes into the fan-out of one node.
        ///
        /// Each of several nodes which are marked as equivalent may have fan-out.
        /// This function takes all of the references to the equivalent nodes and
        /// reconnects them to only one of the equivalent nodes.
        ///
        /// \param x
        getGraph(x)->chooseEquivalent(x.ref);
    }

    static void         getFanoutOfEquivalentNodes(ModRef x, list<ModRef> & result);

    /// @}
    /// \name Global graph queries
    /// @{

    static void         print(oa::oaModule *module);
    
    static void         getAllNodes(oa::oaModule *module, list<ModRef> &result);
    static void         getInputs(oa::oaModule *module, list<ModRef> &result);
    static void         getOutputs(oa::oaModule *module, list<ModRef> &result);
    static void         getLocalStates(oa::oaModule *module, list<ModRef> &result);    

    static void                getAllConnections(oa::oaModBitNet *net,
                                    set<oa::oaModBitNet*> &connectedNets,
                                    set<ModRef> &connectedRefs,
                                    bool searchForwardThroughGraph = true,
                                    bool searchBackwardThroughGraph = true,
                                    bool searchThroughEquivNets = true,
                                    bool searchThroughEquivRefs = false,
                                    bool includeSelf = true);
    static void                getAllConnections(ModRef x, 
                                    set<oa::oaModBitNet*> &connectedNets,
                                    set<ModRef> &connectedRefs,
                                    bool searchForwardThroughGraph = true,
                                    bool searchBackwardThroughGraph = true,
                                    bool searchThroughEquivNets = true,
                                    bool searchThroughEquivRefs = false,
                                    bool includeSelf = true);

    /// @}
    /// \name Memory management
    /// @{
    
    static void         incrementExternalReferences(ModRef x) {
        /// \brief Increment the external reference count of a node.
        ///
        /// With garbage collection, this function
        /// should be called to mark nodes that are being used by
        /// the application.
        /// A non-zero external reference count guarantees that the
        /// functionality of a node is not destroyed during the next
        /// garbage collection.
        /// \param x
        getGraph(x)->incrementExternalReferences(x.ref);
    }
    static void         decrementExternalReferences(ModRef x) {
        /// \brief Decrement the external reference count of a node.
        ///
        /// When the application is finished with a node, its
        /// external reference count can be decremented, such that
        /// if there exist no external references the node may
        /// be freed during the next garbage collection.
        /// \param x    
        getGraph(x)->decrementExternalReferences(x.ref);
    }
    static void         clearExternalReferences(ModRef x) {
        /// \brief Clears the reference count of a particular node.
        ///
        /// \param x
        getGraph(x)->clearExternalReferences(x.ref);
    }

    /// @}

  protected:
  
    static inline oagAi::Graph *   getGraph(oa::oaDesign *design) {
        /// \brief Returns the graph of a design.
        ///
        /// All design modules' functional descriptions will be contained
        /// within this graph.
        /// \param design
        /// \return the graph of a design
        return &(Manager::get(design)->ai);
    }
    static inline oagAi::Graph *   getGraph(oa::oaModule *module) {
        /// \brief Returns the graph of a module's design.
        ///
        /// This module's functional description will be contained
        /// within this graph, along with the functional descriptions
        /// of other modules in the same design.
        /// \param module
        /// \return the graph of a module's design
        return &(Manager::get(module->getDesign())->ai);
    }
    static inline oagAi::Graph *   getGraph(const ModRef &x) {
        /// \brief Returns the graph that a reference is in.
        /// \param x
        return &(Manager::get(x.module->getDesign())->ai);
    }
    
    static void             failIfInDifferentModules(oa::oaModule *x, oa::oaModule *y);
    static void             failIfInDifferentModules(ModRef x, ModRef y) {
        /// \brief Generates an error if the two references are in different modules.
        /// \param x
        /// \param y
        failIfInDifferentModules(x.module, y.module);
    }
    
    static oa::oaModule*    convertModRefListToRefList(const list<ModRef> & source,
                                                       list<oagAi::Ref> & result);
    static void             convertRefListToModRefList(const list<oagAi::Ref> & source,
                                                       oa::oaModule *module,
                                                       list<ModRef> & result);
    static void             convertRefVectorToModRefVector(const vector<oagAi::Ref> & source,
                                                           oa::oaModule *module,
                                                           vector<ModRef> & result);
};

}

#endif
