/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-05-31: ChangeLog started
*/

#if !defined(oagFuncManager_P)
#define oagFuncManager_P

#include "oaDesignDB.h"
#include "oagFunc.h"
#include "oagFuncReasoningEngine.h"
#include "oagAiGraph.h"
#include "oagFuncOccRef.h"
#include <list>
#include <map>
#include <set>

#define CACHE_LAST_MANAGER_LOOKUP

using namespace std;

namespace oagFunc {

// *****************************************************************************
// Manager
//
/// \brief This class manages the functional description of a design.
//
// *****************************************************************************

class Manager {
  friend class Synthesis;
  friend class ModGraph;
  friend class OccGraph;

  protected:

                        Manager(oa::oaDesign *design);
                       ~Manager();

  public:

    static inline bool         hasManager(oa::oaDesign *design) {
      /// \brief Returns true if the given design has an associated Manager.
      assert(design);

      return design->getAppDefs().includes(managerAppDef);
    }
    static inline Manager*     get(oa::oaModule *module) {
      return get(module->getDesign());
    }
    static inline Manager*     get(oa::oaDesign *design) {
      assert(design);

#if defined(CACHE_LAST_MANAGER_LOOKUP)
      if (design == lastManagerDesign) {
        return lastManagerObject;
      } else if (hasManager(design)) {
        lastManagerDesign = design;
        lastManagerObject = static_cast<Manager*>(managerAppDef->get(design));
        return lastManagerObject;
      } else {
        lastManagerDesign = design;
        lastManagerObject = create(design);
        return lastManagerObject;
      }
#else
      if (hasManager(design)) {
        return static_cast<Manager*>(managerAppDef->get(design));
      } else {
        return create(design);   
      }
#endif
    }
    static Manager*            create(oa::oaDesign *design);
    static void                destroy(oa::oaDesign *design);

    /// \name Database operations
    /// @{
    
    static int          getSerializedSize(oa::oaDesign *design);
    static void         serialize(void* buf, oa::oaDesign *design);
    static int          unserialize(void* buf, oa::oaDesign *design);
    
    void                garbageCollect() {
        /// \brief Garbage collects the graph of a design.
        ///
        /// For more details on graph memory management, see the 
        /// oagAi package documentation.
        
        ai.garbageCollect();
    }    

    /// @}
    /// \name Informational queries
    /// @{
    
    bool                isStructural();
    
    int                 getLocalAIsize();
    int                 getHierarchicalAIsize();
    
    int                 getLocalAndAIsize();
    int                 getHierarchicalAndAIsize();

    int                 getLocalSeqNodeCount();
    int                 getHierarchicalSeqNodeCount();
       
    int                 getNumTotalInstances();
    int                 getNumLeafInstances();

    void                print();
    
    /// @}
    /// \name Maintaining links between OA objects and the AI graph
    /// @{
    
  public:

    oa::oaModBitNet *   getNetToAiConnection(oagAi::Ref ref);
    oagAi::Ref          getNetToAiConnection(oa::oaModBitNet *net);
    void                setNetToAiConnection(oa::oaModBitNet *net, oagAi::Ref ref);
    void                removeNetToAiConnection(oa::oaModBitNet *net);

    inline const oagAi::Graph *  getGraph() const { 
        /// \brief Returns the oagAi::Graph associated with the design that this Manager controls.
        return &ai; 
    }
    inline oagAi::Graph *        getGraph() { 
        /// \brief USE WITH CAUTION.  Returns the oagAi::Graph associated with the design that this Manager controls.
        /// Modifying the structure of this graph (and in particular the 
        /// TERMINAL nodes) could invalid the links to OpenAccess design 
        /// objects and result in errors or incorrect behavior.  
        /// The const version of getGraph() is safer.  
        /// Use with extreme caution.
        return &ai; 
    }

    /// @}

  protected:
  
    oagAi::Graph            ai;
    oa::oaDesign           *design;

  private:

    // *****************************************************************************
    // struct TraversalState
    ///
    /// \brief Maintains the current state of a traversal through the OpenAccess design structure.
    ///
    // *****************************************************************************
    struct TraversalState {
        enum{NOT_FOUND, 
             FOUND_AI,
             FOUND_NET}                              status;
        OccRef                                       nextAI;
        oa::oaOccBitNet                             *nextNet;

        TraversalState() { status = NOT_FOUND; }
    };
    
    void                           scanAIconnections( oa::oaOccBitNet *net, TraversalState &state );
    void                           scanUpHierarchy( oa::oaOccBitNet *net, TraversalState &state );
    void                           scanDownHierarchy( oa::oaOccBitNet *net, TraversalState &state );
    oa::oaOccBitNet*               searchForAIConnection(OccRef &g);
    
    oagAi::Ref                     flattenRecursively(oa::oaOccBitNet *net);
    oagAi::Ref                     flattenRecursively(OccRef &g);
    
    map<OccRef,oagAi::Ref>         preflattenedCache;
    list<OccRef>                   visitedStateBits;
    list<OccRef>                   cycleBreaks;
    std::set<OccRef>               visitedSet;

    int                            getHierarchicalAIsize(oa::oaDesign *design);
    int                            getHierarchicalAndAIsize(oa::oaDesign *design);
    int                            getHierarchicalSeqNodeCount(oa::oaDesign *design);
    int                            getNumTotalInstances(oa::oaDesign *design);
    int                            getNumLeafInstances(oa::oaDesign *design);
    void                           getStateBits(list<oa::oaOccBitNet*> &l, oa::oaOccurrence *occ);
    
    oagAi::Ref                     prepareNetToAiConnection(oa::oaModBitNet *net);
    
#if defined(CACHE_LAST_MANAGER_LOOKUP)
    static oa::oaDesign *lastManagerDesign;
    static Manager      *lastManagerObject;
#endif

};

}

#endif
