/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-06-01: ChangeLog started
*/

#include "oaDesignDB.h"
#include "oagFunc.h"
#include "oagFuncManager.h"
#include "oagFuncTimingLibrarySynthesis.h"
#include "oagFuncVerilogSynthesis.h"
#include "oagFuncModGraph.h"
#include "oagFuncOccGraph.h"
#include <iostream>

// #define DEBUG

#include "oagFuncDebug.h"

// *****************************************************************************
// Global static variable definitions.
// *****************************************************************************

extern char                          *oagFuncVerilog_yyfilename;
extern FILE                          *oagFuncVerilog_yyin;
int                                   oagFuncVerilog_yyparse();
extern oagFunc::VerilogDesign        *oagFuncVerilog_yydesign;
extern FILE                          *oagFuncLib_yyin;
int                                   oagFuncLib_yyparse();

namespace oagFunc {

// *****************************************************************************
// Class static variable definitions.
// *****************************************************************************

oa::oaVarDataAppDef<oa::oaDesign>           *streamAppDef;
oa::oaVoidPointerAppDef<oa::oaDesign>       *managerAppDef;
oa::oaIntAppDef<oa::oaModBitNet>            *AiRefAppDef;
    
Observer                                    *observer;

using namespace std;


// *****************************************************************************
// initialize()
//
/// \brief Initializes Func package.  Must be called before any use.
///
/// Initializes data structures for representing and manipulating
/// functional information.  The necessary AppDefs are 
/// created in the OpenAccess database.  An Observer object is also created
/// to intercept the file I/O operations on objects with functional
/// representations.
//
// *****************************************************************************
void            
initialize() {
    streamAppDef = oa::oaVarDataAppDef<oa::oaDesign>::get(oa::oaString("oagFunc_function_stream"));
    managerAppDef = oa::oaVoidPointerAppDef<oa::oaDesign>::get(oa::oaString("oagFunc_manager"));
    AiRefAppDef = oa::oaIntAppDef<oa::oaModBitNet>::get(oa::oaString("oagFunc_oagAi::Ref"));

    observer = new Observer();
}


// *****************************************************************************
// readVerilog()
//
/// \brief Reads a behavioral Verilog description into the netlist view.
///
/// A view other than "netlist" can be specified with additional parameters.
///
/// \param library the library in which to find/create the cells described in the file
/// \param filename the path of the Verilog file to read
//
// *****************************************************************************
void
readVerilog(oa::oaLib *library, const char *filename) {
    assert(library);
    assert(filename);

    oa::oaNativeNS nativeNS;
    oa::oaScalarName viewName(nativeNS, "netlist");

    // read Verilog
    readVerilog(library, viewName, filename);
}


// *****************************************************************************
// readVerilog()
//
/// \brief Reads a behavioral Verilog description.
///
/// \param library the library in which to find/create the cells described in the file
/// \param view the view in which to find/create the cells
/// \param filename the path of the Verilog file to read
//
// *****************************************************************************
void
readVerilog(oa::oaLib *library, const oa::oaScalarName &viewName, const char *filename) {
    assert(library);
    assert(filename);

    if(library->getAccess(oa::oaLibAccess(oa::oacWriteLibAccess))) {
        const oa::oaViewType *viewType = oa::oaViewType::get(oa::oacNetlist);
        oa::oaView *view = oa::oaView::get(library, viewName, viewType);
        assert(view);

        // parse verilog file
        VerilogSynthesis::setLibrary(library);
        VerilogSynthesis::setView(view);
        oagFuncVerilog_yyfilename = const_cast<char *>(filename);
        oagFuncVerilog_yyin = fopen(filename, "r");
        if (!oagFuncVerilog_yyin) {
            cerr << "ERROR: Could not open Verilog file " << filename << endl;
            exit(0);
        }
        oagFuncVerilog_yyparse();
        VerilogSynthesis::synthesize(oagFuncVerilog_yydesign);
        fclose(oagFuncVerilog_yyin);
        library->releaseAccess();
    } else {
        cerr << "ERROR: Could not write into library " << endl;
        exit(0);
    }
}


// *****************************************************************************
// readLiberty()
//
/// \brief Read the functional information from a Synopsys Liberty library description.
///
/// \param library the library in which to find/create the cells described in the file
/// \param view the view in which to find/create the cells
/// \param filename the path of the Liberty file to read
//
// *****************************************************************************
void
readLiberty(oa::oaLib *library, const oa::oaScalarName &viewName, const char *filename) {
    assert(library);
    assert(filename);

    if(library->getAccess(oa::oaLibAccess(oa::oacWriteLibAccess))) {
        const oa::oaViewType *viewType = oa::oaViewType::get(oa::oacNetlist);
        oa::oaView *view = oa::oaView::get(library, viewName, viewType);
        assert(view);

        // parse Liberty file
        TimingLibrarySynthesis::initialize();
        TimingLibrarySynthesis::setLibrary(library);
        TimingLibrarySynthesis::setView(view);
        oagFuncLib_yyin = fopen(filename, "r");
        if (!oagFuncLib_yyin) {
            cerr << "ERROR: Could not open Liberty file " << filename << endl;
            exit(0);
        }
        cout << "Reading Liberty file " << filename << endl;
        oagFuncLib_yyparse();
        fclose(oagFuncLib_yyin);
        library->releaseAccess();
    } else {
        cerr << "ERROR: Could not write into library " << endl;
        exit(0);
    }
}


// *****************************************************************************
// readLiberty()
//
/// \brief Read the functional information from a Synopsys Liberty library description into the netlist view.
///
/// A view other than "netlist" can be specified with additional parameters.
///
/// \param library the library in which to find/create the cells described in the file
/// \param filename the path of the Liberty file to read
//
// *****************************************************************************
void
readLiberty(oa::oaLib *library, const char *filename) {
    assert(library);
    assert(filename);

    oa::oaNativeNS nativeNS;
    oa::oaScalarName viewName(nativeNS, "netlist");

    // read the Liberty file
    readLiberty(library, viewName, filename);
}


// *****************************************************************************
// findBusTerm()
//
/// \brief Returns the bus Term to which a bit Term belongs.
///
/// \param bitTerm
/// \return the bus Term to which the bit Term belongs, or NULL if it can't be found
//
// *****************************************************************************
oa::oaOccBusTerm*
findBusTerm(oa::oaOccBusTermBit *bitTerm) {
    assert(bitTerm);

    oa::oaName generalName;
    bitTerm->getName(generalName);
    if (generalName.getScalar()) {
      // ignore scalars
      return NULL;
    }
    assert(generalName.getVectorBit());

    oa::oaOccBusTermDef *busTermDef = static_cast<oa::oaOccBusTermBit*>(bitTerm)->getDef();
    if (busTermDef) {
      oa::oaIter<oa::oaOccBusTerm> busIter(busTermDef->getBusTerms());
      oa::oaOccBusTerm *busTerm = busIter.getNext();
      if (busTerm) {
        return busTerm;
      }
    }
    
    // should never reach this point
    QUIT_ON_INTERNAL_ERROR;
    return NULL;
}


// *****************************************************************************
// findBusNet()
//
/// \brief Returns the bus net to which a bit net belongs.
///
/// \param bitNet
/// \return the bus to which the bit belongs, or NULL if it can't be found
//
// *****************************************************************************
oa::oaModBusNet*
findBusNet(oa::oaModBitNet *bitNet) {
    assert(bitNet);

    oa::oaName generalName;
    bitNet->getName(generalName);
    if (generalName.getScalar()) {
      // ignore scalars
      return NULL;
    }
    assert(generalName.getVectorBit());

    oa::oaModBusNetDef *busNetDef = static_cast<oa::oaModBusNetBit*>(bitNet)->getDef();
    if (busNetDef) {
      oa::oaIter<oa::oaModBusNet> busIter(busNetDef->getBusNets());
      oa::oaModBusNet *busNet = busIter.getNext();
      if (busNet) {
        return busNet;
      }
    }
    
    // should never reach this point
    QUIT_ON_INTERNAL_ERROR;
    return NULL;
}


}
