/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron Hurst <ahurst@eecs.berkeley.edu>

ChangeLog:
2005-07-20: ChangeLog started
*/

#include <iostream>
#include "oaDesignDB.h"
#include "oagFuncModGraph.h"
#include "oagFuncOccGraph.h"
#include "oagFuncManager.h"
#include "oagFuncPrint.h"
#include "oagFuncSynthesis.h"
#include "oagUtilOption.h"
#include "oagUtilOptionParser.h"


// *****************************************************************************
// findOpenLib()
//
/// \brief Find an already open, opens from disk, or creates a library.
///
// *****************************************************************************
oa::oaLib *
findOpenLib(const oa::oaScalarName libName, const char *libString) {
    oa::oaLib *lib = NULL;

    if ((lib = oa::oaLib::find(libName)))
        ;
    else if (!oa::oaLib::exists(libString)) {
        lib = oa::oaLib::create(libName, libString);
        oa::oaLibDefList *defList = oa::oaLibDefList::get("lib.defs", 'a');
        oa::oaLibDef::create(defList, libName, libString);
        defList->save();
    } else {
        lib = oa::oaLib::open(libName, libString); 
    }

    return lib;
}


// *****************************************************************************
// main()
//
// *****************************************************************************
int main(int argc, const char **argv) {

    std::cout << "*****************************************************************************" << std::endl;
    std::cout << "Tool:   funcTest        Package:    oagFunc     version:  01/2007" << std::endl;
    std::cout << "*****************************************************************************" << std::endl;
    std::cout << std::endl;

    oagUtil::OptionParser options("Simple test program to demonstrate the use "
            "of the OAGear Func package, to read a design from OpenAccess or "
            "Verilog, print some statistics and save the design in "
            "OpenAccess.");

    oagUtil::Option *libOpt = options.add("lib", "Input and output library "
            "name", true, "library");
    oagUtil::Option *viewOpt = options.add("view", "Input and output view name "
            "(default: netlist)", false, "view");
    oagUtil::Option *cellOpt = options.add("cell", "Print info about this cell "
            "or module", false, "cell");
    oagUtil::Option *libertyOpt = options.add("liberty", ".lib library file", 
            false, "file");
    oagUtil::Option *verilogOpt = options.add("verilog", "Verilog file", false, 
            "file");
    oagUtil::Option *leafLibsOpt = options.add("leafLibs", "Library files for leaf cells",
             false, "\"libList");
    oagUtil::Option *leafViewsOpt = options.add("leafViews", "Library views for leaf cells", 
             false, "\"viewList");
    oagUtil::Option *nextStatesOpt = options.add("next_states", "Print the "
            "next states", false);
    oagUtil::Option *aiGraphOpt = options.add("ai", "Print the "
            "full and/inverter graph", false);
    oagUtil::Option *reuseOpt = options.add("reuse", "Re-use existing structure "
            "and append functional descriptions", false);
    oagUtil::Option *hierarchicalNamesOpt = options.add("hierarchical_names", 
            "Print the hierarchcal names", false);
    oagUtil::Option *flattenOpt = options.add("flatten", "Flatten the cell "
            "(cell needs to be specified)", false);
    oagUtil::Option *uniquifyOpt = options.add("uniquify", "Uniquify the cell "
            "(only possible if -flatten is not used)", false, "cell");
    oagUtil::Option *extractOpt = options.add("extract", "Extract all functionality "
            "into a new flat sub-cell (cell needs to be specified)", false);
    oagUtil::Option *saveOpt = options.add("save", "Save into OpenAccess", 
            false);

    if (!options.parse(argc, argv)) {
        std::cerr << options.getMessage();
        exit(0);
    }

    const char *libString = libOpt->getValue();
    const char *viewString 
        = viewOpt->isGiven() ? viewOpt->getValue(): "netlist";

    try {

        // Initialize OpenAccess

        oa::oaDesignInit();
        oa::oaLibDefList::openLibs();
        oagFunc::initialize();

        const oa::oaNativeNS nativeNS;
        const oa::oaScalarName libName(nativeNS, libString);
        const oa::oaScalarName viewName(nativeNS, viewString);

        // Find, create, or load the first library

        oa::oaLib *library = findOpenLib(libName, libString);
        assert(library);

        // Find, create, or load additional libraries (if given)
        
        if (leafLibsOpt->isGiven()) {
            assert(strlen(leafLibsOpt->getValue())<256);
            char opt[256];
            strcpy(opt, leafLibsOpt->getValue());
            const char *tok = strtok(opt, "\" ,");
            while(tok) {
              oa::oaScalarName leafLibName(nativeNS, tok);
              oa::oaLib *leafLib = findOpenLib(leafLibName, tok);
              assert(leafLib);
              oagFunc::Synthesis::addLeafLibrary(leafLib);
              tok = strtok(NULL, "\" ,");
            }
            if (leafViewsOpt->isGiven()) {
              assert(strlen(leafViewsOpt->getValue())<256);
              char opt[256];
              strcpy(opt, leafViewsOpt->getValue());
              char *tok = strtok(opt, "\" ,");
              while(tok) {
                oa::oaScalarName leafViewName(nativeNS, tok);
                oagFunc::Synthesis::addLeafView(leafViewName);
                tok = strtok(NULL, "\" ,");
              }
            } else {
                std::cout << "WARNING: No leaf views provided. Assuming \"netlist\" view" << endl;
                oa::oaScalarName leafViewName(nativeNS, "netlist");
                oagFunc::Synthesis::addLeafView(leafViewName);
            }
        }

        // Should existing structures be re-used?
        
        if (reuseOpt->isGiven()) {
            oagFunc::Synthesis::setOverwriteStructure(false);
        }

        // Read designs from a Liberty file

        if (libertyOpt->isGiven()) {
            // read in a cell library
            oagFunc::readLiberty(library, viewName, libertyOpt->getValue());
        }

        // Read designs from a Verilog file

        if (verilogOpt->isGiven()) {
            // read in functional Verilog description
            oagFunc::readVerilog(library, viewName, verilogOpt->getValue());
        }

        // Manipulate a module (if given)

        if (cellOpt->isGiven()) {

            // i. print initial info

            const char *cellString = cellOpt->getValue();
            const oa::oaScalarName cellName(nativeNS, cellString);
            oa::oaDesign *design;
            if ((design = oa::oaDesign::find(libName, cellName, viewName))) {
                ;
            } else if (saveOpt->isGiven()) {
                // open for read and write
                design = oa::oaDesign::open(libName, cellName, viewName, 'a');
            } else {
                // open for read only
                design = oa::oaDesign::open(libName, cellName, viewName, 'r');
            }

            printFunctionality(design, nextStatesOpt->isGiven(), 
                    hierarchicalNamesOpt->isGiven());

            // ii. do one or more operations

            if (flattenOpt->isGiven()) {
                std::cout << "Flattening \'" << cellString << "\' ..." 
                    << std::endl;
                oagFunc::collapseAllInstancesAllLevels(design->getTopModule());
            }
            if (uniquifyOpt->isGiven()) {
                const oa::oaVerilogNS verilogNS;
                oa::oaSimpleName occName(verilogNS, uniquifyOpt->getValue());
                oa::oaOccInst *occInst = oa::oaOccInst::find(design->getTopOccurrence(), occName);

                if (!occInst) {
                    std::cerr << "ERROR: Could not find instance: " << uniquifyOpt->getValue() << std::endl;
                    exit(0);
                }

                std::cout << "Uniquifying \'" << uniquifyOpt->getValue() << "\' ..." << std::endl;

                oa::oaOccurrence *occurrence = occInst->getMasterOccurrence();
                assert(occurrence);
                oagFunc::uniquify(occurrence);
            }
            if (extractOpt->isGiven()) {
                std::cout << "Extracting all functionality from \'" << cellString << "\' into \'" 
                          << cellString << "_extracted\' ..." << std::endl;
                oagFunc::extract(design->getTopModule());
            }

            // iii. print final info
            if (flattenOpt->isGiven() || uniquifyOpt->isGiven() || extractOpt->isGiven()) {
                printFunctionality(design, nextStatesOpt->isGiven(), hierarchicalNamesOpt->isGiven());
            }

            // iv. print entire AI graph

            if (aiGraphOpt->isGiven()) {
                oagFunc::ModGraph::print(design->getTopModule());
            }
        }

        // Test for combinational cycles

        bool cyclicDesigns = false;
        oa::oaDesign *cycleDesign;
        oa::oaIter<oa::oaDesign> cycleDesignIter(oa::oaDesign::getOpenDesigns());
        while((cycleDesign = cycleDesignIter.getNext())) {
          if (oagFunc::ModGraph::hasCombinationalCycle(cycleDesign->getTopModule())) {
            if (!cyclicDesigns) {
              std::cout << "WARNING: The following designs have combinational cycles: ";
              cyclicDesigns = true;
            }
            oa::oaString designString;
            cycleDesign->getTopModule()->getName(nativeNS, designString);
            cout << designString << " ";
          }
        }
        if (cyclicDesigns) {
          std::cout << endl;
        }

        // Save and close

        if (saveOpt->isGiven()) {
            std::cout << "Saving designs: ";
        }
        oa::oaDesign *openDesign;
        oa::oaIter<oa::oaDesign> designIter(oa::oaDesign::getOpenDesigns());
        while((openDesign = designIter.getNext())) {
            if (saveOpt->isGiven() && 
                openDesign->getLib() == library && openDesign->getMode() != 'r') {
                oa::oaString designString;
                openDesign->getTopModule()->getName(nativeNS, designString);
                std::cout << designString << " ";
                openDesign->save();
            }
            openDesign->purge();
        }
        if (saveOpt->isGiven()) {
            std::cout << std::endl;
        }
        std::cout << std::endl;

        // Handle errors

    } catch(oa::oaException &e) {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        return 1;
    }

    return 0;
}
