/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: David Papa <iamyou@umich.edu>

ChangeLog:
2004-09-15: ChangeLog started
 */

#include <iostream>
#include <string>
#include <vector>
#include <string>

#include <stdlib.h>
#include <assert.h>

#include <qscrollview.h>
#include <qapplication.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qmultilineedit.h>
#include <qwidgetfactory.h>
#include <qsizegrip.h>
#include <qgl.h>

#include "oaDesignDB.h"
#include "oaBase.h"

#include "OAGearCommon.h"
#include "SchemInstView.h"
#include "oagConnect.h"
#include "DebugLog.h"

using std::cerr;
using std::endl;
using std::string;
using std::vector;
using std::pair;

using oa::oaModInst;
using oa::oaIter;
using oa::oaModInstTerm;
using oa::oaModNet;
using oa::oaModTerm;
using oa::oaTermType;
using oa::oacInputTermType;
using oa::oacInputOutputTermType;
using oa::oacOutputTermType;
using oa::oaNativeNS;
using oa::oaString;
using oa::oaPoint;
using oa::oaBox;

namespace oagBazaar
{

    namespace Qoag
    {
        using oagBazaar::operator<<;

        QoaSchemInstView::
            QoaSchemInstView(QWidget* parent,
                    const std::string& lib,
                    const std::string& cell,
                    const std::string& view,
                    oa::oaModInst* center_inst) :
                QGLWidget(parent,"Schematic Inst View GLWidget"),
                _ready(false),
                _mouse_over_inst(0),
                _dbConnection(lib, cell, view),
                _oagDBlk(_dbConnection.getOAGDesignBlock()),
                _edgeDisplayList( glGenLists(1) )
            {
                setFormat(QGLFormat(DoubleBuffer | DepthBuffer));
                setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding);
                setMouseTracking(true);

                learnInst(center_inst);
                setCenterInst(center_inst);
                //setInstLocationsOneCenterInstStyle();

                learnInstNeighbors(center_inst); 
                setInstLocationsCenterInstNeighborStyle();

                _ready = true;
            }

        QSize QoaSchemInstView::sizeHint(void)
        {
            return QSize(400,400);
        }

        QSize QoaSchemInstView::minimumSizeHint(void)
        {
            return QSize(400,400);
        }

        void QoaSchemInstView::
            setCenterInst(oa::oaModInst* inst)
            {
                _center_inst = inst;
            }

        void QoaSchemInstView::
            setCenterInst(internalInstIndexT inst)
            {
                assert( _indexToInst.size() > inst);
                setCenterInst(_indexToInst[inst]);
            }

        void QoaSchemInstView::
            learnInstNeighbors(oa::oaModInst* source_inst)
            {
                assert(_instToIndex.find(source_inst) != _instToIndex.end() );
                learnInstNeighbors(_instToIndex[source_inst]);
            }

        void QoaSchemInstView::
            learnInstNeighbors(internalInstIndexT source_inst_index)
            {
                assert(source_inst_index < _instToIndex.size());
                oaModInst* source_inst = _indexToInst[source_inst_index];
                DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG: Call to learnInstNeighbors, Source inst has "<<
                    source_inst->getInstTerms().getCount()<<" terms."<<endl;

                vector<oaModInst*> input_inst_list;
                getInputInsts(source_inst_index, input_inst_list);
                DebugLog::ostream()<<"SCHEMINSTNEIGHBORDEBUG: Number of inputs being learned: "<<input_inst_list.size()<<endl;
                for(vector<oaModInst*>::iterator it  = input_inst_list.begin(); 
                        it != input_inst_list.end(); ++it)
                {
                    learnInst(*it);
                    DebugLog::ostream()<<"NETADDDEBUG: "<<getNameFromOA(*it)<<","<<getNameFromOA(source_inst)<<endl;
                    addConnection(*it, source_inst);
                }

                vector<oaModInst*> output_inst_list;
                getOutputInsts(source_inst_index, output_inst_list);
                DebugLog::ostream()<<"SCHEMINSTNEIGHBORDEBUG: Number of outputs being learned: "<<output_inst_list.size()<<endl;
                for(vector<oaModInst*>::iterator it  = output_inst_list.begin(); 
                        it != output_inst_list.end(); ++it)
                {
                    learnInst(*it);
                    DebugLog::ostream()<<"NETADDDEBUG: "<<getNameFromOA(source_inst)<<","<<getNameFromOA(*it)<<endl;
                    addConnection(source_inst, *it);
                }

                return;
            }

        //Caution!  This function currently searches all insts, use it sparingly.
        //TODO:  Make the following function call an O(1) lookup, and change to getTerm
        QoaSchemTerm* QoaSchemInstView::findTerm(oa::oaModInstTerm* instTerm)
        {
            /*
               QoaSchemTerm* curr_term=0;
               for(unsigned i = 0;i<_insts.size();++i)
               {
               curr_term = _insts[i]->getTerm(term);
               if(curr_term != 0 )
               return curr_term;
               }

            //couldnt find a matching term, but architecture dictates it should be here.
            //Unless other code changes, reaching this point in the program is a bug.
            //  Therefore assert(0) .

            assert(0);
            return 0;
             */
            if( _oaInstTerm2QoaSchemInstRect.find(instTerm) != _oaInstTerm2QoaSchemInstRect.end() )
            {
                QoaSchemInstRect* SIR = _oaInstTerm2QoaSchemInstRect.find(instTerm)->second;
                DebugLog::ostream()<<" Find term of "<<getNameFromOA(instTerm->getTerm())<<" returning "<<SIR->getName()<<endl;
                return SIR->getInstTerm(instTerm);
            }
            else
                return 0;
        }

        void QoaSchemInstView::clear(void)
        {
            _center_inst = 0;
            _mouse_over_inst = 0;
            _edges.clear();
            _insts.clear();
            _instToIndex.clear();
            _indexToInst.clear();
            _instName.clear();
            _oaInstTerm2QoaSchemInstRect.clear();
            _ready = false;
        }

        void QoaSchemInstView::drawNets(GLuint displayList)
        {
            glNewList(displayList,GL_COMPILE);
            qglColor(yellow);
            glPushMatrix();
            glTranslated(0.0, 0.0, -0.5);
            for(vector< SchemEdgeT >::iterator it = _edges.begin(); it!=_edges.end(); ++it)
            {
                oaModInstTerm *u_instTerm=it->first, *v_instTerm=it->second;
                QoaSchemTerm *uqoaterm=findTerm(u_instTerm), *vqoaterm=findTerm(v_instTerm); 
                oaPoint upt=uqoaterm->getTermCenterPoint(), vpt=vqoaterm->getTermCenterPoint();

                DebugLog::ostream() << "NET DRAW DEBUG: drawing a net, summary: " << endl;
                DebugLog::ostream() <<     "\tConnection from inst: \"" << getNameFromOA( u_instTerm->getInst() ) << 
                    "\" to inst \"" << getNameFromOA( v_instTerm->getInst() ) << "\"." << endl;
                DebugLog::ostream() <<     "\tOn SchemTerms \""<< uqoaterm->getName() << "\" and \"" << 
                    vqoaterm->getName() << "\" respectively." << endl;
                DebugLog::ostream() <<     "\tfrom: (" << upt.x() << "," << upt.y() << ")" << endl; 
                DebugLog::ostream() <<     "\tto: (" << vpt.x() << "," << vpt.y() << ")" << endl;
                DebugLog::ostream() << "NET DRAW DEBUG: end summary." << endl;

                glBegin(GL_LINES);
                glVertex2d(upt.x(), upt.y());
                glVertex2d(vpt.x(), vpt.y());
                glEnd();
            }
            glPopMatrix();
            glEndList();
        }

        void QoaSchemInstView::draw(void)
        {
            if(!_ready)
            {
                DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG:  draw called, but not ready to draw. "<<endl;
                return;
            }

            for(std::vector< QoaSchemInstRect* >::iterator it=_insts.begin(); it!=_insts.end();++it)
            {
                (*it)->debug_double_check_terms_for_consistancy();
                (*it)->draw();
                GLuint dl = (*it)->getGlDisplayList();
                glCallList( dl );
            }

            //draw inst names
            qglColor(green);

            double label_z = -1.0; //<- on top of everything    
            for(internalInstIndexT i = 0;i<_insts.size();++i)
            {
                QoaSchemInstRect* scR = _insts[i];
                double curr_scr_width = scR->right() - scR->left();
                double curr_scr_height = scR->top() - scR->bottom();
                internalPointT p(static_cast<int>(scR->left() + curr_scr_width/30.0),
                        static_cast<int>(scR->top()  - curr_scr_height/90.0) );
                // window coordinates lower left

                //these are window coords
                //NOT GL coords
                QFont normalFont( "Times", int( 10.0 ) ); 

                renderText(p.x(), p.y(), label_z, 
                        _insts[i]->getFullName(),
                        normalFont );
            }

            //end draw names

            //draw term names
            for(internalInstIndexT i = 0;i<_insts.size();++i)
            {
                QoaSchemInstRect* scR = _insts[i];
                double curr_scr_width = scR->right() - scR->left();
                double curr_scr_height = scR->top()  - scR->bottom();

                internalPointT p1(static_cast<int>(scR->left() + 2*curr_scr_width/10.0),
                        static_cast<int>(scR->bottom() - 5*curr_scr_height/10.0) );
                //these are window coords NOT GL coords 
                qglColor(black);
                QFont normalFont( "Times", int( 9.0 ) ); 
                renderText(p1.x(), p1.y(), label_z, 
                        _insts[i]->getLeftTermsFullString(),
                        normalFont );

                qglColor(white);
                internalPointT p2(static_cast<int>(scR->left() + 6.0*curr_scr_width/10.0),
                        static_cast<int>(scR->top() - 5*curr_scr_height/10.0) );
                //these are window coords NOT GL coords
                renderText(p2.x(), p2.y(), label_z, 
                        _insts[i]->getRightTermsFullString(),
                        normalFont );
            }
            //end draw names

            drawNets(_edgeDisplayList);
            glCallList(_edgeDisplayList);

        }

        void QoaSchemInstView::learnInst(oa::oaModInst* inst)
        {
            if(_instToIndex.find(inst) != _instToIndex.end())
            {
                //we know about this inst already
                return;
            }
            //else
            //its a new inst, lets learn it

            internalInstIndexT new_idx = _instToIndex.size();
            _instToIndex[inst]=new_idx;
            _indexToInst.push_back(inst);
            _instName.push_back( getNameFromOA(inst) );

            QoaSchemInstRect *newSCR = new QoaSchemInstRect(_indexToInst[new_idx]);
            _insts.push_back(newSCR);
            for(unsigned i = 0; i < newSCR->getTermCount(); ++i)
            {
                //DebugLog::ostream()<<"TERM FIND DEBUG: "<<newSCR->getOAInstTerm(i)<<endl;
                //assert( _oaTerm2QoaSchemInstRect.find(newSCR->getOATerm(i)) == _oaTerm2QoaSchemInstRect.end());
                _oaInstTerm2QoaSchemInstRect.insert(
                        std::make_pair<oa::oaModInstTerm*,QoaSchemInstRect*>(newSCR->getOAInstTerm(i), newSCR) 
                        );
            }


            assert( _insts.size() == new_idx + 1 );
            assert( _insts.size() == _instToIndex.size() );
            assert( _instToIndex.size() == _indexToInst.size() );
            assert( _indexToInst.size() == _instName.size() );

        }

        void QoaSchemInstView::
            addConnection(oa::oaModInst* u, oa::oaModInst* v)
            {
                oaModInstTerm *u_instTerm=NULL, *v_instTerm=NULL;

                //find inst terms u_instTerm, v_instTerm
                oaIter<oaModInstTerm> instTermIter(u->getInstTerms());
                oaModInstTerm* instTerm;
                while( (instTerm = instTermIter.getNext()) )
                {
                    //TODO:  double check term is in _allTerms
                    oaModNet* Net_on_InstTerm = instTerm->getNet();

                    oaIter<oaModInstTerm> netIter(Net_on_InstTerm->getInstTerms());
                    oaModInstTerm* currInstTerm;
                    while( (currInstTerm = netIter.getNext()) )
                    {
                        const oaModInst* is_v = currInstTerm->getInst();
                        if( is_v == v)
                        {
                            u_instTerm = instTerm;
                            v_instTerm = currInstTerm;
                        }
                    }//end: for all inst terms on net
                }//end: for all inst terms on source

                addConnection( SchemEdgeT(u_instTerm, v_instTerm) );
            }


        void QoaSchemInstView::
            addConnection(const SchemEdgeT& edge )
            {
                _edges.push_back(edge);
            }

        void QoaSchemInstView::mousePressEvent(QMouseEvent* event)
        {
            _ready = false;

            if(event->button() == LeftButton)
            {
                if(_mouse_over_inst)
                {
                    oaModInst* new_center = _mouse_over_inst;
                    clear();
                    learnInst(new_center);
                    setCenterInst(new_center);
                    learnInstNeighbors(new_center);
                    setInstLocationsCenterInstNeighborStyle();
                }
            }

            _ready = true;
        }


        void QoaSchemInstView::_windowToGLCoords(oa::oaPoint& p)
        {
            double newX = p.x() * _SCV_gl_coord_width / width();
            double newY = p.y() * _SCV_gl_coord_height / height();

            //OpenGL has a cartesian coordinate system, with higher values growing upward.
            //Qt has a cartesian system with higher values growing downward.
            //The next line inverts the growth, to convert between these two methods.
            newY = _SCV_gl_coord_height - newY;

            p.set(static_cast<int>(newX), static_cast<int>(newY) );
        }

        void QoaSchemInstView::mouseMoveEvent( QMouseEvent * e )
        {
            mouseOver( e->pos().x(), e->pos().y() );
        }

        void QoaSchemInstView::mouseOver(int x, int y)
        {
            oaPoint p(x,y);
            //convert p from window coordinates to gl coordinates.
            _windowToGLCoords(p);
            oaModInst* old_over = _mouse_over_inst;
            _mouse_over_inst=0;
            for(internalInstIndexT i = 0; i < _insts.size() ; ++i)
            {
                if(_insts[i]->contains(p))
                {
                    _mouse_over_inst = _indexToInst[i];
                    //DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG: Setting the mouse over inst to \""<< _instName[i] << "\"" <<endl;
                    _insts[i]->setSelected(true);
                }
                else
                    _insts[i]->setSelected(false);
            }

            //dont keep redrawing if nothing is changing
            if(_mouse_over_inst != old_over)    updateGL();                      
        }

        void QoaSchemInstView::initializeGL(void)
        {
            qglClearColor(black);
            glShadeModel(GL_FLAT);
            glEnable(GL_DEPTH_TEST);
            glEnable(GL_CULL_FACE);
        }

        void QoaSchemInstView::resizeGL(int width, int height)
        {
            DebugLog::ostream() << "SCHEMINSTVIEWDEBUG: Resizing GL (" << width << "," << height << ")" <<endl;

            //set the size of this widget, and the glViewport
            //to be the same size as the QScrollView viewport
            resize(width,height);
            glViewport(0,0,width,height);

            //The following math creates an equivalent clipping volume
            //to glOrtho except in true 3d

            //begin clipping volume creation
            glMatrixMode(GL_PROJECTION);
            glLoadIdentity();
            GLfloat fAspect = GLfloat( _SCV_gl_coord_width ) / GLfloat( _SCV_gl_coord_height );
            gluPerspective(_SCV_glFovy, fAspect, _SCV_gl_near_clipping_z, _SCV_gl_far_clipping_z);
            double numerator_z = sqrt( _SCV_gl_coord_width*_SCV_gl_coord_width +
                    _SCV_gl_coord_height*_SCV_gl_coord_height );
            double denominator_z = tan( _SCV_glFovy * 3.14159/180.0);
            glTranslated(-_SCV_gl_coord_width/2.0,
                    -_SCV_gl_coord_height/2.0,
                    -numerator_z/denominator_z - 7.0 );  //7.0 == fudge
            //end clipping volume creation

            glMatrixMode(GL_MODELVIEW);
            glLoadIdentity();
            glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        }

        void QoaSchemInstView::paintGL()
        {
            glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
            DebugLog::ostream()<<"SCHEMINSTTERMVIEW: Painting gl..."<<endl;
            draw();
            //draw_test_pattern();
            //draw_test_pattern2();
        }

        void QoaSchemInstView::draw_test_pattern2()
        {
            GLuint dispList = glGenLists(1);
            glNewList( dispList, GL_COMPILE );
            glColor3d( 0, 1.0, 0 ); //<-- full green
            glRectd( 5.0, 5.0, 95.0, 95.0 );
            glEndList();

            glCallList(dispList);

            glDeleteLists( dispList, 1 );
        }

        void QoaSchemInstView::draw_test_pattern()
        {
            qglColor(green);
            glRectd( 5.0, 5.0, 95.0, 95.0 );
        }

        void QoaSchemInstView::setInstLocationsOneCenterInstStyle(void)
        {
            internalInstIndexT center_idx = _instToIndex[_center_inst];
            std::vector<QoaSchemInstRect*> inst_list;
            inst_list.push_back( _insts[center_idx] );
            vector<internalTermIndexT> inst_indices;
            inst_indices.push_back(center_idx);
            double x_column_position = 50.0;
            double top_edge = 100.0;
            double available_height = 100.0;
            double whitespace = 0.666667;
            setInstLocations(inst_list,inst_indices,
                    x_column_position, top_edge,
                    available_height, whitespace);
        }

        //Caution! inst_vec will be cleared.
        //It will subsequently filled with the input and bidirectional insts 
        //    connected to inst with index source_inst_index
        void QoaSchemInstView::getInputInsts(internalInstIndexT source_inst_index,
                std::vector<oa::oaModInst*>& inst_list)
        {
            assert(source_inst_index < _instToIndex.size());
            const oaModInst* source_inst = _indexToInst[source_inst_index];
            DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG: Call to getInputInsts, Source inst has "<<
                source_inst->getInstTerms().getCount()<<" terms."<<endl;

            inst_list.clear();

            oaIter<oaModInstTerm> instTermIter(source_inst->getInstTerms());
            oaModInstTerm* instTerm;
            DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG: Exploring source inst's neighbors"<<endl;
            while( (instTerm = instTermIter.getNext()) )
            {

                oaModTerm* Term_on_InstTerm = instTerm->getTerm();
                oaModNet* Net_on_InstTerm = instTerm->getNet();
                DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG: Adding insts connected by \""<<getNameFromOA(Term_on_InstTerm)<<
                    "\", which has dir: "<< Term_on_InstTerm->getTermType()<<endl;

                //input and bidirectional insts
                if(Term_on_InstTerm->getTermType() == oacInputTermType
                        || Term_on_InstTerm->getTermType() == oacInputOutputTermType)
                {
                    oaIter<oaModInstTerm> netIter(Net_on_InstTerm->getInstTerms());
                    oaModInstTerm* currInstTerm;
                    while( (currInstTerm = netIter.getNext()) )
                    {
                        oaModTerm* Term_on_currNet = currInstTerm->getTerm();
                        if(!Term_on_currNet) continue;
                        oaTermType t = Term_on_currNet->getTermType();
                        if(t==oacOutputTermType || t == oacInputOutputTermType)
                        {
                            oaModInst* currInst = currInstTerm->getInst();
                            if(currInst != source_inst)
                                inst_list.push_back(currInst);
                        }
                        else
                        {
                            //; //uncomment this semi if you comment this error message (for warning purposes)
                            DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG:  ignoring inst IN->IN"<<endl;
                        }
                    }//end: for all inst terms on net
                }//end: if input
            }//end: for all inst terms on source
            DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG:  getInputInsts found "<<inst_list.size()<<" proper input insts."<<endl;
            reverse(inst_list.begin(), inst_list.end());
        }


        //Caution! inst_vec and inst_indices will be cleared.
        //It will subsequently filled with the input and bidirectional insts 
        //    connected to inst with index source_inst_index
        void QoaSchemInstView::getInputInsts(internalInstIndexT source_inst_index,
                std::vector<QoaSchemInstRect*>& inst_list,
                std::vector<internalInstIndexT>& inst_indices)
        {
            assert(source_inst_index < _instToIndex.size());
            const oaModInst* source_inst = _indexToInst[source_inst_index];
            DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG: Call to getInputInsts, Source inst has "<<
                source_inst->getInstTerms().getCount()<<" terms."<<endl;

            inst_list.clear();
            inst_indices.clear();

            vector<oaModInst*> inputInsts;

            getInputInsts(source_inst_index, inputInsts);

            for(vector<oaModInst*>::iterator it = inputInsts.begin(); it!=inputInsts.end(); ++it)
            {
                learnInst(*it);
                inst_list.push_back(_insts[_instToIndex[*it]]);
                inst_indices.push_back(_instToIndex[*it]);
            }

        }

        //Caution! inst_vec will be cleared.
        //It will subsequently filled with the output insts 
        //    connected to inst with index source_inst_index    
        void QoaSchemInstView::getOutputInsts(internalInstIndexT source_inst_index,
                std::vector<oa::oaModInst*>& inst_list)
        {
            assert(source_inst_index < _instToIndex.size());
            const oaModInst* source_inst = _indexToInst[source_inst_index];

            inst_list.clear();

            DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG: Call to getOutputInsts, Source inst has "<<
                source_inst->getInstTerms().getCount()<<" terms."<<endl;

            oaIter<oaModInstTerm> instTermIter(source_inst->getInstTerms());
            oaModInstTerm* instTerm;
            DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG: Exploring source inst's neighbors"<<endl;
            while( (instTerm = instTermIter.getNext()) )
            {

                oaModTerm* Term_on_InstTerm = instTerm->getTerm();
                oaModNet* Net_on_InstTerm = instTerm->getNet();
                DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG: Adding insts connected by \""<<getNameFromOA(Term_on_InstTerm)<<
                    "\", which has dir: "<< Term_on_InstTerm->getTermType()<<endl;

                //input and bidirectional insts
                if(Term_on_InstTerm->getTermType() == oacOutputTermType)
                {
                    oaIter<oaModInstTerm> netIter(Net_on_InstTerm->getInstTerms());
                    oaModInstTerm* currInstTerm;
                    while( (currInstTerm = netIter.getNext()) )
                    {
                        oaModTerm* Term_on_currNet = currInstTerm->getTerm();
                        oaTermType t = Term_on_currNet->getTermType();
                        if(t==oacInputTermType || t == oacInputOutputTermType)
                        {
                            inst_list.push_back(currInstTerm->getInst());
                        }
                        else
                        {
                            //; //uncomment this semi if you comment this error message (for warning purposes)
                            DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG:  ignoring inst OUT->OUT"<<endl;
                        }
                    }//end: for all inst terms on net
                }//end: if output
            }//end: for all inst terms on source
            DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG:  getInputInsts found "<<inst_list.size()<<" proper output insts."<<endl;
        }

        //Caution! inst_vec and inst_indices will be cleared.
        //It will subsequently filled with the output insts 
        //    connected to inst with index source_inst_index  
        void QoaSchemInstView::getOutputInsts(internalInstIndexT source_inst_index,
                std::vector<QoaSchemInstRect*>& inst_list,
                std::vector<internalInstIndexT>& inst_indices)
        {
            assert(source_inst_index < _instToIndex.size());
            const oaModInst* source_inst = _indexToInst[source_inst_index];

            DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG: Call to getOutputInsts, Source inst has "<<
                source_inst->getInstTerms().getCount()<<" terms."<<endl;

            inst_list.clear();
            inst_indices.clear();

            vector<oaModInst*> outputInsts;

            getOutputInsts(source_inst_index, outputInsts);

            for(vector<oaModInst*>::iterator it = outputInsts.begin(); it!=outputInsts.end(); ++it)
            {
                learnInst(*it);
                inst_list.push_back(_insts[_instToIndex[*it]]);
                inst_indices.push_back(_instToIndex[*it]);
            }
        }

        void QoaSchemInstView::setInstLocationsCenterInstNeighborStyle(void)
        {
            //We are working toward levelized circiut schematics
            //level 0 == left
            //level 1 == center
            //level 2 == right
            DebugLog::ostream()<<"SCHEMINSTVIEWDEBUG:  setting locations of center and neighbor insts."<<endl;
            internalInstIndexT center_idx = _instToIndex[_center_inst];

            std::vector<QoaSchemInstRect*> level_0_inst_list;
            vector<internalTermIndexT> level_0_inst_indices;
            getInputInsts(center_idx,level_0_inst_list,level_0_inst_indices);
            double level_0_x_column_position = 10.0;
            double level_0_top_edge = 100.0;
            double level_0_available_height = 100.0;
            double level_0_whitespace = 0.25;
            setInstLocations(level_0_inst_list,level_0_inst_indices,
                    level_0_x_column_position, level_0_top_edge,
                    level_0_available_height, level_0_whitespace);


            std::vector<QoaSchemInstRect*> level_1_inst_list;
            level_1_inst_list.push_back( _insts[center_idx] );
            vector<internalTermIndexT> level_1_inst_indices;
            level_1_inst_indices.push_back(center_idx);
            double level_1_x_column_position = 50.0;
            double level_1_top_edge = 100.0;
            double level_1_available_height = 100.0;
            double level_1_whitespace = 0.666667;
            setInstLocations(level_1_inst_list,level_1_inst_indices,
                    level_1_x_column_position, level_1_top_edge,
                    level_1_available_height, level_1_whitespace);

            std::vector<QoaSchemInstRect*> level_2_inst_list;
            vector<internalTermIndexT> level_2_inst_indices;
            getOutputInsts(center_idx,level_2_inst_list,level_2_inst_indices);
            double level_2_x_column_position = 90.0;
            double level_2_top_edge = 100.0;
            double level_2_available_height = 100.0;
            double level_2_whitespace = 0.25;
            setInstLocations(level_2_inst_list,level_2_inst_indices,
                    level_2_x_column_position, level_2_top_edge,
                    level_2_available_height, level_2_whitespace);

            return;
        }

        //set vertical spacing for uniform whitespace
        //Whitespace is expressed in a percentage normalized to the range 0-1, ex: 80% == 0.8
        void QoaSchemInstView::setInstLocations(
                const std::vector<QoaSchemInstRect*>& inst_vec,
                const std::vector<internalTermIndexT>& inst_indices,
                double x_column_position,
                double top_edge,
                double available_height,
                double whitespace)
        {
            //whitespace is a number [0,1)
            assert( whitespace >=0 && whitespace < 1);

            DebugLog::ostream() << "SCHEMINSTVIEWDEBUG: call to setInstLocations, summary: " << endl;
            DebugLog::ostream() <<    "\tVec Size: " << inst_vec.size() << endl;
            DebugLog::ostream() <<    "\tIndices count: " << inst_indices.size() << endl;
            DebugLog::ostream() <<    "\tColumn x: " << x_column_position << " Top edge: " << top_edge << endl;
            DebugLog::ostream() <<    "\tAvailable height: " << available_height << " Whitespace: " << whitespace << endl;
            DebugLog::ostream() << "SCHEMINSTVIEWDEBUG: end summary." << endl;
            assert(inst_vec.size() == inst_indices.size());

            //equally spaced among portion of the height with spacing accounting for specified whitespace
            //the height of each rect is the portion of available height
            //that ISNT whitespace (1-ws) divided by the number of insts
            //place the available whitespace equally above and below each inst (half above, half below)
            double inst_height = available_height*(1-whitespace) / inst_vec.size() ;
            double y_incr = (available_height) / (inst_vec.size());
            double accum = 0.5 * (available_height*whitespace) / inst_vec.size();
            double inst_width = 20.0; //TODO fix this hardcodedness
            DebugLog::ostream() <<"SCHEMINSTVIEWDEBUG:  y_incr: "<<y_incr<<" accum: "<<accum<<endl;



            for(unsigned it = 0; it < inst_indices.size(); ++it )
            {
                internalTermIndexT inst_idx = inst_indices[it];
                double curr_top_edge = top_edge - accum;
                double l = x_column_position-inst_width/2.0,
                       t = curr_top_edge,
                       r = x_column_position+inst_width/2.0,
                       b = curr_top_edge-inst_height;

                oa::oaBox instLoc(static_cast<int>(l),
                        static_cast<int>(b),
                        static_cast<int>(r), 
                        static_cast<int>(t) );
                DebugLog::ostream() << instLoc ;

                DebugLog::ostream() << "SCHEMINSTVIEWDEBUG: Setting Inst location (l,t,r,b): " << instLoc << endl;
                _insts[inst_idx]->setRect( instLoc );
                accum += y_incr;
            }
        }

        std::ostream& operator<<(std::ostream& os, QoaSchemInstView)
        {
            //TODO
            return os;
        }

    }; //end namespace Qoag

}

