/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: David Papa <iamyou@umich.edu>

ChangeLog:
2004-09-15: ChangeLog started
*/

#include <limits>
#include <iostream>

#include "QoagCommon.h"

using std::numeric_limits;
using std::cerr;
using std::endl;

using oa::oaPoint;

namespace oagBazaar
{

    namespace Qoag
    {
        const int QoaTermGeomCenter=-1;
        const int QoaTermGeomLeft=0;
        const int QoaTermGeomRight=1;

        const double _L_txt_margin=0.05;
        const double _R_txt_margin=0.05;
        const double _L_term_margin=0.10;
        const double _R_term_margin=0.10;
        const double _txt_ht=0.10;
        const QColor _inst_border_color=QColor(Qt::darkRed);
        const QColor _inst_foreground_color=QColor(Qt::darkMagenta);
        const QColor _selected_inst_border_color=QColor(Qt::red);
        const QColor _selected_inst_foreground_color=QColor(Qt::magenta);
        const double _border_size=0.05;
        const double _term_width=0.10;
        const QColor _term_border_color=QColor(Qt::green);
        const QColor _term_foreground_color=QColor(Qt::cyan);

        const int QoaSCVCenter = -1;
        const int QoaSCVLeft = 0;
        const int QoaSCVRight = 1;
        const double _SCV_gl_coord_height = 100.0;
        const double _SCV_gl_coord_width = 100.0;
        const double _SCV_gl_near_clipping_z = -1.0;
        const double _SCV_gl_far_clipping_z = 40.0;
        const double _SCV_glFovy = 60.0;

        bool Cmp_std_pair_internalInstIndexT_internalInstIndexT::
            operator()(std::pair<internalInstIndexT,internalInstIndexT> a,
                    std::pair<internalInstIndexT,internalInstIndexT> b)
            {
                if(a.first != b.first)
                {
                    return a.first < b.first;
                }
                else
                {
                    return a.second < a.second;
                }
            }

        template <> std::string getNameFromOA<oa::oaDesign*>(oa::oaDesign* i)
        {
            oa::oaNativeNS ns;
            oa::oaString name;
            i->getCellName(ns,name);
            return std::string(static_cast<const char*>(name) );
        }


        //frame_width and frame_height are expressed as a portion of 1.0
        void QoaGlFrameRectd(QGLWidget* w, const oa::oaBox& bbox, 
                double frame_width, double frame_height, 
                QColor border_color, QColor foreground_color)
        {
            //DebugLog::ostream()<<"FRAMEDEBUG:  box(l,b,r,t):"
            //                   << bbox.xMin<<","
            //                   << bbox.yMin<<","
            //                   << bbox.xMax<<","
            //                   << bbox.yMax<<","<<endl;

            double width = bbox.right()-bbox.left();
            double height = bbox.top()-bbox.bottom();

            double  x1 = bbox.left()+width*frame_width,
                    x2 = bbox.right()-width*frame_width,
                    y1 = bbox.bottom()+height*frame_height,
                    y2 = bbox.top()-height*frame_height;

            //draw base frame rectangle
            w->qglColor(border_color);
            glRectd(bbox.left(),bbox.bottom(),bbox.right(),bbox.top());
            //draw the foreground of the rectangle
            w->qglColor(foreground_color);
            glRectd(x1, y1, x2, y2);
            //end draw the foreground of the rectangle
            assert(0);
        }

        void QoaGlFrameRectd(QGLWidget*, const oa::oaBox& bbox, 
                double frame_width, double frame_height, 
                const std::vector<double>& border_color, const std::vector<double>& foreground_color)
        {
            // DebugLog::ostream()<<"FRAMEDEBUG:  box(l,b,r,t):"
            //            << bbox.xMin << ","
            //            << bbox.yMin << ","
            //            << bbox.xMax << ","
            //            << bbox.yMax << ","<<endl;
            assert(border_color.size()==3);
            assert(foreground_color.size()==3);
            double width = bbox.right()-bbox.left();
            double height = bbox.top()-bbox.bottom();

            double  x1 = bbox.left()+width*frame_width,
                    x2 = bbox.right()-width*frame_width,
                    y1 = bbox.bottom()+height*frame_height,
                    y2 = bbox.top()-height*frame_height;

            //glPushMatrix();
            //glLoadIdentity();
            glTranslated(0.0, 0.0, -0.01);
            //draw base frame rectangle
            glColor3d(border_color[0], border_color[1], border_color[2]);
            glRectd(bbox.left(),bbox.bottom(),bbox.right(),bbox.top());
            //draw the foreground of the rectangle
            glTranslated(0.0, 0.0, 0.01);
            glColor3d(foreground_color[0], foreground_color[1], foreground_color[2]);
            glRectd(x1, y1, x2, y2);
            //end draw the foreground of the rectangle
            //glPopMatrix();

        }

        //this overload compiles a GL display list, without drawing
        void QoaGlFrameRectd(GLuint displayList, const oa::oaBox& bbox, 
                double frame_width, double frame_height,
                const std::vector<double>& border_color, const std::vector<double>& foreground_color)
        {
            assert(border_color.size()==3);
            assert(foreground_color.size()==3);
            double width = bbox.right()-bbox.left();
            double height = bbox.top()-bbox.bottom();

            //compute dimensions of inner rectangle
            double  x1 = bbox.left() + width*frame_width,
                    x2 = bbox.right() - width*frame_width,
                    y1 = bbox.bottom() + height*frame_height,
                    y2 = bbox.top() - height*frame_height;

            glNewList(displayList,GL_COMPILE);
            //draw base framed rectangle, stupid flat 0.0 version
            glColor3d(border_color[0], border_color[1], border_color[2]);
            glRectd(bbox.left(), bbox.bottom(), bbox.right(), bbox.top());

            glPushMatrix();

            //draw the foreground of the rectangle, stupid flat 0.0 version
            glTranslated(0.0, 0.0, -0.1);
            glColor3d(foreground_color[0], foreground_color[1], foreground_color[2]);
            glRectd(x1, y1, x2, y2);

            glPopMatrix();
            glEndList();


            ////test pattern
            //glNewList(displayList,GL_COMPILE);
            //  glColor3d(0.0,0.5,0.5);
            //  glRectd(5.0,5.0,95.0,95.0);
            //glEndList();
        }

        void QoaGlRectd(QGLWidget* glw, 
                        const oa::oaBox& box, 
                        const std::vector<double>& foreground_color)
        {
            static_cast<void>(glw); //get rid of compiler unused warning
            // DebugLog::ostream()<<"FRAMEDEBUG:  box(l,b,r,t):"
            //            << bbox.xMin << ","
            //            << bbox.yMin << ","
            //            << bbox.xMax << ","
            //            << bbox.yMax << ","<<endl;
            assert(foreground_color.size()==3);

            double  x1 = box.left(),
                    x2 = box.right(),
                    y1 = box.bottom(),
                    y2 = box.top();

            //draw the foreground of the rectangle
            glColor3d(foreground_color[0], foreground_color[1], foreground_color[2]);
            glRectd(x1, y1, x2, y2);
            //end draw the foreground of the rectangle
        }

        void setVectorColor(std::vector<double>& v, double r, double g, double b)
        {
            v.resize(3);
            v[0]=r;
            v[1]=g;
            v[2]=b;
        }

        void _checkBoundsInit(const oa::oaBox& bb, const std::string& IDname )
        {
            if(bb.left() == numeric_limits<int>::max() || bb.bottom() == numeric_limits<int>::max() ||
                    bb.right() == numeric_limits<int>::max() || bb.right() == numeric_limits<int>::max())
            {
                cerr<<"Attempt to draw "<< IDname << " with uninitialized location!  Aborting..."<<endl;
                assert(0);
            }
        }

        oa::oaPoint transformPoint(const oa::oaPoint& POld, 
                const oa::oaBox& BBoxOld, 
                const oa::oaBox& BBoxNew)
        {
            double x_pct = (POld.x() - BBoxOld.left()) / (BBoxOld.right() - BBoxOld.left());
            double y_pct = (POld.y() - BBoxOld.bottom()) / (BBoxOld.top() - BBoxOld.bottom());


            double rval_X = (BBoxNew.right() - BBoxNew.left()) * x_pct + BBoxNew.left();
            double rval_Y = (BBoxNew.top() - BBoxNew.bottom()) * y_pct + BBoxNew.right();

            oaPoint rval(static_cast<int>(rval_X), static_cast<int>(rval_Y) );

            return rval;
        }


        oa::oaBox getPlacementBBox(const oa::oaInst * const inst)
        {
            oa::oaSnapBoundary *sb = oa::oaSnapBoundary::find(inst->getMaster()->getTopBlock());
            oa::oaBox returnVal;
            sb->getBBox(returnVal);
            oa::oaTransform instTrans;
            inst->getTransform(instTrans);
            returnVal.transform(instTrans);
            return returnVal;
        }

        QoagTCLAction::QoagTCLAction( QObject* parent, const char* name):
           QAction(parent, name)
        {
            connectActivatedToExecute();
        }

        QoagTCLAction::QoagTCLAction( const QString& menuText, QKeySequence accel, QObject* parent, const char* name):
           QAction(menuText, accel, parent, name)
        {
            connectActivatedToExecute();
        }

        QoagTCLAction::QoagTCLAction( const QString& text, const QIconSet& icon, const QString& menuText, QKeySequence accel, QObject* parent, const char * name, bool toggle):
           QAction(text, icon, menuText, accel, parent, name, toggle)
        {
            connectActivatedToExecute();
        }

        QoagTCLAction::QoagTCLAction( const QString& text, const QString& menuText, QKeySequence accel, QObject* parent, const char* name, bool toggle):
            QAction(text, menuText, accel, parent, name, toggle)
        {
            connectActivatedToExecute();
        }

        QoagTCLAction::QoagTCLAction( QObject* parent, const char* name, bool toggle):
            QAction(parent, name, toggle)
        {
            connectActivatedToExecute();
        }

        void QoagTCLAction::connectActivatedToExecute(void)
        {
            connect(this, SIGNAL( activated() ), this, SLOT( emitExecute() ));

        }

        void QoagTCLAction::emitExecute()
        {
           emit execute(this);
        }


    };//end namespace Qoag

}
