/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */
#include <iostream>
#include <iomanip>

#include <qtable.h>
#include <qlineedit.h>

#include "simulatorPlugIn.h"
#include "SimEqui/oagResyn.h"
#include "runSimulation.h"
#include "simulationDataTable.h"
#include "designEquivalenceFormT.h"
#include "similarityLayoutViewFormT.h"
#include "netEquivalenceFormT.h"
#include "equivalenceCheckResultsFormT.h"
#include "SimEqui/oagResynUserIntf.h"
#include "QoagCommon/QoagCommonTempl.h"
#include "runCycleSimulation.h"
#include "similarityLayoutViewWnd.h"

using std::vector;
using std::string;

using oa::oaNet;

int RunCycleSimulatorPlugInCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
        //remove compiler unused warnings
        //static_cast<void>(clientData);
        static_cast<void>(interp);
        static_cast<void>(argc);
        static_cast<void>(argv);

        SimulatorPlugIn* plugInPtr = static_cast<SimulatorPlugIn*>(clientData);

        //cout<<"Hello from the PlugIn World!"<<endl;
        runCycleSimulationFormT* runSimulationDialog = new runCycleSimulationFormT;
        runSimulationDialog->readLibs();
        plugInPtr->getAPI()->connectCommand(runSimulationDialog, SIGNAL( runSimulationPressed(const std::string& ) ) );
        plugInPtr->getAPI()->giveDialog(runSimulationDialog);
        //oagResyn::simulationMain(argc, argv);

        return TCL_OK;
}

int RunSimulatorPlugInCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
        //remove compiler unused warnings
        //static_cast<void>(clientData);
        static_cast<void>(interp);
        static_cast<void>(argc);
        static_cast<void>(argv);

        SimulatorPlugIn* plugInPtr = static_cast<SimulatorPlugIn*>(clientData);

        //cout<<"Hello from the PlugIn World!"<<endl;
        runSimulationFormT* runSimulationDialog = new runSimulationFormT;
        runSimulationDialog->readLibs();
        plugInPtr->getAPI()->connectCommand(runSimulationDialog, SIGNAL( runSimulationPressed(const std::string& ) ) );
        plugInPtr->getAPI()->giveDialog(runSimulationDialog);
        //oagResyn::simulationMain(argc, argv);

        return TCL_OK;
}

int SimulatorPlugInCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
    //remove compiler unused warnings
    //static_cast<void>(clientData);
    static_cast<void>(interp);
    //static_cast<void>(argc);
    //static_cast<void>(argv);

   SimulatorPlugIn* plugInPtr = static_cast<SimulatorPlugIn*>(clientData);


   // oagResyn::simulationMain(argc, argv);


      int argc_mod = argc;
      bool have_cycles = false;
   if(argv[argc_mod - 2] == std::string("-cycles"))
   {
     argc_mod -= 2; //cut the cycles off
     have_cycles=true;
   }
   std::auto_ptr<oagResyn::simulatorInterface> simulator;
   try
   {
      simulator = oagResyn::getSimulator(argc_mod, argv);
   }
   catch(oagResyn::oagResynErr& e)
   {
      cerr<<"Error building simulator object in Bazaar:"<<endl<<e.what()<<endl;
      return TCL_ERROR;
   }
   catch (oa::oaException &excp)
   {
       cerr<<"OpenAcess exception building simulator object, Error: "<<endl<<
          static_cast<const char *>(excp.getMsg())<<endl;
      return TCL_ERROR;
   }


   
   unsigned num_cycles = 1;
   if(have_cycles)
   {
     std::stringstream num;
     num << argv[argc-1];
     num >> num_cycles;
   }


   vector<oa::oaNet*> outputs;
   simulator->getOutputs(outputs);
   SimulationDataTableT* simDataTable = new SimulationDataTableT();
   simDataTable->setSimulatorPI(plugInPtr);
   simDataTable->simulationData->setNumRows(outputs.size());
   vector<string> outputValues(outputs.size());

   for(unsigned i = 0; i < num_cycles; ++i)
   {
       simulator->doRandomSimulation();
       oagResyn::simulatorInterface::SimulationVector simulationValue;
       for(unsigned  i = 0; i < outputs.size(); ++i)
       {
           simulator->getVector(outputs[i], simulationValue);
           std::stringstream num;
           num  << std::setfill('0') << std::setw(8) << std::hex << simulationValue;
           outputValues[i] += num.str();
       }
   }

   for(unsigned  i = 0; i < outputs.size(); ++i)
   {
      simDataTable->simulationData->setText(i, 0, QString(oagBazaar::Qoag::getNameFromOA(outputs[i]).c_str()));
      simDataTable->simulationData->setText(i, 1, QString( string("0x"+outputValues[i]).c_str()));

   }
   simDataTable->simulationData->adjustColumn(1);
   
   simDataTable->setSimulator(simulator.release());

   plugInPtr->getAPI()->giveWidget(simDataTable);

   return TCL_OK;
}

int CheckDesignEquivalenceCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
    //remove compiler unused warnings
    //static_cast<void>(clientData);
    static_cast<void>(interp);
    static_cast<void>(argc);
    static_cast<void>(argv);

    SimulatorPlugIn* plugInPtr = static_cast<SimulatorPlugIn*>(clientData);

        designEquivalenceFormT* designEquivalenceDialog = new designEquivalenceFormT;
        designEquivalenceDialog->readLibs1();
        designEquivalenceDialog->readLibs2();
        plugInPtr->getAPI()->connectCommand(designEquivalenceDialog, SIGNAL( checkPressed(const std::string& ) ) );
        plugInPtr->getAPI()->giveDialog(designEquivalenceDialog);


   return TCL_OK;
}


int CheckNetEquivalenceCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
    //remove compiler unused warnings
    //static_cast<void>(clientData);
    static_cast<void>(interp);
    static_cast<void>(argc);
    static_cast<void>(argv);

    SimulatorPlugIn* plugInPtr = static_cast<SimulatorPlugIn*>(clientData);

    netEquivalenceFormT* netEquivalenceDialog = new netEquivalenceFormT;
    netEquivalenceDialog->readLibs();
    plugInPtr->getAPI()->connectCommand(netEquivalenceDialog, SIGNAL( checkPressed(const std::string& ) ) );
    plugInPtr->getAPI()->giveDialog(netEquivalenceDialog);

   return TCL_OK;
}

int CheckEquivalenceCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
    //remove compiler unused warnings
    //static_cast<void>(clientData);
    static_cast<void>(interp);
    //static_cast<void>(argc);
    //static_cast<void>(argv);

    SimulatorPlugIn* plugInPtr = static_cast<SimulatorPlugIn*>(clientData);

   std::auto_ptr<oagResyn::equiCheckOutputInterface> equichecker;
   try
   {
      equichecker= oagResyn::getEquiResult(argc, argv);
   }
   catch(oagResyn::oagResynErr& e)
   {
      cerr<<"Error building equivalence checker object in Bazaar:"<<endl<<e.what()<<endl;
      return TCL_ERROR;
   }
   catch (oa::oaException &excp)
   {
       cerr<<"OpenAcess exception building equivalence checker object, Error: "<<endl<<
          static_cast<const char *>(excp.getMsg())<<endl;
      return TCL_ERROR;
   }

     
    
    equivalenceCheckResultsFormT* equivalenceCheckResults = new equivalenceCheckResultsFormT;

    equivalenceCheckResults->lib1Edit->setText(QString(equichecker->getLib1().c_str()));
    equivalenceCheckResults->lib2Edit->setText(QString(equichecker->getLib2().c_str()));
    equivalenceCheckResults->cell1Edit->setText(QString(equichecker->getCell1().c_str()));
    equivalenceCheckResults->cell2Edit->setText(QString(equichecker->getCell2().c_str()));
    equivalenceCheckResults->net1Edit->setText(QString(equichecker->getNet1().c_str()));
    equivalenceCheckResults->net2Edit->setText(QString(equichecker->getNet2().c_str()));

    equivalenceCheckResults->equivalentEdit->setText( equichecker->isEquivalent()?"true":"false" );

    if(equichecker->isTwoDesign())
    {
        std::stringstream similarityFss;
        similarityFss<< equichecker->getSimilarityFactor()<<"%";
        equivalenceCheckResults->similarityFactorEdit->setText( similarityFss.str().c_str() );
    }
    else
        equivalenceCheckResults->similarityFactorEdit->setText("Undefined");

    const std::vector<oagResyn::SimulationVector>& counterExample = equichecker->getCounterExample();
    const std::vector<oagResyn::SimulationVector>& counterExampleOutput1 = equichecker->getCounterExampleOutput1();
    const std::vector<oagResyn::SimulationVector>& counterExampleOutput2 = equichecker->getCounterExampleOutput2();
    const std::vector<oa::oaNet*>& primaryInputs1 = equichecker->getPrimaryInputs1();
    const std::vector<oa::oaNet*>& primaryInputs2 = equichecker->getPrimaryInputs2();
    const std::vector<oa::oaNet*>& primaryOutputs1 = equichecker->getPrimaryOutputs1();
    const std::vector<oa::oaNet*>& primaryOutputs2 = equichecker->getPrimaryOutputs2();

    equivalenceCheckResults->counterExampleTable->setNumCols(7);
    unsigned numRows = std::max(counterExample.size(), counterExampleOutput1.size());
    numRows = std::max(numRows, counterExampleOutput2.size());
    numRows = std::max(numRows, primaryInputs1.size());
    numRows = std::max(numRows, primaryInputs2.size());
    numRows = std::max(numRows, primaryOutputs1.size());
    numRows = std::max(numRows, primaryOutputs2.size());
    equivalenceCheckResults->counterExampleTable->setNumRows(numRows);

    for(unsigned i = 0; i < primaryInputs1.size(); ++i)
      equivalenceCheckResults->counterExampleTable->setText(i, 0, oagBazaar::Qoag::getNameFromOA(primaryInputs1[i]).c_str());
    for(unsigned i = 0; i < primaryInputs2.size(); ++i)
      equivalenceCheckResults->counterExampleTable->setText(i, 1, oagBazaar::Qoag::getNameFromOA(primaryInputs2[i]).c_str());
    for(unsigned i = 0; i < counterExample.size(); ++i)
    {
      std::stringstream ss;
      ss<<"0x"<< std::hex << counterExample[i];
      equivalenceCheckResults->counterExampleTable->setText(i, 2, ss.str().c_str());
    }
    for(unsigned i = 0; i < primaryOutputs1.size(); ++i)
      equivalenceCheckResults->counterExampleTable->setText(i, 3, oagBazaar::Qoag::getNameFromOA(primaryOutputs1[i]).c_str());
    for(unsigned i = 0; i < counterExampleOutput1.size(); ++i)
    {
      std::stringstream ss;
      ss<<"0x"<< std::hex << counterExampleOutput1[i];
      equivalenceCheckResults->counterExampleTable->setText(i, 4, ss.str().c_str());
    }

    for(unsigned i = 0; i < primaryOutputs2.size(); ++i)
      equivalenceCheckResults->counterExampleTable->setText(i, 5, oagBazaar::Qoag::getNameFromOA(primaryOutputs2[i]).c_str());
    for(unsigned i = 0; i < counterExampleOutput2.size(); ++i)
    {
      std::stringstream ss;
      ss<<"0x"<< std::hex << counterExampleOutput2[i];
      equivalenceCheckResults->counterExampleTable->setText(i, 6, ss.str().c_str());
    }

    plugInPtr->getAPI()->giveWidget(equivalenceCheckResults);

   return TCL_OK;
}

oagBazaar::oagBazaarPlugInInterface * newPlugIn(void)
{
    SimulatorPlugIn* plugIn = new SimulatorPlugIn; //memory leak!
    return plugIn;
}

int SimilarityLayoutViewCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
    //remove compiler unused warnings
    //static_cast<void>(clientData);
    static_cast<void>(interp);
    static_cast<void>(argc);
    static_cast<void>(argv);

    SimulatorPlugIn* plugInPtr = static_cast<SimulatorPlugIn*>(clientData);

    similarityLayoutViewFormT* similarityLayoutViewDialog = new similarityLayoutViewFormT;
    similarityLayoutViewDialog->readLibs1();
    similarityLayoutViewDialog->readLibs2();
    plugInPtr->getAPI()->connectCommand(similarityLayoutViewDialog, SIGNAL( checkPressed(const std::string& ) ) );
    plugInPtr->getAPI()->giveDialog(similarityLayoutViewDialog);


   return TCL_OK;
}


int SimilarityLayoutCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
    //remove compiler unused warnings
    //static_cast<void>(clientData);
    static_cast<void>(interp);
    //static_cast<void>(argc);
    //static_cast<void>(argv);

    //if (argc != 5)
    //{
    //    cerr<<"Usage: "<<argv[0]<<" library1 cell1 library2 cell2 "<<endl;
    //    return TCL_OK;
    //}

    SimulatorPlugIn* plugInPtr = static_cast<SimulatorPlugIn*>(clientData);

    string library1(argv[2]), cell1(argv[6]), view1("physical");
    //string library1(argv[2]), cell1(argv[6]), view1("oagSchematic");
    //string library2(argv[4]), cell2(argv[5]);

   //check_equivalence  -lib simEquiTests -lib2 simEquiTests -cell1 s27 -cell2 s27_bug -randomPattern 1024 -liberty gsclib.lib
  
   std::auto_ptr<oagResyn::equiCheckOutputInterface> equichecker;
   try
   {
      equichecker= oagResyn::getEquiResult(argc, argv);
   }
   catch(oagResyn::oagResynErr& e)
   {
      cerr<<"Error building equivalence checker object in Bazaar:"<<endl<<e.what()<<endl;
      return TCL_ERROR;
   }
   catch (oa::oaException &excp)
   {
       cerr<<"OpenAcess exception building equivalence checker object, Error: "<<endl<<
          static_cast<const char *>(excp.getMsg())<<endl;
      return TCL_ERROR;
   }

    oagBazaar::similarityLayoutViewWnd* similarityLayoutView = 0;
    try
    {
       vector<oaNet*>* diffNets = 0;
       const std::vector<oa::oaNet*>& diffnets = equichecker->getDiffNets1();
       diffNets = new std::vector<oa::oaNet*>(diffnets);
       
       similarityLayoutView = new oagBazaar::similarityLayoutViewWnd(0, library1, cell1, view1, diffNets);
     
       for(std::vector<oa::oaNet*>::const_iterator it = diffnets.begin(); it != diffnets.end(); ++it)
       {
            //cout<<"Different net named: "<<oagBazaar::Qoag::getNameFromOA(*it)<<endl;
       }

    }
    catch(...)
    {
       cerr<<"Exception thrown by "<<argv[0]<<", aborting..."<<endl;
       return TCL_ERROR;
    }

    plugInPtr->getAPI()->giveWidget(similarityLayoutView);

   return TCL_OK;

}

int SimiEquiCreateBuggyDesignCmd(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
{
    //remove compiler unused warnings
    static_cast<void>(clientData);
    static_cast<void>(interp);
    //static_cast<void>(argc);
    //static_cast<void>(argv);

    //if (argc != 5)
    //{
    //    cerr<<"Usage: "<<argv[0]<<" library1 cell1 library2 cell2 "<<endl;
    //    return TCL_OK;
    //}

    //SimulatorPlugIn* plugInPtr = static_cast<SimulatorPlugIn*>(clientData);

   //create_buggy_design  -lib simEquiTests -lib2 simEquiTests -cell1 s27 -cell2 s27_bug -liberty gsclib.lib
  
   try
   {
      oagResyn::increVeriMain(argc, argv);
   }
   catch(oagResyn::oagResynErr& e)
   {
      cerr<<"oagResynError running SimEqui's increVeriMain() in Bazaar:"<<endl<<e.what()<<endl;
      return TCL_ERROR;
   }
   catch (oa::oaException &excp)
   {
       cerr<<"OpenAcess exception running SimEqui's increVeriMain(), Error: "<<endl<<
          static_cast<const char *>(excp.getMsg())<<endl;
      return TCL_ERROR;
   }


   return TCL_OK;

}


void SimulatorPlugIn::load(std::auto_ptr<oagBazaar::oagBazaarPlugInAPI> bazaarAPI)
{
   //DO THIS FIRST BEFORE ANYTHING ELSE 
   _bazaarAPI = bazaarAPI; //take ownership of the API object pointer

   _bazaarAPI->registerTCLCommand(&RunSimulatorPlugInCmd, "run_simulator", this);
   _bazaarAPI->registerTCLCommand(&RunCycleSimulatorPlugInCmd, "run_cycle_simulator", this);
   _bazaarAPI->registerTCLCommand(&SimulatorPlugInCmd, "simulate", this);
   _bazaarAPI->registerTCLCommand(&CheckDesignEquivalenceCmd, "check_design_equivalence", this);
   _bazaarAPI->registerTCLCommand(&CheckNetEquivalenceCmd, "check_net_equivalence", this);
   _bazaarAPI->registerTCLCommand(&CheckEquivalenceCmd, "check_equivalence", this);
   _bazaarAPI->registerTCLCommand(&SimilarityLayoutViewCmd, "similarity_layout_view", this);
   _bazaarAPI->registerTCLCommand(&SimilarityLayoutCmd, "similarity_layout", this);
   _bazaarAPI->registerTCLCommand(&SimiEquiCreateBuggyDesignCmd, "create_buggy_design", this);

   QPopupMenu* simulatorMenu = _bazaarAPI->addMenu("michigan.png", "MSim");
   QToolBar* simulatorToolBar = _bazaarAPI->addToolBar("MSim Tool Bar");

   QAction* runSimulatorAction = _bazaarAPI->createAction("run_cycle_simulator", "Simulator", "Run OAGear's UMSim simulation tool.", "simulator_icon_small.png");
   QAction* simulateAction = _bazaarAPI->createAction("run_simulator", "Random Simulation", "Simulate random input vectors", "random_icon_small.png");
   QAction* checkDesignEQAction = _bazaarAPI->createAction("check_design_equivalence", "Design Equivalence Checker", "Run OAGear's UMSim design equivalence checking tool.", "design_eq_icon.png");
   QAction* checkNetEQAction = _bazaarAPI->createAction("check_net_equivalence", "Signal Equivalence Checker", "Run OAGear's UMSim signal equivalence checking tool.", "signal_eq_icon.png");
   QAction* similarityViewerAction = _bazaarAPI->createAction("similarity_layout_view", "Similarity Layout View", "Run OAGear's UMSim similarity layout viewer.", "signal_eq_icon.png");
   //QAction* createBuggyDesignAction = _bazaarAPI->createAction("create_buggy_design", "Create A Buggy Design", "Create a new copy of a design and inject bugs.", "signal_eq_icon.png");


   _bazaarAPI->addMenuItem(simulatorMenu, runSimulatorAction);
   _bazaarAPI->addToolBarButton(simulatorToolBar, runSimulatorAction);

   _bazaarAPI->addMenuItem(simulatorMenu, simulateAction);
   _bazaarAPI->addToolBarButton(simulatorToolBar, simulateAction);

   _bazaarAPI->addMenuItem(simulatorMenu, checkDesignEQAction);
   _bazaarAPI->addToolBarButton(simulatorToolBar, checkDesignEQAction);

   _bazaarAPI->addMenuItem(simulatorMenu, checkNetEQAction);
   _bazaarAPI->addToolBarButton(simulatorToolBar, checkNetEQAction);

   _bazaarAPI->addMenuItem(simulatorMenu, similarityViewerAction);

   //_bazaarAPI->addMenuItem(simulatorMenu, createBuggyDesignAction);

}



