/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

// Author: David Papa <iamyou@umich.edu>

/*! \file controller.h */

#ifndef _CONTROLLER_H_
#define _CONTROLLER_H_

#include <string>
#include <map>
#include <vector>
#include "tcl.h"
#include "oagConnect.h"
#include "processManager.h"

namespace oagBazaar
{

    class MainWindow;


    ///Singleton that keeps a map from TCL interpreter to Controller
    /**
     * ControllerRegistry is a singleton class that is intended to keep a map from interpreter
     * to controller, so that functions called by the interpreter can affect the controller.
     * it is expected that there is exactly one interpreter per controller.
     * Controller is a friend so that it can register itself with the registry using
     * private functions.
     */
    class ControllerRegistry
    {
        friend class Controller;

        public:

        static Controller& interpToController(Tcl_Interp* interp);

        ~ControllerRegistry();

        private: //functions

        static ControllerRegistry * get(void);

        ///private default constructor that only get() can use
        ControllerRegistry(){}
        ///private copy constructor that only get() can use
        ControllerRegistry(const ControllerRegistry& src); //no implementation provided

        static void registerController(Tcl_Interp* interp, Controller* controller);
        static void unregisterController(Tcl_Interp* interp, Controller* controller);

        private: //data

        static ControllerRegistry * _impl;

        std::map<Tcl_Interp*, Controller*> _interpToController;
    };

    class commandWindowT;

    ///Controller object of the Model-View-Controller design pattern
    class Controller
    {
        public:
            Controller();
            ~Controller();

            /**
             * Register a View object with this controller.  You must call
             * this function before the Controller is usable.
             * \param view A Qt QMainWindow Bazaar MainWindow object
             */ 
            void setView(MainWindow& view);
            MainWindow& getView(void);
            commandWindowT * getCommandWindow(void);
            ProcessManager& getProcessManager(void);

            ///finds and returns the specified connection if it exists throws an exeption otherwise
            oagConnection* getConnection(const std::string& lib, const std::string& cell, const std::string& view);
            ///opens a connection with the OpenAccess database and stores it internally
            oagConnection* openConnection(const std::string& lib, const std::string& cell, const std::string& view);
            ///test whether a connection with OpenAccess is already open
            bool isConnectionOpen(const std::string& lib, const std::string& cell, const std::string& view);

            ///Send a command to the TCL interpreter
            /**
             * Run a command on the TCL interpreter that is associated with
             * this Controller object \param command a whitespace delimited
             * std::string in the typical TCL format (i.e.  cmd arg1 arg2 arg3 ...).
             */
            void runCommand(const std::string& command);

            void registerCommand(Tcl_CmdProc* cmd, const std::string& cmdName, ClientData clientData = 0, Tcl_CmdDeleteProc *deleteProc = 0);

        private:

            MainWindow * _view;

            Tcl_Interp * _interp;
   
            ProcessManager * _processManager;
          
            std::vector< oagConnection* > _openAccessConnections;
    };

}

#endif


