/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

// Author: David Papa <iamyou@umich.edu>

#include <iostream>
#include <algorithm>

#include "assert.h"
#include "controller.h"
#include "tclCommands.h"
#include "mainwindow.h"

using std::cout;
using std::endl;
using std::cerr;

namespace oagBazaar
{


    int HelloWorld(ClientData clientData,
            Tcl_Interp *interp,
            int argc,
            CONST char *argv[])
    {
        //remove compiler unused warnings
        static_cast<void>(clientData);
        static_cast<void>(interp);
        static_cast<void>(argc);
        static_cast<void>(argv);

        cout<<"Hello World!"<<endl;
        return 0;
    }

    Controller::Controller(void):
        _view(0), _interp(Tcl_CreateInterp()), _processManager(new ProcessManager(this, 0, "ProcessManager"))
    { 
        ControllerRegistry::registerController(_interp, this);

        Tcl_CmdProc* hello_world_cmd = &HelloWorld;
        Tcl_CreateCommand(_interp, "hello_world", hello_world_cmd, 0, NULL);

        registerTclCommands(_interp);
    }

    Controller::~Controller()
    {
        delete _processManager; 
        ControllerRegistry::unregisterController(_interp, this);
        Tcl_DeleteInterp(_interp);
        for(std::vector<oagConnection*>::iterator it = _openAccessConnections.begin();
            it != _openAccessConnections.end(); ++it)
        {
           delete *it;
        }
        
    }

    void Controller::setView(MainWindow &mainWindow)
    {
        _view=&mainWindow;
    }

    MainWindow& Controller::getView(void)
    {
        return *_view;
    }

    commandWindowT * Controller::getCommandWindow(void)
    {
        return _view->getCommandWindow();
    }

    ProcessManager& Controller::getProcessManager(void)
    {
        return *_processManager;
    }

    oagConnection* Controller::getConnection(const std::string& lib, const std::string& cell, const std::string& view)
    {
       for(std::vector<oagConnection*>::iterator it = _openAccessConnections.begin();
            it != _openAccessConnections.end(); ++it)
       {
          if((*it)->getLibName()  == lib  && 
             (*it)->getCellName() == cell && 
             (*it)->getViewName() == view)
            return (*it);
       }
       throw; //TODO: throw a meaningful object
    }

    oagConnection* Controller::openConnection(const std::string& lib, const std::string& cell, const std::string& view)
    {
      if(isConnectionOpen(lib, cell, view))
        return getConnection(lib, cell, view);

       oagConnection* newConnection = new oagConnection(lib, cell, view);
       _openAccessConnections.push_back(newConnection);
      return newConnection; 
    }

    bool Controller::isConnectionOpen(const std::string& lib, const std::string& cell, const std::string& view)
    {
       for(std::vector<oagConnection*>::iterator it = _openAccessConnections.begin();
            it != _openAccessConnections.end(); ++it)
          if((*it)->getLibName()  == lib  && 
             (*it)->getCellName() == cell && 
             (*it)->getViewName() == view)
            return true;
       return false;
    }

    void Controller::runCommand(const std::string& command) 
    {
        assert(_view && "Cannot use a Controller without calling setView()");
        int retval=INT_MAX;

        cout << "> " << command << endl;
        char cmdToTCL[command.size() + 1]; //TCL insists that command's memory is writable
        copy( command.begin(), command.end(), cmdToTCL );
        cmdToTCL[command.size()]=0;
        try
        {
            retval = Tcl_Eval(_interp, cmdToTCL);
            if(retval == TCL_OK) 
            {
                cout << _interp->result << endl;
            }
            else 
            {
                cerr << "TCL Error : " << _interp->result << endl;
            } 
            getCommandWindow()->refresh();
        }
        catch (oa::oaException &excp)
        {
            cerr << "OA threw an exception while trying to save_as, Error: " << excp.getMsg() << endl;
            cerr<<"The command causing this error message was \""<<cmdToTCL<<"\"."<<endl;
        }
        catch(...)
        {
            cerr<<"Unknown error thrown from inside tcl command, ignoring and continuing..."<<endl;
            cerr<<"The command causing this error message was \""<<cmdToTCL<<"\"."<<endl;
        } 

        return;
    }


    Controller& ControllerRegistry::interpToController(Tcl_Interp* interp)
    {
        ControllerRegistry * reg = get();
        assert(reg->_interpToController.find(interp) != reg->_interpToController.end());
        return *(reg->_interpToController.find(interp)->second);
    }

    ControllerRegistry * ControllerRegistry::_impl=0;

    ControllerRegistry::~ControllerRegistry()
    {
        if(_impl)
            delete _impl;
    }

    ControllerRegistry * ControllerRegistry::get(void)
    {
        if(!_impl)
            _impl = new ControllerRegistry();
        return _impl;
    }

    void ControllerRegistry::registerController(Tcl_Interp* interp, Controller* controller)
    {
        ControllerRegistry * reg = get();
        reg->_interpToController.insert(std::make_pair(interp, controller));
    }

    void ControllerRegistry::unregisterController(Tcl_Interp* interp, Controller* controller)
    {
        //remove compiler unused warning
        static_cast<void>(controller);

        ControllerRegistry * reg = get();
        reg->_interpToController.erase(interp);
    }

    void Controller::registerCommand(Tcl_CmdProc* cmd, const std::string& cmdName, ClientData clientData, Tcl_CmdDeleteProc *deleteProc)
    {
       registerTCLCommand(_interp, cmd, cmdName, clientData, deleteProc);
    }
}
