/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-05-31: ChangeLog started
*/

#include <iostream>
#include "oagAiNode.h"
#include "oagAiGraph.h"

#if !defined(NULL)
#define NULL 0
#endif

#include "oagAiDebug.h"

using namespace std;

namespace oagAi {

// *****************************************************************************
// Node()
//
/// \brief Generic node constructor.
///
/// The node will be created with a type of NONE.
//
// *****************************************************************************
Node::Node() {
    this->type = NONE;
    this->refCount = 0;
    this->fanoutCount = 0;
    this->external_terminal_connection = NULL;
    this->traversalID = 0;
}


// *****************************************************************************
// Node()
//
/// \brief Terminal node constructor.
///
/// The node will be created as an TERMINAL of the supplied parameter.
///
/// \param driver
//
// *****************************************************************************
Node::Node(Ref driver) {
    this->type = TERMINAL;
    this->terminal_driver = driver;
    this->refCount = 0;
    this->fanoutCount = 0;
    this->external_terminal_connection = NULL;
    this->traversalID = 0;
}


// *****************************************************************************
// Node()
//
/// \brief And node constructor.
///
/// The node will be created as an AND of the two supplied parameters.
///
/// \param left
/// \param right
//
// *****************************************************************************
Node::Node(Ref left, Ref right) {
    this->type = AND;
    this->left = left;
    this->right = right;
    this->refCount = 0;
    this->fanoutCount = 0;
    this->external_terminal_connection = NULL;
    this->traversalID = 0;
}


// *****************************************************************************
// clear()
//
/// \brief Reinitializes a node to its initial state.
///
/// The node will be set to be of type NONE.
//
// *****************************************************************************
void
Node::clear() {
    this->type = NONE;
    this->left = this->left = Graph::getNull();
    this->next = this->equivalent = Graph::getNull();
    this->sequentialData = NULL;
    this->refCount = 0;
    this->fanoutCount = 0;
    this->external_terminal_connection = NULL;
    this->traversalID = 0;
}


// *****************************************************************************
// PersistentNode()
//
/// \brief Persistant node constructor.
//
// *****************************************************************************
PersistentNode::PersistentNode() {
    this->sequentialData = NULL;
    this->equivalent = Graph::getNull();    
}


// *****************************************************************************
// ~PersistentNode()
//
/// \brief Persistant node destructor.
//
// *****************************************************************************
PersistentNode::~PersistentNode() {
    if (type == SEQUENTIAL && sequentialData != NULL)
        delete sequentialData;
}

 
// *****************************************************************************
// serialize()
//
/// \brief Places a serialized version of this node into a buffer
///
/// \param buf the pre-allocated buffer into which the serialized data is placed
//
// *****************************************************************************
void
PersistentNode::serialize(void* buf) {
    assert(buf);

    memcpy(buf, this, sizeof(PersistentNode));
    
    PersistentNode *buf_asPersistentNode = static_cast<PersistentNode*>(buf);
    buf = ++buf_asPersistentNode;

    if (type != SEQUENTIAL || sequentialData == NULL)
        return;

    // copy sequential information
    DEBUG_PRINTLN("\tcopying sequential data");
    
    // write abstraction type
    short *buf_asShort = static_cast<short *>(buf);
    *buf_asShort = sequentialData->abstractionLevel;
    buf = ++buf_asShort;

    switch(sequentialData->abstractionLevel) {
      case SequentialData::BLACK_BOX: {

        // write number of signals
        short numSignals = sequentialData->signals.size();
        buf_asShort = static_cast<short *>(buf);
        *buf_asShort = numSignals;
        buf = ++buf_asShort;
        for(map<string, Ref>::iterator it = sequentialData->signals.begin(); it != sequentialData->signals.end(); it++) {

            // write length of name
            short nameLength = it->first.length();
            buf_asShort = static_cast<short *>(buf);
            *buf_asShort = nameLength;
            buf = ++buf_asShort;

            // write name data
            memcpy(buf, it->first.data(), sizeof(char)*nameLength);
            buf = static_cast<void*>(&(static_cast<char *>(buf)[nameLength]));

            // write signal reference
            Ref *buf_asRef = static_cast<Ref *>(buf);
            *buf_asRef = it->second;
            buf = ++buf_asRef;
            
            DEBUG_PRINTLN("\tasync sig " << it->first << " = " << it->second);
        }
        break;
        }

      case SequentialData::PRIMITIVE: {

        // write gate reference
        Ref *buf_asRef = static_cast<Ref *>(buf);
        *buf_asRef = sequentialData->gate;
        buf = ++buf_asRef;

        break;
      }
      case SequentialData::FEEDBACK:  
        // nothing to write
        break;
      default:
        assert(false);
        exit(0);
    }
}


// *****************************************************************************
// unserialize()
//
/// \brief Creates a copy of a serialized node
///
/// \param buf pointer to a buffer with the serialized data
/// \return the number of bytes read from the buffer
//
// *****************************************************************************
int
PersistentNode::unserialize(void* buf) {
    assert(buf);
    int bytes = 0;

    memcpy(this, buf, sizeof(PersistentNode));
    bytes += sizeof(PersistentNode);
    PersistentNode *buf_asPersistentNode = static_cast<PersistentNode*>(buf);
    buf = ++buf_asPersistentNode;

    if (type != SEQUENTIAL)
        return bytes;

    if (!sequentialData)
        return bytes;
    sequentialData = NULL;

    // copy sequential information
    DEBUG_PRINTLN("\tcopying sequential data");    
    
    // read abstraction type and create sequential data
    short *buf_asShort = static_cast<short *>(buf);
    sequentialData = new SequentialData(
        static_cast<SequentialData::AbstractionLevel>(*buf_asShort));
    bytes += sizeof(short);
    buf = ++buf_asShort;
    switch(sequentialData->abstractionLevel) {
      case SequentialData::BLACK_BOX: {
        // read number of signals
        buf_asShort = static_cast<short *>(buf);
        short numSignals = *buf_asShort;
        bytes += sizeof(short);
        buf = ++buf_asShort;
        for(int i=0; i<numSignals; i++) {
            // read length of name
            buf_asShort = static_cast<short *>(buf);
            short nameLength = *buf_asShort;
            bytes += sizeof(short);
            buf = ++buf_asShort;
            // read name data
            char *str = new char[nameLength+1];
            memcpy(str, buf, sizeof(char)*nameLength);
            str[nameLength] = '\0';
            bytes += sizeof(char)*nameLength;
            buf = static_cast<void*>(&(static_cast<char *>(buf)[nameLength]));
            // read signal reference
            Ref *buf_asRef = static_cast<Ref *>(buf);
            sequentialData->signals[str] = *buf_asRef;
            bytes += sizeof(Ref);
            buf = ++buf_asRef;

            DEBUG_PRINTLN("\tasync sig " << str << " = " << sequentialData->signals[str]);            
            delete [] str;
        }
        break;
        }
      case SequentialData::PRIMITIVE: {
        // read gate reference
        Ref *buf_asRef = static_cast<Ref *>(buf);
        sequentialData->gate = *buf_asRef;
        bytes += sizeof(Ref);
        buf = ++buf_asRef;
        break;
      }
      case SequentialData::FEEDBACK:  
        // nothing to read
        break;
      default:
        assert(false);
        exit(0);
    }
    
    return bytes;
}


// *****************************************************************************
// getSerializedSize()
//
/// \brief Computes the number of bytes necessary for the serialzed representation.
///
/// \return the number of bytes
//
// *****************************************************************************
int
PersistentNode::getSerializedSize() {
    int bytes = sizeof(PersistentNode);

    if (type != SEQUENTIAL || sequentialData == NULL)
        return bytes;

    // compute size of sequential information

    // size of abstraction type
    bytes += sizeof(short);
    switch(sequentialData->abstractionLevel) {
      case SequentialData::BLACK_BOX: {
        // size of number of signals
        bytes += sizeof(short);
        for(map<string, Ref>::iterator it = sequentialData->signals.begin(); it != sequentialData->signals.end(); it++) {
            // size of length of name
            short nameLength = it->first.length();
            bytes += sizeof(short);
            // size of name data
            bytes += sizeof(char)*nameLength;
            // size of signal reference
            bytes += sizeof(Ref);
        }
        break;
        }
      case SequentialData::PRIMITIVE:
        // size of gate reference
        bytes += sizeof(Ref);
        break;
      case SequentialData::FEEDBACK:  
        // nothing to write
        break;
      default:
        assert(false);
        exit(0);
    }

    return bytes;
}
    
}
