/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: David Papa <iamyou@umich.edu>

ChangeLog:
2004-11-18: ChangeLog started
 */

#include <stdio.h>
#include <iostream>
#include <ostream>
#include <fstream>
#include <string>
#include <map>
#include <vector>

#include "oaDesignDB.h"

using std::cout;
using std::endl;
using std::cerr;
using std::ostream;
using std::ofstream;
using std::string;
using std::vector;
using std::map;

using oa::oaBlock;
using oa::oaIter;
using oa::oaInst;
using oa::oaNativeNS;
using oa::oaTerm;
using oa::oaString;
using oa::oaBox;
using oa::oaBoolean;
using oa::oaNet;
using oa::oaUInt4;
using oa::oaSigType;
using oa::oaTermType;
using oa::oaInstTerm;
using oa::oacInputTermType;
using oa::oacInputOutputTermType;
using oa::oacOutputTermType;
using oa::oaPin;
using oa::oaPinFig;
using oa::oaCoreBoxSpec;
using oa::oaPRBoundary;
using oa::oaDesignInit;
using oa::oaLibDefList;
using oa::oaDesign;
using oa::oaScalarName;
using oa::oaException;
using oa::oaRow;
using oa::oaPoint;
using oa::oacLockedPlacementStatus;
using oa::oacFixedPlacementStatus;
using oa::oacClockSigType;

template <typename T> 
std::string getNameFromOA(T i)
{
   oa::oaNativeNS ns;
   oa::oaString name;
   i->getName(ns,name);
   return std::string(static_cast<const char*>(name) );
}
  
template <> 
std::string getNameFromOA<oa::oaDesign*>(oa::oaDesign* i);

template <> std::string getNameFromOA<oa::oaDesign*>(oa::oaDesign* i)
{
      oa::oaNativeNS ns;
      oa::oaString name;
      i->getCellName(ns,name);
      return std::string(static_cast<const char*>(name) );
}

 
oa::oaBox getPlacementBBox(const oa::oaInst * const inst)
{
  oa::oaSnapBoundary *sb = oa::oaSnapBoundary::find(inst->getMaster()->getTopBlock());
  oa::oaBox returnVal;
  sb->getBBox(returnVal);
  return returnVal;
}


void writeNodesFile( const oaBlock *block, ostream& nodes);
void writeNetFile( const oaBlock *block, ostream& nets);
void writeWtsFile( const oaBlock *block, ostream& wts);
void writePlFile( const oaBlock *block, ostream& pl);
void writeSclFile( const  oaBlock *block, ostream& scl);
void writeLogicFile( const  oaBlock *block, ostream& logic);
void initLogicMap(void);

map<string, string> logicMap;

int main(int   argc, char  *argv[])
{
    initLogicMap();
 
    try
    {
        oaDesignInit();
        oaLibDefList::openLibs();
    }
    catch (oa::oaException &excp) {
        cerr<< "Error: "<< excp.getMsg() <<endl;
        exit(1);
    }


    // Check number of arguments.
    if ( argc < 4 || argc > 6 ) {
        cout<< " Use : \t" << argv[0] 
            << " libraryName cellName viewName [bookshelfName]" << endl;
        exit(1);
    }

    // Get the names from the command line
    oaString        libstr(argv[1]);
    oaString        cellstr(argv[2]);
    oaString        viewstr(argv[3]);

    // Treat the names on the command line as belonging to the Native
    // namespace and convert them to oaNames.
    oaNativeNS      ns;
    oaScalarName    libName(ns, libstr);
    oaScalarName    cellName(ns, cellstr);
    oaScalarName    viewName(ns, viewstr);

    oaDesign      *des;
    oaBlock       *block;

    try 
    {
        des  = oa::oaDesign::open(libName, cellName, viewName, 'r') ;
        block = des->getTopBlock() ;
    }
    catch (oaException &excp) 
    {
        cerr << "Can't read CellView ";
        cerr << libstr <<" "<< cellstr << " " << viewstr;
        cerr << " (check lib.defs)" << endl;
        cerr << "\t"<< excp.getMsg() <<endl;
        exit(1);
    }

    cout << "Reading CellView ";
    cout <<  libstr << " " << cellstr << " " << viewstr <<endl;

    // Write the benchmark to a file 
    // if the name is not specified on the command line,
    // it defaults to a concatenation of cell and view names.
    ofstream       auxF,nodesF,netsF,sclF,wtsF,plF,logicF;
    string rootName;
    if(argc >= 5)
        rootName = argv[4];
    else
    {
        rootName = cellstr;
        rootName += '_';
        rootName += viewstr;
    }

    bool createLogicFile = false;
    if(argc==6) createLogicFile = (string(argv[5]) == "-logic");
   
    string   auxFname(rootName),
             nodesFname(rootName),
             netsFname(rootName),
             sclFname(rootName),
             wtsFname(rootName),
             plFname(rootName),
             logicFname(rootName);
    auxFname += ".aux" ;
    nodesFname += ".nodes" ;
    netsFname += ".nets" ;
    sclFname += ".scl" ;
    wtsFname += ".wts" ;
    plFname += ".pl" ;
    logicFname += ".logic" ;


    auxF.open(auxFname.c_str());
    if (auxF == NULL) {
        cout << "Can't write file " << rootName << endl; 
        exit(1);
    }
    cout << "Writing Bookshelf benchmark to root filename " << rootName << endl; 
    auxF << "RowBasedPlacement\t:\t";
    auxF << nodesFname << "\t";
    auxF << netsFname << "\t";
    auxF << sclFname << "\t";
    auxF << wtsFname << "\t";
    auxF << plFname << "\t"; 
    if(createLogicFile) auxF << logicFname << "\t";
    auxF.close();

    try 
    {
        nodesF.open(nodesFname.c_str());
        writeNodesFile(block,nodesF);
        nodesF.close();

        netsF.open(netsFname.c_str());
        writeNetFile  (block, netsF);
        netsF.close();

        wtsF.open(wtsFname.c_str());
        writeWtsFile  (block, wtsF);
        wtsF.close();

        plF.open(plFname.c_str());
        writePlFile  (block, plF);
        plF.close();

        sclF.open(sclFname.c_str());
        writeSclFile(block, sclF);
        sclF.close();

        if(createLogicFile)
        {
          logicF.open(logicFname.c_str());
          writeLogicFile(block, logicF);
          logicF.close();
        }
    }
    catch (oaException &excp) 
    {
        cerr << "Error: " << excp.getMsg() << endl; 
        exit(1);
    }

    return 0;
}

void writeNodesFile( const oa::oaBlock *block, ostream& nodes)
{
    nodes << "UCLA nodes 1.0" << endl;
    nodes << "#Created : by oa2bk v1.0b" <<endl;
    nodes << "#User : fill in later" << endl;
    nodes << "#Platform : fill in later" << endl;
    nodes << endl;
    nodes << "NumNodes : " << block->getInsts().getCount() + block->getTerms().getCount() << endl; 
    nodes << "NumTerminals : " << block->getTerms().getCount() << endl; 
    //TODO:  This isnt the exact number of terminals
    // here i am assuming that there are 0 Firm and Locked insts
    nodes << endl;

    oaIter<oaInst> iIter(block->getInsts());
    oaNativeNS     ns;

    oaIter<oaTerm> itIter(block->getTerms());
    while (oaTerm   *term = itIter.getNext()) 
    {
        oaString        iStr;
        //oaOrient        orient = inst->getOrient();

        term->getName(ns, iStr);

        nodes << "\t" << iStr;
        nodes << "\t1\t1";
        nodes << "\tterminal";
        nodes << endl;
    }

    while (oaInst   *inst = iIter.getNext()) 
    {
        oaString        iStr;
        //oaOrient        orient = inst->getOrient();
        oaBox           bbox;

        inst->getName(ns, iStr);
        //inst->getBBox(bbox);
        bbox = getPlacementBBox(inst);

        oaBoolean isFixed;
        switch (inst->getPlacementStatus())
        {
            case oacLockedPlacementStatus:
            case oacFixedPlacementStatus:
                isFixed = true; break;

            default:
                isFixed = false;
        }

        nodes << "\t" << iStr;
        nodes << "\t" << bbox.getWidth() << "\t" << bbox.getHeight();
        if(isFixed) 
            nodes << "\tterminal";
        nodes << endl;
    }
}

void getPinOffset(oaInst* inst, oaTerm* term, double& pinOffX, double& pinOffY)
{
    assert(inst && "Null inst");
    assert(term && "Null term");
    oaBox instBox;
    inst->getBBox(instBox);
    oaIter<oaPin> oaPinIt(term->getPins());
    oaPin* firstPin = oaPinIt.getNext();
    assert(firstPin && "Term has no pin");
    oaIter<oaPinFig> oaPinFigIt(firstPin->getFigs());
    oaPinFig* firstPinFig = oaPinFigIt.getNext();
    assert(firstPinFig && "Pin has no Fig");
    oaBox pinFigBox;
    firstPinFig->getBBox(pinFigBox);
    double instWidth = (instBox.right() - instBox.left() ),
           instHeight =(instBox.top() - instBox.bottom() ) ;
    double ptX=pinFigBox.left(), ptY=pinFigBox.bottom();
    double instCtrX = instWidth/2.0,
           instCtrY = instHeight/2.0;
    double relCtrPtX = ptX - instCtrX, relCtrPtY = ptY - instCtrY;
    
    pinOffX = relCtrPtX / instWidth;
    pinOffY = relCtrPtY / instHeight;
}

void writeNetFile( const oa::oaBlock *block, ostream& nets)
{
    oaNativeNS     ns;
    // Print the Nets in the CellView and Terms on each Net.
    nets << "UCLA nets 1.0"<<endl;
    nets << "#Created : by oa2bk v1.0b" <<endl;
    nets << "#User : fill in later" << endl;
    nets << "#Platform : fill in later" << endl;
    nets << endl;

    nets << "NumNets : " << block->getNets().getCount() << endl; 



    oa::oaIter<oaNet>   nIter(block ->getNets());

    oaUInt4 degree = 0;
    while (oaNet   *net = nIter.getNext()) 
    {
        //oaBoolean getTermsArg = true;
        degree += net->getTerms(/*getTermsArg*/).getCount()+net->getInstTerms().getCount();
    }
    nets << "NumPins : " << degree << endl;

    nIter.reset();
    while (oaNet   *net = nIter.getNext()) 
    {
        oaString        nStr;
        oaSigType       stype = net->getSigType();

        net->getName(ns, nStr);

        // Print the Net line
        //oaBoolean getTermsArg = true;
        nets << "NetDegree : "<< net->getTerms(/*getTermsArg*/).getCount()+net->getInstTerms().getCount() <<
            "    " << nStr<< "      " <<
            endl;

        // Print a line for each Term on this Net
        oa::oaIter<oaTerm>      tIter(net->getTerms());
        while (oaTerm   *term = tIter.getNext()) 
        {
            oaString    tStr;
            oaTermType  ttype = term->getTermType();

            term->getName(ns, tStr);
            

            nets << "\t" << tStr << " ";
            switch (ttype)
            {
                case oacInputTermType :    nets << "I"; break;
                case oacOutputTermType :   nets << "O"; break;
                case oacInputOutputTermType : 
                default :
                                           nets << "B"; break;
            }

            nets<<" : 0.0 0.0";

            nets<<endl;          
 
        }

        // Print a line for each instTerm on this Net
        oa::oaIter<oaInstTerm>      tiIter(net->getInstTerms());
        while (oaInstTerm   *term = tiIter.getNext()) 
        {
            oaString    tStr;
            oaInst* inst = term->getInst();
            inst->getName(ns, tStr);
            oaTerm* t = term->getTerm();
            if(!t) 
            {
                cerr << "Inst has no term: "<<nStr<<endl;
                cerr << "inst: "<<tStr<<endl;
                exit(1);
            }
            oaTermType  ttype = t->getTermType();

            nets << "\t" << tStr << " ";
            switch (ttype)
            {
                case oacInputTermType :    nets << "I"; break;
                case oacOutputTermType :   nets << "O"; break;
                case oacInputOutputTermType : 
                default :
                                           nets << "B"; break;
            }

            double xOff = 0.0, yOff = 0.0;
            getPinOffset(inst, t, xOff, yOff);
            nets << " : "<<xOff<<" "<<yOff;           
    //        nets << " /"<<getNameFromOA(t);
            nets << endl;

        }

    }
}


void writeWtsFile( const oa::oaBlock *block, ostream& wts)
{
    // Print all the Insts in the CellView, and all InstTerms on each Inst.
    wts << "UCLA wts 1.0" << endl;
    wts << "#Created : by oa2bk v1.0b" <<endl;
    wts << "#User : fill in later" << endl;
    wts << "#Platform : fill in later" << endl;
    wts << endl;


    oa::oaIter<oaInst> iIter(block->getInsts());
    oaNativeNS     ns;

    oa::oaIter<oaTerm> itIter(block->getTerms());
    while (oaTerm   *term = itIter.getNext()) 
    {
        oaString        iStr;

        term->getName(ns, iStr);

        wts << "\t" << iStr << "\t0" << endl;
    }

    while (oaInst   *inst = iIter.getNext()) 
    {
        oaString        iStr;

        inst->getName(ns, iStr);

        oaBoolean isFixed;
        switch (inst->getPlacementStatus())
        {
            case oacLockedPlacementStatus:
            case oacFixedPlacementStatus:
                isFixed = true; break;
            default:
                isFixed = false;
        }

        wts << "\t" << iStr;
        if(isFixed) 
            wts << "\t0";
        else
            wts << "\t1";
        wts << endl;


    }

}

void writePlFile( const oa::oaBlock *block, ostream& pl)
{
    pl << "UCLA pl 1.0" << endl;
    pl << "#Created : by oa2bk v1.0b" <<endl;
    pl << "#User : fill in later" << endl;
    pl << "#Platform : fill in later" << endl;
    pl << endl;

    oaNativeNS     ns;

    oa::oaIter<oaTerm> itIter(block->getTerms());
    while (oaTerm   *term = itIter.getNext()) 
    {
        oaString        iStr;
        //oaOrient        orient = inst->getOrient();

        term->getName(ns, iStr);
        oa::oaIter<oaPin> pinIter(term->getPins());
        oaPin *pin = pinIter.getNext();
        if(pin == NULL)
        {
            pl << "\t" << iStr;
            pl << "\t0\t0" ;
            pl << "\t : N";  //TODO : fix this to use real oaOrient
            pl << endl;
            continue;
        }
        oaBox bbox;
        oa::oaIter<oaPinFig> p_it(pin->getFigs());
        oaPinFig* p = p_it.getNext();
        if(p)
            p->getBBox(bbox);

        pl << "\t" << iStr;
        pl << "\t" << bbox.left()<< "\t"  << bbox.bottom();
        pl << "\t : N";  //TODO : fix this to use real oaOrient
        pl << endl;
    }

    oa::oaIter<oaInst> iIter(block->getInsts());
    while (oaInst   *inst = iIter.getNext()) 
    {
        oaString        iStr;
        //oaOrient        orient = inst->getOrient();
        oaBox           bbox;

        inst->getName(ns, iStr);
        inst->getBBox(bbox);

        pl << "\t" << iStr;
        pl << "\t"<< bbox.left()<< "\t"  << bbox.bottom() ;
        pl << "\t : N"; //TODO: fix this to use real oaOrient
        pl << endl;


    }

}

void writeSclFile(const oaBlock *block, ostream& scl)
{
    oaNativeNS     ns;

    oaCoreBoxSpec cbs;
    oaPRBoundary *prb;
    const oaBlock * bl  = block;
    prb = oaPRBoundary::find(bl);
    prb->getCoreBoxSpec(cbs);

    scl << "UCLA scl 1.0" << endl;
    scl << "#Created : by oa2bk v1.0b" <<endl;
    scl << "#User : fill in later" << endl;
    scl << "#Platform : fill in later" << endl;
    scl << endl;

    //TODO : the following line of code accounts for a bug in the OA implementation
    // the row counts can be over by 1 in some circumstances, hence the -1. bug: [ #262 ]
    //  oaUInt4 nRows = cbs.getNumRows() -1;
    // scl << "Numrows : " << nRows << endl;
    // scl << endl;
    // for(oaUInt4 i = 0; i < nRows; i++) 
    // {
    //    oaBox bbox;
    //    //TODO : the following line of code accounts for a bug in the OA implementation
    //    // the row counts begin at 1, and therefore I adding 1
    //    // this should be changed to simply i if and when the bug is fixed. bug: [ #263 ]
    //    cbs.getRowBBox(i+1,bbox);

    //    oaUInt4 numSites = ( bbox.right() - bbox.left() ) / cbs.getSiteDef()->getWidth();

    //    scl << "CoreRow Horizontal" << endl;
    //    scl << " Coordinate\t:\t"<< bbox.bottom() << endl;
    //    scl << " Height\t:\t"<< cbs.getSiteDef()->getHeight() << endl;
    //    scl << " Sitewidth\t:\t"<< cbs.getSiteDef()->getWidth() << endl;
    //    scl << " Sitespacing\t:\t"<< cbs.getSiteDef()->getWidth() << endl;
    //    scl << " Siteorient\t:\tN"<<endl; //TODO make this actually care about orientations
    //    scl << " Sitesymmetry\t:\tY"<<endl; //TODO make this actually care about symmetry
    //    scl << " SubrowOrigin\t:\t"<< bbox.left() <<"\tNumsites\t:\t" << numSites <<endl;
    //    scl << "End"<<endl;
    // }

     scl << "Numrows : " << block->getRows().getCount() << endl;
     scl << endl;
    oaIter<oaRow> rowIter(block->getRows());
    while (oaRow   *row = rowIter.getNext()) 
    {
        oaPoint pt;
        row->getOrigin(pt);

        scl << "CoreRow Horizontal" << endl;
        scl << " Coordinate\t:\t"<< pt.y() << endl;
        scl << " Height\t:\t"<< row->getSiteDef()->getHeight() << endl;
        scl << " Sitewidth\t:\t"<< row->getSiteDef()->getWidth() << endl;
        scl << " Sitespacing\t:\t"<< row->getSiteDef()->getWidth() << endl;
        scl << " Siteorient\t:\tN"<<endl; //TODO make this actually care about orientations
        scl << " Sitesymmetry\t:\tY"<<endl; //TODO make this actually care about symmetry
        scl << " SubrowOrigin\t:\t"<< pt.x() <<"\tNumsites\t:\t" << row->getNumSites() <<endl;
        scl << "End"<<endl;
    }

}

void getOutputs(oaInst* inst, vector<oaInstTerm*> outputs)
{
    outputs.clear();
    try
    {
        oaIter<oaInstTerm>   oaITIter1(inst->getInstTerms());
        while (oaInstTerm*  instTerm = oaITIter1.getNext()) 
        {
            oaTerm* term = instTerm->getTerm();
            oaTermType termType = term->getTermType();
            if( termType == oacOutputTermType) 
            {
              outputs.push_back(instTerm);
            }               
        }
    }
    catch (oa::oaException &excp) 
    {
        cerr << "Error: " << excp.getMsg() << endl;
        exit(1);
    }
}

void getInputs(oaInst* inst, vector<oaInstTerm*> inputs)
{
    inputs.clear();
    try
    {
        oaIter<oaInstTerm>   oaITIter1(inst->getInstTerms());
        while (oaInstTerm*  instTerm = oaITIter1.getNext()) 
        {
            oaTerm* term = instTerm->getTerm();
            oaTermType termType = term->getTermType();
            if( termType == oacInputTermType) 
            {
                inputs.push_back(instTerm);
            }               
        }

    }
    catch (oa::oaException &excp) 
    {
        cerr << "Error: " << excp.getMsg() << endl;
        exit(1);
    }
}

bool isClockedInst(oaInst* inst)
{
    try
    {
        oaIter<oaInstTerm> instTermIter(inst->getInstTerms());
        oaInstTerm* instTerm;
        while(instTerm = instTermIter.getNext())
        {
            oaNet *net = instTerm->getNet();
            if(net->getSigType() == oacClockSigType)
            {
                //cout<<"Found that the instTerm connects to a clock net"<<endl;
                return true;
            }
        } 

        oaDesign *master = inst->getMaster();
        assert(master);
        oaBlock *masterBlock = master->getTopBlock();
        assert(masterBlock);
        oaIter<oaTerm> masterTermIter(masterBlock->getTerms());
        oaTerm *j;
        while (j = masterTermIter.getNext()) 
        {
            string termName = getNameFromOA(j);
            //cout<<"\tChecking if term "<<termName<<" is clocked"<<endl;
            oaNet *masterNet = j->getNet();
            if (masterNet->getSigType() == oacClockSigType)
            {
                return true;
            }
            else if (termName == "CK")
            {
                return true;
            }
        }
    }
    catch (oa::oaException &excp) 
    {
        cerr << "Error: " << excp.getMsg() << endl;
        exit(1);
    }
}

void initLogicMap(void)
{
    //gates are encoded as truth tables outputs
    //inputs are listed from least significant to most significant
    logicMap["FILL8"]="NO_OUTPUT";
    logicMap["FILL4"]="NO_OUTPUT";
    logicMap["FILL2"]="NO_OUTPUT";
    logicMap["FILL1"]="NO_OUTPUT";
    logicMap["TLATSRX1"]="SEQUENTIAL";
    logicMap["TLATX1"]="SEQUENTIAL";
    logicMap["DFFSRX1"]="SEQUENTIAL";
    logicMap["DFFX1"]="SEQUENTIAL";
    logicMap["SDFFSRX1"]="SEQUENTIAL";
    logicMap["TINVX1"]="OE A Y: 2";
    logicMap["TBUFX4"]="OE A Y: 8";
    logicMap["TBUFX8"]="OE A Y: 8";
    logicMap["TBUFX1"]="OE A Y: 8";
    logicMap["TBUFX2"]="OE A Y: 8";
    logicMap["ADDFX1"]="A B CI S: 150 CO: 232";
    logicMap["ADDHX1"]="A B S: 6 CO: 8";
    logicMap["CLKBUFX2"]="A Y: 2";
    logicMap["CLKBUFX3"]="A Y: 2";
    logicMap["CLKBUFX1"]="A Y: 2";
    logicMap["MX2X1"]="A B S0 Y: 216";
    logicMap["XOR2X1"]="A B Y: 6";
    logicMap["OAI33X1"]="A0 A1 A2 B0 B1 B2 Y: 72340172838076927";
    logicMap["OAI22X1"]="A0 A1 B0 B1 Y: 4383";
    logicMap["OAI21X1"]="A0 A1 B0 Y: 87";
    logicMap["AOI22X1"]="A0 A1 B0 B1 Y: 1911";
    logicMap["AOI21X1"]="A0 A1 B0 Y: 37";
    logicMap["BUFX3"]="A Y: 2";
    logicMap["OR4X1"]="A B C D Y: 65534";
    logicMap["OR2X1"]="A B Y: 14";
    logicMap["NOR4X1"]="A B C D Y: 1";
    logicMap["NOR3X1"]="A B C Y: 1";
    logicMap["NOR2X1"]="A B Y: 1";
    logicMap["NAND4X1"]="A B C D Y: 32767";
    logicMap["BUFX1"]="A Y: 2";
    logicMap["NAND3X1"]="A B C Y: 127";
    logicMap["NAND2X2"]="A B Y: 7";
    logicMap["NAND2X1"]="A B Y: 7";
    logicMap["AND2X1"]="A B Y: 8";
    logicMap["INVX8"]="A Y: 1";
    logicMap["INVX4"]="A Y: 1";
    logicMap["INVX2"]="A Y: 1";
    logicMap["INVX1"]="A Y: 1";
    logicMap["FILL"]="NO_OUTPUT";
    logicMap["TLATSR"]="SEQUENTIAL";
    logicMap["TLAT"]="SEQUENTIAL";
    logicMap["DFFSR"]="SEQUENTIAL";
    logicMap["DFF"]="SEQUENTIAL";
    logicMap["SDFFSR"]="SEQUENTIAL";
    logicMap["TINV"]="OE A Y: 2";
    logicMap["TBUF"]="OE A Y: 8";
    logicMap["ADDF"]="A B CI S: 150 CO: 232";
    logicMap["ADDH"]="A B S: 6 CO: 8";
    logicMap["CLKBUF"]="A Y: 2";
    logicMap["MX2"]="A B S0 Y: 216";
    logicMap["XOR2"]="A B Y: 6";
    logicMap["OAI33"]="A0 A1 A2 B0 B1 B2 Y: 72340172838076927";
    logicMap["OAI22"]="A0 A1 B0 B1 Y: 4383";
    logicMap["OAI21"]="A0 A1 B0 Y: 87";
    logicMap["AOI22"]="A0 A1 B0 B1 Y: 1911";
    logicMap["AOI2"]="A0 A1 B0 Y: 37";
    logicMap["BUF"]="A Y: 2";
    logicMap["BUF4"]="A Y: 2";
    logicMap["OR4"]="A B C D Y: 65534";
    logicMap["OR3"]="A B C Y: 254";
    logicMap["OR2"]="A B Y: 14";
    logicMap["NOR4"]="A B C D Y: 1";
    logicMap["NOR3"]="A B C Y: 1";
    logicMap["NOR2"]="A B Y: 1";
    logicMap["NAND4"]="A B C D Y: 32767";
    logicMap["NAND3"]="A B C Y: 127";
    logicMap["NAND2"]="A B Y: 7";
    logicMap["AND2"]="A B Y: 8";
    logicMap["INV"]="A Y: 1";
    logicMap["INV2"]="A Y: 1";

}

string getLogicValueString(oaInst* inst)
{
    oaDesign *master = inst->getMaster();
    assert(master);
    string masterName = getNameFromOA(master);
    if(logicMap.find(masterName) != logicMap.end())
    {
        return logicMap.find(masterName)->second;
    }
    else
    {
        cerr<<"Do not know the logic value of "<<masterName<<endl;
        return "UNKNOWN_GATE";
    }
}

void writeLogicFile(const oaBlock *block, ostream& logic)
{
    logic << "UCLA logic 1.0" << endl;
    logic << "#Created : by oa2bk v1.0b" <<endl;
    logic << "#User : fill in later" << endl;
    logic << "#Platform : fill in later" << endl;
    logic << endl;

    oaNativeNS     ns;

    // Print a line for each Term on this Net
    oa::oaIter<oaTerm>      tIter(block->getTerms());
    while (oaTerm   *term = tIter.getNext()) 
    {
        oaString    tStr;
        oaTermType  ttype = term->getTermType();

        term->getName(ns, tStr);


        logic << "\t" << tStr << " ";
        switch (ttype)
        {
            case oacInputTermType :    logic << "PRIMARY_INPUT"; break;
            case oacOutputTermType :   logic << "PRIMARY_OUTPUT"; break;
            case oacInputOutputTermType : 
            default :
                                       logic << "PRIMARY_BIDIRECTIONAL"; break;
        }


        logic<<endl;          

    }


    oa::oaIter<oaInst> iIter(block->getInsts());
    while (oaInst   *inst = iIter.getNext()) 
    {
        oaString        iStr;
        oaBox           bbox;

        inst->getName(ns, iStr);
        vector<oaInstTerm*> inputs, outputs; 
        getInputs(inst, inputs);
        getOutputs(inst, outputs);

        logic << iStr <<"\t";
        logic << getLogicValueString(inst)<<endl;

    }
}


