/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Philip Chong <pchong@cadence.com>

ChangeLog:
2004-09-15: ChangeLog started
*/

#include <algorithm>
#include <iostream>
#include <vector>
#include "oaDesignDB.h"
#include "oagUtilOptionParser.h"

using namespace oa;

//-----------------------------------------------------------------------------

/*! Find the total area of cells in the given block
 */
double
findArea(oaBlock *block) {
    double totalArea = 0.0;

    oaIter<oaInst> instIter(block->getInsts());
    while (oaInst *i = instIter.getNext()) {
        oaDesign *master = i->getMaster();
        if (!master) {
            oaString nameString;
            i->getName(oaNativeNS(), nameString);
            std::cerr << "Warning: black box found, assuming zero area:"
                      << nameString << std::endl;
        } else {
            oaBlock *masterBlock = master->getTopBlock();
            assert(masterBlock);
            oaPRBoundary *bound = oaPRBoundary::find(masterBlock);
            assert(bound);
            oaBox bbox;
            bound->getBBox(bbox);
            oaUInt8 cellArea = static_cast<oaUInt8>(bbox.getWidth()) *
                static_cast<oaUInt8>(bbox.getHeight());
            oaTech *tech = master->getTech();
            assert(tech);
            totalArea +=
                tech->dbuToUUArea(oaViewType::get(oacMaskLayout), cellArea);
        }
    }
    return totalArea;
}

//-----------------------------------------------------------------------------

/*! Find the narrowest site def for the given tech
 */
oaSiteDef *
findSiteDef(oaTech *tech) {
    oaSiteDef *r = 0;

    oaIter<oaSiteDef> siteIter(tech->getSiteDefs());
    while (oaSiteDef *i = siteIter.getNext()) {
        if (i->getSiteDefType() == oacCoreSiteDefType) {
            if (!r || r->getWidth() > i->getWidth()) {
                r = i;
            }
        }
    }
    return r;
}

//-----------------------------------------------------------------------------

/*! Find metal3 layer
 */
oaLayer *
findMetal3(oaTech *tech)
{
    oaIter<oaLayer> oaLayerIter(tech->getLayers());
    while (oaLayer *i = oaLayerIter.getNext()) {
        if (i->getType() != oacPhysicalLayerType) {
            continue;
        }
        oaMaterial m = (static_cast<oaPhysicalLayer *>(i))->getMaterial();
        if (m != oacMetalMaterial) {
            continue;
        }
        int metalLayersBelow = 0;
        oaPhysicalLayer *belowLayer = (static_cast<oaPhysicalLayer *>(i))
            ->getLayerBelow(oacMetalMaterial);
        while (belowLayer) {
            metalLayersBelow++;
            belowLayer =
                belowLayer->getLayerBelow(oacMetalMaterial);
        }
        if (metalLayersBelow == 2) {
            return i;
        }
    }
    return 0;
}

//-----------------------------------------------------------------------------

/*!
  This program generates the physical view of a design in Open Access 
  from its netlist view.
*/
int
main(int        argc,
     const char *argv[])
{
    static const oaNativeNS     nativeNS;

    oagUtil::OptionParser parser("Generate the physical view of a design in "
                                 "OpenAccess from its netlist view.");

    oagUtil::Option *techlibOpt = parser.add("techlib",
            "Input technology library name", true, "library");
    oagUtil::Option *libOpt =
        parser.add("lib", "Input library name", true, "library");
    oagUtil::Option *cellOpt =
        parser.add("cell", "Input cell name", true, "cell");
    oagUtil::Option *viewOpt =
        parser.add("view", "Input view name (netlist)", false, "view");
    oagUtil::Option *outViewOpt = 
        parser.add("outView", "Output view name (physical)", false, "view");
    oagUtil::Option *utilOpt =
        parser.add("util", "Design utilization (0.70)", false, "percent");

    if(!parser.parse(argc, argv)) {
        std::cout << parser.getMessage();
        return 1;
    }

    try {
        oaDesignInit();
        oaLibDefList::openLibs();

        const oaScalarName  techLibName(nativeNS, techlibOpt->getValue());
        const oaScalarName  libName(nativeNS, libOpt->getValue());
        const oaScalarName  cellName(nativeNS, cellOpt->getValue());
        const oaScalarName  viewName(nativeNS, viewOpt->isGiven() ?
                                     viewOpt->getValue(): "netlist");
        const oaScalarName  outViewName(nativeNS, outViewOpt->isGiven() ?
                                        outViewOpt->getValue() : "physical");

        double utilization = utilOpt->isGiven() ?
                             std::atof(utilOpt->getValue()) : 0.70;

        // open tech
        oaTech *tech = oaTech::open(techLibName);
        if (!tech) { 
            std::cerr << "ERROR : Could not open techlib" << std::endl;
            return 1; 
        }   

        // open original netlist view
        oaDesign *netlistCV =
            oaDesign::open(libName, cellName, viewName, 'r');
        if (!netlistCV) {
            std::cerr << "ERROR: Could not open original design" << std::endl;
            return 1;
        }

        // merge logically equivalent nets in original netlist view
        oaBlock *topNetlistBlock = netlistCV->getTopBlock();
        assert(topNetlistBlock);
        oaIter<oaNet> netIter(topNetlistBlock->getNets(
              oacNetIterSingleBit | oacNetIterNotImplicit));
        while (oaNet *n = netIter.getNext()) {
            oaBitNet *bn = static_cast<oaBitNet *>(n);
            oaBitNet *en = bn->getPreferredEquivalent();
            if (en != bn) {
                en->merge(bn);
            }
        }

        // create new physical view
        oaDesign *physCV = oaDesign::open(libName, cellName, outViewName,
                                          oaViewType::get(oacMaskLayout), 'w');

        // embed netlist into physical view
        oaModule *topMod = oaModule::embed(physCV, netlistCV);
        physCV->setTopModule(topMod);

        oaBlock *topBlock = physCV->getTopBlock();
        assert(topBlock);

        double totalArea = findArea(topBlock);
        if (totalArea == 0.0) {
            std::cerr << "ERROR : Design has zero area" << std::endl;
            return 1;
        }
        unsigned int numTerms = topBlock->getTerms().getCount();
        if (!numTerms) {
            std::cerr << "ERROR : Design has no terms" << std::endl;
            return 1;
        }

        oaSiteDef *mySiteDef = findSiteDef(tech);
        assert(mySiteDef);

        // compute die width and height
        double width = sqrt(totalArea / utilization);
        oaInt4 dbWidth =
            tech->uuToDBUDistance(oaViewType::get(oacMaskLayout), width);
        oaInt4 numSites = dbWidth / mySiteDef->getWidth() + 1;
        dbWidth = numSites * mySiteDef->getWidth();

        double height = sqrt(totalArea / utilization);
        oaInt4 dbHeight =
            tech->uuToDBUDistance(oaViewType::get(oacMaskLayout), height);
        oaInt4 numRows = dbHeight / mySiteDef->getHeight() + 1;
        dbHeight = numRows * mySiteDef->getHeight();

        oaConstraintGroup *cg =
            oaConstraintGroup::find(tech, "LEFDefaultRouteSpec");
        assert(cg);

        // generate pins for terms and assign to boundary
        double jump = (static_cast<double>(dbWidth) + dbHeight)
                      * 2.0 / numTerms;
        jump = static_cast<int>(jump / mySiteDef->getWidth());
        assert(jump >= 1.0);
        jump *= mySiteDef->getWidth();
        double pos = 0.0;

        std::vector<double> posVector(numTerms);

        for (unsigned int i = 0; i < numTerms; ++i) {
            posVector[i] = pos;
            pos += jump;
        }
        std::random_shuffle(posVector.begin(), posVector.end());

        // find the route layer - Metal3
        oaLayer *routeLayer = findMetal3(tech);
        assert(routeLayer);
        oaLayerNum routeLayerNum(routeLayer->getNumber());

        // get route width constraint for metal 3 to compute size of pins
        oaConstraint *cons = oaLayerConstraint::find(cg, routeLayerNum,
                             oaLayerConstraintDef::get(oacMinWidth));
        oaValue *val = cons->getValue();
        oaInt4 m3Width = static_cast<oaIntValue *>(val)->get();

        oaIter<oaTerm> termIter(topBlock->getTerms());
        while (oaTerm *i = termIter.getNext()) {
            pos = posVector.back();
            double x, y;

            if (pos < dbWidth) {
                x = pos;
                y = 0.0;
            } else if (pos < dbWidth + dbHeight) {
                x = dbWidth;
                y = pos - dbWidth;
            } else if (pos < 2.0 * dbWidth + dbHeight) {
                x = 2.0 * dbWidth + dbHeight - pos;
                y = dbHeight;
            } else {
                x = 0.0;
                y = 2.0 * dbWidth + 2.0 * dbHeight - pos;
            }
            oaInt4 ux = static_cast<oaInt4>(x);
            oaInt4 uy = static_cast<oaInt4>(y);
            oaBox pinBox(ux - m3Width / 2, uy - m3Width / 2,
                         ux + m3Width / 2, uy + m3Width / 2);
            oaRect *pinRect = oaRect::create(topBlock, routeLayerNum,
                              oavPurposeNumberDrawing, pinBox);
            assert(pinRect);
            oaPin *pin = oaPin::create(i);
            assert(pin);
            pinRect->addToPin(pin);
            pin->setPlacementStatus(oacFixedPlacementStatus);
            posVector.pop_back();
        }

        oaPointArray prBoundPoints(4);
        oaPoint p0(0, 0);
        prBoundPoints.append(p0);
        oaPoint p1(0, dbHeight);
        prBoundPoints.append(p1);
        oaPoint p2(dbWidth, dbHeight);
        prBoundPoints.append(p2);
        oaPoint p3(dbWidth, 0);
        prBoundPoints.append(p3);
        oaPRBoundary *boundary = oaPRBoundary::create(topBlock, prBoundPoints);

        oaBox cBox(p0, p2);
        oaCoreBoxSpec coreSpec(cBox);
        coreSpec.setSiteDef(mySiteDef);
        coreSpec.setRowFlipType(oacEvenRowFlipType);
        boundary->setCoreBoxSpec(coreSpec);

        // make explicit rows
        for (int nr = 0; nr < numRows; ++nr) {
            oaPoint rowOrigin(0, nr * mySiteDef->getHeight());
            oaOrient siteOrient = (nr % 2) ? oacR0 : oacMX;
            oaRow::create(topBlock, mySiteDef, rowOrigin, oacR0, 
                          numSites, siteOrient);
        }

        // generate routing tracks
        oaIter<oaLayer> oaLayerIter(tech->getLayers());
        while (oaLayer *i = oaLayerIter.getNext()) {
            if (i->getType() != oacPhysicalLayerType) {
                continue;
            }
            oaPhysicalLayer *pl = static_cast<oaPhysicalLayer *>(i);
            if (pl->getMaterial() != oacMetalMaterial) {
                continue;
            }
            oaLayerNum lNum = i->getNumber();
            bool horiz;
            oaConstraint *cons;
            oaUInt4 pitch;
            oaUInt4 nTracks;
            if (pl->getPrefRoutingDir() == oacHorzPrefRoutingDir) {
                horiz = true;
                cons = oaLayerConstraint::find(cg, lNum,
                       oaLayerConstraintDef::get(oacHorizontalRouteGridPitch));
                assert(cons);
                pitch = static_cast<oaIntValue *>(cons->getValue())->get();
                nTracks = dbHeight / pitch + 1;
            } else if (pl->getPrefRoutingDir() == oacVertPrefRoutingDir) {
                horiz = false;
                cons = oaLayerConstraint::find(cg, lNum,
                       oaLayerConstraintDef::get(oacVerticalRouteGridPitch));
                assert(cons);
                pitch = static_cast<oaIntValue *>(cons->getValue())->get();
                nTracks = dbWidth / pitch + 1;
            } else {
                assert(0);
            }
            oaTrackPattern::create(topBlock, horiz, 0, nTracks, pitch, lNum);
            oaPhysicalLayer *lBelow = pl->getLayerBelow(oacMetalMaterial);
            if (lBelow) {
                lNum = lBelow->getNumber();
                oaTrackPattern::create(topBlock, horiz, 0, nTracks,
                                       pitch, lNum);
            }
            oaPhysicalLayer *lAbove = pl->getLayerAbove(oacMetalMaterial);
            if (lAbove) {
                lNum = lAbove->getNumber();
                oaTrackPattern::create(topBlock, horiz, 0, nTracks,
                                       pitch, lNum);
            }
        }

        tech->close();
        netlistCV->close();
        physCV->save();
        physCV->close();
    }

    catch(oaException &e) {
        std::cerr << "ERROR: " << e.getMsg() << std::endl;
        return 1;
    }

    return 0;
}

//-----------------------------------------------------------------------------
