/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: David Papa <iamyou@umich.edu>

ChangeLog:
2004-09-15: ChangeLog started
*/

#include <iostream>
#include <string>
#include <vector>
//#include <ext/hash_map>
//using __gnu_cxx::hash_map;
#include <map>
using std::map;
#include "oaDesignDB.h"

using std::cout;
using std::cerr;
using std::endl;
using std::string;
using std::vector;

using oa::oaDesign;
using oa::oaInstTerm;
using oa::oaTerm;
using oa::oaIter;
using oa::oaInst;
using oa::oaBlock;
using oa::oaDesignInit;
using oa::oaLibDefList;
using oa::oaNativeNS;
using oa::oaScalarName;
using oa::oaCellView;
using oa::oaTermType;
using oa::oaNet;
using oa::oacClockSigType;
using oa::oacOutputTermType;
using oa::oacInputTermType;

template <typename T> std::string getNameFromOA(T i)
{
   oa::oaNativeNS ns;
   oa::oaString name;
   i->getName(ns,name);
   return std::string(static_cast<const char*>(name) );
}

typedef enum  { White = 0, Grey = 1, Black = 2 } stateColor;
  
template <> std::string getNameFromOA<oa::oaDesign*>(oa::oaDesign* i);

template <> std::string getNameFromOA<oa::oaDesign*>(oa::oaDesign* i)
  {
     oa::oaNativeNS ns;
     oa::oaString name;
     i->getCellName(ns,name);
     return std::string(static_cast<const char*>(name) );
  }

void findPrimaryOutputs(vector<oaInst*>& outputList, const oaBlock * const block)
{
    try
    {
        oaIter<oaInst>   oaInstIter(block->getInsts());
        while (oaInst*  inst = oaInstIter.getNext()) 
        {
            oaIter<oaInstTerm>   oaITIter(inst->getInstTerms());
            while (oaInstTerm*  instTerm = oaITIter.getNext()) 
            {
                //if its an output and there is no net
                oaTerm* term = instTerm->getTerm();
                oaTermType termType = term->getTermType();
                if( termType == oacOutputTermType) 
                {
                    oaNet* net = instTerm->getNet();
                    int degree = net->getInstTerms().getCount();
                    assert(degree >= 0 && "Net degree must be at least 0");
                    if(degree < 2)
                        outputList.push_back(inst);
                }
            }
        }
    }
    catch (oa::oaException &excp) 
    {
        printf("Error: %s\n", (const char *)excp.getMsg());
        exit(1);
    }
}

void findPrimaryInputs(vector<oaInst*>& inputList, const oaBlock * const block)
{
    try
    {
        oaIter<oaInst>   oaInstIter(block->getInsts());
        while (oaInst*  inst = oaInstIter.getNext()) 
        {
            oaIter<oaInstTerm>   oaITIter(inst->getInstTerms());
            while (oaInstTerm*  instTerm = oaITIter.getNext()) 
            {
                //if its an input and there is no net
                oaTerm* term = instTerm->getTerm();
                oaTermType termType = term->getTermType();
                if( termType == oacInputTermType) 
                {
                    oaNet* net = instTerm->getNet();
                    int degree = net->getInstTerms().getCount();
                    assert(degree >= 0 && "Net degree must be at least 0");
                    if(degree < 2)
                        inputList.push_back(inst);
                    else
                    {
                       bool noDriverOnNet = true;
                       oaIter<oaInstTerm> netFindDriverIt(net->getInstTerms());
                   
                       while(oaInstTerm* itITonNet = netFindDriverIt.getNext())
                       {
                          noDriverOnNet = noDriverOnNet && 
                                         ( itITonNet->getTerm()->getTermType() != oacOutputTermType);

                          //if a signal has no output pin driving, but its a clock signal
                          //then it is also not a primary input for these purposes
                          oaDesign *master = itITonNet->getInst()->getMaster();
                          assert(master);
                          oaBlock *masterBlock = master->getTopBlock();
                          assert(masterBlock);
                          oaIter<oaTerm> masterTermIter(masterBlock->getTerms());
                          while (oaTerm *j = masterTermIter.getNext()) 
                          {
                             if(getNameFromOA(j) == getNameFromOA(itITonNet->getTerm()))
                             {
                               oaNet *masterNet = j->getNet();
                               if (masterNet->getSigType() == oacClockSigType)
                               {
                                  noDriverOnNet = false;
                                  break;
                               }
                             }
                          }

                          if(!noDriverOnNet)
                            break; 
                       }
                       if(noDriverOnNet)
                           inputList.push_back(inst);
                    }
                }
            }
        }
    }
    catch (oa::oaException &excp) 
    {
        printf("Error: %s\n", (const char *)excp.getMsg());
        exit(1);
    }
    //end find Primary Inputs
}

void findClockedInsts(vector<oaInst*>& clockedList, const oaBlock* const block)
{
    try
    {
        oaIter<oaInst>   oaInstIter(block->getInsts());
        while (oaInst*  inst = oaInstIter.getNext()) 
        {
            oaDesign *master = inst->getMaster();
            assert(master);
            oaBlock *masterBlock = master->getTopBlock();
            assert(masterBlock);
            oaIter<oaTerm> masterTermIter(masterBlock->getTerms());
            while (oaTerm *j = masterTermIter.getNext()) 
            {
                oaNet *masterNet = j->getNet();
                if (masterNet->getSigType() == oacClockSigType)
                {
                    clockedList.push_back(inst);
                    break;
                }
            }
        }
    }
    catch (oa::oaException &excp) 
    {
        printf("Error: %s\n", (const char *)excp.getMsg());
        exit(1);
    }
}

vector<unsigned> findInputConsumers(unsigned instNum, const vector<oaInst*>& idxToInst, const map<oaInst*, unsigned>& instToIdx)
{
    vector<unsigned> inputConsumers;
    try
    {
        assert(instNum < idxToInst.size() && 
               "Attempting to find input consumers of an index out of range");
        oaInst* inst = idxToInst[instNum];
        oaIter<oaInstTerm>   oaITIter(inst->getInstTerms());
        while (oaInstTerm*  instTerm = oaITIter.getNext()) 
        {
            //if its an input find the driver of the net 
            oaTerm* term = instTerm->getTerm();
            oaTermType termType = term->getTermType();
            if( termType == oacInputTermType) 
            {
                oaNet* net = instTerm->getNet();
                oaIter<oaInstTerm> oaITonNetIter(net->getInstTerms());
                while(oaInstTerm* instTermOnNet = oaITonNetIter.getNext())
                {
                   oaTerm* termOnNet = instTermOnNet->getTerm();
                   oaTermType termTypeOnNet = termOnNet->getTermType();
                   if( termTypeOnNet == oacOutputTermType)
                   {
                      oaInst* instOnNet = instTermOnNet->getInst();
                      unsigned instIdx = (instToIdx.find(instOnNet)->second);
                      inputConsumers.push_back(instIdx);
                      break;
                   }
                }
            }
        }
    }
    catch (oa::oaException &excp) 
    {
        printf("Error: %s\n", (const char *)excp.getMsg());
        exit(1);
    }
   
    return inputConsumers;
}

vector<unsigned> findOutputConsumers(unsigned instNum, const vector<oaInst*>& idxToInst, const map<oaInst*, unsigned>& instToIdx)
{
    vector<unsigned> outputConsumers;
    try
    {
        assert(instNum < idxToInst.size() && 
               "Attempting to find input consumers of an index out of range");
        oaInst* inst = idxToInst[instNum];
        //cout<<"DPDEBUG: finding output consumers of inst: \""<<getNameFromOA(inst)<<endl;
        oaIter<oaInstTerm>   oaITIter(inst->getInstTerms());
        while (oaInstTerm*  instTerm = oaITIter.getNext()) 
        {
            //if its an output find the sinks of the net 
            oaTerm* term = instTerm->getTerm();
            oaTermType termType = term->getTermType();
            if( termType == oacOutputTermType) 
            {
                oaNet* net = instTerm->getNet();
                oaIter<oaInstTerm> oaITonNetIter(net->getInstTerms());
                while(oaInstTerm* instTermOnNet = oaITonNetIter.getNext())
                {
                   //cout<<"\tDPDEBUG: exploring net found instTerm for inst: \""<<getNameFromOA(instTermOnNet->getInst())<<endl;
                   oaTerm* termOnNet = instTermOnNet->getTerm();
                   oaTermType termTypeOnNet = termOnNet->getTermType();
                   //cout<<"\tDPDEBUG: type of term \""<<getNameFromOA(termOnNet)<<"\" is: "<<termTypeOnNet<<endl;
                   if( termTypeOnNet == oacInputTermType)
                   {
                      unsigned instIdx = (instToIdx.find(instTermOnNet->getInst())->second);
                      outputConsumers.push_back(instIdx);
                   }
                   
                   //oaInst* adjInst = instTermOnNet->getInst();
                   //cout<<"\tDPDEBUG stats for inst \""<<getNameFromOA(adjInst)<<"\":"<<endl;
                   //oaIter<oaInstTerm> itDebugIt(adjInst->getInstTerms());
                   //while(oaInstTerm* dbgIT = itDebugIt.getNext())
                   //{
                   //   cout<<"\t\tDPDEBUG inst term connects term \""<<
                   //                getNameFromOA(dbgIT->getTerm())<<"\" with type "<<dbgIT->getTerm()->getTermType()<<endl;
                   //} 
                }
            }
        }
    }
    catch (oa::oaException &excp) 
    {
        printf("Error: %s\n", (const char *)excp.getMsg());
        exit(1);
    }
   
    return outputConsumers;
}

double sumOverOutputs(unsigned instNum, const vector<double>& outputPathCount, const vector<oaInst*>& idxToInst, const map<oaInst*, unsigned>& instToIdx)
{
    double rval = 0.0;
    vector<unsigned> outputConsumers = findOutputConsumers(instNum, idxToInst, instToIdx); 
    for(unsigned i = 0; i < outputConsumers.size(); ++i)
        rval += outputPathCount[outputConsumers[i]]; 
    return rval;
}

void DFS_Visit_input(unsigned instNum, vector<unsigned>& input_color, vector<unsigned>& output_color, vector<double>& outputPathCount, const vector<oaInst*>& idxToInst, const map<oaInst*, unsigned>& instToIdx)
{
    if(output_color[instNum] == Black)
    {
      input_color[instNum] == Black;
      return;
    }
    input_color[instNum] = Grey; //a white vertex has just been discovered
    vector<unsigned> outputConsumers = findOutputConsumers(instNum, idxToInst, instToIdx);
    //for each v that consumes an output of instNum //Explore edge(u,v) 
    for(unsigned adjInstIt = 0; adjInstIt < outputConsumers.size(); ++adjInstIt) 
    {
        unsigned adjInstNum = outputConsumers[adjInstIt];
        //if the output is black then it is a clocked element or primary output
        //and we should stop here, we will begin from clocked elements that 
        //are white from the inputDFS function
        if( input_color[adjInstNum] == White && output_color[adjInstNum] != Black )
        {
            DFS_Visit_input(adjInstNum, input_color, output_color, outputPathCount, idxToInst, instToIdx);
        }
        if( output_color[adjInstNum] == Black )
          input_color[adjInstNum] = Black;
    }
    input_color[instNum] = Black; //Blacken u; it is finished
    //whenever we blacken a node (the input) we know the total
    //number of paths on its output because we have definately 
    //visited all of its outputs already
    outputPathCount[instNum] = sumOverOutputs(instNum, outputPathCount, idxToInst, instToIdx);
}

void inputDFS(vector<unsigned>& input_color, vector<unsigned>& output_color, vector<double>& outputPathCount, const vector<oaInst*>& idxToInst, const map<oaInst*, unsigned>& instToIdx, const vector<unsigned>& clockedList, vector<unsigned>& outputList)
{
  for(unsigned i = 0; i < input_color.size(); ++i)
  {
    input_color[i] = White;
    output_color[i] = White;
  }
  for(unsigned i = 0; i < clockedList.size(); ++i)
  {
    output_color[clockedList[i]] = Black;
    outputPathCount[clockedList[i]] = 1.0; //TODO, clocked elements may have multiple outputs
  }
  for(unsigned i = 0; i < outputList.size(); ++i)
  {
    output_color[outputList[i]] = Black;
    outputPathCount[outputList[i]] = 1.0;
  }

  for(unsigned i = 0; i < input_color.size(); ++i)
  {
     if(input_color[i] == White)
       DFS_Visit_input(i, input_color, output_color, outputPathCount, idxToInst, instToIdx);
  }
}

double sumOverInputs(unsigned instNum, const vector<double>& inputPathCount, const vector<oaInst*>& idxToInst, const map<oaInst*, unsigned>& instToIdx)
{
    double rval = 0.0;
    vector<unsigned> inputConsumers = findInputConsumers(instNum, idxToInst, instToIdx); 
    for(unsigned i = 0; i < inputConsumers.size(); ++i)
        rval += inputPathCount[inputConsumers[i]]; 
    return rval;
}

void DFS_Visit_output(unsigned instNum, vector<unsigned>& input_color, vector<unsigned>& output_color, vector<double>& inputPathCount, const vector<oaInst*>& idxToInst, const map<oaInst*, unsigned>& instToIdx)
{
    if(input_color[instNum] == Black)
    {
      output_color[instNum] == Black;
      return;
    }
    output_color[instNum] = Grey; //a white vertex has just been discovered
    vector<unsigned> inputConsumers = findInputConsumers(instNum, idxToInst, instToIdx);
    //for each v that consumes an input of instNum //Explore edge(u,v) 
    for(unsigned adjInstIt = 0; adjInstIt < inputConsumers.size(); ++adjInstIt) 
    {
        unsigned adjInstNum = inputConsumers[adjInstIt];
        //if the input is black then it is a clocked element or primary input
        //and we should stop here, we will begin from clocked elements that 
        //are white from the outputDFS function
        if( output_color[adjInstNum] == White && input_color[adjInstNum] != Black )
        {
            DFS_Visit_output(adjInstNum, input_color, output_color, inputPathCount, idxToInst, instToIdx);
        }
        if( input_color[adjInstNum] == Black )
          output_color[adjInstNum] = Black;
    }
    output_color[instNum] = Black;  //Blacken instNum; it is finished
    //whenever we blacken a node (the output) we know the total
    //number of paths on its input because we have definately 
    //visited all of its inputs already
    inputPathCount[instNum] = sumOverInputs(instNum, inputPathCount, idxToInst, instToIdx);
}

void outputDFS(vector<unsigned>& input_color, vector<unsigned>& output_color, vector<double>& inputPathCount, const vector<oaInst*>& idxToInst, const map<oaInst*, unsigned>& instToIdx, const vector<unsigned>& clockedList, const vector<unsigned>& inputList)
{
    for(unsigned i = 0; i < output_color.size(); ++i)
    {
        input_color[i] = White;
        output_color[i] = White;
    }
    for(unsigned i = 0; i < clockedList.size(); ++i)
    {
        input_color[clockedList[i]] = Black;
        inputPathCount[clockedList[i]] = 1.0; //TODO, clocked elements may have multiple outputs
    }
    for(unsigned i = 0; i < inputList.size(); ++i)
    {
        input_color[inputList[i]] = Black;
        inputPathCount[inputList[i]] = 1.0;
    }

    for(unsigned instNum = 0; instNum < output_color.size(); ++instNum)
    {
        if(output_color[instNum] == White)
            DFS_Visit_output(instNum, input_color, output_color, inputPathCount, idxToInst, instToIdx);
    }
}

void createBidirectionalInstIdxMap(vector<oaInst*>& idxToInst, map<oaInst*, unsigned>& instToIdx, oaBlock* block)
{
  idxToInst.clear();
  instToIdx.clear();
  unsigned instCt = block->getInsts().getCount();
  idxToInst.reserve(instCt);
  oaIter<oaInst> instIt(block->getInsts());
  while(oaInst* inst = instIt.getNext())
  {
    instToIdx[inst]=idxToInst.size();
    idxToInst.push_back(inst);
  }
}

int main(int argc, char *argv[])
{
    try
    {
        oaDesignInit();
        oaLibDefList::openLibs();
    }
    catch (oa::oaException &excp) {
        printf("Error: %s\n", (const char *)excp.getMsg());
        exit(1);
    }

    // Check number of arguments.
    if ( argc < 2 ) {
        cout<<" Use : "<<endl<<argv[0]<<" libraryName cellName viewName "<<endl;
    }

    // Get the names from the command line
    string Lib;
    if(argc < 2)
    {
      cout<<"Using default value for libraryName : \"designs\""<<endl;
      Lib = "designs"; 
    }
    else
      Lib = argv[1];

    string Cell;
    if(argc < 3)
    {
      cout<<"Using default value for cellName : \"s386_bench\""<<endl;
      Cell = "s386_bench";
    }
    else
      Cell = argv[2];

    string View;
    if(argc < 4)
    {
      cout<<"Using default value for viewName : \"physical\""<<endl;
      View = "physical";
    }
    else
      View = argv[3];
   

    // Get the names from the command line
    oa::oaString        libstr(Lib.c_str());
    oa::oaString        cellstr(Cell.c_str());
    oa::oaString        viewstr(View.c_str());

    // Treat the names on the command line as belonging 
    //to the Native
    // namespace and convert them to oaNames.
    oa::oaNativeNS      ns;
    oa::oaScalarName    libName(ns, libstr);
    oa::oaScalarName    cellName(ns, cellstr);
    oa::oaScalarName    viewName(ns, viewstr);

    oa::oaBlock* block;
    oa::oaDesign* des;

    unsigned instCount;

    try
    {
        oa::oaLib::find(libName);
    }
    catch (oa::oaException &excp) {
        printf("Error: %s\n", (const char *)excp.getMsg());
        exit(1);
    }

    try 
    {
        try
        {
            des  = oa::oaDesign::open(libName, cellName, viewName, 'a') ;
        }
        catch (oa::oaException &excp) 
        {
            printf("Can't read CellView %s %s %s (check libDefFile)\n", 
                    (const char *)libstr,
                    (const char *)cellstr, (const char *)viewstr);
            printf("\t%s\n", (const char *)excp.getMsg());
            exit(1);
        }

        block = des->getTopBlock() ;
        oa::oaBox* CoreRegionBBox = new oa::oaBox(); 
        block->getBBox(*CoreRegionBBox);
        cout << "Opened design \""<<getNameFromOA(des)<<"\", CoreBox: ("<<CoreRegionBBox->left()   << "," <<
            CoreRegionBBox->bottom() << "," <<
            CoreRegionBBox->right()  << "," <<
            CoreRegionBBox->top()    << ")."<<endl;
        instCount = block->getInsts().getCount();
        cout << "Design has a total of " << instCount << " insts." <<endl;
        delete CoreRegionBBox;
    }
    catch (oa::oaException &excp) {
        printf("Error: %s\n", (const char *)excp.getMsg());
        exit(1);
    }


    //find Clocked Elements
    vector<oaInst*> clockedList;
    findClockedInsts(clockedList, block);

    //find Primary Outputs
    vector<oaInst*> outputList;
    findPrimaryOutputs(outputList, block);

    //find Primary Inputs
    vector<oaInst*> inputList;
    findPrimaryInputs(inputList, block);

    cout<<"DPDEBUG: Counts of PI: "<<inputList.size()<<" PO: "<<outputList.size()<<" CLK: "<<clockedList.size()<<endl;

    //Performing TOPOLOGICAL-SORT from CLRS
    //Nodes in the graph will be represented by insts
    //Nodes will have two fields, input state and output state
    //each field will have one of three states {white, grey, black}
    //    while = unvisited
    //    grey = discovered but not finished
    //    black = finished, or stop exploring here I am a leaf node

    vector<unsigned> input_color(instCount);
    vector<unsigned> output_color(instCount);
    vector<oaInst*> idxToInst;
    map<oaInst*, unsigned> instToIdx;
    createBidirectionalInstIdxMap(idxToInst, instToIdx, block);

    //convert lists to vector<unsigned>
    vector<unsigned> clockedListU;
    cout<<"Clocked elements:"<<endl;
    for(unsigned i = 0; i < clockedList.size(); ++i)
    {
        clockedListU.push_back( instToIdx.find(clockedList[i])->second );
        cout<<"\tInst \""<<getNameFromOA(clockedList[i])<<endl;
    }
    vector<unsigned> outputListU;
    cout<<"Output elements:"<<endl;
    for(unsigned i = 0; i < outputList.size(); ++i)
    {
        outputListU.push_back( instToIdx.find(outputList[i])->second );
        cout<<"\tInst \""<<getNameFromOA(outputList[i])<<endl; 
    }
    vector<unsigned> inputListU;
    cout<<"Input elements:"<<endl;
    for(unsigned i = 0; i < inputList.size(); ++i) 
    {
        inputListU.push_back( instToIdx.find(inputList[i])->second );
        cout<<"\tInst \""<<getNameFromOA(inputList[i])<<endl; 
    }




    vector<double> outputPathCount(instCount);
    inputDFS( input_color, output_color, outputPathCount, idxToInst, instToIdx, clockedListU, outputListU); 
    vector<double> inputPathCount(instCount);
    outputDFS(input_color, output_color, inputPathCount, idxToInst, instToIdx, clockedListU, inputListU);

    cout<<"Inst Path Report:"<<endl;
    for(unsigned i = 0; i < instCount; ++i)
    {
        cout<<"Inst \""<<getNameFromOA(idxToInst[i])<<"\"  InputPaths: "<<inputPathCount[i]<<" OutputPaths: "<<outputPathCount[i]<<endl;
    }


    cout<<"Net Path Report:"<<endl;
    try
    {
        oaIter<oaNet>   oaNetIter(block->getNets());
        while (oaNet*  net = oaNetIter.getNext()) 
        {
            double inputCt  = 0.0;
            double outputCt = 0.0;
            oaIter<oaInstTerm>   oaITIter(net->getInstTerms());
            while (oaInstTerm*  itIt = oaITIter.getNext()) 
            {
                oaTermType oatt = itIt->getTerm()->getTermType();
                if ( oatt == oacInputTermType)
                    inputCt += inputPathCount[ instToIdx[ itIt->getInst() ] ];
                else if ( oatt == oacOutputTermType)
                    outputCt += outputPathCount[ instToIdx[ itIt->getInst() ] ];
            }
            cout<<"Net \""<<getNameFromOA(net)<<"\" : "<< 
                ( inputCt * outputCt ) << endl;

        }
    }
    catch (oa::oaException &excp) {
        printf("Error: %s\n", (const char *)excp.getMsg());
        exit(1);
    }

}


