/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Zhong Xiu <zxiu@andrew.cmu.edu>

ChangeLog:
2004-09-15: ChangeLog started
*/

#if !defined(oagTimerTPoint_P)
#define oagTimerTPoint_P

#include <float.h>
#include <vector>
#include <map>
#include "oaDesignDB.h"

#include "oagTimerModel.h"

#undef AFSHIN

using namespace oa;
using namespace std;

namespace oagTimer {

/*! the types of a timing point */
typedef enum {
  TIMING_POINT_NULL, //!< null pointer type
  TIMING_POINT_SIGNAL_IN, //!< input type instTerm
  TIMING_POINT_SIGNAL_OUT, //!< output type instTerm
  TIMING_POINT_CLOCK, //!< clock type instTerm
  TIMING_POINT_PI, //!< primary input type term
  TIMING_POINT_PO, //!< primary output type term
  TIMING_POINT_PI_CLOCK //!< primary input and clock type term
} timingPointType;

class TPointMaster;

//! The TPoint class defines the timing point.
/*! 
  The TPoint class defines the timing point which is
  associated with either an oaTerm or an oaInstTerm. It's used
  to do either Full or Incremental timing analysis. The timing
  point contains all the necessary timing information, like
  delay, load, arrival time, required arrival time and so on.
*/

struct timingData {
  int clockIndex;
  /*! The rise arrival time at this timing point. */
  TimeType riseArr;
  /*! The required rise arrival time at this timing point. */
  TimeType riseReq;
  /*! The fall arrival time at this timing point. */
  TimeType fallArr;
  /*! The required fall arrival time at this timing point. */
  TimeType fallReq;
  /*! The early fall time at this timing point. */
  TimeType fallEarly;
  /*! The early rise time at this timing point. */
  TimeType riseEarly;
  /*! The rise slew rate at this timing point. */
  DelayType riseSlew;
  /*! The fall slew rate at this timing point. */
  DelayType fallSlew;
};

struct DPData {
    oaString cellSize;
    double inCap;
    double rAT;
    double arr;
    map<oaOccInstTerm*,oaString> faninMap;
    map<oaOccInstTerm*,oaString> fanoutMap;
};

struct lrData {
    oaString cellSize;
    double inCap;
    double fObj;
    map<oaOccInstTerm*,oaString> fanoutMap;
};

class TPoint {
  friend class Timer;
  friend class DuetSensitivity;
  friend class Opt;
  friend class Util;
  friend class TimerUtil;
  friend class lpSA;
  friend class LR;
  friend class DP;
  friend class SubTimer;

  public:
    /** @name Constructors and desctructor
     */
    //@{
    /*! The constructor of an TPoint associated with an oaTerm 
        or an oaInstTerm. */
    TPoint() : type(TIMING_POINT_NULL) {
      riseArr = riseReq = fallArr = fallReq = 0.0;
      netLoad = load = 0.0;
      res = 0.0;
      netDelay = 0.0;
      fallSlew = riseSlew = 0.0;
      atValid = ratValid = netValid = false;
      revTopValid = inverting = swapFlag = flag = slewValid = riseSlewValid = fallSlewValid = false;
      //currentRiseSlew = currentFallSlew = prevRiseSlew = prevFallSlew = 0.0;
      delay1 = delay2 = 0.0; 
      riseDelay = fallDelay = 0.0; 
      isToFF = isFromFF = isFromPI = isToPO = false;
      oldRiseSlew = oldFallSlew = 0.0;
      multiCycleConstraint = 1;
      counter = 0;
      lm_old = lm = 0.0;
      dpIndex = lmIndex = -1;
      dpArr = -DBL_MAX;
      index = 0;
      clear();
    }
    /*! The destructor. */
    ~TPoint() {}
    //@}

  public:
    /** @name Clear data
     */
    //@{
    void clear();
    void clearSlew();
    //@}

  private:
    /*! The load value at this timing point. */
    double load;
    double netLoad;
    double res;

  public:
    int multiCycleConstraint;
    std::vector<timingData*> multiClockData;
    std::string getBlockName(oaOccObject *block);

    /*! For use in DP */
    std::vector<DPData*> pinDPData;
    std::map<oaOccInstTerm*,oaString> DPSolutions;
    double dpArr;
    int dpIndex;

    /*! For use in LR with DP */
    std::vector<lrData> lrDataVec;
    int lrDataIndex; //used in output pin only
    oaString lrSolution;

  private:
    /*! The rise arrival time at this timing point. */
    TimeType riseArr;
    /*! The required rise arrival time at this timing point. */
    TimeType riseReq;
    /*! The fall arrival time at this timing point. */
    TimeType fallArr;
    /*! The required fall arrival time at this timing point. */
    TimeType fallReq;
    /*! The rise slew rate at this timing point. */
    DelayType riseSlew;
    /*! The fall slew rate at this timing point. */
    DelayType fallSlew;

    /*! The net delay value of this timing point. 
     * For a SIGNAL_OUT pin this is the worst net delay seen at the fanouts*/
    DelayType netDelay;

    /*! Used in lpSA to assign gate index in the circuit */
    int index;

    /*! Use in incremental timing optimization to 
     * speedup arrival time propagation
     */
    /*
    DelayType currentRiseSlew;
    DelayType currentFallSlew;
    DelayType prevRiseSlew;
    DelayType prevFallSlew;*/

  public:
    /** @name Query functions
     */
    //@{
    /*! This function returns the rise arrival time at this timing point. */
    TimeType getRiseArr() const { 
        TimeType ra = ModelType::MAX_TIME();
        for (unsigned k = 0; k < multiClockData.size(); ++k) { 
            timingData *d = multiClockData[k];
            if (d->riseArr < ra) {
                ra = d->riseArr;
            }
        }
        return ra;
    }
    TimeType getRiseArr(int i) const { return multiClockData[i]->riseArr; }
    /*! This function returns the fall arrival time at this timing point. */
    TimeType getFallArr() const { 
        TimeType fa = ModelType::MAX_TIME();
        for (unsigned int k = 0; k < multiClockData.size(); ++k) { 
            timingData *d = multiClockData[k];
            if (d->fallArr < fa) {
                fa = d->fallArr;
            }
        }
        return fa;
    }
    TimeType getFallArr(int i) const { return multiClockData[i]->fallArr; }
    /*! This function returns the required rise arrival time 
        at this timing point. */
    TimeType getRiseReq() const { 
        TimeType rr = ModelType::MAX_TIME();
        for (unsigned k = 0; k < multiClockData.size(); ++k) { 
            timingData *d = multiClockData[k];
            if (d->riseReq < rr) {
                rr = d->riseReq;
		//std::cout << "   RR d->rise REQ: " << rr << std::endl;
            }
        }
        return rr;
    }
    TimeType getRiseReq(int i) const { return multiClockData[i]->riseReq; }
    /*! This function returns the required fall arrival time 
        at this timing point. */
    TimeType getFallReq() const { 
        TimeType fr = ModelType::MAX_TIME();
        for (unsigned int k = 0; k < multiClockData.size(); ++k) { 
            timingData *d = multiClockData[k];
            if (d->fallReq < fr) {
                fr = d->fallReq;
		//std::cout << "   FR d->fall REQ: " << fr << std::endl;
            }
        }
        return fr;
    }
    TimeType getFallReq(int i) const { return multiClockData[i]->fallReq; }
    /*! This function returns the rise slack at this timing point. */

    DelayType getRiseSlack() const {
      //afshin-begin
      DelayType rs = ModelType::MAX_DELAY();
      for (unsigned int k = 0; k < multiClockData.size(); ++k) { 
        timingData *d = multiClockData[k];
        if (d->riseReq == ModelType::MAX_TIME() || d->riseArr == (-ModelType::MAX_TIME())) {
          continue;
        }
        if (d->riseReq - d->riseArr < rs) {
          rs = d->riseReq - d->riseArr;
        }
      }
      return rs;
      //afshin-end
      if (riseReq == ModelType::MAX_TIME() || riseArr == -ModelType::MAX_TIME()) {
        return ModelType::MAX_DELAY();
      }
      DelayType t = riseReq - riseArr;
//      return (t > 0.0) ? t : 0.0;
      return t;
    }

    DelayType getRiseSlack(int i) const {
      DelayType rs = ModelType::MAX_DELAY();
      timingData *d = multiClockData[i];
      if (d->riseReq == ModelType::MAX_TIME() || d->riseArr == -ModelType::MAX_TIME()) {
        return rs;
      }
      rs = d->riseReq - d->riseArr;
      return rs;
    }

    /*! This function returns the fall slack at this timing point. */
    DelayType getFallSlack() const {
      //afshin-begin
      DelayType fs = ModelType::MAX_DELAY();
      for (unsigned int k = 0; k < multiClockData.size(); ++k) { 
        timingData *d = multiClockData[k];
        if (d->fallReq == ModelType::MAX_TIME() || d->fallArr == -ModelType::MAX_TIME()) {
          continue;
        }
        if (d->fallReq - d->fallArr < fs) {
          fs = d->fallReq - d->fallArr;
        }
      }
      return fs;
      //afshin-end
      if (fallReq == ModelType::MAX_TIME() || fallArr == -ModelType::MAX_TIME()) {
        return ModelType::MAX_DELAY();
      }
      DelayType t = fallReq - fallArr;
//      return (t > 0.0) ? t : 0.0;
      return t;
    }

    DelayType getFallSlack(int i) const {
      DelayType fs = ModelType::MAX_DELAY();
      timingData *d = multiClockData[i];
      if (d->fallReq == ModelType::MAX_TIME() || d->fallArr == -ModelType::MAX_TIME()) {
        return fs;
      }
      fs = d->fallReq - d->fallArr;
      return fs;
    }

    /*! This function returns the slack at this timing point. */
    DelayType getSlack() const {
      DelayType t1 = getRiseSlack();
      DelayType t2 = getFallSlack();
      DelayType r = (t1 < t2) ? t1 : t2;
      return r;
    }
    /*! 
      This function returns the slack value at the clock pin, 
      here the clock pin is the starting point of a path.
    */
    DelayType getStartingPointSlack() const {
      if (type == TIMING_POINT_CLOCK) {
        TimeType t1, t2;
        if (riseReq == ModelType::MAX_TIME() || riseArr == -ModelType::MAX_TIME()) {
          return ModelType::MAX_DELAY();
        }
        t1 = riseReq;
        t2 = fallReq;
        DelayType r = (t1 < t2) ? (t1 - ModelType::ZERO_TIME()) : (t2 - ModelType::ZERO_TIME());
        return r;
      } else {
        return getSlack();
      }
    }
    /*! 
      This function returns the slack value at the clock pin, 
      here the clock pin is the ending point of a path.
    */
    DelayType getEndingPointSlack(DelayType clkPeriod) const {
      return getSlack(); //afshin
      if (type == TIMING_POINT_CLOCK) {
        DelayType t1;
        if (riseReq == ModelType::MAX_TIME() || riseArr == -ModelType::MAX_TIME()) {
          return ModelType::MAX_DELAY();
        }
        t1 = clkPeriod - riseArr;
        DelayType t2 = getFallSlack();
        DelayType r = (t1 < t2) ? t1 : t2;
        return r;
      } else {
        return getSlack();
      }
    }

    /*! This function returns the load value at this timing point. */
    double getLoad() const {
      return load;
    }
    /*! This function returns the rise slew rate at this timing point. */
    DelayType getRiseSlew() const {
      return riseSlew;
    }
    DelayType getRiseSlew(int i) const {
      return multiClockData[i]->riseSlew;
    }
    /*! This function returns the fall slew rate at this timing point. */
    DelayType getFallSlew() const {
      return fallSlew;
    }
    DelayType getFallSlew(int i) const {
      return multiClockData[i]->fallSlew;
    }
    //@}

    /*! This flag indicates input arc valid used for getting reverse topological in DesignTool. */
    bool revTopValid;
    /*! This flag indicates if the required arrival time is valid. */
    bool ratValid;
    /*! This flag indicates if the arrival time is valid. */
    bool atValid;

    /*! This flag indicates if NetLoad, and NetDelay are valid */
    bool netValid;

    /*! This flag indicate whether this input pin is connected from PI */
    bool isFromPI; 
    /*! This flag indicate whether this output pin is connected to PO */
    bool isToPO; 
    /*! This flag indicate whether this input pin is connected from FF */
    bool isFromFF; 
    /*! This flag indicate whether this output pin is connected to FF */
    bool isToFF; 

    /*! This flag indicates if slew is valid, for used in incremental timing only */
    bool flag;
    bool swapFlag; 
    bool slewValid;
    bool riseSlewValid; 
    bool fallSlewValid;
    bool inverting;
    DelayType delay1;
    DelayType delay2;
    DelayType riseDelay;
    DelayType fallDelay;
    DelayType oldRiseSlew;
    DelayType oldFallSlew;

    /*! Lagrange Multiplier:  used only for LR optimization*/
    double lm;
    double lm_old;
    int lmIndex;

    /*! The type of this timing point. */
    timingPointType type;

    /** @name Type functions
     */
    //@{
    /*! This function returns true if the pin is a clock pin type. */
    bool isClock() const {
      return type == TIMING_POINT_CLOCK || type == TIMING_POINT_PI_CLOCK;
    }

    /*! This function returns true if it's a PI. */
    bool isPI() const {
      return type == TIMING_POINT_PI;
    }

    /*! This function returns true if it's a PO. */
    bool isPO() const {
      return type == TIMING_POINT_PO;
    }
    //@}

    /** @name Primary inputs timing information
     */
    //@{
    void piDriverForward(TPointMaster *tm, DelayType defaultSlew);
    //@}

///////////////////////////////////////////////////////////////////////////////

  private:
    // Static data members.
    static oaVoidPointerAppDef<oaOccInstTerm> *_instTermAppDef;
    static oaVoidPointerAppDef<oaOccTerm> *_termAppDef;
    static oaVoidPointerAppDef<oaOccInstTerm> *_clockInstTermAppDef;
    static oaVoidPointerAppDef<oaOccTerm> *_clockTermAppDef;

    /// Initialize the appDefs associated with TPoints.
    static void initAppDefs();

  public:
    /// Create a TPoint for the given OccTerm/OccInstTerm.
    ///
    /// 0 is returned if the OccTerm/OccInstTerm does not support a TPoint
    /// (i.e. is a power/ground term).
    ///
    /// It is a fatal error to create a TPoint for an
    /// OccTerm/OccInstTerm which already has a TPoint.
    ///
    /// \param oPtr OccTerm/OccInstTerm for which to create a TPoint.
    /// \return The new TPoint.
    static TPoint *create(oaOccObject *oPtr);

    /// Create a TPoint for the given OccInstTerm.
    ///
    /// 0 is returned if the OccInstTerm does not support a TPoint
    /// (i.e. is a power/ground term).
    ///
    /// It is an error to create a TPoint for an OccInstTerm which
    /// already has a TPoint.
    ///
    /// \param oPtr OccInstTerm for which to create a TPoint.
    /// \return The new TPoint.
    static TPoint *create(oaOccInstTerm *oPtr);

    /// Create a TPoint for the given OccTerm.
    ///
    /// 0 is returned if the OccTerm does not support a TPoint
    /// (i.e. is a power/ground term).
    ///
    /// It is a fatal error to create a TPoint for an OccTerm which
    /// already has a TPoint.
    ///
    /// \param oPtr OccTerm for which to create a TPoint.
    /// \return The new TPoint.
    static TPoint *create(oaOccTerm *oPtr);

    /// Get the TPoint for the given OccTerm/OccInstTerm.
    /// It is a fatal error to try to get a TPoint for
    /// an OccTerm/OccInstTerm which does not have a TPoint.
    ///
    /// \param oPtr OccTerm/OccInstTerm to query.
    /// \return The corresponding TPoint.
    static TPoint *get(oaOccObject *oPtr);

    /// Get the clock TPoint for the given OccTerm/OccInstTerm.
    /// If the OccTerm/OccInstTerm does not have a clock TPoint,
    /// create a new clock TPoint and return it.  Note that this behavior
    /// differs from the TPoint::get() method.
    ///
    /// \param bo OccTerm/OccInstTerm to query.
    /// \return The corresponding clock TPoint.
    static TPoint *getClock(oaOccObject *oPtr);

    /// Destroys the TPoint and clock TPoint
    /// for the given OccTerm/OccInstTerm.
    ///
    /// \param oPtr OccTerm/OccInstTerm to operate on.
    static void destroy(oaOccObject *oPtr);

    /// Invalidate the fanout cone associated with this TPoint.
    ///
    /// \param oPtr OccTerm/OccInstTerm associated with this TPoint.
    void invalidateFanout(oaOccObject *oPtr);

    /// Invalidate the fanin cone associated with this TPoint.
    ///
    /// \param oPtr OccTerm/OccInstTerm associated with this TPoint.
    void invalidateFanin(oaOccObject *oPtr);

    /// Invalidate the fanout cone associated with this Clock TPoint.
    ///
    /// \param oPtr OccTerm/OccInstTerm associated with this Clock TPoint.
    void invalidateClockFanout(oaOccObject *oPtr);

    /// Invalidate the fanin cone associated with this Clock TPoint.
    ///
    /// \param oPtr OccTerm/OccInstTerm associated with this Clock TPoint.
    void invalidateClockFanin(oaOccObject *oPtr);

    int counter;
///////////////////////////////////////////////////////////////////////////////

};

}

#endif
