/************************************************************
* 
* File: oagTimerOptMinDelay.cpp
* Author: Santiago Mok (smok@ee.ucla.edu)
* Created: 01-26-2011
* Last Modified: Wed 16 Feb 2011 12:32:38 PM PST
*
************************************************************/
#include "oagTimerOptMinDelay.h"
#include "oagTimerUtil.h"
#include "oagTimerDesignTool.h"

//#include "solverIF.h"

#include "assert.h"
#include <vector>
namespace oagTimer{

using namespace oa;
using namespace std;

/*---------------------------------------------------------*/
OptMinDelay::OptMinDelay(oaDesign *d, Timer *t){
      design = d;
      timing = t;
      tau = 0.01;
      alpha1=0.25;
      alpha2=tau*alpha1;

};
/*---------------------------------------------------------*/
OptMinDelay::~OptMinDelay(){}
/*---------------------------------------------------------*/
void
OptMinDelay::run(){
    Util ut;
    std::cout << "Initial Design Power:" 
	<< ut.getTotalLeakPower(design) 
	<< "  Target Delay:" << timing->getPeriod() << std::endl;
    G = DesignTool::getTopological(design,timing); 
    int graphSize = G.size(); 
    bool tolSet = false;
    bool flag;

    for(globalLM=0.1; globalLM<=5.0; globalLM+=0.1){
	int counter=0;
	saveCurrSizing();
	do{
	    ++counter;
	    flag = false;
	    for(int i=0; i<graphSize; i++){
		double dSum = DBL_MAX;
		double minSum = getDelaySum();
		CellData *cell = CellData::get(G[i]);
		assert(cell);
		/*std::cout << "minSum:" << minSum << std::endl;
		std::cout << "Inst:" << cell->instName <<
		    " - " << cell->master->name << std::endl;*/
		vector<oaString> sizes = cell->getOtherSizes();
		//std::cout << " Sizes:" << sizes.size() << std::endl;
		for(int j=0; j<sizes.size(); j++){
		    //std::cout << " ->" << sizes[j] << std::endl;
		    cell->swap(sizes[j]);
		    ut.invalidateFanIOTiming(design,timing,cell->inst);
		    timing->updateAll();

		    dSum = getDelaySum();
		    //std::cout << " " << dSum << "/" << pSum << std::endl;
		    if( (dSum) >= minSum){
			cell->reverseSwap();
			ut.invalidateFanIOTiming(design,timing,cell->inst);
			timing->updateAll();
			//std::cout << " Not TAKEN" << std::endl;
		    }else{
			minSum = dSum;
			cell->commitSwap();
			flag = true;
			//std::cout << " TAKEN" << std::endl;
		    }
		}
	    }
	    //std::cout << counter << std::endl;
	}while(flag);
	double worstArr = timing->getWorstArr();
	//updateLM();

	std::cout << globalLM << ":" << worstArr 
	    << ":" << ut.getTotalLeakPower(design) << std::endl; 
    }
}
/*---------------------------------------------------------*/
void
OptMinDelay::updateLM(){
    int indx=0;
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> oitIter(occ->getInstTerms());
    while(oaOccInstTerm *oit = oitIter.getNext()){ 
	TPoint *tp = TPoint::get(oit);
	assert(tp);
	if(tp->type == TIMING_POINT_SIGNAL_IN){
	    tp->lm_old = tp->lm;
	    double slack = timing->getSlack(oit);
	    if(slack < 0) 
		tp->lm = tp->lm - alpha1*slack;
	    else 
		tp->lm = tp->lm - alpha2*slack;
	    if(tp->lm < 0) tp->lm = 0;
	    /*std::cout << indx << ":" << tp->lm << std::endl;
	    ++indx;*/
	}
    }
}
/*---------------------------------------------------------*/
void
OptMinDelay::reverseLM(){
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> oitIter(occ->getInstTerms());
    while(oaOccInstTerm *oit = oitIter.getNext()){ 
	TPoint *tp = TPoint::get(oit);
	assert(tp);
	if(tp->type == TIMING_POINT_SIGNAL_IN){
	    tp->lm = tp->lm_old;
	}
    }
}
/*---------------------------------------------------------*/
vector<int> 
OptMinDelay::getIndices(oaModInst *inst){
    vector<int> indices;
    oaOccNet *net;
    bool toPO = true;
    oaOccInst *occInst = DesignTool::getOccInst(design, inst);
    //Get the input LMIndex;
    oaIter<oaOccInstTerm> oitIter(occInst->getInstTerms()); 
    while(oaOccInstTerm *oit = oitIter.getNext()){
	TPoint *tp = TPoint::get(oit);
	assert(tp);
	if(tp->type == TIMING_POINT_SIGNAL_IN){
	    indices.push_back(tp->lmIndex);
	}else if(tp->type == TIMING_POINT_SIGNAL_OUT){
	    net = oit->getNet();
	}
    }
    //Get the output LMIndex
    oaIter<oaOccInstTerm> netIter(net->getInstTerms()); 
    while(oaOccInstTerm *oit = netIter.getNext()){
	TPoint *tp = TPoint::get(oit);
	assert(tp);
	if(tp->type == TIMING_POINT_SIGNAL_IN){
	    toPO = false;
	    indices.push_back(-1*tp->lmIndex);
	}
    }
    if(toPO) indices.clear();
    return indices;
}
/*---------------------------------------------------------*/
double
OptMinDelay::getDelaySum(){
    double s = 0.0;
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> oitIter(occ->getInstTerms());
    while(oaOccInstTerm *oit = oitIter.getNext()){ 
	TPoint *tp = TPoint::get(oit);
	assert(tp);
	if(tp->type == TIMING_POINT_SIGNAL_IN){
	    //std::cout << Util::getCellDelay(oit) << std::endl;
	    //s += tp->lm*Util::getCellDelay(oit);
	    s += globalLM*Util::getCellDelay(oit);
	}
    }
    return s;
}
/*---------------------------------------------------------*/
void 
OptMinDelay::saveCurrSizing(){
    if(!currSizing.empty()) currSizing.clear();
    oaModule *mod = design->getTopModule();
    oaIter<oaModInst> mIter(mod->getInsts());
    while(oaModInst *m = mIter.getNext()){
	CellData *c = CellData::get(m);
	assert(c);
	currSizing[m] = c->master->name; 
    }
}
/*---------------------------------------------------------*/
void 
OptMinDelay::reverseSizing(){
    map<oaModInst*,oaString>::iterator it;
    for(it=currSizing.begin(); it!=currSizing.end(); ++it){
	CellData *c = CellData::get(it->first);
	assert(c);
	if(c->master->name == it->second) continue;
	c->swap(it->second);
	c->commitSwap();
	ut.invalidateFanIOTiming(design,timing,c->inst);
    }
    timing->updateAll();
}
/*---------------------------------------------------------*/
void 
OptMinDelay::save() {
    // Save the configuration for later 
    std::cout << " ** Saving Best Configuration ** " << std::endl;
    bestDesign.clear();
    oaModule *mod = design->getTopModule();
    oaIter<oaModInst> mIter(mod->getInsts());
    while(oaModInst *m = mIter.getNext()){
	CellData *c = CellData::get(m);
	assert(c);
	bestDesign[m] = c->master->name; 
    }
}
/*---------------------------------------------------------*/
void 
OptMinDelay::reload() {
    // reload saved design 
    std::cout << " ** Reloading Best Configuration ** " << std::endl;
    if(bestDesign.empty()){ 
	cout << "  no best design to load " << endl;
    }
    map<oaModInst*,oaString>::iterator iter;
    for(iter=bestDesign.begin(); iter!=bestDesign.end(); ++iter){
	CellData *cell = CellData::get(iter->first);
	assert(cell);
	cell->swap(iter->second);
	cell->commitSwap();
	ut.invalidateFanIOTiming(design,timing,cell->inst);
    }
    timing->updateAll();
}
/*---------------------------------------------------------*/
void
OptMinDelay::printLMAssignment(){
    //Check LM assignment
    oaOccurrence *occ = design->getTopOccurrence();
    oaIter<oaOccInstTerm> oitIter(occ->getInstTerms());
    while(oaOccInstTerm *oit = oitIter.getNext()){ 
	TPoint *tp = TPoint::get(oit);
	assert(tp);
	if(tp->type == TIMING_POINT_SIGNAL_IN){
	    std::cout << Util::getBlockName(oit) 
		<< ":" << tp->lmIndex << std::endl;
	}
    }
    for(int i=0; i<G.size(); i++){
	vector<int> indx = getIndices(G[i]);
	CellData *cell = CellData::get(G[i]);
	std::cout << cell->instName << ":";
	for(int j=0; j<indx.size(); j++){	
	    std::cout << " " << abs(indx[j]);
	}
	std::cout << std::endl;
    }
}
/*---------------------------------------------------------*/
}//namespace
