/************************************************************
* 
* File: oagTimerPeepHole.h
* Author: Santiago Mok (smok@ee.ucla.edu)
* Created: 09-29-2009
* Last Modified: Wed 06 Oct 2010 06:16:31 PM PDT
*
************************************************************/
#if !defined(oagTimerOpt_P)
#define oagTimerOpt_P

#include <vector>
#include <map>
#include <algorithm>
#include <iostream>

#include "oaDesignDB.h"
#include "oagTimerCellData.h"
#include "oagTimerDesignData.h"
#include "oagTimerDesignTool.h"
#include "oagTimerSensitivity.h"
#include "oagTimerPeepHole.h"
#include "oagTimerUtil.h"

namespace oagTimer{
/*---------------------------------------------------------*/
using namespace oa;
using namespace std;
/*---------------------------------------------------------*/
typedef enum {
    UPSIZE_ROOT,
    DOWNSIZE_ROOT,
    /*IMPROVING, //Account for improving moves only
    NON_IMPROVING, //Account for non improving peephole candidates
    TRANSITIVE_CHECK, //Check for transitive fanouts
    */
} Modes;
/*---------------------------------------------------------*/
typedef map<oaModInst*,oaString> CellSizingMap;
/*---------------------------------------------------------*/
struct FOCnt{
    oaModInst *inst;
    int numFO;
    int weight;
};
struct SizingData{
    CellSizingMap sizes;
    double leakage;
};
struct optData{
    int oneOptData;
    int twoOptData;
    int moreOptData;
};
/*---------------------------------------------------------*/
struct cellScore{
    oaModInst *inst;
    double score;
};
struct combTimingData{
    //instVector insts;
    oaModInst *inst;
    DelayType score;
};
struct rootData{
    oaModInst *inst;
    double score;
    int fanoutLevel;
};

bool rankHigh(rootData a, rootData b){
    if(a.score > b.score) return true;
    return false;
}

bool rank_cell_score(cellScore a, cellScore b){
    if(a.score > b.score) return true;
    return false;
}
bool rank_high(combTimingData a, combTimingData b){
    if(a.score > b.score) return true;
    return false;
}
bool rank_low(combTimingData a, combTimingData b){
    if(a.score < b.score) return true;
    return false;
}
bool compare_nFO(const FOCnt& a, const FOCnt& b){
    return a.numFO < b.numFO;
}
bool compare_weightFO(const FOCnt& a, const FOCnt& b){
    return a.weight < b.weight;
}
static bool incr_slack_func(sensitivityData a, sensitivityData b){
    if(a.sensitivity < b.sensitivity) return true;
    else return false;
}
/*---------------------------------------------------------*/
class Opt {  
    friend class PeepHole;

    public:
	Opt(oaDesign *d, Timer* t);
	~Opt();
	
	//float getSensitivity(oaInst *old_inst, oaInst *new_inst){}

	/* Perform PeepHole Optimization */
	void run();
	void peepHoleOpt(int lvl, int i);
	void doOneOPT();

    private:
	bool retry();
	bool isValidTiming(TimeType new_arr, TimeType new_slack, double curr_arr);
	bool validComb(float val);
	/* Create list of all inst (cells) in the design 
	 * and initialize visited flag to 0
	 */
	void initDesign();
	void querySensitivities();
	//bool scanDesign();

	void markSwappedFanouts(instVector insts);
	bool isAllVisited(vector<oaModInst*> insts);
	bool isFanoutMarked(instVector insts);

	bool scanPeepHoles(sensitivityData s);
	void scanPeepHoles(oaModInst*);
	void scanPeepHoles(oaModInst* root, instVector fo,int size);

	/*! This function set the given new master 
	 * of the given CellData inst and check if the new 
	 * swapping is timing feasible
	 * @PARAM CellData: cell instance to be swap
	 * @PARAM oaString: new master lib cell name
	 * @RETURN bool: wether the new master is timing feasible
	 */
	bool setNewMaster(CellData *cell, oaString newMaster);

	/*! This function try out various upsizes for root and 
	 * downsizes for fanout cell succesively and select the first
	 * combination that decreases leakage power and is timing 
	 * feasible
	 * @PARAM none
	 * @RETURN void
	 */
	void doSizing();

	/*Check this*/
	bool isVisited(oaModInst* inst);

	/* Swap the given combination list within 
	 * the peephole and keep the configuration 
	 * that meet min(leakage and delay)
	 * return - true: swap was commmitted
	 *	    false: swap was reversed
	 */
	bool doSwap(SizingData list);
	bool doSwap(sensitivityData d);

	/* Mark current and immediate fanout cells
	 * as visited
	 */
	void markVisited(vector<oaModInst*> f);
	
	/* Query for the possible set of swaps in 
	 * the peephole
	 */
	void genAllCombinations();
	bool genCombinations(vector<oaModInst*>::iterator &list_iter,
		SizingData &data_list);
	/* Clear current sizing combination */
	void initDataStruct(SizingData &_data);

	/*!Prune peepholes that seems not feasible for sizing 
	 * The processing occur after all possible enumeration of size n fanout are
	 * generated
	 * 1) Remove PH if root and all fanouts in the peephole are min size
	 * 2) Remove PH if all fanouts in the peephole does not contain other sizes
	 * 3) Remove PH if all fanouts in the peephole are critical
	 * @PARAM vector of enumerated combinations
	 * @RETURN vector of sorted enumeration
	 */
	instVector Preprocess_Peepholes(oaModInst* root, instVector vec);

	/*! Sort the fanout in decreasing order based on the min input slack value
	 * The processing occur after all possible enumeration of size n fanout are
	 * generated
	 * @PARAM vector of enumerated combinations
	 * @RETURN vector of sorted enumeration
	 */
	vector<combTimingData> PreprocessComb_Slack(vector<instVector> vec);

	/*! Sort the fanout in increasing order based on the max input slew value
	 * The processing occur after all possible enumeration of size n fanout are
	 * generated
	 * @PARAM vector of enumerated combinations
	 * @RETURN vector of sorted enumeration
	 */
	vector<combTimingData> PreprocessComb_Slew(vector<instVector> vec);

	vector<combTimingData> rankFanoutWeight(vector<instVector> vec);
	/*! This function query the difference in delay for changing its master
	 * to the next avaiable downsize
	 * @PARAM CellData: the cell instance to change it master
	 * @RETURN double: the delay sensitivity of downsizing
	 */
	double getDownSizeDelaySensitivity(CellData *c);

	double getUpSizeSensitivity(oaModInst* i);
	double getDownSizeSensitivity(oaModInst* i);

	void getUpSizeSensitivities(CellData *c);
	double getDownSizeSensitivities(CellData *c);

	double getInInstTermSlack(oaModInst* inst);
	void clearContainers();

	/* Debugging */
	void printVisitedCells();
	void printCombinations(SizingData _s);
	void printAllFanouts(vector<InstsMap> vm);
	void printKOptTable(oaModInst* h, vector<instVector> T);
	void printRootNodeSorting(vector<instTimingData> data_vector, int type);
	map<oaOccInstTerm*,double> getTimingInformation(vector<oaModInst*> insts);
	void printTimingData(map<oaOccInstTerm*,double> list);
	void printSensitivityList();
    private:
	oaDesign *design;
	Timer* timing;

	// Peephole width
	PeepHole *ph;
	Util ut;
	int level;
	int currFanoutLevel;
	int currRankLevel;
	bool any_move_taken;
	bool currPHMoveTaken;
	Modes mode;
	vector<oaModInst*> currPH_list;
	int max_fo;
	int currNotTaken;

	//Cells in current Design
	instVector design_Insts;
	instVector takenSwapInstsChecklist;
	vector<instTimingData> design_Insts_type2;
	vector<rootData> multiLevelRootInstances;

	vector<sensitivityData> sensitivity_sorted_Insts;
	vector<sensitivityData> upsized_sensitivity_Insts;
	vector<sensitivityData> downsized_sensitivity_Insts;
	instVector PeepholeChecklist;

    /* Statistic Info*/
	int totalNumOfTrials;
	int numOfTrials;
	int numOfTaken;

	int totalNumOfTaken;
	int numOfZeroOPTMove;
	int numOfOneOPTMove;
	int numOfTwoOPTMove;
	map<oaModInst*,int> sizedRootNodeCount;

	int numOfOneTaken;
	int numOfTimingInfeasiblePH;
	int curr_counter;
	int numOfPeepholeTrials;
	int curr_ph_counter;
	int total_ph_counter;
	double numOfTakeninBucket;
	double numOfTrialinBucket;
	int numOfFanoutsTakenTwice;

	bool fo_taken;
	//bool flag;
	bool issueReturn;
	vector<int> takenFanoutCounter;
};

}//namespace
#endif
