/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

// Author: Philip Chong <pchong@cadence.com>

#include <iostream>
#include "oagTimerLinearWireModel.h"
#include "oagUtilOaUtil.h"

namespace oagTimer {

using namespace oa;
using namespace oagUtil;

///////////////////////////////////////////////////////////////////////////////
/// Constructor.
///
/// \param tech Tech to use.

LinearWireModel::LinearWireModel(oaTech *tech)
{
    set(tech);
}

///////////////////////////////////////////////////////////////////////////////
/// Set the tech library associated with this wire model instance.
///
/// \param tech Tech to use.

void
LinearWireModel::set(oaTech *tech) {
    _tech = tech;
    _capPerLength = 0.0;
    _resPerLength = 0.0;

    if (!tech) {
        return;
    }

    // find metal3
    oaIter<oaLayer> oaLayerIter(_tech->getLayers());
    oaLayer *m3Layer = 0;
    while (!m3Layer) {
        oaLayer *l = oaLayerIter.getNext();
        if (!l) {
            break;
        }
        if (l->getType() != oacPhysicalLayerType) {
            continue;
        }
        oaMaterial m = (static_cast<oaPhysicalLayer*>(l))->getMaterial();
        if (m != oacMetalMaterial) {
            continue;
        }
        int metalLayersBelow = 0;
        oaPhysicalLayer *belowLayer =
            (static_cast<oaPhysicalLayer*>
            (l))->getLayerBelow(oacMetalMaterial);
        while (belowLayer) {
            metalLayersBelow ++;
            belowLayer = belowLayer->getLayerBelow(oacMetalMaterial);
        }
        if (metalLayersBelow == 2) {
            m3Layer = l;
        }
    }
    if (!m3Layer) {
        std::cerr << "Missing METAL3 layer" << std::endl;
        return;
    }

    oaConstraintGroup *cg
        = oaConstraintGroup::find(_tech, "LEFDefaultRouteSpec");
    if (!cg) {
        std::cerr << "Missing oacMinWidthRule" << std::endl;
        return;
    }
    oaConstraint *cons =
        oaLayerConstraint::find(cg, m3Layer->getNumber(),
        oaLayerConstraintDef::get(oacMinWidth));
    oaValue *val = cons->getValue();
    oaInt4 minWidth = static_cast<oaIntValue *>(val)->get();
    double width = _tech->
        dbuToUUDistance(oaViewType::get(oacMaskLayout), minWidth);

    oaProp *prop = oaProp::find(m3Layer, "RESISTANCE");
    if (!prop) {
        std::cerr << "Missing RESISTANCE" << std::endl;
        return;
    }
    double res = (static_cast<oaFloatProp *>(prop))->getValue();

    prop = oaProp::find(m3Layer, "CAPACITANCE");
    if (!prop) {
        std::cerr << "Missing CAPACITANCE" << std::endl;
        return;
    }
    double cap = (static_cast<oaFloatProp *>(prop))->getValue();

    prop = oaProp::find(m3Layer, "EDGECAPACITANCE");
    if (!prop) {
        std::cerr << "Missing EDGECAPACITANCE" << std::endl;
        return;
    }
    double edgeCap = (static_cast<oaFloatProp *>(prop))->getValue();

    _capPerLength = cap * width + edgeCap;
    _resPerLength = res / width;
}

///////////////////////////////////////////////////////////////////////////////
/// Get the wire capacitance associated with the given net.
///
/// \todo Check correctness and units
///
/// \param net The net.
/// \return Capacitance in picofarads.

double
LinearWireModel::getWireCap(oaNet *net) {
    oaUInt4 dbNetLen = OaUtil::getHPWL(net);
    double netLen = _tech->
        dbuToUUDistance(oaViewType::get(oacMaskLayout), dbNetLen);
    return _capPerLength * netLen;
}

///////////////////////////////////////////////////////////////////////////////
/// Get the wire delay associated with the given net.
///
/// \todo Check correctness and units.
///
/// \param net The net.
/// \return Delay in nanoseconds.

double
LinearWireModel::getWireDelay(oaNet *net) {
    oaUInt4 dbNetLen = OaUtil::getHPWL(net);
    double netLen = _tech->
        dbuToUUDistance(oaViewType::get(oacMaskLayout), dbNetLen);
    return _capPerLength * _resPerLength * netLen;
}

///////////////////////////////////////////////////////////////////////////////

}
