/************************************************************
* oagTimerElmoreWireModel.cpp
* Perform Elmore delay analysis on interconnects
* by: Santiago Mok <smok@ee.ucla.edu>
* advised by: Prof. Puneet Gupta
* Electrical Engr Dept, UCLA
*
************************************************************/
#include <iostream>
#include <assert.h>
#include <string.h>
#include <map>
#include <vector>
#include <math.h>
#include "oagTimerElmoreWireModel.h"
#include "oagTimerTPointMaster.h"
#include "oagTimerTimer.h"

//#define TRACE
namespace oagTimer{
    
using namespace oa;
using namespace std;
//Contructor
ElmoreWireModel::ElmoreWireModel(){
    time_factor = 1;
    cap_factor = 1;
}
//Destructor
ElmoreWireModel::~ElmoreWireModel(){
    clearAll();
}

bool ElmoreWireModel::isPinNode(oaNode *node){
    oaIter<oaDesignObject> objIter (node->getConns() );
    oaDesignObject *obj = objIter.getNext();
    if(!obj)
	return false;
    oaType oa_type = obj->getType();
    if( oa_type == oacInstTermType ||
	oa_type == oacScalarTermType ||
	oa_type == oacBusTermBitType ||
	oa_type == oacBusTermType ||
	oa_type == oacBundleTermType){
	    return true;
    }
    return false;
}

void ElmoreWireModel::setUnit(const char* time,const char* cap){
    //Time factor
    if( strcmp(time,"ms") == 0 || strcmp(time,"1ms") == 0 ) 
	time_factor = 1e+3;
    else if( strcmp(time,"us") == 0 || strcmp(time,"1us") == 0 ) 
	time_factor = 1e+6;
    else if( strcmp(time,"ns") == 0 || strcmp(time,"1ns") == 0 ) 
	time_factor = 1e+9;
    else if( strcmp(time, "ps") == 0 || strcmp(time,"1ps") == 0 )
	time_factor = 1e+12;
    else if( strcmp(time, "fs") == 0 || strcmp(time,"1fs") == 0 )
	time_factor = 1e+15;
    //Cap factor
    if( strcmp(cap, "uf") == 0)
	cap_factor = 1e+6;
    else if( strcmp(cap, "nf") == 0)
	cap_factor = 1e+9;
    else if( strcmp(cap, "pf") == 0)
	cap_factor = 1e+12;
    else if( strcmp(cap, "ff") == 0)
	cap_factor = 1e+15;
    else if( strcmp(cap, "af") == 0)
	cap_factor = 1e+18;
}
/*---------------------------------------------------------*/
bool ElmoreWireModel::isTopPathNode(oaNode *_currNode){
    _PathMap::iterator it;
    it = topPath.find(_currNode);
    if (it != topPath.end())
	return true;
    return false;
}
/*---------------------------------------------------------*/
double ElmoreWireModel::getInputPinCap(oaNode *p){
    inputPinMap::iterator iter;
    iter = inputPinCap.find(p);
    if (iter != inputPinCap.end())
	return inputPinCap[p];
    return 0; 
}
/*---------------------------------------------------------*/
void ElmoreWireModel::clearPathCap(){
   _currPathCap = 0; 
}
/*---------------------------------------------------------*/
void ElmoreWireModel::clearCurrentPath(){
    _currPathRes = 0;
    _currNodeCap = 0;
    _currElmorePath = 0;
}
/*---------------------------------------------------------*/
void 
ElmoreWireModel::clearAll(){
    clearCurrentPath();
    clearPathCap();
    _currNodeVector.clear();
    inputPinCap.clear();
    topPath.clear();
    luMap.clear();
    inputTerm.clear();
    _terms.clear();
    _instTerms.clear();
}
/*---------------------------------------------------------*/
void 
ElmoreWireModel::storeTopPath(oaNode *currentNode, oaFloat _res, oaFloat _elm){
    _PathMap::iterator it;
    it = topPath.find(currentNode);
    if(it != topPath.end())
	return;
    _pathData top;
    top.totalRes = _res;
    top.totalElmoreValue = _elm;
    topPath[currentNode] = top;
    _currNodeVector.push_back(currentNode);
}
/*---------------------------------------------------------*/
void 
ElmoreWireModel::updateTopPathElmoreValue(oaFloat pathCap){
    _PathMap::iterator it;
    for(it = topPath.begin(); it != topPath.end(); ++it){
	oaFloat res = it->second.totalRes;
	it->second.totalElmoreValue += (res * pathCap);
    }
}
/*---------------------------------------------------------*/

void 
ElmoreWireModel::storeInpPin(oaNode *currNode, oaFloat _res, oaFloat _elm){
    _inputPinMap::iterator it; 
    it = inputTerm.find(currNode);
    if(it != inputTerm.end())
	return;
	_inputTermsData inputPin;
	inputPin.res = _res;
	inputPin.elmoreValue = _elm;
	inputPin.topNode = _currNodeVector;
	inputTerm[currNode] = inputPin;
}
/*---------------------------------------------------------*/
oaTerm* ElmoreWireModel::getTerm(oaNode *n){
   termMap::iterator iter;
   iter = _terms.find(n);
   if(iter != _terms.end())
	return iter->second;
    return 0;
}
/*---------------------------------------------------------*/
oaInstTerm* ElmoreWireModel::getInstTerm(oaNode *n){
   instTermMap::iterator iter;
   iter = _instTerms.find(n);
   if(iter != _instTerms.end())
	return iter->second;
    return 0;
}
/*---------------------------------------------------------*/
void 
ElmoreWireModel::findNext(nodeData &node, oaNode *previousNode){
    //Search in the "From Device" list first
    node._nextNode.clear();
    //node.left = NULL;
    //node.right = NULL;
    oaNode* currentNode = node.current;
    oaIter<oaDevice> fromDevIter (currentNode->getFromDevices());
    while(oaDevice *device = fromDevIter.getNext()){
	oaStdDevice *stdDev = static_cast<oaStdDevice*>(device);
	if(stdDev->getType() == oacResistorType){
	    oaNode *otherNode = stdDev->getOtherNode(currentNode);
	    if(otherNode != previousNode){
		if(!node._nextNode.empty()){
		    branch = true;
		}
		if(isPinNode(currentNode)){
		    pinBranch = true;
		}
		node._res.push_back( static_cast<oaStdDevice*>(device)->getValue(_currentAP) );
		node._nextNode.push_back(otherNode);
	    }	
	}
    }
 //Then Search in the "To Device" list
    oaIter<oaDevice> toDevIter (currentNode->getToDevices());
    while(oaDevice *toDevice = toDevIter.getNext()){
	oaStdDevice *stdDev = static_cast<oaStdDevice*>(toDevice);
	if(stdDev->getType() == oacResistorType){
	    oaNode *otherNode = stdDev->getOtherNode(currentNode);
	    if(otherNode != previousNode){
		if(!node._nextNode.empty()){
		    branch = true;
		}
		if(isPinNode(currentNode)){
		    pinBranch = true;
		}
		node._res.push_back( static_cast<oaStdDevice*>(toDevice)->getValue(_currentAP) );
		node._nextNode.push_back(otherNode);
	    }
	}
    }
}
/*---------------------------------------------------------*/
void 
ElmoreWireModel::oaTraversePath(oaNode *currentNode, oaNode *previousNode){
    
    branch = false;
    pinBranch =  false;
    nodeData node;
    node.current = currentNode;
    findNext(node, previousNode);
    _retPathCap = 0; 
    
    //save top data to map
    if(branch){
	updateTopPathElmoreValue(_currPathCap);
	storeTopPath(currentNode, _currPathRes, _currElmorePath);
	clearPathCap();
	clearCurrentPath();
    }

    if(isPinNode(currentNode)){
	if(currentNode != parentNode){
	   _currNodeCap = getInputPinCap(currentNode); 
	   _currPathCap += _currNodeCap;
	   _retPathCap += _currNodeCap;
	   _currElmorePath += _currNodeCap * _currPathRes;

	   //Save current elmore path for input terms
	   updateTopPathElmoreValue(_currPathCap);
	   storeInpPin(currentNode, _currPathRes, _currElmorePath);

	   if(pinBranch){
	       storeTopPath(currentNode, _currPathRes, _currElmorePath);
	   }
	   clearPathCap();
	   clearCurrentPath();
	}
    }

    while(!node._nextNode.empty()){
	oaNode *nextNode = node._nextNode.back();
	node._nextNode.pop_back();
	_currPathRes += node._res.back();
	node._res.pop_back();
	if(nextNode->getType() == oacGroundedNodeType){
	    _currNodeCap = static_cast<oaGroundedNode*>(nextNode)->getValue(_currentAP);
	    _currPathCap += _currNodeCap;
	}else{
	    _currNodeCap = 0;
	}
	_currElmorePath += _currPathRes * _currNodeCap;
	oaTraversePath(nextNode, currentNode);
    }

    //node returning up 
    if(currentNode != parentNode){
	if(currentNode->getType() == oacGroundedNodeType && !isTopPathNode(currentNode)){
	    _retPathCap +=  static_cast<oaGroundedNode*>(currentNode)->getValue(_currentAP);
	}
	if(isTopPathNode(currentNode)){
	    //updateTopPathElmoreValue(_retPathCap);

	    //update input Pin Elmore Value for branching pin input
	    if(isPinNode(currentNode)){
		_inputPinMap::iterator it;
		it = inputTerm.find(currentNode);
		if(it != inputTerm.end()){
		    it->second.elmoreValue += (it->second.res * _retPathCap);
		}
	    }
	    clearPathCap();
	    
	    _PathMap::iterator it;
	    it = topPath.find(currentNode);
	    luMap[currentNode] = it->second;
	    topPath.erase(currentNode);
	    _currNodeVector.pop_back();
	}
    }
}
double ElmoreWireModel::getWireCap(oaNet *net){
    oaDesign *design = net->getDesign();
    oaCollection<oaAnalysisPoint, oaDesign>
	collecAPs( design->getAnalysisPoints() );
    oaIter<oaAnalysisPoint>
	iterAPinDES( collecAPs );
    oaAnalysisPoint
	*ap;
    double cap;
    clearAll();
    iterAPinDES.reset();
    ap = iterAPinDES.getNext();
    if(ap == NULL){
	cerr << "Elmore Wire Model Error: No spef file loaded or no analysis point associated with the spef file\n";
	cerr << "Please include a spef file for elmore analysis\n";
	exit(1);
    }
    while(ap){
	oaBoolean _pnExists = oaParasiticNetwork::exists(net,ap);
	if( !_pnExists ){
	    if (Timer::displayWarning)
	      cerr << "Warning: No parasitic network exist for net, setting 0 wire cap for current net" << endl;
	    cap = 0;
	    return 0;
	}else{
	    cap = 0;
	    oaParasiticNetwork *pn = oaParasiticNetwork::load(net,ap);
	    oaIter<oaNode> nodesIter (pn->getNodes());
	    while(oaNode *node = nodesIter.getNext()){
		if(node->getType() == oacGroundedNodeType){
		    cap += static_cast<oaGroundedNode*>(node)->getValue(ap);
		}
	    }
	    /*oaIter<oaInstTerm> instTermIter (net->getInstTerms() );
	    while( oaInstTerm *instTerm = instTermIter.getNext()){
		if(instTerm->getTerm()->getTermType() == oacInputTermType){
		    oaModTerm *masterTerm = instTerm->getOccInstTerm()->getModInstTerm()->getTerm();
		    assert(masterTerm);
		    TPointMaster *tm = TPointMaster::get(masterTerm);
		    cap += tm->cap;
		}
	    }*/
	    pn->unload();
	}
    ap = iterAPinDES.getNext();
    }
    return cap*cap_factor;
}
double ElmoreWireModel::getWireRes(oaNet *net){

    oaDesign *design = net->getDesign();
    oaCollection<oaAnalysisPoint, oaDesign>
	collecAPs( design->getAnalysisPoints() );
    oaIter<oaAnalysisPoint>
	iterAPinDES( collecAPs );
    oaAnalysisPoint
	*ap;
    double res = 0;
    //clearAll();
    iterAPinDES.reset();
    ap = iterAPinDES.getNext();
    if(ap == NULL){
	cerr << "Elmore Wire Model Error: No spef file loaded or no analysis point associated with the spef file\n";
	cerr << "Please include a spef file for elmore analysis\n";
	exit(1);
    }
    while(ap){
	oaBoolean _pnExists = oaParasiticNetwork::exists(net,ap);
	if( !_pnExists ){
	    if (oagTimer::Timer::displayWarning)
	      cerr << "Warning: No parasitic network exist for net, setting 0 wire res for current net" << endl;
	    res = 0;
	    return 0;
	}else{
	    oaParasiticNetwork *pn = oaParasiticNetwork::load(net,ap);
	    oaIter<oaDevice> nodesIter (pn->getDevices());
	    while(oaDevice *dev = nodesIter.getNext()){
		if(dev->getType() == oacResistorType)
		    res += static_cast<oaStdDevice*>(dev)->getValue(ap);
	    }
	    pn->unload();
	}
    ap = iterAPinDES.getNext();
    }
    return res;
}
double ElmoreWireModel::getWireCapEff(double netLoad, double netRes, DelayType Td, DelayType Tx){
    LibData *libD =  new LibData();
    double C_eff, min_slew_thres, max_slew_thres;

    min_slew_thres = libParseData.slew_low_rise*0.01;
    max_slew_thres = libParseData.slew_high_rise*0.01;
    if(min_slew_thres <= 0 || max_slew_thres <= 0){
	min_slew_thres = 0.2; 
	max_slew_thres = 0.8; 
    }
    double C_tot = netLoad;
    double C1 = 5.0/6.0*C_tot;
    double C2 = 1.0/6.0*C_tot;

    double R_pi = 12.0/25.0*netRes;
    double Rd = (Tx/time_factor)/((C_tot/cap_factor)*(log(max_slew_thres)-log(min_slew_thres)));
    C_eff = C2 + ((Rd/(Rd+R_pi))*C1);

    return C_eff;
}
//Definition form WireModel.h
DelayType ElmoreWireModel::getWireDelay(oaNet *net){}
	

DelayType ElmoreWireModel::getWireDelay(oaNet *net, _outputMap &outputPin, _inputMap &inputPin){
    oaDesign *design = net->getDesign();
    oaCollection<oaAnalysisPoint, oaDesign>
	collecAPs( design->getAnalysisPoints() );
    oaIter<oaAnalysisPoint>
	iterAPinDES( collecAPs );
    oaAnalysisPoint
	*ap;
    clearAll();
    double max_Value = 0;
    iterAPinDES.reset();
    ap = iterAPinDES.getNext();
    if(ap == NULL){
	cerr << "Elmore Wire Model Error: No spef file loaded or no analysis point associated with the spef file\n";
	cerr << "Please include a spef file for elmore analysis\n";
	exit(1);
    }
    while(ap){
	oaBoolean _pnExists = oaParasiticNetwork::exists(net,ap);
	     
	if(!_pnExists){
	  if (oagTimer::Timer::displayWarning)
	    cerr << "Warning: No parasitic network exist for net, setting 0 wire delay for current net" << endl;
	    max_Value = 0;
	    return 0;
	}
	else{
	    oaNativeNS NS;
	    oaString 
		termName, instTermName;

	    oaParasiticNetwork *pn = oaParasiticNetwork::load(net,ap);
	    _currentAP = ap;
	    parentNode = NULL;
	    
	    oaIter<oaTerm> termIter (net->getTerms());
	    while( oaTerm *term = termIter.getNext()){
		oaIter<oaNode> nIter (pn->getConnNodes(term));
		while(oaNode *node = nIter.getNext()){
		    if(term->getTermType() == oacInputTermType){
			parentNode = node;
		    
		    }else if(term->getTermType() == oacOutputTermType){
			_terms[node] = term;	
		    }
		}
	    }
	    
	    oaIter<oaInstTerm> instTermIter (net->getInstTerms() );
	    while( oaInstTerm *instTerm = instTermIter.getNext()){
		oaIter<oaNode> nIter (pn->getConnNodes(instTerm));
		while(oaNode *node = nIter.getNext()){
		    if(instTerm->getTerm()->getTermType() == oacOutputTermType){
			parentNode = node;
		    }else if(instTerm->getTerm()->getTermType() == oacInputTermType){
			_instTerms[node] = instTerm;
			oaModTerm *masterTerm = instTerm->getOccInstTerm()->getModInstTerm()->getTerm();
			assert(masterTerm);
			TPointMaster *tm = TPointMaster::get(masterTerm);
			inputPinCap[node] = (tm->cap / cap_factor);
		    }
		}
	    }
	    oaTraversePath(parentNode,parentNode);

	    _inputPinMap::iterator it; 
	    vector<oaNode*>::iterator itr;
	    _PathMap::iterator iter;
	    for( it = inputTerm.begin(); it != inputTerm.end(); ++it){
		oaNode *currNode = it->first;
		vector<oaNode*> node = it->second.topNode;
		for( itr = node.begin(); itr != node.end(); ++itr){
		    oaNode *n = *itr;
		    iter = luMap.find(n);
		    if(iter != luMap.end()){
			it->second.elmoreValue += iter->second.totalElmoreValue;
		    }
		}
		node.clear();
		oaFloat elmoreVal = it->second.elmoreValue * time_factor;
		oaNativeNS NS;
		oaString tName, instName;
		if(oaTerm *term = getTerm(currNode)){
		    outputPin[term] = elmoreVal;
		    term->getName(NS,tName);	
		}else if(oaInstTerm *instTerm = getInstTerm(currNode)){
		    inputPin[instTerm] = elmoreVal;
		    instTerm->getInst()->getName(NS,instName);	
		}
		if(max_Value < it->second.elmoreValue){
		    max_Value = it->second.elmoreValue;
		}

	    }
	    pn->unload();
	}
	
	_currentAP = NULL;
	ap = iterAPinDES.getNext();
    }//ap
    return (max_Value * time_factor);
	
}//getWireDelay
	
}//namespace
