#if !defined(oagSswOdcSatSweepingEngine_P)
#define oagSswOdcSatSweepingEngine_P

#include <map>
#include <set>
#include <vector>
#include "oaBase.h"
#include "oagSswObsAiGraph.h"
#include "oagSswAiLevelizer.h"
#include "oagSswAiSimEngine.h"
#include "oagSswAiBitVectorDict.h"
#include "oagSswAiSatSolver.h"
#include "oagSswLogger.h"
#include "oagSswSet.h"
#include "oagSswMap.h"

namespace oagSsw
{

/// \brief Performs SAT sweeping with local observability don't-cares on an
/// observable AI graph.
class OdcSatSweepingEngine {
  public:
    /// \brief Constructs an engine that will search for replacements for all nodes in
    /// the graph.
                            OdcSatSweepingEngine(ObsAiGraph             &graph);
    /// \brief Constructs an engine that will search for replacements for a specified
    /// set of nodes in the graph.
                            OdcSatSweepingEngine(ObsAiGraph             &graph,
                                                 const list<oagAi::Ref> &nodes);

    /// \brief Specifies the pool of possible replacement nodes.
    ///
    /// It is not necessary, but harmless, to supply both polarities of a node;
    /// both inverted and non-inverted refs will always be used.
    void                    setRepPool(const list<oagAi::Ref> &reps);
    
    /// \brief Runs SAT sweeping with local observability.
    ///
    /// For each node x, searches for a replacement that is equivalent under local
    /// observability of x.  When a replacement is found, the two nodes are
    /// merged:  The fanout of x is moved to the replacement.
    /// Only combinational equivalences are detected.
    ///
    /// Even if structural hashing is enabled, after merging it is possible for
    /// two nodes in the graph to have the same inputs.
    /// \param numObsLevels how many levels of local observability to consider
    void                    run(oa::oaUInt4 numObsLevels);
    void                    runObsLevel(oa::oaUInt4 numObsLevels);
    
    /// \brief Returns true if the node has been replaced by an equivalent.
    bool                    hasRep(oagAi::Ref x);
    /// \brief Returns the replacement of a node, or a null ref if it has not been
    /// replaced.
    oagAi::Ref              getRep(oagAi::Ref x);

    /// Returns the output Logger used for SAT sweeping with LODCs.
    Logger                  *getLogger()    { return logger; }

    /// \name Tuning Parameters
    //@{
    /// Objective for ranking indexing bits for the sim vector dictionary.
    typedef enum {
        /// Ranks the bits in decreasing order of frequency of observation (the
        /// highest-ranked bits are least often masked).
        MAX_OBS_OBJ,
        /// Ranks the bits in increasing order of expected effort in searching.
        MIN_EFFORT_OBJ
    } DictBitObj;

    class Params {
      public:
                            Params() {
                                initNumSimBits      = 32;
                                maxSatBacktracks    = INT_MAX;
                                maxSimVecBinSize    = 16;
                                dictBitObjective    = MAX_OBS_OBJ;
                            }

        oa::oaUInt4         initNumSimBits;
        oa::oaUInt4         maxSatBacktracks;
        oa::oaUInt4         maxSimVecBinSize;
        DictBitObj          dictBitObjective;
    };
    Params                  params;
    //@}
    
    /// \brief Statistics.
    struct Stats {
        /// \brief The number of merges during the last run.
        oa::oaUInt4         numMerges;

        /// \brief The number of SAT checks with each result during the most
        /// recent run.
        struct {
            oa::oaUInt4     numSatisfiable;
            oa::oaUInt4     numUnsatisfiable;
            oa::oaUInt4     numUndecidable;
        } sat;
        struct {
            oa::oaFloat     odcSatCheckTime;
            oa::oaFloat     computeObsVectorsTime;
            oa::oaFloat     findReplacementTime;
            oa::oaFloat     mergeFailedTime;
            oa::oaFloat     searchTrieTime;
            oa::oaFloat     mergeTime;
            oa::oaFloat     computeSimTrieTime;
            oa::oaFloat     extendSimTime;
            oa::oaFloat     lastWordSimTime;
        } runtime;
    };

    Stats                   stats;
   
    struct OdcTimer {
        oa::oaTimer         odcSatCheckTimer;
        oa::oaTimer         computeObsVectorsTimer;
        oa::oaTimer         findReplacementTimer;
        oa::oaTimer         mergeFailedTimer;
        oa::oaTimer         searchTrieTimer;
        oa::oaTimer         mergeTimer;
        oa::oaTimer         computeSimTrieTimer;
        oa::oaTimer         extendSimTimer;
        oa::oaTimer         lastWordSimTimer;
    };
 
    OdcTimer                odcTimer;            

  private:
    /// Helper type for observability of a node at a level.
    typedef std::pair<oagAi::Ref, oa::oaUInt4> ObsRef;

    void                    clearStats();

    /// \brief Collects combinational inputs from among a list of nodes.
    ///
    /// Inserts into the inputs field each node in the list that is:
    /// \li A terminal node whose driver is null
    /// \li Sequential
    /// The list given may contain nodes that are not inputs; they will be
    /// ignored.
    ///
    /// \param pool a list of nodes
    void                    addInputsFrom(const list<oagAi::Ref> &pool);
    
    /// Groups the search and replacement nodes by level.
    void                    levelize();

    /// Initializes the simulation vectors with random inputs.
    void                    initializeSimVectors();
    void                    computeObsVectors(const set<oagAi::Ref> &nodes,
                                              oa::oaUInt4           numObsLevels);
    /// \brief Compute an observability vector from the current simulation
    /// vectors.
    ///
    /// \param node
    /// \param numObsLevels how many levels of observability to consider
    /// \retval obsVector stores the result
    void                    computeObsVector(oagAi::Ref     node,
                                             oa::oaUInt4    numObsLevels,
                                             BitVector      &obsVector);
    /// Re-computes the observability vector of a node.
    void                    updateObsVector(oagAi::Ref      x,
                                             oa::oaUInt4    numObsLevels);

    /// \brief Updates the simulation vectors of the nodes in the local
    /// transitive fanout of a node.
    void                    updateLocalSimVectors(oagAi::Ref     x,
                                                  oa::oaUInt4    numObsLevels);

    /// \brief Adds an assignment of input values to the simulation vectors.
    ///
    /// Inputs not mentioned in the assignment are assigned randomly.
    /// The simulation vectors of non-inputs are not changed.
    void                    extendSimVectors(const Map<oagAi::Ref, bool>::Type &assignment);

    /// Computes the bit ranking for the sim vector dictionary.
    void                    computeDictBitOrder(const set<oagAi::Ref>   &nodes,
                                                vector<oaUInt4>         &bitOrder);
    /// \brief Stores in the dictionary the rep nodes up to some level.
    ///
    /// Old entries are removed.
    void                    populateSimVecDict(oaUInt4 level);

    /// \brief Add clauses that represent observability to the solver.
    void                    addObsClauses(AiSatSolver           &satSolver,
                                          oagAi::Ref            node,
                                          Lit                   nodeObsLit,
                                          oa::oaUInt4           numObsLevels,
                                          std::set<ObsRef>      &alreadyAdded,
                                          std::set<oagAi::Ref>  &alreadyAddedTransitiveFanin,
                                          const vector<bool>    &tfo);
    /// \brief Add transitive fanin of a node to solver.
    void                    addTransitiveFaninToSolver(AiSatSolver          &satSolver,
                                                       oagAi::Ref           node,
                                                       std::set<oagAi::Ref> &alreadyAddedTransitiveFanin);
    
    /// \brief Finds an equivalent (modulo observability) ref for a node.
    ///
    /// \param x
    /// \param numObsLevels how many levels of local observability to consider
    oagAi::Ref              findReplacement(oagAi::Ref   x,
                                            oa::oaUInt4  numObsLevels);

    /// \brief Verifies equivalence (modulo observability) of two refs.
    /// 
    /// Returns true if \a x can be merged onto \a y without changing the
    /// observable functions of the graph; otherwise, returns an assignment to
    /// the inputs under which the refs have different values.
    /// 
    /// \param x the node whose observability is considered
    /// \param y the possible replacement
    /// \param numObsLevels how many levels of local observability to consider
    /// \retval assignment contains a distinguishing assignment when false is
    /// returned
    /// \return true if \a x can be safely merged onto \a y
    bool                    verifyEquivalence(oagAi::Ref        x,
                                              oagAi::Ref        y,
                                              oa::oaUInt4       numObsLevels,
                                              Map<oagAi::Ref,
                                                  bool>::Type   &assignment);

    /// \brief Merges one node to another.
    ///
    /// The fanout of the first node is moved to the replacement ref.
    void                    merge(oagAi::Ref x,
                                  oagAi::Ref replacement);
    /// Merge recursively.
    void                    mergeRec(oagAi::Ref x,
                                     oagAi::Ref replacement);

    oaUInt4                 countMerged();

    ObsAiGraph              &graph;
    Set<oagAi::Ref>::Type   searchNodes;
    set<oagAi::Ref>         inputs;
    
    set<oagAi::Ref>                     repPool;
    Map<oagAi::Ref, oagAi::Ref>::Type   reps;

    AiLevelizer                         levelizer;
    vector<set<oagAi::Ref> >            searchLevels;
    vector<set<oagAi::Ref> >            repLevels;

    AiSimEngine                         simEngine;
    AiBitVectorDict                     simVecDict;
    map<oagAi::Ref, BitVector>          obsVectors;

    oaUInt4                 maxRef;
    Logger                      *logger;
};

}

#endif
