#include <cstdlib>
#include <iostream>
#include <iomanip>
#include "oagSswAiGraphUtil.h"
#include "oagSswOdcSatSweepingEngine.h"
#include "oagSswAiSatSolver.h"
#include "oagSswLogger.h"
#include "oagSswMap.h"

using namespace std;
using namespace oa;
using namespace oagAi;
using namespace oagMiniSat;

namespace oagSsw
{

#define QUIT_ON_INTERNAL_ERROR { assert(false); cerr << "ERROR: Internal error" << endl; exit(0); }

const string LOGGER_NAME = "oagSswOdc";

/// Sorts a list of statistics and returns their original indices in sorted
/// order.
///
/// \param stats
/// \retval sorted
void
sortIndicesBy(const vector<oaUInt4> &stats,
              vector<oaUInt4>       &sorted);

OdcSatSweepingEngine::OdcSatSweepingEngine(ObsAiGraph & graph)
: graph(graph),
  levelizer(graph),
  simEngine(graph),
  simVecDict(simEngine)
{
    list<Ref> allNodes;

    graph.getAll(allNodes);
    searchNodes.insert(allNodes.begin(), allNodes.end());
    addInputsFrom(allNodes);
    setRepPool(allNodes);

    levelizer.initialLevels();

    maxRef = 0;
    for (list<Ref>::iterator it = allNodes.begin();
         it != allNodes.end();
         ++it) {
        if ((oaUInt4)(*it) > maxRef)
            maxRef = (oaUInt4)(*it);
    } 
    simEngine.setSimVectorListLength(maxRef / 2 + 1);
    
    logger = Logger::getInstance(LOGGER_NAME);

    initializeSimVectors();
    clearStats();
}

OdcSatSweepingEngine::OdcSatSweepingEngine(ObsAiGraph             &graph, 
                                           const list<oagAi::Ref> &nodes)
: graph(graph),
  levelizer(graph),
  simEngine(graph),
  simVecDict(simEngine)
{
    for (list<Ref>::const_iterator it = nodes.begin();
         it != nodes.end();
         ++it) {
        searchNodes.insert(graph.getNonInverted(*it));
    }

    list<Ref> tfi;
    AiGraphUtil::getTransitiveFanin(graph, nodes, tfi);
    tfi.insert(tfi.end(), nodes.begin(), nodes.end());
    addInputsFrom(tfi);

    list<Ref> tfo;
    AiGraphUtil::getTransitiveFanout(graph, tfi, tfo);
    tfo.insert(tfo.end(), tfi.begin(), tfi.end());
    setRepPool(tfo);
    
    levelizer.initialLevels();

    list<Ref> allNodes;
    graph.getAll(allNodes);
    maxRef = 0;
    for (list<Ref>::iterator it = allNodes.begin();
         it != allNodes.end();
         ++it) {
        if ((oaUInt4)(*it) > maxRef)
            maxRef = (oaUInt4)(*it);
    } 
    
    simEngine.setSimVectorListLength(maxRef / 2 + 1);
 
    logger = Logger::getInstance(LOGGER_NAME);

    initializeSimVectors();
    clearStats();
}

void
OdcSatSweepingEngine::clearStats()
{
    stats.numMerges = 0;

    stats.sat.numSatisfiable    = 0;
    stats.sat.numUnsatisfiable  = 0;
    stats.sat.numUndecidable    = 0;
    
    stats.runtime.odcSatCheckTime       = 0.0;
    stats.runtime.computeObsVectorsTime = 0.0;
    stats.runtime.findReplacementTime   = 0.0;
    stats.runtime.searchTrieTime        = 0.0;
    stats.runtime.mergeFailedTime       = 0.0;
    stats.runtime.mergeTime             = 0.0;
    stats.runtime.computeSimTrieTime    = 0.0;
    stats.runtime.extendSimTime         = 0.0;
    stats.runtime.lastWordSimTime       = 0.0;
}

void
OdcSatSweepingEngine::addInputsFrom(const list<Ref> &pool)
{
    for (list<Ref>::const_iterator it = pool.begin(); it != pool.end(); ++it) {
        Ref x = *it;
        switch (graph.getNodeType(x)) {
          case Node::TERMINAL:
            if (!graph.isNull(graph.getTerminalDriver(x))) {
                break;
            }
            // fall through for null-driven terminals
          case Node::SEQUENTIAL:
            inputs.insert(graph.getNonInverted(x));
            break;
            
          default:
            ; // do nothing
        }
    }
}

void
OdcSatSweepingEngine::levelize()
{
    searchLevels.clear();
    for (Set<Ref>::iterator it = searchNodes.begin(); it != searchNodes.end(); ++it) {
        Ref x = *it;
        oaUInt4 level = levelizer.levelOf(x);
        if (searchLevels.size() < level + 1) {
            searchLevels.resize(level + 1);
        }
        searchLevels[level].insert(graph.getNonInverted(x));
    }

    repLevels.clear();
    for (set<Ref>::iterator it = repPool.begin(); it != repPool.end(); ++it) {
        Ref x = *it;
        oaUInt4 level = levelizer.levelOf(x);
        if (repLevels.size() < level + 1) {
            repLevels.resize(level + 1);
        }
        repLevels[level].insert(graph.getNonInverted(x));
    }
}

bool 
OdcSatSweepingEngine::hasRep(oagAi::Ref x)
{
    x = graph.getNonInverted(x);
    return reps.find(x) != reps.end();
}

oagAi::Ref
OdcSatSweepingEngine::getRep(oagAi::Ref x)
{
    x = graph.getNonInverted(x);

    Map<oagAi::Ref, oagAi::Ref>::iterator it = reps.find(x);
    if (it == reps.end()) {
        return graph.getNull();
    } else {
        Ref rep = it->second;
        if (hasRep(rep)) {
            rep = getRep(rep);
            reps[x] = rep;
        }
        return rep;
    }
}

void
OdcSatSweepingEngine::setRepPool(const list<oagAi::Ref> &reps)
{
    repPool.clear();
    for (list<Ref>::const_iterator it = reps.begin(); it != reps.end(); ++it) {
        Ref x = *it;
        repPool.insert(graph.getNonInverted(x));
    }
    addInputsFrom(reps);
    repPool.insert(graph.constantZero());
}


void
OdcSatSweepingEngine::populateSimVecDict(oaUInt4 level)
{
    vector<oaUInt4> bitOrder;
    computeDictBitOrder(searchLevels[level], bitOrder);

    simVecDict.initialize(bitOrder, params.maxSimVecBinSize);
    for (oaUInt4 i = 0; i <= level; ++i) {
        for (set<Ref>::iterator it = repLevels[i].begin();
                it != repLevels[i].end();
                ++it) {
            Ref x = *it;
            simVecDict.insert(x);
            simVecDict.insert(graph.notOf(x));
        }
    }

}

void
OdcSatSweepingEngine::run(oaUInt4 numObsLevels)
{
    oaTimer timer;
    runObsLevel(0);
    if (numObsLevels > 0) {
        runObsLevel(numObsLevels);
    }
    OAGSSW_LOGLN(logger, "Elapsed time: " << setprecision(3) << timer.getElapsed() << "s");
    OAGSSW_LOGLN(logger, "merged nodes in graph " << countMerged() );
}


void
OdcSatSweepingEngine::runObsLevel(oaUInt4 numObsLevels)
{
    OAGSSW_LOGLN(logger, "Running SAT sweeping with "
                         << numObsLevels << " levels of observability...");
    OAGSSW_LOGLN(logger, searchNodes.size() << " search nodes");
    OAGSSW_LOGLN(logger, inputs.size() << " inputs");
    OAGSSW_LOGLN(logger, repPool.size() << " nodes in rep pool");
    //OAGSSW_LOGLN(logger, "sim vector list length " << simEngine.simVectors.size());
    //OAGSSW_LOGLN(logger, "merged nodes in graph before odc sweeping " << countMerged() );
    OAGSSW_LOGLN(logger, stats.numMerges << " merged nodes in graph before sweeping with " 
                        << numObsLevels << " levels of observability");

    //initializeSimVectors();
    
    levelize();
    for (oaUInt4 i = 0; i < searchLevels.size(); ++i) {
        oaUInt4 oldNumMerges = stats.numMerges;

        //odcTimer.computeSimTrieTimer.reset();
        computeObsVectors(searchLevels[i], numObsLevels);
        // Note: All refs in searchLevels set are non-inverted refs.
        populateSimVecDict(i);
        //stats.runtime.computeSimTrieTime += odcTimer.computeSimTrieTimer.getElapsed();
        
        //OAGSSW_LOGLN(logger, searchLevels[i].size() << " search nodes in graph level " << i);
        for (set<Ref>::iterator it = searchLevels[i].begin();
             it != searchLevels[i].end();
             ++it) {
            Ref x = *it;
            if (hasRep(x))
                continue;
            Ref y = findReplacement(x, numObsLevels);

            // The original level of x may have changed as a result of an
            // earlier recursive merge.
            //odcTimer.mergeTimer.reset();
            if (!graph.isNull(y) && levelizer.levelOf(x) >= levelizer.levelOf(y)) {
                //OAGSSW_LOGLN(logger, "merge x " << x << " y " << y);
                merge(x, y);
                updateLocalSimVectors(y, numObsLevels);
            }
            //stats.runtime.mergeTime += odcTimer.mergeTimer.getElapsed();
        }

        //OAGSSW_LOGLN(logger, (stats.numMerges - oldNumMerges) << " nodes merged in graph level " << i);
        //OAGSSW_LOGLN(logger, "after level " << i << " current sim length " <<simEngine.getLength());
    }

    OAGSSW_LOGLN(logger, "Finished SAT sweeping.");
    OAGSSW_LOGLN(logger, stats.numMerges << " merged nodes in graph after sweeping with " 
                        << numObsLevels << " levels of observability");
    /*
    OAGSSW_LOGLN(logger, "odc sat check time: " << setprecision(3) << stats.runtime.odcSatCheckTime << "s");
    OAGSSW_LOGLN(logger, "compute obs vectors time: " << setprecision(3) << stats.runtime.computeObsVectorsTime << "s");
    OAGSSW_LOGLN(logger, "find replacement time: " << setprecision(3) << stats.runtime.findReplacementTime << "s");
    OAGSSW_LOGLN(logger, "search trie time: " << setprecision(3) << stats.runtime.searchTrieTime << "s");
    OAGSSW_LOGLN(logger, "operations after non-merge time: " << setprecision(3) << stats.runtime.mergeFailedTime << "s");
    OAGSSW_LOGLN(logger, "merge and update time: " << setprecision(3) << stats.runtime.mergeTime << "s");
    OAGSSW_LOGLN(logger, "comupte sim and construct trie time: " << setprecision(3) << stats.runtime.computeSimTrieTime << "s");
    OAGSSW_LOGLN(logger, "extend sim time: " << setprecision(3) << stats.runtime.extendSimTime << "s");
    OAGSSW_LOGLN(logger, "sim last word time: " << setprecision(3) << stats.runtime.lastWordSimTime << "s");
    */

}

oagAi::Ref
OdcSatSweepingEngine::findReplacement(oagAi::Ref   x,
                                      oa::oaUInt4  numObsLevels)
{
    //odcTimer.findReplacementTimer.reset();
    updateObsVector(x, numObsLevels);

    list<Ref> matches;
    //odcTimer.searchTrieTimer.reset();
    simVecDict.search(x, obsVectors[x], matches);
    //stats.runtime.searchTrieTime += odcTimer.searchTrieTimer.getElapsed();
    for (list<Ref>::iterator it = matches.begin();
         it != matches.end();
         ++it) {
        Ref y = *it;
        if (graph.getNonInverted(y) == graph.getNonInverted(x) 
            || hasRep(y))
            continue;

        const BitVector &obsVec = obsVectors[x];
        if (simEngine.maskedEquivalent(x, y, obsVec)) {
            Map<Ref, bool>::Type assignment;
            if (verifyEquivalence(x, y, numObsLevels, assignment)) {
                if (levelizer.levelOf(x) >= levelizer.levelOf(y)) {
                    //stats.runtime.findReplacementTime += odcTimer.findReplacementTimer.getElapsed();
                    return y;
                }
            } else {
                //odcTimer.mergeFailedTimer.reset();
                //odcTimer.extendSimTimer.reset();
                extendSimVectors(assignment);
                //stats.runtime.extendSimTime += odcTimer.extendSimTimer.getElapsed();
                //odcTimer.lastWordSimTimer.reset();
                /*
                list<Ref> allObservable; 
                graph.getAllObservable(allObservable);
                for (list<Ref>::iterator it = allObservable.begin();
                     it != allObservable.end();
                     it++) {
                    simEngine.simulateTransitiveFanIn(*it, true);
                }
                */
                simEngine.simulateAll(true);
                //stats.runtime.lastWordSimTime += odcTimer.lastWordSimTimer.getElapsed();
                updateObsVector(x, numObsLevels);
                //stats.runtime.mergeFailedTime += odcTimer.mergeFailedTimer.getElapsed();
            }
        }
    }

    //stats.runtime.findReplacementTime += odcTimer.findReplacementTimer.getElapsed();
    return graph.getNull();
}

void
OdcSatSweepingEngine::initializeSimVectors()
{
    oaUInt4 numBits = params.initNumSimBits;

    simEngine.setLength(numBits);
    simEngine.randomizeVars(0, numBits - 1);
    simEngine.simulateAll();
}

void
OdcSatSweepingEngine::computeObsVectors(const set<oagAi::Ref> &nodes,
                                        oa::oaUInt4           numObsLevels)
{
    //odcTimer.computeObsVectorsTimer.reset();
    
    for (set<Ref>::const_iterator it = nodes.begin();
         it != nodes.end();
         ++it) {
        computeObsVector(*it, numObsLevels, obsVectors[*it]);
    }
    //stats.runtime.computeObsVectorsTime += odcTimer.computeObsVectorsTimer.getElapsed();
}

void
OdcSatSweepingEngine::computeObsVector(oagAi::Ref   node,
                                       oa::oaUInt4  numObsLevels,
                                       BitVector    &obsVector)
{
    obsVector.setSize(simEngine.getLength());
    
    if (numObsLevels == 0 || graph.isObservable(node)) {
        for (oaUInt4 i = 0; i < obsVector.words.size(); i++) {
            obsVector.words[i] = ~0;
        }
        return;
    } 

    for (oaUInt4 i = 0; i < obsVector.words.size(); i++) {
        obsVector.words[i] = 0;
    }
    const list<Ref> &fanout = graph.getFanout(node);
    BitVector obsVectorFanout;
    for (list<Ref>::const_iterator it = fanout.begin();
         it != fanout.end();
         ++it) {
        Ref out = *it;
        if (hasRep(out)) {
            continue;
        }
        switch (graph.getNodeType(out)) {
          case Node::AND:
            {
                Ref sibling = AiGraphUtil::getAndOther(graph, out, node);
                assert(!graph.isNull(sibling));

                computeObsVector(out, numObsLevels - 1, obsVectorFanout);
                for (oaUInt4 i = 0; i < obsVector.words.size(); i++) {
                    obsVector.words[i] |= simEngine.getWord(sibling, i)
                                          & obsVectorFanout.words[i];
                }
            }
            break;

          case Node::TERMINAL:
            // This case is equivalent to having the terminal's fanout driven
            // directly by the node.
            computeObsVector(out, numObsLevels, obsVectorFanout);
            for (oaUInt4 i = 0; i < obsVector.words.size(); i++) {
                obsVector.words[i] |= obsVectorFanout.words[i];
            }
            break;

          case Node::SEQUENTIAL:
            // Latching of state does not count as observation.
            break;

          default:
            QUIT_ON_INTERNAL_ERROR;
        }
    }
}

struct Counts {
  oaUInt4 i[2];
  Counts() { i[0]=i[1]=0; }
};

void
OdcSatSweepingEngine::computeDictBitOrder(const set<oagAi::Ref> &nodes,
                                          vector<oaUInt4>       &bitOrder)
{
    oaUInt4 numSimBits = simEngine.getLength();
    bitOrder.resize(numSimBits);
    
    if (params.dictBitObjective == MAX_OBS_OBJ) {
        vector<oaUInt4> dcCount(numSimBits, 0);
        for (set<Ref>::const_iterator it = nodes.begin();
         it != nodes.end();
         ++it) {
            if (!hasRep(*it)) {
                for (oaUInt4 i = 0; i < numSimBits; i++) {
                    if (!obsVectors[*it].getBit(i))
                        dcCount[i]++;
                }
            }
        }
        sortIndicesBy(dcCount, bitOrder);
    } else if (params.dictBitObjective == MIN_EFFORT_OBJ) {
        const Counts c; 
        vector<Counts> nDC(numSimBits, c);
        vector<Counts> nCare(numSimBits, c);
        for (set<Ref>::const_iterator it = nodes.begin();
         it != nodes.end();
         ++it) {
            if (!hasRep(*it)) {
                for (oaUInt4 i = 0; i < numSimBits; i++) {
                    oaUInt4 bBit = simEngine.getBit(*it, i);
                    oaUInt4 oBit = obsVectors[*it].getBit(i);
                    if (oBit) {
                        nCare[i].i[bBit]++;
                    }
                    else {
                        nDC[i].i[bBit]++;
                    }
                }
            }
        }
        vector<oaUInt4> expEffort(numSimBits, 0);
        for (oaUInt4 i = 0; i < numSimBits; i++) {
            oaUInt4 d0 = nDC[i].i[0];
            oaUInt4 d1 = nDC[i].i[1];
            oaUInt4 c0 = nCare[i].i[0];
            oaUInt4 c1 = nCare[i].i[1];
            expEffort[i] = (d0 + d1) * (d0 + d1 + c0 + c1)
                + c0 * (c0 + d0)
                + c1 * (c1 + d1);
        }
        sortIndicesBy(expEffort, bitOrder);
    } else {
        QUIT_ON_INTERNAL_ERROR;
    }
}

void
OdcSatSweepingEngine::updateObsVector(oagAi::Ref    x,
                                      oa::oaUInt4   numObsLevels)
{
    computeObsVector(x, numObsLevels, obsVectors[x]); 
}

void
OdcSatSweepingEngine::addTransitiveFaninToSolver(AiSatSolver        &satSolver,
                                                 oagAi::Ref         node,
                                                 set<oagAi::Ref>    &alreadyAddedTransitiveFanin)
{
    if (alreadyAddedTransitiveFanin.find(node) !=  alreadyAddedTransitiveFanin.end()) {
        return;
    }
    alreadyAddedTransitiveFanin.insert(node);
    Ref left, right, driver;
    switch(graph.getNodeType(node)) {
      case Node::AND:
        left = graph.getAndLeft(node);
        right = graph.getAndRight(node);
        addTransitiveFaninToSolver(satSolver, left, alreadyAddedTransitiveFanin);
        addTransitiveFaninToSolver(satSolver, right, alreadyAddedTransitiveFanin);
        satSolver.addAndGate(satSolver.litOf(left),
                             satSolver.litOf(right),
                             satSolver.litOf(graph.getNonInverted(node)));
        break;
      case Node::TERMINAL:
        driver = graph.getTerminalDriver(node);
        if (!graph.isNull(driver)) {
            addTransitiveFaninToSolver(satSolver, driver, alreadyAddedTransitiveFanin);
            satSolver.addNotGate(satSolver.litOf(driver),
                                 satSolver.litOf(graph.notOf(graph.getNonInverted(node))));
        }
        break;
      case Node::SEQUENTIAL:
      case Node::CONSTANT0:
        break;
      default:
        QUIT_ON_INTERNAL_ERROR;
    }
}

void
OdcSatSweepingEngine::addObsClauses(AiSatSolver     &satSolver,
                                    oagAi::Ref      node,
                                    Lit             nodeObsLit,
                                    oaUInt4         numObsLevels,
                                    set<ObsRef>     &alreadyAdded,
                                    set<oagAi::Ref> &alreadyAddedTransitiveFanin,
                                    const vector<bool>    &tfo)
{
    node = graph.getNonInverted(node);

    if (alreadyAdded.find(ObsRef(node, numObsLevels)) != alreadyAdded.end()) {
        return;
    }

    alreadyAdded.insert(ObsRef(node, numObsLevels));

    addTransitiveFaninToSolver(satSolver, node, alreadyAddedTransitiveFanin);
    if (numObsLevels == 0 || graph.isObservable(node)) {
        satSolver.addUnit(nodeObsLit);
        return;
    } 

    vector<Ref> validOutNodes;
    vector<Ref> validSiblingNodes;
    const list<Ref> &fanout = graph.getFanout(node);
    for (list<Ref>::const_iterator it = fanout.begin();
         it != fanout.end();
         ++it) {
        Ref out = *it;
        if (hasRep(out)) {
            continue;
        }
        switch (graph.getNodeType(out)) {
          case Node::AND:
            {
                Ref sibling = AiGraphUtil::getAndOther(graph, out, node);
                assert(!graph.isNull(sibling));
                validOutNodes.push_back(out);
                //validSiblingNodes.push_back(sibling);
                if (!tfo[(oaUInt4)(sibling) >> 1]) {
                    validSiblingNodes.push_back(sibling);
                } else {
                    //OAGSSW_LOGLN(logger, "found start in sibling");
                    validSiblingNodes.push_back(graph.constantOne());
                }
            }
            break;

          case Node::TERMINAL:
            // This case is equivalent to having the terminal's fanout driven
            // directly by the node.
            {
                OAGSSW_LOGLN(logger, "not AND node!");
                validOutNodes.push_back(out);
                validSiblingNodes.push_back(graph.constantOne());
            }
            break;

          case Node::SEQUENTIAL:
            // Latching of state does not count as observation.
            break;

          default:
            QUIT_ON_INTERNAL_ERROR;
        }
    }

    // construct clauses
    oaUInt4 numValidOut = validOutNodes.size();
    vec<Lit> edgeObsLits(numValidOut);
    for (oaInt4 i = 0; i < edgeObsLits.size(); i++)
    {
        edgeObsLits[i] = Lit(satSolver.newVar());
    } 
    satSolver.addOrGate(edgeObsLits, nodeObsLit);
    if (numValidOut == 0) {
        return;
    }
    vector<Lit> fanoutNodeObsLits(numValidOut);
    for (oaUInt4 i = 0; i < numValidOut; i++)
    {
        //OAGSSW_LOGLN(logger, "out " << validOutNodes[i] << " neighbor " << validSiblingNodes[i]);
        fanoutNodeObsLits[i] = Lit(satSolver.newVar());
        satSolver.addAndGate(satSolver.litOf(validSiblingNodes[i]),
                             fanoutNodeObsLits[i],
                             edgeObsLits[i]);
        if (graph.getNodeType(node) == Node::AND) {
            addObsClauses(satSolver, validOutNodes[i], fanoutNodeObsLits[i], 
                          numObsLevels - 1, alreadyAdded, alreadyAddedTransitiveFanin, tfo);
        } else if (graph.getNodeType(node) == Node::TERMINAL) {
            addObsClauses(satSolver, validOutNodes[i], fanoutNodeObsLits[i], 
                          numObsLevels, alreadyAdded, alreadyAddedTransitiveFanin, tfo);
        }
    }
}

bool
OdcSatSweepingEngine::verifyEquivalence(oagAi::Ref                      x,
                                        oagAi::Ref                      y,
                                        oa::oaUInt4                     numObsLevels,
                                        Map<oagAi::Ref, bool>::Type     &assignment)
{
    //OAGSSW_LOGLN(logger, "verify " << x <<" " << y);
    //odcTimer.odcSatCheckTimer.reset();
    AiSatSolver satSolver(graph); 
    
    set<Ref> alreadyAddedTransitiveFanin;
    
    addTransitiveFaninToSolver(satSolver, x, alreadyAddedTransitiveFanin);
    addTransitiveFaninToSolver(satSolver, y, alreadyAddedTransitiveFanin);

    vec<Lit> xObsClause;
    Lit xObsLit = Lit(satSolver.newVar());
    satSolver.addUnit(xObsLit);
    set<ObsRef> alreadyAdded;

    vector<bool> tfo;
    tfo.resize(maxRef / 2 + 1);
    list<Ref> tfoList;
    graph.getTransitiveFanout(x, tfoList);
    for (list<Ref>::iterator it = tfoList.begin();
         it != tfoList.end();
         it++) {
        tfo[(oaUInt4)(*it) >> 1] = true;
    }
    addObsClauses(satSolver, x, xObsLit, numObsLevels, alreadyAdded, alreadyAddedTransitiveFanin, tfo);
    tfo.clear();
    
    satSolver.addNotGate(satSolver.litOf(x), 
                         satSolver.litOf(y)); 

    //OAGSSW_LOGLN(logger, "finsh adding clauses fo verifying " << x <<" " << y);
    lbool res = satSolver.solve(params.maxSatBacktracks);

    if (res == l_True) {
        // SAT
        stats.sat.numSatisfiable++;

        for (set<Ref>::iterator it = inputs.begin();
             it != inputs.end();
             ++it) {
            Ref x = graph.getNonInverted(*it);
            lbool value = satSolver.getValue(x);
            if (value == l_True) {
                assignment[x] = true;
            } else if (value == l_False) {
                assignment[x] = false;
            }
        }
        //stats.runtime.odcSatCheckTime += odcTimer.odcSatCheckTimer.getElapsed();
        return false;
    } else if (res == l_False) {
        //UNSAT
        stats.sat.numUnsatisfiable++;

        //stats.runtime.odcSatCheckTime += odcTimer.odcSatCheckTimer.getElapsed();
        return true;
    } else {
        //UNDECIDED
        stats.sat.numUndecidable++;

        //stats.runtime.odcSatCheckTime += odcTimer.odcSatCheckTimer.getElapsed();
        return false;
    }
}


void
OdcSatSweepingEngine::extendSimVectors(const Map<oagAi::Ref, bool>::Type &assignment)
{
    oaUInt4 i = simEngine.getLength();
    simEngine.setLength(i + 1);

    for (set<Ref>::iterator it = inputs.begin(); it != inputs.end(); ++it) {
        Ref x = graph.getNonInverted(*it);

        Map<Ref, bool>::Type::const_iterator it2 = assignment.find(x);
        if (it2 != assignment.end()) {
            bool bit = it2->second;
            simEngine.setBit(x, i, bit);
        } else if (x == graph.constantZero()) {
            simEngine.setBit(x, i, false);
        } else {
            bool bit = ((rand() & 1) == 0);
            simEngine.setBit(x, i, bit);
        }
    }
}

void
OdcSatSweepingEngine::merge(oagAi::Ref  x,
                            oagAi::Ref  replacement)
{
    mergeRec(x, replacement);
    //levelizer.clear();
}

void
OdcSatSweepingEngine::mergeRec(oagAi::Ref  x,
                               oagAi::Ref  replacement)
{
    // Only merge nodes that the user specified to be merged.
    if (searchNodes.find(x) == searchNodes.end()) {
        return;
    }

    if (hasRep(x)) {
        return;
    }
    assert(!hasRep(replacement));

    //Logger *logger = Logger::getInstance("oagSswOdc/merge");
    //OAGSSW_LOGLN(logger, "Merge: " << x << " -> " << replacement);

    graph.resubstitute(x, replacement);
    if (graph.isObservable(x)) {
        graph.setObservable(replacement, true);
        //OAGSSW_LOGLN(logger, "x " << x << " was observable " );
    }
    stats.numMerges++;

    levelizer.updateLevel(replacement);
    reps[graph.getNonInverted(x)] = replacement;

    if (graph.structuralHashingEnabled) {
        typedef pair<Ref, Ref> RefPair;

        map<RefPair, Ref> fanoutHash;
        list<RefPair> toMerge;
        const list<Ref> &fanout = graph.getFanout(replacement);
        for (list<Ref>::const_iterator it = fanout.begin();
                it != fanout.end();
                ++it) {
            Ref out = *it;

            if (graph.getNodeType(out) != Node::AND || hasRep(out)) {
                continue;
            }
        
            RefPair inputsLR    = RefPair(graph.getAndLeft(out), graph.getAndRight(out));
            Ref     hashedLR    = fanoutHash[inputsLR];
            if (graph.isNull(hashedLR)
                    || levelizer.levelOf(hashedLR) > levelizer.levelOf(out)) {
                fanoutHash[inputsLR] = out;
            }

            RefPair inputsRL    = RefPair(graph.getAndRight(out), graph.getAndLeft(out));
            Ref     hashedRL    = fanoutHash[inputsRL];
            if (graph.isNull(hashedRL)
                    || levelizer.levelOf(hashedRL) > levelizer.levelOf(out)) {
                fanoutHash[inputsRL] = out;
            }
        }

        for (list<Ref>::const_iterator it = fanout.begin();
                it != fanout.end();
                ++it) {
            Ref out = *it;

            if (graph.getNodeType(out) != Node::AND || hasRep(out)) {
                continue;
            }
        
            RefPair inputs  = RefPair(graph.getAndLeft(out), graph.getAndRight(out));
            Ref     hashed  = fanoutHash[inputs];
            if (hashed != out) {
               toMerge.push_back(RefPair(out, hashed));
            }
        }

        for (list<RefPair>::iterator it = toMerge.begin();
             it != toMerge.end();
             ++it) {
            mergeRec(it->first, it->second);
        }
    }
}

void
OdcSatSweepingEngine::updateLocalSimVectors(oagAi::Ref  x,
                                            oa::oaUInt4 numObsLevels)
{
    if (numObsLevels == 0 || graph.isObservable(x)) {
        return;
    }

    if (graph.getNodeType(x) == Node::AND) {
        simEngine.simulateAnd(x);

        const list<Ref> &fanout = graph.getFanout(x);
        for (list<Ref>::const_iterator it = fanout.begin();
             it != fanout.end();
             ++it) {
            if (!hasRep(*it)) {
                updateLocalSimVectors(*it, numObsLevels - 1);
            }
        }
    }
}

oaUInt4
OdcSatSweepingEngine::countMerged() {
    oaUInt4 counter = 0;
    
    list<Ref> allNodes;
    graph.getAll(allNodes);
    for (list<Ref>::iterator it = allNodes.begin();
         it != allNodes.end();
         ++it) {
        if (hasRep(*it)) { 
            if (!(graph.getNodeType(*it) == Node::AND))
                OAGSSW_LOGLN(logger, "Not AND node merged");
            counter++;
        }
    } 
    return counter;
}
 
void
sortIndicesBy(const vector<oaUInt4> &stats,
              vector<oaUInt4>       &sorted)
{
    typedef pair<oaUInt4, oaUInt4> UIntPair;

    vector<UIntPair> pairs(stats.size());

    for (oaUInt4 i = 0; i < stats.size(); ++i) {
        pairs[i].first = stats[i];
        pairs[i].second = i;
    }
    
    // Here we take advantage of the fact that pairs compare lexicographically.
    sort(pairs.begin(), pairs.end());

    sorted.resize(pairs.size());
    for (oaUInt4 i = 0; i < pairs.size(); ++i) {
        sorted[i] = pairs[i].second;
    }
}

}
