#include <cassert>
#include <iostream>
#include "oagSswAiGraphUtil.h"

using namespace std;
using namespace oagAi;

namespace oagSsw
{

#define QUIT_ON_ERROR { assert(false); exit(0); }
#define QUIT_ON_INTERNAL_ERROR { assert(false); cerr << "ERROR: Internal error" << endl; exit(0); }


// *****************************************************************************
// getTransitiveFanin()
//
/// \brief Returns a list of all nodes in the transitive fan-in.
///
/// The nodes in the transitive fan-in are appended to the provided list.
/// The initial nodes are only included when they belong to the fan-in cone
/// of other initial nodes.  All references will be non-inverted.
///
/// The backward traversal will terminate at sequential nodes, terminal nodes
/// that point to a null reference, and the constant zero node.  If an initial
/// reference is to a sequential node, the traversal from that node will
/// continue and not terminate immediately.
///
/// There must be no cycles in the fan-in cone.
///
/// \param graph the graph containing the vertices
/// \param refs the vertices whose transitive fan-in is collected
/// \retval transitiveFanin the list onto which to append the results
//
// *****************************************************************************
void AiGraphUtil::getTransitiveFanin(oagAi::Graph & graph,
    const list<oagAi::Ref> & refs,
    list<oagAi::Ref> & transitiveFanin) {
  // Based on source from oagAi::Graph::getTransitiveFanin
  map<Ref, bool> alreadyVisited;
  list<Ref> toBeVisited;

  for (list<Ref>::const_iterator it = refs.begin(); it != refs.end(); ++it) {
    Ref x = graph.getNonInverted(*it);

    toBeVisited.push_back(x);
    alreadyVisited[x] = true;

    while(!toBeVisited.empty()) {
      Ref current = toBeVisited.front();

      Ref   fanin1, fanin2 = graph.getNull();
      switch(graph.getNodeType(current)) {
      case Node::AND:
        fanin1 = graph.getAndLeft(current);
        fanin2 = graph.getAndRight(current);
        break;
      case Node::TERMINAL:
        fanin1 = graph.getTerminalDriver(current);
        break;
      case Node::SEQUENTIAL:
        fanin1 = graph.getNextState(current);
        break;
      case Node::CONSTANT0:
        goto continue_loop;
      default:
        QUIT_ON_INTERNAL_ERROR;
      }

      if (graph.isNull(fanin1)) {
        // we're done; there weren't any fanin
        goto continue_loop;
      }

      fanin1 = graph.getNonInverted(fanin1);
      // have we already explored this node?
      if (alreadyVisited.find(fanin1) == alreadyVisited.end()) {
        alreadyVisited[fanin1] = true;
        if (graph.isTerminal(fanin1) && graph.isNull(graph.getTerminalDriver(fanin1)) 
            || graph.getNonInverted(fanin1) == graph.constantZero()
            || graph.isSequential(fanin1)) {
          transitiveFanin.push_back(fanin1);
        } else {
          transitiveFanin.push_back(fanin1);  
          toBeVisited.push_back(fanin1);
        }
      }

      if (graph.isNull(fanin2)) {
        // we're done; there was only one fanin
        goto continue_loop;
      }

      fanin2 = graph.getNonInverted(fanin2);    
      // have we already explored this node?
      if (alreadyVisited.find(fanin2) == alreadyVisited.end()) {
        alreadyVisited[fanin2] = true;
        if (graph.isTerminal(fanin2) && graph.isNull(graph.getTerminalDriver(fanin2)) 
            || graph.getNonInverted(fanin2) == graph.constantZero()
            || graph.isSequential(fanin2)) {
          transitiveFanin.push_back(fanin2);   
        } else {
          transitiveFanin.push_back(fanin2);  
          toBeVisited.push_back(fanin2);
        }
      }

continue_loop:
      toBeVisited.pop_front();
    }
  }
}

// *****************************************************************************
// getTransitiveFanout()
//
/// \brief Returns a list of all nodes in the transitive fan-out.
///
/// The nodes in the transitive fan-out are appended to the provided list.
/// The initial nodes are only included when they belong to the fan-out cone of
//// other initial nodes.  All references will be non-inverted.
///
/// The forward traversal will terminate at sequential nodes and nodes that
/// do not have any fan-out.  If an initial reference is to a sequential node,
/// the traversal of that node will continue and not terminate immediately.
///
/// There must be no cycles in the fan-out cone.
///
/// \param graph the graph containing the vertices
/// \param refs the vertices whose transitive fan-out is collected
/// \retval transitiveFanout the list onto which to append the results
//
// *****************************************************************************
void
AiGraphUtil::getTransitiveFanout(oagAi::Graph & graph,
    const list<oagAi::Ref> & refs,
    list<oagAi::Ref> & transitiveFanout) {
  // Based on source from oagAi::Graph::getTransitiveFanout
  map<Ref, bool> alreadyVisited;
  list<Ref> toBeVisited;

  for (list<Ref>::const_iterator it = refs.begin(); it != refs.end(); ++it) {
    Ref x = graph.getNonInverted(*it);
    
    toBeVisited.push_back(x);
    alreadyVisited[x] = true;

    while(!toBeVisited.empty()) {
        Ref current = graph.getNonInverted(toBeVisited.front());
      
        const list<Ref> &immediateFanout = graph.getFanout(current);

        for(list<Ref>::const_iterator it = immediateFanout.begin(); 
            it != immediateFanout.end(); 
            ++it) {

            Ref next = graph.getNonInverted(*it);
            // there shouldn't be fanout to a null or constant node
            assert(!graph.isNull(next));
            assert(graph.getNodeType(next) != Node::CONSTANT0);

            // have we already explored this node?
            if (alreadyVisited.find(next) != alreadyVisited.end()) {
                continue;
            }
            alreadyVisited[next] = true;
            
            if (graph.getNodeType(next) == Node::SEQUENTIAL) {
                transitiveFanout.push_back(next);
            } else {
                transitiveFanout.push_back(next);
                toBeVisited.push_back(next);
            }
        }
    
        toBeVisited.pop_front();
    }
  }
}

oagAi::Ref
AiGraphUtil::getAndOther(oagAi::Graph   &graph, 
                         oagAi::Ref     x,
                         oagAi::Ref     input)
{
    input = graph.getNonInverted(input);
    assert(graph.getNodeType(x) == Node::AND);
    if (graph.getNonInverted(graph.getAndLeft(x)) == input) {
        return graph.getAndRight(x);
    }
    if (graph.getNonInverted(graph.getAndRight(x)) == input) {
        return graph.getAndLeft(x);
    }
    return graph.getNull();
}

}
