/*
Author: Kai-hui Chang <changkh@eecs.umich.edu>
*/

#include <iostream>
#include <string>
#include "oagFunc.h"
#include "oagFuncOccGraph.h"
#include "oagFuncSimOcc.h"
#include "oagResynLibMan.h"
#include "oagFuncManager.h"

// #define DEBUG

using namespace oagFunc;
using namespace std;
using namespace oa;

namespace oagResyn {

// *****************************************************************************
// addCell()
//
/// \brief Add a cell to the library. 
//
// *****************************************************************************
void LibMan::addCell(string& cellName)
{
    cellMapType::iterator cellIter;

    cellIter= cellNameMap.find(cellName);
    if (cellIter == cellNameMap.end())
    {
        cellNameMap[cellName]= new(libCell);
        cellNames.push_back(cellName);
    }
}

// *****************************************************************************
// lookUpCellType()
//
/// \brief Look up a cell in the library according to type.
//
// *****************************************************************************
libCell* LibMan::lookUpCellType(cellLibType type)
{
    map<cellLibType, libCell*>::iterator cellIter;

    cellIter= cellTypeMap.find(type);
    if (cellIter == cellTypeMap.end())
        return NULL;
    else
        return ((*cellIter).second);
}

// *****************************************************************************
// lookUpCell()
//
/// \brief Look up a cell to the library. 
//
// *****************************************************************************
libCell* LibMan::lookUpCell(string& newCell)
{
    cellMapType::iterator cellIter;

    cellIter= cellNameMap.find(newCell);
    if (cellIter == cellNameMap.end())
        return NULL;
    else
        return ((*cellIter).second);
}

// *****************************************************************************
// printLib()
//
/// \brief Print cells in the library.
//
// *****************************************************************************
void LibMan::printLib() 
{
    cellMapType::iterator cellIter;
    libCell* myCell;
    string myStr;

    cout<<"Cell library:"<<endl;
    for (cellIter= cellNameMap.begin(); cellIter != cellNameMap.end(); cellIter++)
    {
        myStr= (*cellIter).first;
        myCell= (*cellIter).second;
        cout<<myStr<<endl;
        printCell(myCell);
    }
}


// *****************************************************************************
// genLogicInfo()
//
/// \brief Generate Logic info and put it in libCell. 
///
/// \param design The oaDesign for generating logic info (usually a cell)
/// \return false if any error.
//
// *****************************************************************************
bool LibMan::genLogicInfo(oaDesign *design) 
{
    cellMapType::iterator cellIter;
    libCell* myCell;
    string myStr;
    oaDesign* myDesign;
    const oaNativeNS nativeNS;

    for (cellIter= cellNameMap.begin(); cellIter != cellNameMap.end(); cellIter++)
    {
        myStr= (*cellIter).first;
        #ifdef DEBUG
        cout<<"Cell name is "<<myStr<<endl;
        #endif
        myCell= (*cellIter).second;
        // If information about this cell has already been generated, skip
        if (myCell->satCell)    
            continue;
        oaScalarName cellName(nativeNS, myStr.c_str());
        oaScalarName libName, viewName;
        design->getLibName(libName);
        design->getViewName(viewName);
        myDesign= oaDesign::find(libName, cellName, viewName);
        if (myDesign)
        {
            ;
        } else
        {
            if (myDesign == NULL)
            {
                oaScalarName str(nativeNS, "netlist");
                myDesign= oaDesign::open(libName, cellName, str, 'r');
                if (myDesign == NULL)
                    return false;
            }
        }
        myCell->design= myDesign;
        if (genCellLogic(myCell, myDesign) == false)
            return false;
        if (genCellSAT(myCell, myDesign) == false)
            return false;
    }
    return true;
}

// *****************************************************************************
// genCellLogic()
//
/// \brief Generate Logic info for a libCell. 
///
/// \return False if any error.
//
// *****************************************************************************
bool LibMan::genCellLogic(libCell* myCell, oaDesign *myDesign) 
{
    int outputNO, inputNO;
    int mintermSize;
    int i, j, inputCount, outputCount;
    int vectorSize;
    int iterateSize;
    int bitCount;
    oagFunc::SimulationVector vector;
    list<oagFunc::OccRef>::iterator refIter;
    bitset<tTableSize> tTable;
    const oaNativeNS nativeNS;
    oaString str;
    int type= 0;
    
    list<oagFunc::OccRef> inputs, outputs, states;
    oaOccurrence *occurrence = myDesign->getTopOccurrence();
    oagFunc::OccGraph::getInputs(occurrence, inputs);
    oagFunc::OccGraph::getOutputs(occurrence, outputs);
    oagFunc::OccGraph::getStates(occurrence, states);
    oagFunc::SimOcc mySim(myDesign);

    outputNO= outputs.size();
    inputNO= inputs.size();
    // Cannot handle sequential logic and inputs > maxCellInput
    if (states.size() > 0 || inputNO > maxCellInput)
    {
        cout<<"Cannot handle sequential logic or inputs > "<<maxCellInput<<", abort."<<endl;
        return false;
    }
    refIter= outputs.begin(); 
    for (i= 0; i < outputNO; i++)
    {
        oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*refIter);
        assert(net);
        net->getName(nativeNS, str);
        myCell->tTables.push_back(tTable);
        myCell->outputNames.push_back(string(str));
        refIter++;
    }
    refIter= inputs.begin(); 
    for (i= 0; i < inputNO; i++)
    {
        oa::oaOccBitNet *net = oagFunc::OccGraph::getNetToAiConnection(*refIter);
        assert(net);
        net->getName(nativeNS, str);
        myCell->inputNames.push_back(string(str));
        refIter++;
    }
    mintermSize= 1 << inputNO;
    vectorSize= sizeof(oagFunc::SimulationVector) * 8;
    iterateSize= mintermSize / vectorSize;
    if (iterateSize < 1)
        iterateSize= 1;
    for (i= 0; i < iterateSize; i++)
    {
        inputCount= 0;
        for (refIter= inputs.begin(); refIter != inputs.end(); refIter++)
        {
            for (j= 0; j < vectorSize; j++)
            {
                bitCount= i * vectorSize + j;
                if (bitCount & (1 << inputCount))
                    vector |= (1 << j);
                else
                    vector &= ~(1 << j);
            }
            mySim.setVector(*refIter, vector);
            inputCount++;
        }
        mySim.runFull();
        outputCount= 0;
        for (refIter= outputs.begin(); refIter != outputs.end(); refIter++)
        {
            mySim.getVector(*refIter, vector);
            for (j= 0; j < vectorSize; j++)
            {
                bitCount= i * vectorSize + j;
                if (bitCount >= mintermSize)
                    break;
                if (vector & (1 << j))
                    myCell->tTables[outputCount][bitCount]= true;
                else
                    myCell->tTables[outputCount][bitCount]= false;
            }
            outputCount++;
        }
    }
    // Add some basic types to cellTypeMap
    if (outputNO == 1)
    {
        if (inputNO < 3)
            type= myCell->tTables[0].to_ulong();
        if (inputNO == 1)
        {
            if (type == 1)
                cellTypeMap[INV]= myCell;
            else
                cellTypeMap[BUFFER]= myCell;
        }
        else if (inputNO == 2)
        {
            switch (type)
            {
            case 8:
                cellTypeMap[AND]= myCell;
                break;
            case 7:
                cellTypeMap[NAND]= myCell;
                break;
            case 14:
                cellTypeMap[OR]= myCell;
                break;
            case 1:
                cellTypeMap[NOR]= myCell;
                break;
            case 6:
                cellTypeMap[XOR]= myCell;
                break;
            case 9:
                cellTypeMap[XNOR]= myCell;
                break;
            }
        }
    }
    myCell->inputNo= inputNO;
    return true;
}

// *****************************************************************************
// printCell()
//
/// \brief Print information saved in cell. 
///
/// For debugging.
//
// *****************************************************************************
void LibMan::printCell(libCell* cell)
{
    libCell* myCell= cell;
    string myStr;
    vector<string>::iterator strIter;
    unsigned int i;
    int j;
    int mintermSize;

    cout<<"Input: ";
    for (strIter= myCell->inputNames.begin(); strIter != myCell->inputNames.end(); strIter++)
    {
        cout<<*strIter<<" ";
    }
    cout<<endl;
    cout<<"Output: ";
    for (strIter= myCell->outputNames.begin(); strIter != myCell->outputNames.end(); strIter++)
    {
        cout<<*strIter<<" ";
    }
    cout<<endl;
    cout<<"Truth Table: "<<endl;
    mintermSize= 1 << myCell->inputNames.size();
    for (i= 0; i < myCell->outputNames.size(); i++)
    {
        for (j= mintermSize - 1; j >= 0; j--)
        {
            if (myCell->tTables[i][j])
                cout<<"1";
            else
                cout<<"0";
        }
        cout<<endl;
        if (myCell->inputNo <= 4)
            cout<<"("<<myCell->tTables[i].to_ulong()<<")"<<endl;
    }
    if (cell->satCell)
    {
        libSATCell* mySat= cell->satCell;
        vector< vector<int> >::iterator iter1;
        vector<int>::iterator iter2;
        
        cout<<"SAT info: Extra NO= "<<mySat->extraNO<<endl;
        for (iter1= mySat->clauses.begin(); iter1 != mySat->clauses.end(); iter1++)
        {
            for (iter2= (*iter1).begin(); iter2 != (*iter1).end(); iter2++)
            {
                cout<<*iter2<<" ";
            }
            cout<<"0"<<endl;
        }
    }
}

// *****************************************************************************
// genCellSAT()
//
/// \brief Generate SAT info for a libCell. 
///
/// \return False if any error.
//
// *****************************************************************************
bool LibMan::genCellSAT(libCell* myLibCell, oaDesign *myDesign) 
{
    int outputNO, inputNO;
    int i;
    list<oagFunc::OccRef>::iterator refIter;
    const oaNativeNS nativeNS;
    oaString str;
    oagFunc::OccRef curr, left, right, nleft, nright, nref2;
    list<oagFunc::OccRef> inputs, outputs, states;
    oaOccurrence *occurrence = myDesign->getTopOccurrence();
    oagFunc::OccGraph::getInputs(occurrence, inputs);
    oagFunc::OccGraph::getOutputs(occurrence, outputs);
    map<oagFunc::OccRef, int> refNOMap;
    list<oagFunc::OccRef> refQueue;
    int extraBase;
    vector<int> clause;
    int extraCount;
    libSATCell* myCell;

    myCell= new(libSATCell);
    myLibCell->satCell= myCell;
    outputNO= outputs.size();
    inputNO= inputs.size();
    // Cannot handle sequential logic and inputs > maxCellInput
    if (states.size() > 0 || inputNO > maxCellInput)
    {
        cout<<"Cannot handle sequential logic or inputs > "<<maxCellInput<<", abort."<<endl;
        return false;
    }
    myCell->inputNO= inputNO;
    myCell->outputNO= outputNO;
    myCell->extraNO= 0;
    if (myCell->inputNO <= 4 && myCell->outputNO == 1 && useHardCodeCNF)
    {
        unsigned long type;

        type= myLibCell->tTables[0].to_ulong();
        if (myCell->inputNO == 1)
        {
            switch (type)
            {
            case 1: // INV
                clause.push_back(1);
                clause.push_back(2);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-1);
                clause.push_back(-2);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            case 2: // BUF
                clause.push_back(1);
                clause.push_back(-2);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-1);
                clause.push_back(2);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            }
        }
        else if (myCell->inputNO == 2)
        {
            switch (type)
            {
            case 8: // AND
                break; // late AIG handle it
            case 7: // NAND
                clause.push_back(-1);
                clause.push_back(-2);
                clause.push_back(-3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(1);
                clause.push_back(3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(2);
                clause.push_back(3);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            case 14: // OR
                clause.push_back(1);
                clause.push_back(2);
                clause.push_back(-3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-1);
                clause.push_back(3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-2);
                clause.push_back(3);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
             case 1: // NOR
                clause.push_back(1);
                clause.push_back(2);
                clause.push_back(3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-1);
                clause.push_back(-3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-2);
                clause.push_back(-3);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
             case 6: // XOR
                clause.push_back(1);
                clause.push_back(2);
                clause.push_back(-3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-1);
                clause.push_back(-2);
                clause.push_back(-3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(1);
                clause.push_back(-2);
                clause.push_back(3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-1);
                clause.push_back(2);
                clause.push_back(3);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            default: ;
            }
        } else if (myCell->inputNO == 3)
        {
            switch (type)
            {
            case 7: // AOI21
                clause.push_back(1);
                clause.push_back(3);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(2);
                clause.push_back(3);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-3);
                clause.push_back(-4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-4);
                clause.push_back(-1);
                clause.push_back(-2);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            case 202: // MX2X1
                clause.push_back(-4);
                clause.push_back(-3);
                clause.push_back(2);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-4);
                clause.push_back(1);
                clause.push_back(3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(4);
                clause.push_back(3);
                clause.push_back(-1);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(4);
                clause.push_back(-3);
                clause.push_back(-2);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            case 1: // NOR    
                clause.push_back(1);
                clause.push_back(2);
                clause.push_back(3);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-1);
                clause.push_back(-4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-2);
                clause.push_back(-4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-3);
                clause.push_back(-4);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            case 127: // NAND
                clause.push_back(-1);
                clause.push_back(-2);
                clause.push_back(-3);
                clause.push_back(-4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(1);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(2);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(3);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            case 31: // OAI21
                clause.push_back(3);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(4);
                clause.push_back(1);
                clause.push_back(2);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-4);
                clause.push_back(-3);
                clause.push_back(-1);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-4);
                clause.push_back(-3);
                clause.push_back(-2);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            }
        } else if (myCell->inputNO == 4)
        {
            switch (type)
            {
            case 4383: // OAI22
                clause.push_back(5);
                clause.push_back(1);
                clause.push_back(2);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(5);
                clause.push_back(3);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-5);
                clause.push_back(-1);
                clause.push_back(-3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-5);
                clause.push_back(-1);
                clause.push_back(-4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-5);
                clause.push_back(-2);
                clause.push_back(-3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-5);
                clause.push_back(-2);
                clause.push_back(-4);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            case 1911:
                clause.push_back(5);
                clause.push_back(1);
                clause.push_back(3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(5);
                clause.push_back(2);
                clause.push_back(3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(5);
                clause.push_back(1);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(5);
                clause.push_back(2);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-5);
                clause.push_back(-1);
                clause.push_back(-2);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-5);
                clause.push_back(-3);
                clause.push_back(-4);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            case 32767: // NAND4
                clause.push_back(-5);
                clause.push_back(-1);
                clause.push_back(-2);
                clause.push_back(-3);
                clause.push_back(-4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(5);
                clause.push_back(1);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(5);
                clause.push_back(2);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(5);
                clause.push_back(3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(5);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            case 65534: // OR4
                clause.push_back(-5);
                clause.push_back(1);
                clause.push_back(2);
                clause.push_back(3);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(5);
                clause.push_back(-1);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(5);
                clause.push_back(-2);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(5);
                clause.push_back(-3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(5);
                clause.push_back(-4);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            case 1: // NOR4
                clause.push_back(5);
                clause.push_back(1);
                clause.push_back(2);
                clause.push_back(3);
                clause.push_back(4);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-5);
                clause.push_back(-1);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-5);
                clause.push_back(-2);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-5);
                clause.push_back(-3);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-5);
                clause.push_back(-4);
                myCell->clauses.push_back(clause);
                clause.clear();
                return true;
            }
        }
    }
    extraBase= inputNO + outputNO + 1;
    extraCount= extraBase;
    refIter= inputs.begin(); 
    for (i= 1; i <= inputNO; i++)
    {
        refNOMap[*refIter]= i;
        refIter++;
    }
    refIter= outputs.begin(); 
    for (i= 0; i < outputNO; i++)
    {
        oagFunc::OccRef ref2;    
        ref2= OccGraph::getTerminalDriver(*refIter);
        if (OccGraph::isNull(ref2) || OccGraph::isTerminal(ref2))
        {
            refNOMap[*refIter]= inputNO + i + 1;
            // special case: terminal directly connects to terminal.
            // Either BUF or INV
            // Looks like if isNull: buf, isTerminal: inv 
            if (OccGraph::isNull(ref2) == false)
            {
                // INV, driver's nonInvert version is the input
                assert(OccGraph::isInverted(ref2));
                nref2= OccGraph::getNonInverted(ref2);
                clause.push_back(refNOMap[nref2]);
                clause.push_back(refNOMap[*refIter]);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-refNOMap[nref2]);
                clause.push_back(-refNOMap[*refIter]);
                myCell->clauses.push_back(clause);
                clause.clear();
            }
            else
            {
                // BUF. Need to find out which one connects to the output.
                std::set<oagFunc::OccRef> connectedRefs;
                std::set<oa::oaOccBitNet*> connectedNets;
                
                OccGraph::getAllConnections(*refIter, connectedNets, 
                                            connectedRefs, true, false, true, 
                                            true, false, false);
                for(std::set<OccRef>::iterator connectedIter = connectedRefs.begin();
                    connectedIter != connectedRefs.end(); ++connectedIter) 
                {
                    OccRef connectedRef = *connectedIter;
                    assert(OccGraph::getNodeType(connectedRef) == oagAi::Node::TERMINAL);
                    clause.push_back(-refNOMap[connectedRef]);
                    clause.push_back(refNOMap[*refIter]);
                    myCell->clauses.push_back(clause);
                    clause.clear();
                    clause.push_back(refNOMap[connectedRef]);
                    clause.push_back(-refNOMap[*refIter]);
                    myCell->clauses.push_back(clause);
                    clause.clear();
                }    
            }
        }
        else
        {
            if (OccGraph::isInverted(ref2))
            {
                // If the driver is inverted output, then add an INV
                // between the driver and the output.
                nref2 = OccGraph::getNonInverted(ref2);
                refNOMap[ref2]= inputNO + i + 1;
                refNOMap[nref2]= extraCount++;
                refQueue.push_back(nref2);
                clause.push_back(refNOMap[ref2]);
                clause.push_back(refNOMap[nref2]);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-refNOMap[ref2]);
                clause.push_back(-refNOMap[nref2]);
                myCell->clauses.push_back(clause);
                clause.clear();
            }
            else
            {
                refNOMap[ref2]= inputNO + i + 1;
                refQueue.push_back(ref2);
            }
        }

        // Output is terminal, need to use its "driver"
        while (refQueue.empty() == false)
        {
            curr= refQueue.front();
            #ifdef DEBUG
            cout<<"Popped from queue: "<<refNOMap[curr];
            cout<<", type is "<<OccGraph::getNodeType(curr)<<endl;
            #endif
            refQueue.pop_front();

            left = OccGraph::getAndLeft(curr);
            right = OccGraph::getAndRight(curr);
            nleft = OccGraph::getNonInverted(left);
            nright = OccGraph::getNonInverted(right);
            // Check if left/right number assigned. If so, use that number.
            // If not, add those nodes into queue (unless it is CONST)
            if (OccGraph::getNodeType(nleft) != oagAi::Node::CONSTANT0)
            {
                if (refNOMap.find(nleft) == refNOMap.end())
                {
                    refNOMap[nleft]= extraCount++;
                    refQueue.push_back(nleft);
                    #ifdef DEBUG
                    cout<<"Add left to queue: "<<refNOMap[nleft];
                    cout<<", type is "<<OccGraph::getNodeType(nleft)<<endl;
                    #endif
                }
            }
            else
            {
                ;
                #ifdef DEBUG
                cout<<"Left is CONST"<<endl;
                #endif
            }
            if (OccGraph::getNodeType(nright) != oagAi::Node::CONSTANT0)
            {
                if (refNOMap.find(nright) == refNOMap.end())
                {
                    refNOMap[nright]= extraCount++;
                    refQueue.push_back(nright);
                    #ifdef DEBUG
                    cout<<"Add right to queue: "<<refNOMap[nright];
                    cout<<", type is "<<OccGraph::getNodeType(nright)<<endl;
                    #endif
                }
            }
            else
            {
                ;
                #ifdef DEBUG
                cout<<"Right is CONST"<<endl;
                #endif
            }
            #ifdef DEBUG
            cout<<"Left is: "<<refNOMap[nleft]<<", right is: "<<refNOMap[nright];
            cout<<", ltype is "<<OccGraph::getNodeType(nleft)<<", rtype is "<<OccGraph::getNodeType(nright)<<endl;
            #endif
            // Check if anything is 0, itself becomes 0
            if ((OccGraph::getNodeType(nleft) == oagAi::Node::CONSTANT0 && 
                 OccGraph::isInverted(left) == false) || 
                 (OccGraph::getNodeType(nright) == oagAi::Node::CONSTANT0 &&
                 OccGraph::isInverted(right) == false))
            {
                // Itself must be 0
                clause.push_back(-refNOMap[curr]);
                myCell->clauses.push_back(clause);
                clause.clear();
            }
            else if ((OccGraph::getNodeType(nleft) == oagAi::Node::CONSTANT0 && 
                 OccGraph::isInverted(left) == true) || 
                 (OccGraph::getNodeType(nright) == oagAi::Node::CONSTANT0 &&
                 OccGraph::isInverted(right) == true))
            {
                // Itself must be 1
                clause.push_back(refNOMap[curr]);
                myCell->clauses.push_back(clause);
                clause.clear();
            }
            else if ((OccGraph::getNodeType(nleft) == oagAi::Node::CONSTANT0 &&
                      OccGraph::isInverted(left) == true))
            {
                // Same as right, -Z A 0 A -Z 0
                clause.push_back(-refNOMap[curr]);
                clause.push_back(refNOMap[right]);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(refNOMap[curr]);
                clause.push_back(-refNOMap[right]);
                myCell->clauses.push_back(clause);
                clause.clear();
            }
            else if ((OccGraph::getNodeType(nright) == oagAi::Node::CONSTANT0 &&
                      OccGraph::isInverted(right) == true))
            {
                // Same as left, -Z A 0 A -Z 0
                clause.push_back(-refNOMap[curr]);
                clause.push_back(refNOMap[left]);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(refNOMap[curr]);
                clause.push_back(-refNOMap[left]);
                myCell->clauses.push_back(clause);
                clause.clear();
            }
            else 
            {
                int sign1, sign2;
                // -Z A 0 -Z B 0 -A -B Z 0
                if (OccGraph::isInverted(left) == false &&
                     OccGraph::isInverted(right) == false)
                {
                    // None is negated
                    sign1= 1; 
                    sign2= 1;
                }
                else if (OccGraph::isInverted(left) == false &&
                         OccGraph::isInverted(right) == true)
                {
                    // right is negated
                    sign1= 1; 
                    sign2= -1;
                }
                else if (OccGraph::isInverted(left) == true &&
                         OccGraph::isInverted(right) == false)
                {
                    // left is negated
                    sign1= -1; 
                    sign2= 1;
                }
                else  // both are negateda
                {
                    sign1= -1; 
                    sign2= -1;
                }

                clause.push_back(-refNOMap[curr]);
                clause.push_back(refNOMap[nleft] * sign1);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-refNOMap[curr]);
                clause.push_back(refNOMap[nright] * sign2);
                myCell->clauses.push_back(clause);
                clause.clear();
                clause.push_back(-refNOMap[nleft] * sign1);
                clause.push_back(-refNOMap[nright] * sign2);
                clause.push_back(refNOMap[curr]);
                myCell->clauses.push_back(clause);
                clause.clear();
            }
        }
        refIter++;
    }
    myCell->extraNO= extraCount - inputNO - outputNO - 1;
    return true;
}

// *****************************************************************************
// getCellNames()
//
/// \brief Get all cell names in this library.
//
// *****************************************************************************
void LibMan::getCellNames(std::vector<std::string>& strvec)
{
    strvec= cellNames;
}

// *****************************************************************************
// libMan()
//
/// \brief Constructor
//
// *****************************************************************************
LibMan::LibMan()
{
    useHardCodeCNF= false;
}

} // End of namespace
// vim: ci et sw=4 sts=4
