
/* 
Author: Kai-hui Chang <changkh@eecs.umich.edu>
*/

///
/// \file oagResynInternalCkt.h
/// Internal data structures.

#if !defined(oagResynInternalCkt_P)
#define oagResynInternalCkt_P

#include "oaDesignDB.h"
#include "oagResynLibMan.h"
#include "oagResyn.h"
#include <vector>
#include <ext/hash_map>
#include <map>

namespace oagResyn {

/// Hash for hash_table oaNet
struct hash1 {
//  size_t operator()(oaNet* __x) const { return static_cast<size_t>(__x); }
  size_t operator()(oa::oaNet* __x) const { return (size_t)__x; }
};

/// Hash for hash_table oaInst
struct hash2 {
//  size_t operator()(oaInst* __x) const { return static_cast<size_t>(__x); }
  size_t operator()(oa::oaInst* __x) const { return (size_t)__x; }
};

/// Get name from an OA object.
template <typename T> std::string getNameFromOA(T i)
{
    oa::oaNativeNS ns;
    oa::oaString name;
    i->getName(ns,name);
    return std::string(static_cast<const char*>(name) );
}

template <typename T> const std::string getConstNameFromOA(T i)
{
    oa::oaNativeNS ns;
    oa::oaString name;
    i->getName(ns,name);
    return (const std::string)std::string(static_cast<const char*>(name) );
}

/// For hash_map<oaNet*>  
class eqNetMap
{
public:
    bool operator()(const oa::oaNet* n1, const oa::oaNet* n2) const
    {
        return n1 == n2;
    }
};

/// For hash_map<oaInst*>  
class eqInstMap
{
public:
    bool operator()(const oa::oaInst* n1, const oa::oaInst* n2) const
    {
        return n1 == n2;
    }
};

/*
template <> std::string getNameFromOA<oa::oaDesign*>(oa::oaDesign* i);

template <> std::string getNameFromOA<oa::oaDesign*>(oa::oaDesign* i)
{
    oa::oaNativeNS ns;
    oa::oaString name;
    i->getCellName(ns,name);
    return std::string(static_cast<const char*>(name) );
}
*/


/// Additional data structure for oaNet.
class simWire
{
public:
    simWire() {isPI= false; isPO= false; isRegOut= false; regQWire= NULL; marked= false; isRegOutQN= false; isRegD= false; };
    /// Simulation value of the net.
    SimulationVector simVec, nextSimVec; // nextSimVec is for event-driven simulation
    /// Signature of the net.
    std::vector<SimulationVector> signature;
    /// If true, the net is a primary input.
    bool isPI;
    /// If true, the net is a primary output.
    bool isPO;
    /// Is register's out (QN or Q)
    bool isRegOut;
    bool isRegOutQN; ///< Is register's out and is QN
    bool isRegD; ///< Is register's D
    bool marked; ///< Misc. usage, please clear after use.
    oa::oaNet* regQWire; ///< If this wire is QN, it points to the Q net
    oa::oaNet* origWire; ///< Points to oaNet.
};

/// Additional data structure for oaInst.
class simInst 
{
public:
    simInst() {visited= false; isStateReg= false; marked= false; };
    bool visited;
    /// Points to cell data structure.
    libCell* cell;
    /// If true, is a state register.
    bool isStateReg;
    /// Points to oaInst.
    oa::oaInst* origInst;
    bool marked; ///< Temp flag, please clear after use
    std::vector<int> inputOrder; ///< Input order of ports
    std::vector<simWire*> inputSimWires; ///< Points to SimWire according to input order
    std::vector<int> outputOrder; ///< Output order of ports
    std::vector<simWire*> outputSimWires; ///< Points to SimWire according to output order
};

/// Builds additional data structures for the netlist.
class internalCkt {
friend class Sim;
friend class equiCheck;

public:
    internalCkt(oa::oaDesign *design);
    ~internalCkt();
    void setup();
    void cleanUp();

    void analyzeCircuit();
    void buildInternalCircuit();
    void printInOutState();
    void levelizeInternalCircuit();
    void getStates(std::vector<oa::oaInst*>& result);
    void getStateNets(std::vector<oa::oaNet*>& result);
    void getStateInNets(std::vector<oa::oaNet*>& result);
    void getInputs(std::vector<oa::oaNet*>& result);
    void getOutputs(std::vector<oa::oaNet*>& result);
    void getClockNets(std::vector<oa::oaNet*>& result);
    void getAllNets(std::vector<oa::oaNet*>& result);
    void getInputConeInsts(std::vector<oa::oaNet*>& nets, 
         std::vector<oa::oaInst*>& insts, std::vector<oa::oaNet*>& inputNets);
    void setVector(oa::oaNet* net, SimulationVector vec, int isw);
    simInst* instGetSimInst(oa::oaInst* inst);
    simWire* netGetSimWire(oa::oaNet* net);

protected:
    oa::oaDesign *design;
    oa::oaBlock *block;
    void setVectorEquiNet(oa::oaNet* net, SimulationVector vec, int isw);
    void instGetModName(oa::oaInst* inInst, std::string& modName);
    std::vector<oa::oaInst*> stateInsts;
    std::vector<oa::oaNet*> inputNets, outputNets, stateNets, stateInNets;
    std::vector<oa::oaInst*> levelizedInst;
    std::set<oa::oaNet*> clockNets; 
    std::set<oa::oaNet*> allBitNets;
    std::vector<oa::oaNet*> allBitNetsVec;
    oa::oaNet *tie0Wire, *tie1Wire;
    __gnu_cxx::hash_map<oa::oaNet*, simWire*, hash1, eqNetMap > netSimWireMap;
    __gnu_cxx::hash_map<oa::oaInst*, simInst*, hash2, eqInstMap > instSimInstMap;
//    map<oaNet*, simWire* > netSimWireMap;
//    map<oaInst*, simInst* > instSimInstMap;
};

/// Set vector value for a net.
inline void internalCkt::setVector(oa::oaNet* net, 
                                   SimulationVector vec, 
                                   int isw= 0)
{
    simWire* myWire;
    oa::oaBitNet* bitNet= (oa::oaBitNet*)net;

    if (net == tie1Wire || net == tie0Wire)
        return;
    myWire= netGetSimWire(net);
    if (isw == 0)
        myWire->simVec= vec;
    else
        myWire->nextSimVec= vec;
    if (bitNet->getEquivalentNets().getCount() > 0)
        setVectorEquiNet(net, vec, isw);
}

}

#endif

// vim: ci et sw=4 sts=4
