#include "oaDesignDB.h"
#include "oagAiGraph.h"
#include "oagUtilOptionParser.h"
#include "oagRedun.h"
#include "oagFunc.h"
#include <iostream>

using namespace std;
using namespace oa;
using namespace oagAi;
using namespace oagRedun;

namespace {
oaNativeNS ns;
oaString   str;

template <class T>
const char *
getName(T* inst) {
    inst->getName(ns, str);
    return str;
}

template <>
const char *
getName(oaOccInstTerm* it) {
    oaString s1;
    it->getInst()->getName(ns, str);
    it->getTerm()->getName(ns, s1);
    str += ':';
    str += s1;
    return str;
}
template <>
const char *
getName(oaInstTerm* it) {
    oaString s1;
    it->getInst()->getName(ns, str);
    it->getTerm()->getName(ns, s1);
    str += ':';
    str += s1;
    return str;
}
}

namespace oagRedun {

class TimingStuckAtRedunCB : public StuckAtRedunCB {
public:
    TimingStuckAtRedunCB(oa::oaNet *t0, oa::oaNet *t1) : StuckAtRedunCB(t0, t1) { }
    virtual bool pinSort(oa::oaInstTerm *a, oa::oaInstTerm *b) {
        return false; 
    }
};

class NoopStuckAtRedunCB : public StuckAtRedunCB {
    const bool verbose;
public:
    NoopStuckAtRedunCB(oa::oaNet *, oa::oaNet *, bool v = true) 
        : StuckAtRedunCB(0, 0), verbose(v) { }
    virtual void removeRedundancy(oa::oaInstTerm *t, unsigned sav) { 
        if (verbose)
            printf("Pin %s is untestable for stuck-at-%d\n", getName(t), sav);
    }
};

class VStuckAtRedunCB : public StuckAtRedunCB {
    const bool verbose;
public:
    VStuckAtRedunCB(oa::oaNet *t0, oa::oaNet *t1, bool v = true) 
        : StuckAtRedunCB(t0, t1), verbose(v) { }
    /// This function gets called to remove redundancies
    virtual void removeRedundancy(oa::oaInstTerm *t, unsigned stuckAtVal) {
        if (verbose)
            printf("Removing stuck-at-%d redundancy on %s\n", stuckAtVal, getName(t));
	if (stuckAtVal == 0) t->addToNet(tie0);
	if (stuckAtVal == 1) t->addToNet(tie1);
    }
};

}

double
findArea(oaBlock *block) {
    double totalArea = 0.0;

    oaIter<oaInst> instIter(block->getInsts());
    while (oaInst *i = instIter.getNext()) {
        oaDesign *master = i->getMaster();
        if (!master) {
            oaString nameString;
            i->getName(oaNativeNS(), nameString);
            std::cerr << "Warning: black box found, assuming zero area:" << nameString << std::endl;
        } else {
            oaBlock *masterBlock = master->getTopBlock();
            assert(masterBlock);
            oaPRBoundary *bound = oaPRBoundary::find(masterBlock);
            assert(bound);
            oaBox bbox;
            bound->getBBox(bbox);
            oaUInt8 cellArea = static_cast<oaUInt8>(bbox.getWidth()) *
                static_cast<oaUInt8>(bbox.getHeight());
            oaTech *tech = master->getTech();
            assert(tech);
            totalArea +=
                tech->dbuToUUArea(oaViewType::get(oacMaskLayout), cellArea);
        }
    }
    return totalArea;
}


void
doRR(oaDesign *design, bool testonly, bool verbose, unsigned seed)
{
    oaName tie0Name(ns, "tie0");
    oaName tie1Name(ns, "tie1");

    StuckAtRedunCB *cb;
    if (testonly) {
        cb = new NoopStuckAtRedunCB(0, 0, verbose);
    } else {
        oaBlock *block = design->getTopBlock();
        oaNet *tie0 = oaNet::find(block, tie0Name);
        oaNet *tie1 = oaNet::find(block, tie1Name);
        if (!tie0) {
            tie0 = oaNet::create(block, tie0Name, oacTieLoSigType);
        }
        if (!tie1) {
            tie1 = oaNet::create(block, tie1Name, oacTieHiSigType);
        }

        cb = new VStuckAtRedunCB(tie0, tie1, verbose);
    }
    removeStuckAtRedundancies(design, cb, seed);
    removeUnusedLogic(design);
    delete cb;
}

int
main(int argc, const char *argv[])
{
    using namespace oagUtil;

    OptionParser      options("Run redundancy removal on a design.");
    Option *libOpt  = options.add("lib", "Input library name", true, "LIB");
    Option *cellOpt = options.add("cell", "Input cell name", true, "CELL");
    Option *viewOpt = options.add("view", "Input view name (default: netlist)", false, "VIEW");
    Option *rrOpt = options.add("sweepOnly", "Only remove unused logic", false);
    Option *testOpt = options.add("testOnly", "ATPG only - do not remove redundancies", false);
    Option *saveAsOpt = options.add("saveAs", "Output cell name", false, "CELL");
    Option *seedOpt = options.add("randomSeed", "Seed for random number generator", false, "NUM");
    Option *verbOpt = options.add("verbose", "Print out redundancies", false);

    if (!options.parse(argc, argv)) {
        cerr << options.getMessage();
        exit(EXIT_FAILURE);
    }

    try {
        oaDesignInit();
	oaLibDefList::openLibs();

        oagFunc::initialize();

        unsigned seed;
        if (seedOpt->isGiven()) {
            seed = atoi(seedOpt->getValue());
            cout << "Random seed: " << seed << endl;
        } else {
            seed = 0;
        }

        const oaVerilogNS  ns;
        const oaScalarName libName(ns, libOpt->getValue());
        const oaScalarName cellName(ns, cellOpt->getValue());
        const oaScalarName viewName(ns, viewOpt->isGiven() ? viewOpt->getValue()
                                                           : "netlist");

        oaLib *lib = oaLib::find(libName);
        if (!lib) {
            cerr << "Error: Library not found: " << libOpt->getValue() << endl;
            exit(EXIT_FAILURE);
        }

        oaDesign *design = oaDesign::open(libName, cellName, viewName, 'r');
        double startarea = findArea(design->getTopBlock());
        printf("Starting cell area: %g\n", startarea);
        removeUnusedLogic(design);
        double sweeparea = findArea(design->getTopBlock());
        printf("Swept cell area: %g (%g%%)\n", sweeparea, (startarea-sweeparea)/startarea*100);
        if (!rrOpt->isGiven()) {
            doRR(design, testOpt->isGiven(), verbOpt->isGiven(), seed);
            double rrarea = findArea(design->getTopBlock());
            printf("RR'd cell area: %g (%g%%)\n", rrarea, (sweeparea-rrarea)/sweeparea*100);
        }
        if (saveAsOpt->isGiven())
            design->saveAs(libName, oaScalarName(ns, saveAsOpt->getValue()), viewName);
    } catch(oaException &e) {
        cerr << "OA Exception: " << e.getMsg() << endl;
        exit(EXIT_FAILURE);
    } catch (std::exception &e) {
        cerr << "Exception: " << e.what() << endl;
        exit(EXIT_FAILURE);
    }
    
    return 0;
}

// vim:et:
