/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-08-18: ChangeLog started
*/

#include "oagMapperUtils.h"
#include "oagFunc.h"
#include "oagFuncModGraph.h"
#include "oagFuncManager.h"
#include "oagFuncSimMod.h"
#include "oagAiNode.h"
#include <map>
#include <sstream>
#include <algorithm>

// #define DEBUG

#include "oagMapperDebug.h"

using namespace std;
using namespace oagFunc;

namespace oagMapper {

// *****************************************************************************
// Utils()
//
/// \brief Constructor.
///
// *****************************************************************************
Utils::Utils() {
    andGate = NULL;
    andInput[0] = andInput[1] = andOutput = NULL;
    
    notGate = NULL;
    notInput = notOutput = NULL;
    
    seqGate = NULL;
    seqInput = seqOutput = NULL;
    seqReset = seqPreset = seqClock = NULL;
}


// *****************************************************************************
// identifyNot()
//
/// \brief Identifies the inverter gate in a library.
///
/// \param lib
/// \return an inverter gate, or NULL if none
// *****************************************************************************
oa::oaDesign *      
Utils::identifyNot(const std::list<oa::oaDesign*> & lib) {
  notGate =  NULL;
  notOutput = notInput = NULL;
  const unsigned int len = 1;

  for(list<oa::oaDesign*>::const_iterator it = lib.begin();
      it != lib.end(); it++) {
    oa::oaDesign *design = *it;
    oa::oaModule *module = design->getTopModule();
    // must have functional description attached to module
    if (!module) continue;
    
    // get inputs.  reject if |inputs| != 1
    list<ModRef> inputs;
    ModGraph::getInputs(module, inputs);
    if (inputs.size() != len) continue;
    
    // get output.  reject if |outputs| != 1
    list<ModRef> outputs;
    ModGraph::getOutputs(module, outputs);
    if (outputs.size() != 1) continue;
    
    // create simulation engine
    SimMod sim(design);

    // prepare input vector
    oagFunc::SimMod::SimVec inputVec[len], outputVec;
    bzero(inputVec, sizeof(oagFunc::SimMod::SimVec)*len);
    int v = 0;
    for(; v<(1<<len); v++) {
      for(unsigned int i=0; i<len; i++) {
        inputVec[i] |= ((v >> i) & 0x1) << v;
      }
    }
      
    // simulate a word
    list<ModRef>::iterator inputIter = inputs.begin();
    for(unsigned int i=0; i<len; i++, inputIter++) sim.setVector(*inputIter, inputVec[i]);
    sim.runIncremental();
    sim.getVector(outputs.front(), outputVec);
    outputVec &= (0x1 << (0x1 << len))-1; // mask valid bits

    // DEBUG
    /*
    oa::oaString modString;
    module->getName(oa::oaVerilogNS(), modString);
    cout << modString << " = " << outputVec << endl;
    */

    // does this match an inverter?
    if (outputVec == 0x0001) {
      return (notGate = design);
    }
  }

  // could not find a not gate
  return NULL;
}


// *****************************************************************************
// identifyNotTerminals()
//
/// \brief Identifies the terminals of an inverter gate.
///
/// The inverter gate must have been previous identified (either automatically or explicitly).
///
// *****************************************************************************
void   
Utils::identifyNotTerminals() {
  assert(notGate);

  // remember terminals
  oa::oaModTerm *term;
  oa::oaIter<oa::oaModTerm> termIter(notGate->getTopModule()->getTerms(oacTermIterSingleBit));
  while((term = termIter.getNext())) {
    if (term->getTermType() == oa::oacInputTermType) {
      notInput = term;
    } else if (term->getTermType() == oa::oacOutputTermType) {
      notOutput = term;
    }
  }
}


// *****************************************************************************
// identifyAnd()
//
/// \brief Identifies the AND2 gate in a library.
///
/// \param lib
/// \return an AND2 gate, or NULL if none
// *****************************************************************************
oa::oaDesign *      
Utils::identifyAnd(const std::list<oa::oaDesign*> & lib) {
  andGate =  NULL;
  andOutput = andInput[0] = andInput[1] = NULL;
  const unsigned int len = 2;

  for(list<oa::oaDesign*>::const_iterator it = lib.begin();
      it != lib.end(); it++) {
    oa::oaDesign *design = *it;
    oa::oaModule *module = design->getTopModule();
    // must have functional description attached to module
    if (!module) continue;
    
    // get inputs.  reject if |inputs| != 2
    list<ModRef> inputs;
    ModGraph::getInputs(module, inputs);
    if (inputs.size() != len) continue;
    
    // get output.  reject if |outputs| != 1
    list<ModRef> outputs;
    ModGraph::getOutputs(module, outputs);
    if (outputs.size() != 1) continue;
    
    // create simulation engine
    SimMod sim(design);

    // prepare input vector
    oagFunc::SimMod::SimVec inputVec[len], outputVec;
    bzero(inputVec, sizeof(oagFunc::SimMod::SimVec)*len);
    int v = 0;
    for(; v<(1<<len); v++) {
      for(unsigned int i=0; i<len; i++) {
        inputVec[i] |= ((v >> i) & 0x1) << v;
      }
    }
      
    // simulate a word
    list<ModRef>::iterator inputIter = inputs.begin();
    for(unsigned int i=0; i<len; i++, inputIter++) sim.setVector(*inputIter, inputVec[i]);
    sim.runIncremental();
    sim.getVector(outputs.front(), outputVec);
    outputVec &= (0x1 << (0x1 << len))-1; // mask valid bits

    // DEBUG
    /*
    oa::oaString modString;
    module->getName(oa::oaVerilogNS(), modString);
    cout << modString << " = " << outputVec << endl;
    */

    // does this match an inverter?
    if (outputVec == 0x8) {
      return (andGate = design);
    }
  }

  // could not find a not gate
  return NULL;
}


// *****************************************************************************
// identifyAndTerminals()
//
/// \brief Identifies the terminals of an inverter gate.
///
/// The inverter gate must have been previous identified (either automatically or explicitly).
//
// *****************************************************************************
void   
Utils::identifyAndTerminals() {
  assert(andGate);

  // remember terminals
  oa::oaModTerm *term;
  oa::oaIter<oa::oaModTerm> termIter(andGate->getTopModule()->getTerms(oacTermIterSingleBit));
  while((term = termIter.getNext())) {
    if (term->getTermType() == oa::oacInputTermType && !andInput[0]) {
      andInput[0] = term;
    } else if (term->getTermType() == oa::oacInputTermType && andInput[0]) {
      andInput[1] = term;
    } else if (term->getTermType() == oa::oacOutputTermType) {
      andOutput = term;
    }
  }
}


// *****************************************************************************
// identifySeqTerminalsByName()
//
/// \brief Identifies the function of each terminal of a SEQ gate.
///
/// The gate terminals are identified by name.  Every terminal described here
/// should be a scalar terminal.  
///
/// The following MUST be present:
/// \li "D" : data / next state input
/// \li "Q" : output
/// \li "CLK", "CK", "C" : clock
///
/// One of the following may also be present:
/// \li "RN" : asynchronous reset (negative edge triggered)
/// \li "R" : asynchronous reset (positive edge triggered)
///
/// One of the following may also be present:
/// \li "SN" : asynchronous preset (negative edge triggered)
/// \li "S" : asynchronous preset (positive edge triggered)
//
// *****************************************************************************
void
Utils::identifySeqTerminalsByName() {
    assert(seqGate);
    oa::oaModule *module = seqGate->getTopModule();
    assert(module);
    
    const oa::oaVerilogNS verilogNS;
    seqInput = seqOutput = seqClock = seqReset = seqPreset = NULL;    
    
    // input pin
    oa::oaModScalarTerm *term = oa::oaModScalarTerm::find(module,
        oa::oaScalarName(verilogNS, "D"));
    if (!term) {
        cerr << "ERROR: Could not find \"D\" terminal on sequential gate" << endl;
        QUIT_ON_ERROR;
    }
    seqInput = term;
    
    // output pin
    term = oa::oaModScalarTerm::find(module,
        oa::oaScalarName(verilogNS, "Q"));
    if (!term) {
        cerr << "ERROR: Could not find \"Q\" terminal on sequential gate" << endl;
        QUIT_ON_ERROR;
    }
    seqOutput = term;
    
    // clock pin
    term = oa::oaModScalarTerm::find(module,
        oa::oaScalarName(verilogNS, "CK"));
    if (!term) {
        term = oa::oaModScalarTerm::find(module,
            oa::oaScalarName(verilogNS, "CLK"));   
    }
    if (!term) {
        term = oa::oaModScalarTerm::find(module,
            oa::oaScalarName(verilogNS, "C"));   
    }
    if (!term) {
        cerr << "ERROR: Could not find clock terminal on sequential gate" << endl;
        QUIT_ON_ERROR;
    }
    seqClock = term;
    seqClockTrigger = POSEDGE; // assume posedge trigger
    
    // reset pin
    term = oa::oaModScalarTerm::find(module,
        oa::oaScalarName(verilogNS, "RN"));
    if (term) {
        cout << "NOTE: Found a negedge triggered reset pin on sequential gate" << endl;
        seqReset = term;
        seqResetTrigger = NEGEDGE;        
    } else {
        term = oa::oaModScalarTerm::find(module,
            oa::oaScalarName(verilogNS, "R"));
        if (term) {
            cout << "NOTE: Found a posedge triggered reset pin on sequential gate" << endl;
            seqReset = term;
            seqResetTrigger = POSEDGE;
        } else {
          cout << "WARNING: Did not find a reset pin on sequential gate" << endl;
        }
    }

    // preset pin
    term = oa::oaModScalarTerm::find(module,
        oa::oaScalarName(verilogNS, "SN"));
    if (term) {
        cout << "NOTE: Found a negedge triggered preset pin on sequential gate" << endl;
        seqPreset = term;
        seqPresetTrigger = NEGEDGE;        
    } else {
        term = oa::oaModScalarTerm::find(module,
            oa::oaScalarName(verilogNS, "S"));
        if (term) {
            cout << "NOTE: Found a posedge triggered preset pin on sequential gate" << endl;
            seqPreset = term;
            seqPresetTrigger = POSEDGE;
        } else {
          // cout << "WARNING: Did not find a preset pin on sequential gate" << endl;
        }
    }
}


// *****************************************************************************
// removeAsyncResetsFromLogic()
//
/// \brief Removes sequential control signals from logic.
///
/// All sequential control signals are synthesized in a very general manner.
/// They are added as generic triggers on a sequential node, and any 
/// additional functional behavior is implemented in logic surrounding the sequential 
/// node.
///
/// Two common examples of this are asynchronous resets and presets.
/// These are triggers, but also affect the function being latched (i.e.
/// the '0' function will be latched on an asynchronous reset and
/// the '1' function will be latched on an asynchronous preset).
///
/// In library gates that have asynchronous resets and presets, these inputs
/// will both trigger the gate and set its state to the appropriate value.
/// There is no need to keep the logic on the input of the sequential element
/// that had previously driven the next state input to the appropriate reset
/// values.  In fact, this logic is entirely useless and may confuse some
/// equivalence checkers.
///
/// This function removes the asynchronous resets and presets from the
/// input logic of sequential elements, replacing them with a constant that
/// reflects the unasserted value of the signal.
///
/// \param module
// *****************************************************************************
void
Utils::removeAsyncResetsFromLogic(oa::oaModule *module) {
  assert(module);

  // is this design already structural?
  if (!Manager::hasManager(module->getDesign())) {
    return;
  }

  // note that the resubstitute command does not affect anything in
  // the sequentialData structures.  it will only substitute
  // the values in the input logic

  // resubstitute 0 for all posedge resets
  for(set<oa::oaModBitNet *>::iterator it = posAsyncResets.begin();
      it != posAsyncResets.end(); it++) {
    ModRef term = ModGraph::getNetToAiConnection(*it);
    assert(!ModGraph::isNull(term));
    ModGraph::resubstitute(term, ModGraph::constantZero(term.module));
  }
 
  // resubstitute 1 for all negedge resets
  for(set<oa::oaModBitNet *>::iterator it = negAsyncResets.begin();
      it != negAsyncResets.end(); it++) {
    ModRef term = ModGraph::getNetToAiConnection(*it);
    assert(!ModGraph::isNull(term));
    ModGraph::resubstitute(term, ModGraph::constantOne(term.module));    
  }

}


// *****************************************************************************
// identifyControls()
//
/// \brief Attempts to identify the control signals of all sequential gates in a module.
///
/// \param module
//
// *****************************************************************************
void
Utils::identifyControls(oa::oaModule *module) {
  list<ModRef> seq;
  ModGraph::getLocalStates(module, seq);    
  for(list<ModRef>::iterator it = seq.begin(); it!=seq.end(); it++) {
    identifyControls(*it);
  }
}


// *****************************************************************************
// identifyControls()
//
/// \brief Attempts to identify the control signals of a sequential node as resets, clocks, etc.
///
/// \param seq
//
// *****************************************************************************
void
Utils::identifyControls(oagFunc::ModRef seq) {

  oagAi::Node::SequentialData *data = oagFunc::ModGraph::getSequentialData(seq);
  if (!data) {
    return;
  }
  
  oa::oaModule *module = seq.module;
  assert(module);

  // get the roots of the fan-in cone
  list<ModRef> roots;
  ModGraph::getFaninRoots(seq, roots);

  for(map<string, oagAi::Ref>::iterator sigIter = data->signals.begin();
      sigIter != data->signals.end(); sigIter++) {

    ModRef ref(sigIter->second, module);

    // only can deal with TERMINALS
    if (!ModGraph::isTerminal(ref)) {
      continue;
    }

    // follow strings of terminals to driver
    while (true) {
      ModRef driver = ModGraph::getTerminalDriver(ref);
      if (!ModGraph::isTerminal(driver)) {
        break;
      }
      ref = driver;
    }

    // get the connected net
    oa::oaModNet *net = ModGraph::getNetToAiConnection(ref);
    if (!net) {
      continue;
    }
    oa::oaModBitNet *bitNet = oagFunc::toBitNet(net);

    bool inFanin = false;
    inFanin |= find(roots.begin(), roots.end(), ref) != roots.end();
    inFanin |= find(roots.begin(), roots.end(), ModGraph::notOf(ref)) != roots.end();
    
    bool posedge = (sigIter->first.find("posedge") != sigIter->first.npos);
    bool clocked_on = (sigIter->first == "clocked_on");
    bool negedge = (sigIter->first.find("negedge") != sigIter->first.npos);
    bool clear = (sigIter->first == "clear");

#if defined(DEBUG)
    oa::oaString netString;
    bitNet->getName(oa::oaVerilogNS(), netString);
#endif

    if (inFanin && posedge || clear) {
      DEBUG_PRINTLN(netString << " = pos-triggered reset");
      posAsyncResets.insert(bitNet);
    } else if (inFanin && negedge) {
      DEBUG_PRINTLN(netString << " = neg-triggered reset");
      negAsyncResets.insert(bitNet);
    } else if (!inFanin && posedge || clocked_on) {
      DEBUG_PRINTLN(netString << " = pos-triggered clock");
      posClocks.insert(bitNet);
    } else if (!inFanin && negedge) {
      DEBUG_PRINTLN(netString << " = neg-triggered clock");
      negClocks.insert(bitNet);
    }
  }
}


// *****************************************************************************
// connectControl()
//
/// \brief Connects a control signal to an instance.
///
/// \param term the term of the instance to be connected
/// \param inst the instance to be connected
/// \param net the net to be connected
/// \param inverted true, if the signal should be inverted
//
// *****************************************************************************
void
Utils::connectControl(oa::oaModTerm *term, oa::oaModInst *inst,
                      oa::oaModBitNet *net, bool inverted) {
  assert(term);
  assert(inst);
  assert(net);

  oa::oaModule *module = net->getModule();
  assert(module);

#if defined(DEBUG)
  oa::oaString termString, netString, instString;
  term->getName(oa::oaVerilogNS(), termString);
  net->getName(oa::oaVerilogNS(), netString);
  inst->getName(oa::oaVerilogNS(), instString);
  DEBUG_PRINTLN("Connecting " << termString << " of " << instString << " -> " << netString);
#endif

  if (oa::oaModInstTerm::find(inst, term)) {
    DEBUG_PRINTLN("Aborted.  Term is already connected");
    return;
  }

  if (inverted) {
    assert(notGate);
    // if not, insert an inverted on the input   
    oa::oaModScalarInst *notInst = oa::oaModScalarInst::create(module, notGate);
    oa::oaModInstTerm::create(net, notInst, notInput);
    oa::oaModScalarNet *localInvertedClockNet = oa::oaModScalarNet::create(module);
    oa::oaModInstTerm::create(localInvertedClockNet, notInst, notOutput);
    oa::oaModInstTerm::create(localInvertedClockNet, inst, term);
  } else {
    oa::oaModInstTerm::create(net, inst, term);
  }
  
}


// *****************************************************************************
// connectSeqControls()
//
/// \brief Connects the sequential controls, such as clocks, resets, etc.
///
/// The SEQUENTIAL node must be of BLACK_BOX abstraction type.
///
/// For a clock or reset to be connected, it must (1) have been previously
/// identified as such using the appropriate methods of this class and (2) be
/// present in the sequential data of the individual node (3) be either
/// a direct or inverted reference to a TERMINAL node.
///
/// Both clocks and resets should black box signals with labels of the form 
/// "trigger_X_posedge" or "trigger_X_negedge" (generated from behavioral
/// verilog synthesis), where X is a digit between
/// 0 and 9.  However, only one type of each signal can be attached (the 
/// one with the lowest number).
///
/// Clocks can also be black box signals with the label "clocked_on", and
/// resets from black box signals labeled "clear".  These are generated
/// from Liberty timing library synthesis.
///
/// Presets are currently unsupported and will be tied to the inactive value.
///
/// \param ref the sequential node
/// \param inst the corresponding module instance
//
// *****************************************************************************
void
Utils::connectAllControls(oagFunc::ModRef ref, oa::oaModInst *inst) {
    assert(inst);

    oagAi::Node::SequentialData *data = oagFunc::ModGraph::getSequentialData(ref);   
    assert(data);

    // the node must be a BLACK_BOX abstraction level
    assert(data->abstractionLevel == oagAi::Node::SequentialData::BLACK_BOX);

    // scan through all posedge triggers 
    for(char p = '0'; p <= '9'; p++) {
      stringstream label;
      label << "trigger_" << p << "_posedge";

      if (data->signals.find(label.str()) != data->signals.end()) {
        DEBUG_PRINTLN("Trigger " << label.str() << " is present");

        // found a signal
        ModRef ref(data->signals[label.str()], inst->getModule());

        // if this isn't a terminal... not so straightforward
        if (!ModGraph::isTerminal(ref)) {
          continue;
        }

        // follow strings of terminals to driver
        while (true) {
          ModRef driver = ModGraph::getTerminalDriver(ref);
          if (!ModGraph::isTerminal(driver)) {
            break;
          }
          ref = driver;
        }

        oa::oaModNet *net = ModGraph::getNetToAiConnection(ref);
        if (!net) {
          continue;
        }
        oa::oaModBitNet *bitNet = toBitNet(net);
        if (seqClock && find(posClocks.begin(), posClocks.end(), bitNet) != posClocks.end()) {
          connectControl(seqClock, inst, bitNet, false);
        }
        if (seqClock && find(negClocks.begin(), negClocks.end(), bitNet) != negClocks.end()) {
          connectControl(seqClock, inst, bitNet, true);
        }
        if (seqReset && find(posAsyncResets.begin(), posAsyncResets.end(), bitNet) != posAsyncResets.end()) {
          connectControl(seqReset, inst, bitNet, false);
        }
        if (seqReset && find(negAsyncResets.begin(), negAsyncResets.end(), bitNet) != negAsyncResets.end()) {
          connectControl(seqReset, inst, bitNet, true);
        }
      }
    }

    // scan through all negedge triggers 
    for(char p = '0'; p <= '9'; p++) {
      stringstream label;
      label << "trigger_" << p << "_negedge";

      if (data->signals.find(label.str()) != data->signals.end()) {
        DEBUG_PRINTLN("Trigger " << label.str() << " is present");

        // found a signal
        ModRef ref(data->signals[label.str()], inst->getModule());

        // if this isn't a terminal... not so straightforward
        if (!ModGraph::isTerminal(ref)) {
          continue;
        }

        // follow strings of terminals to driver
        while (true) {
          ModRef driver = ModGraph::getTerminalDriver(ref);
          if (!ModGraph::isTerminal(driver)) {
            break;
          }
          ref = driver;
        }

        oa::oaModNet *net = ModGraph::getNetToAiConnection(ref);
        if (!net) {
          continue;
        }
        oa::oaModBitNet *bitNet = toBitNet(net);
        if (seqClock && find(posClocks.begin(), posClocks.end(), bitNet) != posClocks.end()) {
          connectControl(seqClock, inst, bitNet, true);
        }
        if (seqClock && find(negClocks.begin(), negClocks.end(), bitNet) != negClocks.end()) {
          connectControl(seqClock, inst, bitNet, false);
        }
        if (seqReset && find(posAsyncResets.begin(), posAsyncResets.end(), bitNet) != posAsyncResets.end()) {
          connectControl(seqReset, inst, bitNet, true);
        }
        if (seqReset && find(negAsyncResets.begin(), negAsyncResets.end(), bitNet) != negAsyncResets.end()) {
          connectControl(seqReset, inst, bitNet, false);
        }
      }
    }

    // check for "clocked_on"
    if (data->signals.find("clocked_on") != data->signals.end()) {
      DEBUG_PRINTLN("Trigger clocked_on is present");
      
      // found a signal
      ModRef ref(data->signals["clocked_on"], inst->getModule());
      
      // if this isn't a terminal... not so straightforward
      if (ModGraph::isTerminal(ref)) {

        // follow strings of terminals to driver
        while (true) {
          ModRef driver = ModGraph::getTerminalDriver(ref);
          if (!ModGraph::isTerminal(driver)) {
            break;
          }
          ref = driver;
        }

        oa::oaModNet *net = ModGraph::getNetToAiConnection(ref);
        if (net) {
          oa::oaModBitNet *bitNet = toBitNet(net);
          if (seqClock && find(posClocks.begin(), posClocks.end(), bitNet) != posClocks.end()) {
            connectControl(seqClock, inst, bitNet, false);
          }
          if (seqClock && find(negClocks.begin(), negClocks.end(), bitNet) != negClocks.end()) {
            connectControl(seqClock, inst, bitNet, true);
          }
        }
      }
    }

    // check for "clear"
    if (data->signals.find("clear") != data->signals.end()) {
        DEBUG_PRINTLN("Trigger clear is present");
        
        // found a signal
        ModRef ref(data->signals["clear"], inst->getModule());

        // if this isn't a terminal... not so straightforward
        if (ModGraph::isTerminal(ref)) {

          // follow strings of terminals to driver
          while (true) {
            ModRef driver = ModGraph::getTerminalDriver(ref);
            if (!ModGraph::isTerminal(driver)) {
              break;
            }
            ref = driver;
          }

          oa::oaModNet *net = ModGraph::getNetToAiConnection(ref);
          if (net) {
            oa::oaModBitNet *bitNet = toBitNet(net);
            if (seqReset && find(posAsyncResets.begin(), posAsyncResets.end(), bitNet) != posAsyncResets.end()) {
              connectControl(seqReset, inst, bitNet, false);
            }
            if (seqReset && find(negAsyncResets.begin(), negAsyncResets.end(), bitNet) != negAsyncResets.end()) {
              connectControl(seqReset, inst, bitNet, true);
            }
          }
        }
    }
}


// *****************************************************************************
// mergeEquivalentNets()
//
/// \brief Replaces all groups of equivalent nets with a single net.
///
/// \param module
//
// *****************************************************************************
void
Utils::mergeEquivalentNets(oa::oaModule *module) {
    assert(module);

    // merge all nets
    bool changed = true;
    while(changed) {
        changed = false;

        // make all bus net bits explicit
        oa::oaModNet *net;
        oa::oaIter<oa::oaModNet> netIter2(module->getNets(oacNetIterNotImplicit));
        while((net = netIter2.getNext())) {
            if(!net->isImplicit() && net->getNumBits() > 1) {
                net->scalarize();
            }
        }

        oa::oaIter<oa::oaModNet> netIter(module->getNets(oacNetIterSingleBit));
        while(!changed && (net = netIter.getNext())) {
            oa::oaModBitNet *preferred = oagFunc::toBitNet(net)->getPreferredEquivalent();

            oa::oaModBitNet *equivNet;
            oa::oaIter<oa::oaModBitNet> equivIter(preferred->getEquivalentNets());      
            while((equivNet = equivIter.getNext())) {

                // move instTerms
                oa::oaModInstTerm *instTerm;
                oa::oaIter<oa::oaModInstTerm> instTermIter(equivNet->getInstTerms(oacInstTermIterAll));      
                while((instTerm = instTermIter.getNext())) {
                    if (instTerm->isImplicit()) {
                        instTerm->scalarize();
                    }
                    instTerm->addToNet(preferred);
                }
                
                // move terms
                oa::oaModTerm *term;
                oa::oaIter<oa::oaModTerm> termIter(equivNet->getTerms(oacTermIterAll));
                while((term = termIter.getNext())) {
                    if (term->isImplicit()) {
                        term->scalarize();
                    }
                    term->moveToNet(preferred);
                }
               
                // resubstitute graph
                oagFunc::ModRef target = oagFunc::ModGraph::getNetToAiConnection(equivNet);
                if (!oagFunc::ModGraph::isNull(target)) {
                    oagFunc::ModRef replacement = oagFunc::ModGraph::prepareNetToAiConnection(preferred);
                    oagFunc::ModGraph::resubstitute(target, replacement);
                    oagFunc::ModGraph::removeNetToAiConnection(equivNet);
                }
                
                // destroy net or at least break equivalence
                equivNet->destroy();
                changed = true;
            }
        }
    }   
}


// *****************************************************************************
// printGateUsage()
//
/// \brief Prints the number of times each gate is used as an instance.
///
/// Unimplemented.
//
// *****************************************************************************
void
Utils::printGateUsage(oa::oaModule *target) {  
  assert(target);

  const int NAME_COL_WIDTH = 15;
  const int COUNT_COL_WIDTH = 8;

  long total = 0;
  
  map<oa::oaModule *,int> histogram;

  oa::oaModInst *inst;
  oa::oaIter<oa::oaModInst> instIter(target->getInsts());
  while((inst = instIter.getNext())) {
    oa::oaModule *module = inst->getMasterModule();
    assert(module);

    // increment count
    if (histogram.find(module) == histogram.end()) {
      histogram[module] = 1;
    } else {
      histogram[module] = histogram[module] + 1;
    }
    total++;
  }

  // print statistics
  cout << "\tComponent gate usage" << endl;
  for(map<oa::oaModule *,int>::iterator cellIter = histogram.begin();
      cellIter != histogram.end(); cellIter++) {
    oa::oaString modString;
    cellIter->first->getName(oa::oaVerilogNS(), modString);
    cout << "\t\t";
    cout.width(NAME_COL_WIDTH);
    cout << modString;
    cout.width(COUNT_COL_WIDTH);
    cout << cellIter->second;
    cout << endl;
  }

  cout << "\t\t";
  cout.width(NAME_COL_WIDTH);
  cout << "TOTAL";
  cout.width(COUNT_COL_WIDTH);
  cout << total << endl;
}


// *****************************************************************************
// removeDanglingNets()
//
/// \brief Removes all the nets in a module that are not connected to either an Inst or a port.
///
/// The terminal nodes in the underlying AIG are also removed. 
///
/// \param module
//
// *****************************************************************************
void
Utils::removeDanglingNets(oa::oaModule *module) {
    const oa::oaVerilogNS verilogNS;

    // map all nets
    oa::oaModNet *net;
    oa::oaIter<oa::oaModNet> netIter(module->getNets(oacNetIterSingleBit));
    while((net = netIter.getNext())) {
      oa::oaModBitNet *bitNet = oagFunc::toBitNet(net);
      if (net->getInstTerms(oacInstTermIterAll).isEmpty() && 
          net->getTerms(oacTermIterAll).isEmpty()) {

#if defined(DEBUG)
        oa::oaString netString;
        net->getName(verilogNS, netString);
        DEBUG_PRINTLN("Removing dangling net " << netString);
#endif

        // resubstitute and remove node from graph
        ModRef termRef = ModGraph::getNetToAiConnection(bitNet);
        if(!ModGraph::isNull(termRef)) {
          assert(ModGraph::isTerminal(termRef));
          ModRef driverRef = ModGraph::getTerminalDriver(termRef); 
          ModGraph::resubstitute(termRef, driverRef);
          
          // destroy connection
          ModGraph::removeNetToAiConnection(bitNet);
        }
        
        // destroy net
        net->destroy();
      }
    }
}


}
