/* (c) Copyright 2004-2005, Cadence Design Systems, Inc.  All rights reserved. 

This file is part of the OA Gear distribution.  See the COPYING file in
the top level OA Gear directory for copyright and licensing information. */

/*
Author: Aaron P. Hurst <ahurst@eecs.berkeley.edu>
 
ChangeLog:
2005-08-18: ChangeLog started
*/

#if !defined(oagMapperTable_P)
#define oagMapperTable_P

#include "oaDesignDB.h"
#include "oagFunc.h"
#include "oagFuncModGraph.h"
#include "oagMapperUtils.h"
#include <map>

using namespace std;

namespace oagMapper {

// *****************************************************************************
// Table
//
/// \brief A table-based mapper.
///
/// Implements the abstract functionality inside of a module using a set of
/// concrete library cells.  The matching is Boolean and cut-based and is
/// accomplished by generating and matching exhaustive truth-tables for the
/// library gates and for small slices of the target design.
///
/// First, a set of library cells need to be provided through the 
/// addLibraryGate method.  These cells should be (i) combinational (ii) have
/// a single output (iii) have at most MAX_CUT_SIZE inputs each.  One of the
/// cells must also be an inverter (it will be detected automatically).
///
/// Once the library is initialized, the target design can be mapped using
/// one the available techmap() routines.
///
/// If the design had any pre-existing hierarchy, it will be left untouched.
/// Only abstract functionality implemented within this module itself will be
/// mapped.  If this is not the desired behavior, consider flattening the
/// design first (i.e. oagFunc::collapseAllInstancesAllLevels()).
///
// *****************************************************************************

class Table {

  static const int MAX_CUT_SIZE = 5;

  struct TableEntry {
    static const unsigned int MAX_WORDS = 1;
    static const unsigned int BITS_PER_WORD = sizeof(unsigned int)*8;

    unsigned int         func[MAX_WORDS];

    inline void setBit(unsigned int bit, bool val) {
      assert(bit/32 < MAX_WORDS);
      func[bit/32] &= ~(0x1 << (bit%32)); // clear always
      func[bit/32] |= ((val & 0x1) << (bit%32)); // set appropriately
    }
    inline bool getBit(unsigned int bit) {
      assert(bit/32 < MAX_WORDS);
      return (func[bit/32] >> (bit%32)) & 0x1;
    }
   
    unsigned char   N_input;      // bitfield of input inversions
    unsigned char   directInput;  // input number of direct connection
    unsigned char   directFlag;   // bit flag of direct connection
    unsigned char   constantFlag; // bit flag of constant function
    unsigned char   P;            // permutation ID of inputs

    oa::oaDesign   *cell;           // where the functional description lies
    oa::oaDesign   *implementCell;  // the cell to actually be implemented
    float           cost;

    TableEntry() { cell = NULL; N_input = 0; }
    TableEntry(const TableEntry &copy) { memcpy(this, &copy, sizeof(TableEntry)); }

    inline bool match(const TableEntry & t) {
      for(unsigned int i=0;i<MAX_WORDS;i++) {
        if (func[i] != t.func[i]) {
          return false;
        }
      }
      return true;
    }
  };

  public:
  
    typedef enum{POSEDGE, NEGEDGE} TriggerType;

    /// \name Main mapping routines
    /// @{
  
    Table(int cutsPerNode);

    void                useAlternateView(const oa::oaScalarName &viewName);
    
    void                addLibraryGate(oa::oaDesign* design);

    void                techmapDelay(oa::oaModule *target);
    void                techmapArea(oa::oaModule *target);

    oagMapper::Utils    mapUtils;
    
  protected:

    /// @}
    /// \name Simulation and matching
    /// @{

    void                     addTrivialGates();
      
    int                      cutsPerNode;
    std::vector<TableEntry>  tables[MAX_CUT_SIZE+1];

    void                     initializeSimulation();
    void                     simulate(oa::oaDesign *design,
                                      TableEntry &outResult, oagFunc::ModRef & out, 
                                      vector<oagFunc::ModRef> cut);
    unsigned int             exhaustiveInputVectors[MAX_CUT_SIZE][TableEntry::MAX_WORDS];

      

    /// @}
    /// \name Intermediate Mapping State
    /// @{

    map<oagFunc::ModRef,TableEntry *>           choice_p, choice_n;
    map<oagFunc::ModRef,oagFunc::ModGraph::Cut> cut_p, cut_n;
    map<oagFunc::ModRef,double>                 cost_p, cost_n;
    map<oagFunc::ModRef,oa::oaModBitNet*>       mapped;
    
    double     totalArea, totalDelay;
    int        gateCount;
    int        seqCount;
    
    double getCumulativeAreaCost(oagFunc::ModGraph::Cut *cut, TableEntry *choice);
    double getCumulativeDelayCost(oagFunc::ModGraph::Cut *cut, TableEntry *choice);

    /// @}
    /// \name Implementing concrete structures
    /// @{
  protected:

    void                implementAll(oa::oaModule *target);
    oa::oaModInst *     implementSeqNode(oagFunc::ModRef x);
    oa::oaModBitNet *   implementNode(oagFunc::ModRef x);

    /// @}
    /// \name Library information
    /// @{
    
    std::list<oa::oaDesign*> libraryCells;

    oa::oaScalarName         alternateViewName;
    bool                     useAlternateViews;

    TableEntry              *notEntry;

    void   setAreaCosts();
    void   setDelayCosts();

    /// @}

};

}
#endif
